import {
  pgTable,
  text,
  varchar,
  timestamp,
  jsonb,
  index,
  integer,
  decimal,
  boolean,
  uuid,
  primaryKey,
} from "drizzle-orm/pg-core";
import { relations } from "drizzle-orm";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// Session storage table (required for Replit Auth)
export const sessions = pgTable(
  "sessions",
  {
    sid: varchar("sid").primaryKey(),
    sess: jsonb("sess").notNull(),
    expire: timestamp("expire").notNull(),
  },
  (table) => [index("IDX_session_expire").on(table.expire)],
);

// User storage table (supports both Replit Auth and username/password auth)
export const users = pgTable("users", {
  id: varchar("id").primaryKey().notNull(),
  username: varchar("username").unique(),
  email: varchar("email").unique(),
  passwordHash: varchar("password_hash"),
  firstName: varchar("first_name"),
  lastName: varchar("last_name"),
  profileImageUrl: varchar("profile_image_url"),
  userType: varchar("user_type").notNull().default("learner"), // learner, corporate_admin, partner_admin, system_admin, qa_admin, trainer, testing_center, website_admin
  organizationId: varchar("organization_id"),
  isActive: boolean("is_active").default(true),
  phone: varchar("phone"),
  title: varchar("title"),
  company: varchar("company"),
  accreditationStatus: varchar("accreditation_status"), // pending_approval_1, pending_approval_2, pending_approval_3, approved, rejected
  accreditationLevel: varchar("accreditation_level"), // testing_center, training_center, both (for centers)
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Organizations table for multi-tenant architecture
export const organizations = pgTable("organizations", {
  id: varchar("id").primaryKey().notNull(),
  name: varchar("name").notNull(),
  type: varchar("type").notNull(), // corporate, training_center
  branding: jsonb("branding"), // logos, colors, themes
  settings: jsonb("settings"), // configuration options
  subscriptionPlan: varchar("subscription_plan").default("basic"),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Courses table
export const courses = pgTable("courses", {
  id: varchar("id").primaryKey().notNull(),
  title: varchar("title").notNull(),
  description: text("description"),
  content: jsonb("content"), // SCORM content, modules, chapters
  difficulty: varchar("difficulty").default("beginner"),
  duration: integer("duration"), // in hours
  category: varchar("category"),
  tags: jsonb("tags"),
  isActive: boolean("is_active").default(true),
  organizationId: varchar("organization_id"),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Course enrollments
export const enrollments = pgTable("enrollments", {
  id: varchar("id").primaryKey().notNull(),
  userId: varchar("user_id").notNull(),
  courseId: varchar("course_id").notNull(),
  progress: decimal("progress", { precision: 5, scale: 2 }).default("0"),
  status: varchar("status").default("active"), // active, completed, suspended
  completedAt: timestamp("completed_at"),
  enrolledAt: timestamp("enrolled_at").defaultNow(),
});

// Assessments table
export const assessments = pgTable("assessments", {
  id: varchar("id").primaryKey().notNull(),
  title: varchar("title").notNull(),
  type: varchar("type").notNull(), // exam, quiz, pqp, cpat, competency
  courseId: varchar("course_id"),
  questions: jsonb("questions"),
  settings: jsonb("settings"), // time limits, question randomization, etc.
  isProctored: boolean("is_proctored").default(false),
  passingScore: decimal("passing_score", { precision: 5, scale: 2 }),
  organizationId: varchar("organization_id"),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Assessment attempts
export const assessmentAttempts = pgTable("assessment_attempts", {
  id: varchar("id").primaryKey().notNull(),
  userId: varchar("user_id").notNull(),
  assessmentId: varchar("assessment_id").notNull(),
  answers: jsonb("answers"),
  score: decimal("score", { precision: 5, scale: 2 }),
  status: varchar("status").default("in_progress"), // in_progress, completed, failed, cancelled
  startedAt: timestamp("started_at").defaultNow(),
  completedAt: timestamp("completed_at"),
  proctoringSessions: jsonb("proctoring_sessions"),
});

// Certificates table
export const certificates = pgTable("certificates", {
  id: varchar("id").primaryKey().notNull(),
  userId: varchar("user_id").notNull(),
  courseId: varchar("course_id"),
  assessmentId: varchar("assessment_id"),
  type: varchar("type").notNull(), // course_completion, certification, assessment
  title: varchar("title").notNull(),
  blockchainHash: varchar("blockchain_hash"),
  verificationCode: varchar("verification_code").unique(),
  validUntil: timestamp("valid_until"),
  issuedAt: timestamp("issued_at").defaultNow(),
  metadata: jsonb("metadata"),
});

// Learning analytics
export const learningAnalytics = pgTable("learning_analytics", {
  id: varchar("id").primaryKey().notNull(),
  userId: varchar("user_id").notNull(),
  courseId: varchar("course_id"),
  assessmentId: varchar("assessment_id"),
  eventType: varchar("event_type").notNull(), // login, course_start, lesson_complete, etc.
  eventData: jsonb("event_data"),
  timestamp: timestamp("timestamp").defaultNow(),
  sessionId: varchar("session_id"),
});

// Live examination sessions
export const examinationSessions = pgTable("examination_sessions", {
  id: varchar("id").primaryKey().notNull(),
  userId: varchar("user_id").notNull(),
  assessmentId: varchar("assessment_id").notNull(),
  status: varchar("status").default("scheduled"), // scheduled, active, completed, terminated
  proctorId: varchar("proctor_id"),
  scheduledTime: timestamp("scheduled_time"),
  startedAt: timestamp("started_at"),
  endedAt: timestamp("ended_at"),
  monitoringData: jsonb("monitoring_data"),
  technicalIssues: jsonb("technical_issues"),
});

// Accreditation applications table
export const accreditationApplications = pgTable("accreditation_applications", {
  id: varchar("id").primaryKey().notNull(),
  userId: varchar("user_id"),
  applicationType: varchar("application_type").notNull(), // training_center, testing_center, both
  organizationName: varchar("organization_name").notNull(),
  contactPerson: varchar("contact_person").notNull(),
  contactEmail: varchar("contact_email").notNull(),
  contactPhone: varchar("contact_phone").notNull(),
  address: text("address").notNull(),
  city: varchar("city").notNull(),
  country: varchar("country").notNull(),
  businessLicense: varchar("business_license").notNull(),
  taxRegistration: varchar("tax_registration").notNull(),
  organizationType: varchar("organization_type").notNull(), // private, public, non_profit
  establishedYear: integer("established_year").notNull(),
  numberOfEmployees: integer("number_of_employees").notNull(),
  facilitiesDescription: text("facilities_description").notNull(),
  technicalInfrastructure: jsonb("technical_infrastructure").notNull(),
  qualifiedPersonnel: jsonb("qualified_personnel").notNull(),
  certificationPrograms: jsonb("certification_programs").notNull(),
  operatingHours: jsonb("operating_hours").notNull(),
  capacityInformation: jsonb("capacity_information").notNull(),
  qualityAssurance: jsonb("quality_assurance").notNull(),
  supportingDocuments: jsonb("supporting_documents").notNull(),
  facilityPhotos: jsonb("facility_photos").notNull(),
  agreementToTerms: boolean("agreement_to_terms").notNull(),
  declarationOfAccuracy: boolean("declaration_of_accuracy").notNull(),
  status: varchar("status").default("pending"), // pending, under_review, approved, rejected
  reviewerId: varchar("reviewer_id"),
  reviewComments: text("review_comments"),
  rejectionReason: text("rejection_reason"),
  approvedBy: jsonb("approved_by"), // stores approval history
  // QA Checklist for review process
  qaCheckpoint1: boolean("qa_checkpoint_1").default(false), // Document verification complete
  qaCheckpoint2: boolean("qa_checkpoint_2").default(false), // Facility assessment complete  
  qaCheckpoint3: boolean("qa_checkpoint_3").default(false), // Final approval ready
  submittedAt: timestamp("submitted_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Testing Centers table
export const testingCenters = pgTable("testing_centers", {
  id: varchar("id").primaryKey().notNull(),
  name: varchar("name").notNull(),
  contactPerson: varchar("contact_person"),
  email: varchar("email"),
  phone: varchar("phone"),
  address: text("address"),
  city: varchar("city"),
  country: varchar("country"),
  capacity: integer("capacity").default(10),
  facilities: jsonb("facilities"),
  operatingHours: jsonb("operating_hours"),
  accreditationStatus: varchar("accreditation_status").default("active"), // active, suspended, revoked
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Exam bookings table
export const examBookings = pgTable("exam_bookings", {
  id: varchar("id").primaryKey().notNull(),
  userId: varchar("user_id").notNull(),
  assessmentId: varchar("assessment_id").notNull(),
  testingCenterId: varchar("testing_center_id").notNull(),
  scheduledDate: timestamp("scheduled_date"),
  scheduledTime: varchar("scheduled_time"),
  status: varchar("status").default("pending_approval"), // pending_approval, approved, rejected, rescheduled, completed, cancelled
  bookedBy: varchar("booked_by"), // user_id of the person who booked (for corporate accounts)
  approvalHistory: jsonb("approval_history"),
  temporaryLogin: varchar("temporary_login"),
  temporaryPassword: varchar("temporary_password"),
  loginExpiresAt: timestamp("login_expires_at"),
  notes: text("notes"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Question bank table
export const questionBank = pgTable("question_bank", {
  id: varchar("id").primaryKey().notNull(),
  questionText: text("question_text").notNull(),
  questionType: varchar("question_type").notNull(), // multiple_choice, true_false, short_answer
  answerOptions: jsonb("answer_options"),
  correctAnswer: varchar("correct_answer"),
  difficultyLevel: varchar("difficulty_level").notNull(), // easy, intermediate, hard
  topic: varchar("topic"),
  associatedCertificate: varchar("associated_certificate"),
  version: integer("version").default(1),
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Exam rules configuration table
export const examRules = pgTable("exam_rules", {
  id: varchar("id").primaryKey().notNull(),
  certificateId: varchar("certificate_id").notNull(),
  totalQuestions: integer("total_questions").notNull(),
  easyQuestions: integer("easy_questions").default(0),
  intermediateQuestions: integer("intermediate_questions").default(0),
  hardQuestions: integer("hard_questions").default(0),
  timeLimit: integer("time_limit"), // in minutes
  passingScore: decimal("passing_score", { precision: 5, scale: 2 }),
  topics: jsonb("topics"),
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// System Setup - Certificates
export const systemCertificates = pgTable("system_certificates", {
  id: varchar("id").primaryKey().notNull(),
  name: varchar("name").notNull(),
  nameEn: varchar("name_en").notNull(),
  nameAr: varchar("name_ar"),
  nameFr: varchar("name_fr"),
  description: text("description"),
  descriptionEn: text("description_en"),
  descriptionAr: text("description_ar"),
  descriptionFr: text("description_fr"),
  certificateType: varchar("certificate_type").notNull(), // professional, executive, academic
  duration: integer("duration"), // in hours
  validityPeriod: integer("validity_period"), // in months
  prerequisites: jsonb("prerequisites"),
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// System Setup - Levels
export const systemLevels = pgTable("system_levels", {
  id: varchar("id").primaryKey().notNull(),
  certificateId: varchar("certificate_id").notNull(),
  levelNumber: integer("level_number").notNull(),
  name: varchar("name").notNull(),
  nameEn: varchar("name_en").notNull(),
  nameAr: varchar("name_ar"),
  nameFr: varchar("name_fr"),
  description: text("description"),
  requirements: jsonb("requirements"),
  passingScore: decimal("passing_score", { precision: 5, scale: 2 }),
  timeLimit: integer("time_limit"), // in minutes
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// System Setup - Chapters
export const systemChapters = pgTable("system_chapters", {
  id: varchar("id").primaryKey().notNull(),
  levelId: varchar("level_id").notNull(),
  chapterNumber: integer("chapter_number").notNull(),
  title: varchar("title").notNull(),
  titleEn: varchar("title_en").notNull(),
  titleAr: varchar("title_ar"),
  titleFr: varchar("title_fr"),
  content: jsonb("content"), // learning materials, videos, documents
  objectives: jsonb("objectives"),
  duration: integer("duration"), // in minutes
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Exam Configurations
export const examConfigurations = pgTable("exam_configurations", {
  id: varchar("id").primaryKey().notNull(),
  certificateId: varchar("certificate_id").notNull(),
  levelId: varchar("level_id"),
  examName: varchar("exam_name").notNull(),
  examType: varchar("exam_type").notNull(), // certification, placement, practice
  questionCount: integer("question_count").notNull(),
  timeLimit: integer("time_limit").notNull(), // in minutes
  passingScore: decimal("passing_score", { precision: 5, scale: 2 }).notNull(),
  randomizeQuestions: boolean("randomize_questions").default(true),
  allowReview: boolean("allow_review").default(false),
  showResults: boolean("show_results").default(true),
  isProctored: boolean("is_proctored").default(false),
  maxAttempts: integer("max_attempts").default(3),
  retakeDelay: integer("retake_delay").default(24), // hours
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// eBooks
export const systemEBooks = pgTable("system_ebooks", {
  id: varchar("id").primaryKey().notNull(),
  certificateId: varchar("certificate_id").notNull(),
  levelId: varchar("level_id"),
  chapterId: varchar("chapter_id"),
  title: varchar("title").notNull(),
  titleEn: varchar("title_en").notNull(),
  titleAr: varchar("title_ar"),
  titleFr: varchar("title_fr"),
  author: varchar("author"),
  description: text("description"),
  fileUrl: varchar("file_url"),
  fileSize: integer("file_size"), // in bytes
  downloadCount: integer("download_count").default(0),
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// eLearning Content
export const systemELearningContent = pgTable("system_elearning_content", {
  id: varchar("id").primaryKey().notNull(),
  certificateId: varchar("certificate_id").notNull(),
  levelId: varchar("level_id"),
  chapterId: varchar("chapter_id"),
  title: varchar("title").notNull(),
  titleEn: varchar("title_en").notNull(),
  titleAr: varchar("title_ar"),
  titleFr: varchar("title_fr"),
  contentType: varchar("content_type").notNull(), // video, interactive, simulation, quiz
  content: jsonb("content"), // video URLs, interactive elements, etc.
  duration: integer("duration"), // in minutes
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Notification templates table
export const notificationTemplates = pgTable("notification_templates", {
  id: varchar("id").primaryKey().notNull(),
  templateName: varchar("template_name").notNull(),
  templateType: varchar("template_type").notNull(), // email, sms
  eventType: varchar("event_type").notNull(), // registration, accreditation_status, exam_booking, etc.
  subject: varchar("subject"),
  body: text("body"),
  variables: jsonb("variables"),
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Corporate account credits table
export const corporateCredits = pgTable("corporate_credits", {
  id: varchar("id").primaryKey().notNull(),
  organizationId: varchar("organization_id").notNull(),
  totalCredits: integer("total_credits").default(0),
  usedCredits: integer("used_credits").default(0),
  availableCredits: integer("available_credits").default(0),
  packageType: varchar("package_type"),
  purchaseDate: timestamp("purchase_date"),
  expiryDate: timestamp("expiry_date"),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Page sections table for CMS functionality
export const pageSections = pgTable("page_sections", {
  id: varchar("id").primaryKey().notNull(),
  pageId: varchar("page_id").notNull(), // "our-story", "about-us", etc.
  sectionTemplateId: varchar("section_template_id"),
  templateType: varchar("template_type"),
  title: varchar("title").notNull(),
  content: text("content"),
  image: varchar("image"),
  data: jsonb("data"),
  sortOrder: integer("sort_order").default(0),
  isActive: boolean("is_active").default(true),
  createdBy: varchar("created_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Define relations
export const userRelations = relations(users, ({ one, many }) => ({
  organization: one(organizations, {
    fields: [users.organizationId],
    references: [organizations.id],
  }),
  enrollments: many(enrollments),
  assessmentAttempts: many(assessmentAttempts),
  certificates: many(certificates),
  analytics: many(learningAnalytics),
  examinationSessions: many(examinationSessions),
  accreditationApplications: many(accreditationApplications),
  examBookings: many(examBookings),
}));

export const organizationRelations = relations(organizations, ({ many }) => ({
  users: many(users),
  courses: many(courses),
  assessments: many(assessments),
}));

export const courseRelations = relations(courses, ({ one, many }) => ({
  organization: one(organizations, {
    fields: [courses.organizationId],
    references: [organizations.id],
  }),
  creator: one(users, {
    fields: [courses.createdBy],
    references: [users.id],
  }),
  enrollments: many(enrollments),
  assessments: many(assessments),
  certificates: many(certificates),
}));

export const enrollmentRelations = relations(enrollments, ({ one }) => ({
  user: one(users, {
    fields: [enrollments.userId],
    references: [users.id],
  }),
  course: one(courses, {
    fields: [enrollments.courseId],
    references: [courses.id],
  }),
}));

export const assessmentRelations = relations(assessments, ({ one, many }) => ({
  course: one(courses, {
    fields: [assessments.courseId],
    references: [courses.id],
  }),
  organization: one(organizations, {
    fields: [assessments.organizationId],
    references: [organizations.id],
  }),
  creator: one(users, {
    fields: [assessments.createdBy],
    references: [users.id],
  }),
  attempts: many(assessmentAttempts),
  examinationSessions: many(examinationSessions),
}));

export const assessmentAttemptRelations = relations(assessmentAttempts, ({ one }) => ({
  user: one(users, {
    fields: [assessmentAttempts.userId],
    references: [users.id],
  }),
  assessment: one(assessments, {
    fields: [assessmentAttempts.assessmentId],
    references: [assessments.id],
  }),
}));

export const certificateRelations = relations(certificates, ({ one }) => ({
  user: one(users, {
    fields: [certificates.userId],
    references: [users.id],
  }),
  course: one(courses, {
    fields: [certificates.courseId],
    references: [courses.id],
  }),
  assessment: one(assessments, {
    fields: [certificates.assessmentId],
    references: [assessments.id],
  }),
}));

export const learningAnalyticsRelations = relations(learningAnalytics, ({ one }) => ({
  user: one(users, {
    fields: [learningAnalytics.userId],
    references: [users.id],
  }),
  course: one(courses, {
    fields: [learningAnalytics.courseId],
    references: [courses.id],
  }),
  assessment: one(assessments, {
    fields: [learningAnalytics.assessmentId],
    references: [assessments.id],
  }),
}));

export const examinationSessionRelations = relations(examinationSessions, ({ one }) => ({
  user: one(users, {
    fields: [examinationSessions.userId],
    references: [users.id],
  }),
  assessment: one(assessments, {
    fields: [examinationSessions.assessmentId],
    references: [assessments.id],
  }),
  proctor: one(users, {
    fields: [examinationSessions.proctorId],
    references: [users.id],
  }),
}));

// Relations for new tables
export const accreditationApplicationRelations = relations(accreditationApplications, ({ one }) => ({
  user: one(users, {
    fields: [accreditationApplications.userId],
    references: [users.id],
  }),
}));

export const examBookingRelations = relations(examBookings, ({ one }) => ({
  user: one(users, {
    fields: [examBookings.userId],
    references: [users.id],
  }),
  assessment: one(assessments, {
    fields: [examBookings.assessmentId],
    references: [assessments.id],
  }),
  testingCenter: one(testingCenters, {
    fields: [examBookings.testingCenterId],
    references: [testingCenters.id],
  }),
  bookedByUser: one(users, {
    fields: [examBookings.bookedBy],
    references: [users.id],
  }),
}));

export const testingCenterRelations = relations(testingCenters, ({ many }) => ({
  examBookings: many(examBookings),
}));

export const questionBankRelations = relations(questionBank, ({ one }) => ({
  creator: one(users, {
    fields: [questionBank.createdBy],
    references: [users.id],
  }),
}));

export const examRulesRelations = relations(examRules, ({ one }) => ({
  creator: one(users, {
    fields: [examRules.createdBy],
    references: [users.id],
  }),
}));

export const notificationTemplateRelations = relations(notificationTemplates, ({ one }) => ({
  creator: one(users, {
    fields: [notificationTemplates.createdBy],
    references: [users.id],
  }),
}));

export const corporateCreditsRelations = relations(corporateCredits, ({ one }) => ({
  organization: one(organizations, {
    fields: [corporateCredits.organizationId],
    references: [organizations.id],
  }),
}));

export const pageSectionRelations = relations(pageSections, ({ one }) => ({
  creator: one(users, {
    fields: [pageSections.createdBy],
    references: [users.id],
  }),
}));

// Zod schemas for validation
export const upsertUserSchema = createInsertSchema(users).omit({
  createdAt: true,
  updatedAt: true,
});
export const insertOrganizationSchema = createInsertSchema(organizations).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertCourseSchema = createInsertSchema(courses).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertEnrollmentSchema = createInsertSchema(enrollments).omit({
  id: true,
  enrolledAt: true,
});
export const insertAssessmentSchema = createInsertSchema(assessments).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertAssessmentAttemptSchema = createInsertSchema(assessmentAttempts).omit({
  id: true,
  startedAt: true,
});
export const insertCertificateSchema = createInsertSchema(certificates).omit({
  id: true,
  issuedAt: true,
});
export const insertLearningAnalyticsSchema = createInsertSchema(learningAnalytics).omit({
  id: true,
  timestamp: true,
});
export const insertExaminationSessionSchema = createInsertSchema(examinationSessions).omit({
  id: true,
});

// Zod schemas for new tables
export const insertAccreditationApplicationSchema = createInsertSchema(accreditationApplications).omit({
  id: true,
  submittedAt: true,
  updatedAt: true,
});
export const insertTestingCenterSchema = createInsertSchema(testingCenters).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertExamBookingSchema = createInsertSchema(examBookings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertQuestionBankSchema = createInsertSchema(questionBank).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertExamRulesSchema = createInsertSchema(examRules).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertNotificationTemplateSchema = createInsertSchema(notificationTemplates).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertCorporateCreditsSchema = createInsertSchema(corporateCredits).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// Zod schemas for system setup tables
export const insertSystemCertificateSchema = createInsertSchema(systemCertificates).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertSystemLevelSchema = createInsertSchema(systemLevels).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertSystemChapterSchema = createInsertSchema(systemChapters).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertExamConfigurationSchema = createInsertSchema(examConfigurations).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertSystemEBookSchema = createInsertSchema(systemEBooks).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});
export const insertSystemELearningContentSchema = createInsertSchema(systemELearningContent).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// CMS Tables
export const cmsPages = pgTable("cms_pages", {
  id: varchar("id").primaryKey().notNull(),
  slug: varchar("slug").unique().notNull(),
  title: varchar("title").notNull(),
  content: text("content"),
  metaTitle: varchar("meta_title"),
  metaDescription: text("meta_description"),
  status: varchar("status").default("draft"), // draft, published, archived
  pageType: varchar("page_type").notNull(), // about, news, resource, product, service
  featuredImage: varchar("featured_image"),
  publishedAt: timestamp("published_at"),
  createdBy: varchar("created_by"),
  updatedBy: varchar("updated_by"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const cmsMedia = pgTable("cms_media", {
  id: varchar("id").primaryKey().notNull(),
  filename: varchar("filename").notNull(),
  originalName: varchar("original_name").notNull(),
  mimeType: varchar("mime_type").notNull(),
  size: integer("size").notNull(),
  url: varchar("url").notNull(),
  alt: varchar("alt"),
  caption: text("caption"),
  uploadedBy: varchar("uploaded_by"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const cmsMenus = pgTable("cms_menus", {
  id: varchar("id").primaryKey().notNull(),
  name: varchar("name").notNull(),
  location: varchar("location").notNull(), // header, footer, sidebar
  items: jsonb("items").notNull(), // menu structure
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const cmsSettings = pgTable("cms_settings", {
  id: varchar("id").primaryKey().notNull(),
  siteName: varchar("site_name").notNull(),
  siteDescription: text("site_description"),
  logo: varchar("logo"),
  favicon: varchar("favicon"),
  contactEmail: varchar("contact_email"),
  contactPhone: varchar("contact_phone"),
  address: text("address"),
  socialLinks: jsonb("social_links"),
  seoSettings: jsonb("seo_settings"),
  maintenanceMode: boolean("maintenance_mode").default(false),
  customCss: text("custom_css"),
  customJs: text("custom_js"),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const newsArticles = pgTable("news_articles", {
  id: varchar("id").primaryKey().notNull(),
  title: varchar("title").notNull(),
  slug: varchar("slug").unique().notNull(),
  excerpt: text("excerpt"),
  content: text("content"),
  featuredImage: varchar("featured_image"),
  category: varchar("category"),
  tags: jsonb("tags"),
  author: varchar("author"),
  status: varchar("status").default("draft"),
  publishedAt: timestamp("published_at"),
  views: integer("views").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const resources = pgTable("resources", {
  id: varchar("id").primaryKey().notNull(),
  title: varchar("title").notNull(),
  slug: varchar("slug").unique().notNull(),
  description: text("description"),
  content: text("content"),
  resourceType: varchar("resource_type").notNull(), // whitepaper, case_study, guide, template
  fileUrl: varchar("file_url"),
  thumbnailUrl: varchar("thumbnail_url"),
  downloadCount: integer("download_count").default(0),
  accessLevel: varchar("access_level").default("public"), // public, members_only, premium
  tags: jsonb("tags"),
  createdBy: varchar("created_by"),
  status: varchar("status").default("draft"),
  publishedAt: timestamp("published_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const storeProducts = pgTable("store_products", {
  id: varchar("id").primaryKey().notNull(),
  name: varchar("name").notNull(),
  slug: varchar("slug").unique().notNull(),
  description: text("description"),
  shortDescription: text("short_description"),
  price: decimal("price", { precision: 10, scale: 2 }),
  salePrice: decimal("sale_price", { precision: 10, scale: 2 }),
  sku: varchar("sku").unique(),
  category: varchar("category"),
  tags: jsonb("tags"),
  images: jsonb("images"),
  downloadable: boolean("downloadable").default(false),
  downloadFiles: jsonb("download_files"),
  stock: integer("stock"),
  status: varchar("status").default("draft"),
  featured: boolean("featured").default(false),
  createdBy: varchar("created_by"),
  publishedAt: timestamp("published_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertCmsPageSchema = createInsertSchema(cmsPages).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertCmsMediaSchema = createInsertSchema(cmsMedia).omit({
  id: true,
  createdAt: true,
});

export const insertCmsMenuSchema = createInsertSchema(cmsMenus).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertCmsSettingsSchema = createInsertSchema(cmsSettings).omit({
  id: true,
  updatedAt: true,
});

export const insertNewsArticleSchema = createInsertSchema(newsArticles).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertResourceSchema = createInsertSchema(resources).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertStoreProductSchema = createInsertSchema(storeProducts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPageSectionSchema = createInsertSchema(pageSections).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// Type exports
export type UpsertUser = z.infer<typeof upsertUserSchema>;
export type User = typeof users.$inferSelect;
export type Organization = typeof organizations.$inferSelect;
export type InsertOrganization = z.infer<typeof insertOrganizationSchema>;
export type Course = typeof courses.$inferSelect;
export type InsertCourse = z.infer<typeof insertCourseSchema>;
export type Enrollment = typeof enrollments.$inferSelect;
export type InsertEnrollment = z.infer<typeof insertEnrollmentSchema>;
export type Assessment = typeof assessments.$inferSelect;
export type InsertAssessment = z.infer<typeof insertAssessmentSchema>;
export type AssessmentAttempt = typeof assessmentAttempts.$inferSelect;
export type InsertAssessmentAttempt = z.infer<typeof insertAssessmentAttemptSchema>;
export type Certificate = typeof certificates.$inferSelect;
export type InsertCertificate = z.infer<typeof insertCertificateSchema>;
export type LearningAnalytics = typeof learningAnalytics.$inferSelect;
export type InsertLearningAnalytics = z.infer<typeof insertLearningAnalyticsSchema>;
export type ExaminationSession = typeof examinationSessions.$inferSelect;
export type InsertExaminationSession = z.infer<typeof insertExaminationSessionSchema>;

// Type exports for new tables
export type AccreditationApplication = typeof accreditationApplications.$inferSelect;
export type InsertAccreditationApplication = z.infer<typeof insertAccreditationApplicationSchema>;
export type TestingCenter = typeof testingCenters.$inferSelect;
export type InsertTestingCenter = z.infer<typeof insertTestingCenterSchema>;
export type ExamBooking = typeof examBookings.$inferSelect;
export type InsertExamBooking = z.infer<typeof insertExamBookingSchema>;
export type QuestionBank = typeof questionBank.$inferSelect;
export type InsertQuestionBank = z.infer<typeof insertQuestionBankSchema>;
export type ExamRules = typeof examRules.$inferSelect;
export type InsertExamRules = z.infer<typeof insertExamRulesSchema>;
export type NotificationTemplate = typeof notificationTemplates.$inferSelect;
export type InsertNotificationTemplate = z.infer<typeof insertNotificationTemplateSchema>;
export type CorporateCredits = typeof corporateCredits.$inferSelect;
export type InsertCorporateCredits = z.infer<typeof insertCorporateCreditsSchema>;

// Type exports for system setup tables
export type SystemCertificate = typeof systemCertificates.$inferSelect;
export type InsertSystemCertificate = z.infer<typeof insertSystemCertificateSchema>;
export type SystemLevel = typeof systemLevels.$inferSelect;
export type InsertSystemLevel = z.infer<typeof insertSystemLevelSchema>;
export type SystemChapter = typeof systemChapters.$inferSelect;
export type InsertSystemChapter = z.infer<typeof insertSystemChapterSchema>;
export type ExamConfiguration = typeof examConfigurations.$inferSelect;
export type InsertExamConfiguration = z.infer<typeof insertExamConfigurationSchema>;
export type SystemEBook = typeof systemEBooks.$inferSelect;
export type InsertSystemEBook = z.infer<typeof insertSystemEBookSchema>;
export type SystemELearningContent = typeof systemELearningContent.$inferSelect;
export type InsertSystemELearningContent = z.infer<typeof insertSystemELearningContentSchema>;

// CMS Type exports
export type CmsPage = typeof cmsPages.$inferSelect;
export type InsertCmsPage = z.infer<typeof insertCmsPageSchema>;
export type CmsMedia = typeof cmsMedia.$inferSelect;
export type InsertCmsMedia = z.infer<typeof insertCmsMediaSchema>;
export type CmsMenu = typeof cmsMenus.$inferSelect;
export type InsertCmsMenu = z.infer<typeof insertCmsMenuSchema>;
export type CmsSettings = typeof cmsSettings.$inferSelect;
export type InsertCmsSettings = z.infer<typeof insertCmsSettingsSchema>;
export type NewsArticle = typeof newsArticles.$inferSelect;
export type InsertNewsArticle = z.infer<typeof insertNewsArticleSchema>;
export type Resource = typeof resources.$inferSelect;
export type InsertResource = z.infer<typeof insertResourceSchema>;
export type StoreProduct = typeof storeProducts.$inferSelect;
export type InsertStoreProduct = z.infer<typeof insertStoreProductSchema>;
export type PageSection = typeof pageSections.$inferSelect;
export type InsertPageSection = z.infer<typeof insertPageSectionSchema>;
