import {
  users,
  organizations,
  courses,
  enrollments,
  assessments,
  assessmentAttempts,
  certificates,
  learningAnalytics,
  examinationSessions,
  accreditationApplications,
  testingCenters,
  examBookings,
  questionBank,
  examRules,
  notificationTemplates,
  corporateCredits,
  cmsPages,
  cmsMenus,
  newsArticles,
  resources,
  storeProducts,
  cmsMedia,
  cmsSettings,
  systemCertificates,
  systemLevels,
  systemChapters,
  pageSections,
  type User,
  type UpsertUser,
  type Organization,
  type InsertOrganization,
  type Course,
  type InsertCourse,
  type Enrollment,
  type InsertEnrollment,
  type Assessment,
  type InsertAssessment,
  type AssessmentAttempt,
  type InsertAssessmentAttempt,
  type Certificate,
  type InsertCertificate,
  type LearningAnalytics,
  type InsertLearningAnalytics,
  type ExaminationSession,
  type InsertExaminationSession,
  type AccreditationApplication,
  type InsertAccreditationApplication,
  type TestingCenter,
  type InsertTestingCenter,
  type ExamBooking,
  type InsertExamBooking,
  type QuestionBank,
  type InsertQuestionBank,
  type ExamRules,
  type InsertExamRules,
  type NotificationTemplate,
  type InsertNotificationTemplate,
  type CorporateCredits,
  type InsertCorporateCredits,
  type CmsPage,
  type InsertCmsPage,
  type NewsArticle,
  type InsertNewsArticle,
  type Resource,
  type InsertResource,
  type StoreProduct,
  type InsertStoreProduct,
  type CmsMedia,
  type InsertCmsMedia,
  type CmsSettings,
  type InsertCmsSettings,
  type SystemCertificate,
  type InsertSystemCertificate,
  type SystemLevel,
  type InsertSystemLevel,
  type SystemChapter,
  type InsertSystemChapter,
  type PageSection,
  type InsertPageSection,
} from "@shared/schema";
import { db } from "./db";
import { eq, and, desc, asc, sql, ilike, count } from "drizzle-orm";
import { nanoid } from "nanoid";
import bcrypt from "bcryptjs";

export interface IStorage {
  // User operations (required for Replit Auth)
  getUser(id: string): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  getUserByEmail(email: string): Promise<User | undefined>;
  createUser(user: UpsertUser): Promise<User>;
  upsertUser(user: UpsertUser): Promise<User>;
  updateUser(id: string, updates: Partial<UpsertUser>): Promise<User>;
  getUsersByOrganization(organizationId: string): Promise<User[]>;

  // Organization operations
  createOrganization(org: InsertOrganization): Promise<Organization>;
  getOrganization(id: string): Promise<Organization | undefined>;
  updateOrganization(id: string, updates: Partial<InsertOrganization>): Promise<Organization>;

  // Course operations
  createCourse(course: InsertCourse): Promise<Course>;
  getCourse(id: string): Promise<Course | undefined>;
  getCoursesByOrganization(organizationId: string): Promise<Course[]>;
  updateCourse(id: string, updates: Partial<InsertCourse>): Promise<Course>;
  searchCourses(query: string, organizationId?: string): Promise<Course[]>;

  // Enrollment operations
  enrollUser(enrollment: InsertEnrollment): Promise<Enrollment>;
  getUserEnrollments(userId: string): Promise<Enrollment[]>;
  updateEnrollmentProgress(id: string, progress: number): Promise<Enrollment>;
  completeEnrollment(id: string): Promise<Enrollment>;

  // Assessment operations
  createAssessment(assessment: InsertAssessment): Promise<Assessment>;
  getAssessment(id: string): Promise<Assessment | undefined>;
  getAssessmentsByCourse(courseId: string): Promise<Assessment[]>;
  getAssessmentsByOrganization(organizationId: string): Promise<Assessment[]>;

  // Assessment attempt operations
  createAssessmentAttempt(attempt: InsertAssessmentAttempt): Promise<AssessmentAttempt>;
  getAssessmentAttempt(id: string): Promise<AssessmentAttempt | undefined>;
  getUserAssessmentAttempts(userId: string, assessmentId?: string): Promise<AssessmentAttempt[]>;
  updateAssessmentAttempt(id: string, updates: Partial<InsertAssessmentAttempt>): Promise<AssessmentAttempt>;

  // Certificate operations
  createCertificate(certificate: InsertCertificate): Promise<Certificate>;
  getUserCertificates(userId: string): Promise<Certificate[]>;
  verifyCertificate(verificationCode: string): Promise<Certificate | undefined>;

  // Analytics operations
  logLearningEvent(analytics: InsertLearningAnalytics): Promise<LearningAnalytics>;
  getUserAnalytics(userId: string, fromDate?: Date, toDate?: Date): Promise<LearningAnalytics[]>;
  getOrganizationAnalytics(organizationId: string, fromDate?: Date, toDate?: Date): Promise<LearningAnalytics[]>;

  // Examination session operations
  createExaminationSession(session: InsertExaminationSession): Promise<ExaminationSession>;
  getExaminationSession(id: string): Promise<ExaminationSession | undefined>;
  updateExaminationSession(id: string, updates: Partial<InsertExaminationSession>): Promise<ExaminationSession>;
  getActiveExaminationSessions(): Promise<ExaminationSession[]>;
  getUserExaminationSessions(userId: string): Promise<ExaminationSession[]>;

  // Dashboard analytics
  getLearnerDashboardStats(userId: string): Promise<any>;
  getCorporateDashboardStats(organizationId: string): Promise<any>;
  getPartnerDashboardStats(organizationId: string): Promise<any>;
  getSystemDashboardStats(): Promise<any>;

  // Accreditation operations - Enhanced for FRD compliance
  getAccreditationApplications(): Promise<AccreditationApplication[]>;
  getAccreditationApplication(id: string): Promise<AccreditationApplication | undefined>;
  createAccreditationApplication(application: InsertAccreditationApplication): Promise<AccreditationApplication>;
  updateAccreditationStatus(id: string, status: string, comments: string, reviewerId: string): Promise<AccreditationApplication>;
  getAccreditationApplicationsByStatus(status: string): Promise<AccreditationApplication[]>;
  getAccreditationApplicationsByType(applicationType: string): Promise<AccreditationApplication[]>;
  getUserAccreditationApplications(userId: string): Promise<AccreditationApplication[]>;

  // Question Bank operations
  getQuestionBank(): Promise<QuestionBank[]>;
  getQuestion(id: string): Promise<QuestionBank | undefined>;
  createQuestion(question: InsertQuestionBank): Promise<QuestionBank>;
  updateQuestion(id: string, updates: Partial<InsertQuestionBank>): Promise<QuestionBank>;
  deleteQuestion(id: string): Promise<void>;

  // Testing Center operations
  getTestingCenters(): Promise<TestingCenter[]>;
  getTestingCenter(id: string): Promise<TestingCenter | undefined>;
  createTestingCenter(center: InsertTestingCenter): Promise<TestingCenter>;
  updateTestingCenter(id: string, updates: Partial<InsertTestingCenter>): Promise<TestingCenter>;

  // Exam Booking operations
  getAllExamBookings(): Promise<ExamBooking[]>;
  getUserExamBookings(userId: string): Promise<ExamBooking[]>;
  getExamBooking(id: string): Promise<ExamBooking | undefined>;
  createExamBooking(booking: InsertExamBooking): Promise<ExamBooking>;
  updateExamBooking(id: string, updates: Partial<InsertExamBooking>): Promise<ExamBooking>;

  // Corporate Credits operations
  getCorporateCredits(organizationId: string): Promise<CorporateCredits | undefined>;
  createCorporateCredits(credits: InsertCorporateCredits): Promise<CorporateCredits>;
  updateCorporateCredits(id: string, updates: Partial<InsertCorporateCredits>): Promise<CorporateCredits>;

  // Notification Template operations
  getNotificationTemplates(): Promise<NotificationTemplate[]>;
  getNotificationTemplate(id: string): Promise<NotificationTemplate | undefined>;
  createNotificationTemplate(template: InsertNotificationTemplate): Promise<NotificationTemplate>;
  updateNotificationTemplate(id: string, updates: Partial<InsertNotificationTemplate>): Promise<NotificationTemplate>;

  // Exam Rules operations
  getExamRules(): Promise<ExamRules[]>;
  getExamRule(id: string): Promise<ExamRules | undefined>;
  createExamRule(rule: InsertExamRules): Promise<ExamRules>;
  updateExamRule(id: string, updates: Partial<InsertExamRules>): Promise<ExamRules>;

  // CMS operations
  getCmsPages(): Promise<CmsPage[]>;
  getCmsPage(id: string): Promise<CmsPage | undefined>;
  getCmsPageBySlug(slug: string): Promise<CmsPage | undefined>;
  createCmsPage(page: InsertCmsPage): Promise<CmsPage>;
  updateCmsPage(id: string, updates: Partial<InsertCmsPage>): Promise<CmsPage>;
  deleteCmsPage(id: string): Promise<void>;

  // Page Sections operations
  getPageSections(pageId: string): Promise<PageSection[]>;
  getPageSection(id: string): Promise<PageSection | undefined>;
  createPageSection(section: InsertPageSection): Promise<PageSection>;
  updatePageSection(id: string, updates: Partial<InsertPageSection>): Promise<PageSection>;
  deletePageSection(id: string): Promise<void>;
  reorderPageSections(pageId: string, sections: { id: string; sortOrder: number }[]): Promise<void>;

  // CMS Menu Items operations
  getCmsMenuItems(): Promise<any[]>;
  getCmsMenuItem(id: string): Promise<any | undefined>;
  createCmsMenuItem(item: any): Promise<any>;
  updateCmsMenuItem(id: string, updates: any): Promise<any>;
  deleteCmsMenuItem(id: string): Promise<void>;

  // News articles operations
  getNewsArticles(): Promise<NewsArticle[]>;
  getNewsArticle(id: string): Promise<NewsArticle | undefined>;
  createNewsArticle(article: InsertNewsArticle): Promise<NewsArticle>;
  updateNewsArticle(id: string, updates: Partial<InsertNewsArticle>): Promise<NewsArticle>;
  deleteNewsArticle(id: string): Promise<void>;

  // Resources operations
  getResources(): Promise<Resource[]>;
  getResource(id: string): Promise<Resource | undefined>;
  createResource(resource: InsertResource): Promise<Resource>;
  updateResource(id: string, updates: Partial<InsertResource>): Promise<Resource>;
  deleteResource(id: string): Promise<void>;

  // Store products operations
  getStoreProducts(): Promise<StoreProduct[]>;
  getStoreProduct(id: string): Promise<StoreProduct | undefined>;
  createStoreProduct(product: InsertStoreProduct): Promise<StoreProduct>;
  updateStoreProduct(id: string, updates: Partial<InsertStoreProduct>): Promise<StoreProduct>;
  deleteStoreProduct(id: string): Promise<void>;

  // Media operations
  getMediaFiles(): Promise<CmsMedia[]>;
  getMediaFile(id: string): Promise<CmsMedia | undefined>;
  createMediaFile(media: InsertCmsMedia): Promise<CmsMedia>;
  deleteMediaFile(id: string): Promise<void>;

  // Settings operations
  getCmsSettings(): Promise<CmsSettings | undefined>;
  updateCmsSettings(settings: InsertCmsSettings): Promise<CmsSettings>;

  // System Certificate operations
  getSystemCertificates(): Promise<SystemCertificate[]>;
  getSystemCertificate(id: string): Promise<SystemCertificate | undefined>;
  createSystemCertificate(certificate: InsertSystemCertificate): Promise<SystemCertificate>;
  updateSystemCertificate(id: string, updates: Partial<InsertSystemCertificate>): Promise<SystemCertificate>;
  deleteSystemCertificate(id: string): Promise<void>;

  // System Level operations
  getSystemLevels(certificateId: string): Promise<SystemLevel[]>;
  getSystemLevel(id: string): Promise<SystemLevel | undefined>;
  createSystemLevel(level: InsertSystemLevel): Promise<SystemLevel>;
  updateSystemLevel(id: string, updates: Partial<InsertSystemLevel>): Promise<SystemLevel>;
  deleteSystemLevel(id: string): Promise<void>;

  // System Chapter operations
  getSystemChapters(levelId: string): Promise<SystemChapter[]>;
  getSystemChapter(id: string): Promise<SystemChapter | undefined>;
  createSystemChapter(chapter: InsertSystemChapter): Promise<SystemChapter>;
  updateSystemChapter(id: string, updates: Partial<InsertSystemChapter>): Promise<SystemChapter>;
  deleteSystemChapter(id: string): Promise<void>;
}

export class DatabaseStorage implements IStorage {
  // User operations (required for Replit Auth)
  async getUser(id: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user;
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user;
  }

  async getUserByEmail(email: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.email, email));
    return user;
  }

  async createUser(userData: UpsertUser): Promise<User> {
    const userId = nanoid();
    
    // Hash password if provided in passwordHash field
    let hashedPassword = userData.passwordHash;
    if (hashedPassword && !hashedPassword.startsWith('$2')) {
      hashedPassword = await bcrypt.hash(hashedPassword, 10);
    }
    
    const [user] = await db
      .insert(users)
      .values({ 
        ...userData, 
        id: userId,
        passwordHash: hashedPassword
      })
      .returning();
    return user;
  }

  async upsertUser(userData: UpsertUser): Promise<User> {
    const [user] = await db
      .insert(users)
      .values(userData)
      .onConflictDoUpdate({
        target: users.id,
        set: {
          ...userData,
          updatedAt: new Date(),
        },
      })
      .returning();
    return user;
  }

  async updateUser(id: string, updates: Partial<UpsertUser>): Promise<User> {
    const [user] = await db
      .update(users)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(users.id, id))
      .returning();
    return user;
  }

  // Organization operations
  async createOrganization(org: InsertOrganization): Promise<Organization> {
    const [organization] = await db
      .insert(organizations)
      .values({ ...org, id: nanoid() })
      .returning();
    return organization;
  }

  async getOrganization(id: string): Promise<Organization | undefined> {
    const [organization] = await db
      .select()
      .from(organizations)
      .where(eq(organizations.id, id));
    return organization;
  }

  async updateOrganization(id: string, updates: Partial<InsertOrganization>): Promise<Organization> {
    const [organization] = await db
      .update(organizations)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(organizations.id, id))
      .returning();
    return organization;
  }

  async getUsersByOrganization(organizationId: string): Promise<User[]> {
    return await db
      .select()
      .from(users)
      .where(eq(users.organizationId, organizationId));
  }

  // Course operations
  async createCourse(course: InsertCourse): Promise<Course> {
    const [newCourse] = await db
      .insert(courses)
      .values({ ...course, id: nanoid() })
      .returning();
    return newCourse;
  }

  async getCourse(id: string): Promise<Course | undefined> {
    const [course] = await db
      .select()
      .from(courses)
      .where(eq(courses.id, id));
    return course;
  }

  async getCoursesByOrganization(organizationId: string): Promise<Course[]> {
    return await db
      .select()
      .from(courses)
      .where(and(eq(courses.organizationId, organizationId), eq(courses.isActive, true)))
      .orderBy(desc(courses.createdAt));
  }

  async updateCourse(id: string, updates: Partial<InsertCourse>): Promise<Course> {
    const [course] = await db
      .update(courses)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(courses.id, id))
      .returning();
    return course;
  }

  async searchCourses(query: string, organizationId?: string): Promise<Course[]> {
    const conditions = [
      eq(courses.isActive, true),
      ilike(courses.title, `%${query}%`),
    ];

    if (organizationId) {
      conditions.push(eq(courses.organizationId, organizationId));
    }

    return await db
      .select()
      .from(courses)
      .where(and(...conditions))
      .orderBy(desc(courses.createdAt));
  }

  // Enrollment operations
  async enrollUser(enrollment: InsertEnrollment): Promise<Enrollment> {
    const [newEnrollment] = await db
      .insert(enrollments)
      .values({ ...enrollment, id: nanoid() })
      .returning();
    return newEnrollment;
  }

  async getUserEnrollments(userId: string): Promise<Enrollment[]> {
    return await db
      .select()
      .from(enrollments)
      .where(eq(enrollments.userId, userId))
      .orderBy(desc(enrollments.enrolledAt));
  }

  async updateEnrollmentProgress(id: string, progress: number): Promise<Enrollment> {
    const [enrollment] = await db
      .update(enrollments)
      .set({ progress: progress.toString() })
      .where(eq(enrollments.id, id))
      .returning();
    return enrollment;
  }

  async completeEnrollment(id: string): Promise<Enrollment> {
    const [enrollment] = await db
      .update(enrollments)
      .set({ 
        status: 'completed',
        progress: '100',
        completedAt: new Date()
      })
      .where(eq(enrollments.id, id))
      .returning();
    return enrollment;
  }

  // Assessment operations
  async createAssessment(assessment: InsertAssessment): Promise<Assessment> {
    const [newAssessment] = await db
      .insert(assessments)
      .values({ ...assessment, id: nanoid() })
      .returning();
    return newAssessment;
  }

  async getAssessment(id: string): Promise<Assessment | undefined> {
    const [assessment] = await db
      .select()
      .from(assessments)
      .where(eq(assessments.id, id));
    return assessment;
  }

  async getAssessmentsByCourse(courseId: string): Promise<Assessment[]> {
    return await db
      .select()
      .from(assessments)
      .where(eq(assessments.courseId, courseId))
      .orderBy(asc(assessments.createdAt));
  }

  async getAssessmentsByOrganization(organizationId: string): Promise<Assessment[]> {
    return await db
      .select()
      .from(assessments)
      .where(eq(assessments.organizationId, organizationId))
      .orderBy(desc(assessments.createdAt));
  }

  // Assessment attempt operations
  async createAssessmentAttempt(attempt: InsertAssessmentAttempt): Promise<AssessmentAttempt> {
    const [newAttempt] = await db
      .insert(assessmentAttempts)
      .values({ ...attempt, id: nanoid() })
      .returning();
    return newAttempt;
  }

  async getAssessmentAttempt(id: string): Promise<AssessmentAttempt | undefined> {
    const [attempt] = await db
      .select()
      .from(assessmentAttempts)
      .where(eq(assessmentAttempts.id, id));
    return attempt;
  }

  async getUserAssessmentAttempts(userId: string, assessmentId?: string): Promise<AssessmentAttempt[]> {
    const conditions = [eq(assessmentAttempts.userId, userId)];
    if (assessmentId) {
      conditions.push(eq(assessmentAttempts.assessmentId, assessmentId));
    }

    return await db
      .select()
      .from(assessmentAttempts)
      .where(and(...conditions))
      .orderBy(desc(assessmentAttempts.startedAt));
  }

  async updateAssessmentAttempt(id: string, updates: Partial<InsertAssessmentAttempt>): Promise<AssessmentAttempt> {
    const [attempt] = await db
      .update(assessmentAttempts)
      .set(updates)
      .where(eq(assessmentAttempts.id, id))
      .returning();
    return attempt;
  }

  // Certificate operations
  async createCertificate(certificate: InsertCertificate): Promise<Certificate> {
    const [newCertificate] = await db
      .insert(certificates)
      .values({ 
        ...certificate, 
        id: nanoid(),
        verificationCode: nanoid(16).toUpperCase(),
        blockchainHash: nanoid(64) // In real implementation, this would be actual blockchain hash
      })
      .returning();
    return newCertificate;
  }

  async getUserCertificates(userId: string): Promise<Certificate[]> {
    return await db
      .select()
      .from(certificates)
      .where(eq(certificates.userId, userId))
      .orderBy(desc(certificates.issuedAt));
  }

  async verifyCertificate(verificationCode: string): Promise<Certificate | undefined> {
    const [certificate] = await db
      .select()
      .from(certificates)
      .where(eq(certificates.verificationCode, verificationCode));
    return certificate;
  }

  // Analytics operations
  async logLearningEvent(analytics: InsertLearningAnalytics): Promise<LearningAnalytics> {
    const [event] = await db
      .insert(learningAnalytics)
      .values({ ...analytics, id: nanoid() })
      .returning();
    return event;
  }

  async getUserAnalytics(userId: string, fromDate?: Date, toDate?: Date): Promise<LearningAnalytics[]> {
    const conditions = [eq(learningAnalytics.userId, userId)];
    
    if (fromDate) {
      conditions.push(sql`${learningAnalytics.timestamp} >= ${fromDate}`);
    }
    if (toDate) {
      conditions.push(sql`${learningAnalytics.timestamp} <= ${toDate}`);
    }

    return await db
      .select()
      .from(learningAnalytics)
      .where(and(...conditions))
      .orderBy(desc(learningAnalytics.timestamp));
  }

  async getOrganizationAnalytics(organizationId: string, fromDate?: Date, toDate?: Date): Promise<LearningAnalytics[]> {
    const conditions = [sql`${learningAnalytics.userId} IN (SELECT id FROM ${users} WHERE organization_id = ${organizationId})`];
    
    if (fromDate) {
      conditions.push(sql`${learningAnalytics.timestamp} >= ${fromDate}`);
    }
    if (toDate) {
      conditions.push(sql`${learningAnalytics.timestamp} <= ${toDate}`);
    }

    return await db
      .select()
      .from(learningAnalytics)
      .where(and(...conditions))
      .orderBy(desc(learningAnalytics.timestamp));
  }

  // Examination session operations
  async createExaminationSession(session: InsertExaminationSession): Promise<ExaminationSession> {
    const [newSession] = await db
      .insert(examinationSessions)
      .values({ ...session, id: nanoid() })
      .returning();
    return newSession;
  }

  async getExaminationSession(id: string): Promise<ExaminationSession | undefined> {
    const [session] = await db
      .select()
      .from(examinationSessions)
      .where(eq(examinationSessions.id, id));
    return session;
  }

  async updateExaminationSession(id: string, updates: Partial<InsertExaminationSession>): Promise<ExaminationSession> {
    const [session] = await db
      .update(examinationSessions)
      .set(updates)
      .where(eq(examinationSessions.id, id))
      .returning();
    return session;
  }

  async getActiveExaminationSessions(): Promise<ExaminationSession[]> {
    return await db
      .select()
      .from(examinationSessions)
      .where(eq(examinationSessions.status, 'active'))
      .orderBy(asc(examinationSessions.startedAt));
  }

  async getUserExaminationSessions(userId: string): Promise<ExaminationSession[]> {
    return await db
      .select()
      .from(examinationSessions)
      .where(eq(examinationSessions.userId, userId))
      .orderBy(desc(examinationSessions.scheduledTime));
  }

  // Dashboard analytics
  async getLearnerDashboardStats(userId: string): Promise<any> {
    const userEnrollments = await this.getUserEnrollments(userId);
    const userCertificates = await this.getUserCertificates(userId);
    const userAttempts = await this.getUserAssessmentAttempts(userId);

    const activeCourses = userEnrollments.filter(e => e.status === 'active').length;
    const completedCourses = userEnrollments.filter(e => e.status === 'completed').length;
    const certificates = userCertificates.length;
    const assessmentsPending = userAttempts.filter(a => a.status === 'in_progress').length;

    return {
      activeCourses,
      completedCourses,
      certificates,
      assessmentsPending,
      totalLearningTime: userEnrollments.reduce((total, e) => total + (parseFloat(e.progress || '0') * 2), 0), // Rough estimate
    };
  }

  async getCorporateDashboardStats(organizationId: string): Promise<any> {
    const orgUsers = await this.getUsersByOrganization(organizationId);
    const orgCourses = await this.getCoursesByOrganization(organizationId);
    
    const totalUsers = orgUsers.length;
    const activeUsers = orgUsers.filter(u => u.createdAt && new Date(u.createdAt) > new Date(Date.now() - 30 * 24 * 60 * 60 * 1000)).length;
    const totalCourses = orgCourses.length;

    // Get completion rates
    const allEnrollments = await db
      .select()
      .from(enrollments)
      .where(sql`${enrollments.userId} IN (${sql.join(orgUsers.map(u => sql`${u.id}`), sql`, `)})`);

    const completedEnrollments = allEnrollments.filter(e => e.status === 'completed').length;
    const completionRate = allEnrollments.length > 0 ? (completedEnrollments / allEnrollments.length) * 100 : 0;

    return {
      totalUsers,
      activeUsers,
      totalCourses,
      completionRate: Math.round(completionRate * 10) / 10,
      compliance: 94.7, // This would be calculated based on compliance requirements
      trainingROI: 284, // This would be calculated based on business metrics
    };
  }

  async getPartnerDashboardStats(organizationId: string): Promise<any> {
    const orgUsers = await this.getUsersByOrganization(organizationId);
    const orgCourses = await this.getCoursesByOrganization(organizationId);

    const activeStudents = orgUsers.filter(u => u.userType === 'learner').length;
    const successRate = 91.4; // This would be calculated from actual completion/success metrics
    const qualityScore = 4.8; // This would come from feedback and assessment data
    const monthlyRevenue = activeStudents * 125; // Example calculation

    return {
      activeStudents,
      monthlyRevenue,
      successRate,
      qualityScore,
      totalCourses: orgCourses.length,
    };
  }

  async getSystemDashboardStats(): Promise<any> {
    const [totalUsersResult] = await db.select({ count: count() }).from(users);
    const [totalOrgsResult] = await db.select({ count: count() }).from(organizations);
    const [totalCoursesResult] = await db.select({ count: count() }).from(courses);
    const [activeSessionsResult] = await db.select({ count: count() }).from(examinationSessions).where(eq(examinationSessions.status, 'active'));

    return {
      totalUsers: totalUsersResult.count,
      totalOrganizations: totalOrgsResult.count,
      totalCourses: totalCoursesResult.count,
      activeSessions: activeSessionsResult.count,
      uptime: 99.9,
      alerts: 3, // This would come from monitoring systems
    };
  }

  // Accreditation operations - Enhanced for FRD compliance
  async getAccreditationApplications(): Promise<AccreditationApplication[]> {
    const applications = await db.select().from(accreditationApplications).orderBy(desc(accreditationApplications.submittedAt));
    console.log('Retrieved applications from DB:', applications.map(app => ({ id: app.id, organizationName: app.organizationName, qaCheckpoint1: app.qaCheckpoint1, qaCheckpoint2: app.qaCheckpoint2, qaCheckpoint3: app.qaCheckpoint3 })));
    return applications;
  }

  async getAccreditationApplication(id: string): Promise<AccreditationApplication | undefined> {
    const [application] = await db.select().from(accreditationApplications).where(eq(accreditationApplications.id, id));
    return application;
  }

  async createAccreditationApplication(application: InsertAccreditationApplication): Promise<AccreditationApplication> {
    const [newApplication] = await db
      .insert(accreditationApplications)
      .values({ 
        ...application, 
        id: nanoid(),
        status: 'pending',
        submittedAt: new Date()
      })
      .returning();
    return newApplication;
  }

  async getUserAccreditationApplication(userId: string): Promise<AccreditationApplication | undefined> {
    const [application] = await db.select().from(accreditationApplications)
      .where(eq(accreditationApplications.userId, userId));
    return application;
  }

  async updateAccreditationStatus(
    id: string, 
    status: string, 
    comments: string, 
    reviewerId: string, 
    qaChecklist?: { qaCheckpoint1?: boolean; qaCheckpoint2?: boolean; qaCheckpoint3?: boolean }
  ): Promise<AccreditationApplication> {
    console.log('Storage updateAccreditationStatus called with:', { id, status, qaChecklist });
    
    const current = await this.getAccreditationApplication(id);
    if (!current) {
      console.log('Application not found for ID:', id);
      throw new Error(`Application not found with ID: ${id}`);
    }
    
    console.log('Found application:', { id: current.id, orgName: current.organizationName });

    // Prepare update data - simplified version focusing on QA functionality
    const updateData: any = {
      status, 
      reviewComments: comments, 
      reviewerId, 
      rejectionReason: status === 'rejected' ? comments : null,
      updatedAt: new Date()
    };

    // Include QA checklist updates if provided
    if (qaChecklist) {
      if (qaChecklist.qaCheckpoint1 !== undefined) {
        updateData.qaCheckpoint1 = qaChecklist.qaCheckpoint1;
      }
      if (qaChecklist.qaCheckpoint2 !== undefined) {
        updateData.qaCheckpoint2 = qaChecklist.qaCheckpoint2;
      }
      if (qaChecklist.qaCheckpoint3 !== undefined) {
        updateData.qaCheckpoint3 = qaChecklist.qaCheckpoint3;
      }
    }

    const [updatedApplication] = await db
      .update(accreditationApplications)
      .set(updateData)
      .where(eq(accreditationApplications.id, id))
      .returning();
    return updatedApplication;
  }

  async getAccreditationApplicationsByStatus(status: string): Promise<AccreditationApplication[]> {
    return await db.select().from(accreditationApplications)
      .where(eq(accreditationApplications.status, status))
      .orderBy(desc(accreditationApplications.submittedAt));
  }

  async getAccreditationApplicationsByType(applicationType: string): Promise<AccreditationApplication[]> {
    return await db.select().from(accreditationApplications)
      .where(eq(accreditationApplications.applicationType, applicationType))
      .orderBy(desc(accreditationApplications.submittedAt));
  }

  async getUserAccreditationApplications(userId: string): Promise<AccreditationApplication[]> {
    return await db.select().from(accreditationApplications)
      .where(eq(accreditationApplications.userId, userId))
      .orderBy(desc(accreditationApplications.submittedAt));
  }

  // Question Bank operations
  async getQuestionBank(): Promise<QuestionBank[]> {
    return await db.select().from(questionBank).where(eq(questionBank.isActive, true)).orderBy(desc(questionBank.createdAt));
  }

  async getQuestion(id: string): Promise<QuestionBank | undefined> {
    const [question] = await db.select().from(questionBank).where(eq(questionBank.id, id));
    return question;
  }

  async createQuestion(question: InsertQuestionBank): Promise<QuestionBank> {
    const [newQuestion] = await db
      .insert(questionBank)
      .values({ ...question, id: nanoid() })
      .returning();
    return newQuestion;
  }

  async updateQuestion(id: string, updates: Partial<InsertQuestionBank>): Promise<QuestionBank> {
    const [updatedQuestion] = await db
      .update(questionBank)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(questionBank.id, id))
      .returning();
    return updatedQuestion;
  }

  async deleteQuestion(id: string): Promise<void> {
    await db.update(questionBank).set({ isActive: false }).where(eq(questionBank.id, id));
  }

  // Testing Center operations
  async getTestingCenters(): Promise<TestingCenter[]> {
    return await db.select().from(testingCenters).where(eq(testingCenters.isActive, true)).orderBy(asc(testingCenters.name));
  }

  async getTestingCenter(id: string): Promise<TestingCenter | undefined> {
    const [center] = await db.select().from(testingCenters).where(eq(testingCenters.id, id));
    return center;
  }

  async createTestingCenter(center: InsertTestingCenter): Promise<TestingCenter> {
    const [newCenter] = await db
      .insert(testingCenters)
      .values({ ...center, id: nanoid() })
      .returning();
    return newCenter;
  }

  async updateTestingCenter(id: string, updates: Partial<InsertTestingCenter>): Promise<TestingCenter> {
    const [updatedCenter] = await db
      .update(testingCenters)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(testingCenters.id, id))
      .returning();
    return updatedCenter;
  }

  // Exam Booking operations
  async getAllExamBookings(): Promise<ExamBooking[]> {
    return await db.select().from(examBookings).orderBy(desc(examBookings.scheduledDate));
  }

  async getUserExamBookings(userId: string): Promise<ExamBooking[]> {
    return await db.select().from(examBookings).where(eq(examBookings.userId, userId)).orderBy(desc(examBookings.scheduledDate));
  }

  async getExamBooking(id: string): Promise<ExamBooking | undefined> {
    const [booking] = await db.select().from(examBookings).where(eq(examBookings.id, id));
    return booking;
  }

  async createExamBooking(booking: InsertExamBooking): Promise<ExamBooking> {
    const [newBooking] = await db
      .insert(examBookings)
      .values({ ...booking, id: nanoid() })
      .returning();
    return newBooking;
  }

  async updateExamBooking(id: string, updates: Partial<InsertExamBooking>): Promise<ExamBooking> {
    const [updatedBooking] = await db
      .update(examBookings)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(examBookings.id, id))
      .returning();
    return updatedBooking;
  }

  // Corporate Credits operations
  async getCorporateCredits(organizationId: string): Promise<CorporateCredits | undefined> {
    const [credits] = await db.select().from(corporateCredits).where(eq(corporateCredits.organizationId, organizationId));
    return credits;
  }

  async createCorporateCredits(credits: InsertCorporateCredits): Promise<CorporateCredits> {
    const [newCredits] = await db
      .insert(corporateCredits)
      .values({ ...credits, id: nanoid() })
      .returning();
    return newCredits;
  }

  async updateCorporateCredits(id: string, updates: Partial<InsertCorporateCredits>): Promise<CorporateCredits> {
    const [updatedCredits] = await db
      .update(corporateCredits)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(corporateCredits.id, id))
      .returning();
    return updatedCredits;
  }

  // Notification Template operations
  async getNotificationTemplates(): Promise<NotificationTemplate[]> {
    return await db.select().from(notificationTemplates).where(eq(notificationTemplates.isActive, true)).orderBy(asc(notificationTemplates.templateName));
  }

  async getNotificationTemplate(id: string): Promise<NotificationTemplate | undefined> {
    const [template] = await db.select().from(notificationTemplates).where(eq(notificationTemplates.id, id));
    return template;
  }

  async createNotificationTemplate(template: InsertNotificationTemplate): Promise<NotificationTemplate> {
    const [newTemplate] = await db
      .insert(notificationTemplates)
      .values({ ...template, id: nanoid() })
      .returning();
    return newTemplate;
  }

  async updateNotificationTemplate(id: string, updates: Partial<InsertNotificationTemplate>): Promise<NotificationTemplate> {
    const [updatedTemplate] = await db
      .update(notificationTemplates)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(notificationTemplates.id, id))
      .returning();
    return updatedTemplate;
  }

  // Exam Rules operations
  async getExamRules(): Promise<ExamRules[]> {
    return await db.select().from(examRules).where(eq(examRules.isActive, true)).orderBy(desc(examRules.createdAt));
  }

  async getExamRule(id: string): Promise<ExamRules | undefined> {
    const [rule] = await db.select().from(examRules).where(eq(examRules.id, id));
    return rule;
  }

  async createExamRule(rule: InsertExamRules): Promise<ExamRules> {
    const [newRule] = await db
      .insert(examRules)
      .values({ ...rule, id: nanoid() })
      .returning();
    return newRule;
  }

  async updateExamRule(id: string, updates: Partial<InsertExamRules>): Promise<ExamRules> {
    const [updatedRule] = await db
      .update(examRules)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(examRules.id, id))
      .returning();
    return updatedRule;
  }

  // CMS operations - working with actual database
  async getCmsPages(): Promise<CmsPage[]> {
    try {
      const pages = await db.select().from(cmsPages).orderBy(asc(cmsPages.title));
      return pages;
    } catch (error) {
      console.error('Error fetching CMS pages:', error);
      return [];
    }
  }

  async getCmsPage(id: string): Promise<CmsPage | undefined> {
    try {
      const [result] = await db.select().from(cmsPages).where(eq(cmsPages.id, id));
      return result;
    } catch (error) {
      console.error('Error fetching CMS page:', error);
      return undefined;
    }
  }

  async getCmsPageBySlug(slug: string): Promise<CmsPage | undefined> {
    try {
      const [result] = await db.select().from(cmsPages).where(eq(cmsPages.slug, slug));
      return result;
    } catch (error) {
      console.error('Error fetching CMS page by slug:', error);
      return undefined;
    }
  }

  async createCmsPage(data: InsertCmsPage): Promise<CmsPage> {
    try {
      const id = nanoid();
      const pageData = {
        ...data,
        id
      };
      
      const [page] = await db.insert(cmsPages).values(pageData).returning();
      return page;
    } catch (error) {
      console.error('Error creating CMS page:', error);
      throw error;
    }
  }

  async updateCmsPage(id: string, data: Partial<InsertCmsPage>): Promise<CmsPage> {
    try {
      const updateData = {
        ...data,
        updatedAt: new Date()
      };
      
      const [result] = await db
        .update(cmsPages)
        .set(updateData)
        .where(eq(cmsPages.id, id))
        .returning();
      
      if (!result) throw new Error('Page not found');
      
      return result;
    } catch (error) {
      console.error('Error updating CMS page:', error);
      throw error;
    }
  }

  async deleteCmsPage(id: string): Promise<void> {
    try {
      await db.delete(cmsPages).where(eq(cmsPages.id, id));
    } catch (error) {
      console.error('Error deleting CMS page:', error);
      throw error;
    }
  }

  // Page Sections operations
  async getPageSections(pageId: string): Promise<PageSection[]> {
    try {
      const sections = await db
        .select()
        .from(pageSections)
        .where(and(eq(pageSections.pageId, pageId), eq(pageSections.isActive, true)))
        .orderBy(asc(pageSections.sortOrder));
      return sections;
    } catch (error) {
      console.error('Error fetching page sections:', error);
      return [];
    }
  }

  async getPageSection(id: string): Promise<PageSection | undefined> {
    try {
      const [section] = await db.select().from(pageSections).where(eq(pageSections.id, id));
      return section;
    } catch (error) {
      console.error('Error fetching page section:', error);
      return undefined;
    }
  }

  async createPageSection(sectionData: InsertPageSection): Promise<PageSection> {
    try {
      // Auto-increment sortOrder if not provided or if we need to place at end
      let finalSortOrder = sectionData.sortOrder;
      
      if (!finalSortOrder || finalSortOrder === undefined) {
        // Get the maximum sortOrder for this page and add 1
        const existingSections = await db
          .select()
          .from(pageSections)
          .where(and(eq(pageSections.pageId, sectionData.pageId), eq(pageSections.isActive, true)))
          .orderBy(desc(pageSections.sortOrder))
          .limit(1);
        
        finalSortOrder = existingSections.length > 0 ? (existingSections[0].sortOrder || 0) + 1 : 0;
      }

      const [section] = await db
        .insert(pageSections)
        .values({
          ...sectionData,
          sortOrder: finalSortOrder,
          id: nanoid(),
          createdAt: new Date(),
          updatedAt: new Date(),
        })
        .returning();
      return section;
    } catch (error) {
      console.error('Error creating page section:', error);
      throw error;
    }
  }

  async updatePageSection(id: string, updates: Partial<InsertPageSection>): Promise<PageSection> {
    try {
      // First get the existing section to merge data properly
      const existing = await this.getPageSection(id);
      if (!existing) throw new Error('Section not found');
      
      // Deep merge the data field if both exist
      let mergedData = existing.data;
      if (updates.data && existing.data) {
        mergedData = { ...existing.data, ...updates.data };
      } else if (updates.data) {
        mergedData = updates.data;
      }
      
      const [section] = await db
        .update(pageSections)
        .set({
          ...updates,
          data: mergedData,
          updatedAt: new Date(),
        })
        .where(eq(pageSections.id, id))
        .returning();
      
      if (!section) throw new Error('Section not found after update');
      return section;
    } catch (error) {
      console.error('Error updating page section:', error);
      throw error;
    }
  }

  async deletePageSection(id: string): Promise<void> {
    try {
      await db.update(pageSections).set({ isActive: false }).where(eq(pageSections.id, id));
    } catch (error) {
      console.error('Error deleting page section:', error);
      throw error;
    }
  }

  async reorderPageSections(pageId: string, sections: { id: string; sortOrder: number }[]): Promise<void> {
    try {
      // Update sort order for each section in a transaction
      for (const section of sections) {
        await db
          .update(pageSections)
          .set({ sortOrder: section.sortOrder })
          .where(eq(pageSections.id, section.id));
      }
    } catch (error) {
      console.error('Error reordering page sections:', error);
      throw error;
    }
  }

  // News articles operations
  async getNewsArticles(): Promise<NewsArticle[]> {
    return await db.select().from(newsArticles).orderBy(desc(newsArticles.createdAt));
  }

  async getNewsArticle(id: string): Promise<NewsArticle | undefined> {
    const [article] = await db.select().from(newsArticles).where(eq(newsArticles.id, id));
    return article;
  }

  async createNewsArticle(articleData: InsertNewsArticle): Promise<NewsArticle> {
    const [article] = await db
      .insert(newsArticles)
      .values({
        ...articleData,
        id: nanoid(),
        createdAt: new Date(),
        updatedAt: new Date(),
      })
      .returning();
    return article;
  }

  async updateNewsArticle(id: string, updates: Partial<InsertNewsArticle>): Promise<NewsArticle> {
    const [article] = await db
      .update(newsArticles)
      .set({
        ...updates,
        updatedAt: new Date(),
      })
      .where(eq(newsArticles.id, id))
      .returning();
    return article;
  }

  async deleteNewsArticle(id: string): Promise<void> {
    await db.delete(newsArticles).where(eq(newsArticles.id, id));
  }

  // Resources operations
  async getResources(): Promise<Resource[]> {
    return await db.select().from(resources).orderBy(desc(resources.createdAt));
  }

  async getResource(id: string): Promise<Resource | undefined> {
    const [resource] = await db.select().from(resources).where(eq(resources.id, id));
    return resource;
  }

  async createResource(resourceData: InsertResource): Promise<Resource> {
    const [resource] = await db
      .insert(resources)
      .values({
        ...resourceData,
        id: nanoid(),
        createdAt: new Date(),
        updatedAt: new Date(),
      })
      .returning();
    return resource;
  }

  async updateResource(id: string, updates: Partial<InsertResource>): Promise<Resource> {
    const [resource] = await db
      .update(resources)
      .set({
        ...updates,
        updatedAt: new Date(),
      })
      .where(eq(resources.id, id))
      .returning();
    return resource;
  }

  async deleteResource(id: string): Promise<void> {
    await db.delete(resources).where(eq(resources.id, id));
  }

  // Store products operations
  async getStoreProducts(): Promise<StoreProduct[]> {
    return await db.select().from(storeProducts).orderBy(desc(storeProducts.createdAt));
  }

  async getStoreProduct(id: string): Promise<StoreProduct | undefined> {
    const [product] = await db.select().from(storeProducts).where(eq(storeProducts.id, id));
    return product;
  }

  async createStoreProduct(productData: InsertStoreProduct): Promise<StoreProduct> {
    const [product] = await db
      .insert(storeProducts)
      .values({
        ...productData,
        id: nanoid(),
        createdAt: new Date(),
        updatedAt: new Date(),
      })
      .returning();
    return product;
  }

  async updateStoreProduct(id: string, updates: Partial<InsertStoreProduct>): Promise<StoreProduct> {
    const [product] = await db
      .update(storeProducts)
      .set({
        ...updates,
        updatedAt: new Date(),
      })
      .where(eq(storeProducts.id, id))
      .returning();
    return product;
  }

  async deleteStoreProduct(id: string): Promise<void> {
    await db.delete(storeProducts).where(eq(storeProducts.id, id));
  }

  // Media operations
  async getMediaFiles(): Promise<CmsMedia[]> {
    return await db.select().from(cmsMedia).orderBy(desc(cmsMedia.createdAt));
  }

  async getMediaFile(id: string): Promise<CmsMedia | undefined> {
    const [media] = await db.select().from(cmsMedia).where(eq(cmsMedia.id, id));
    return media;
  }

  async createMediaFile(mediaData: InsertCmsMedia): Promise<CmsMedia> {
    const [media] = await db
      .insert(cmsMedia)
      .values({
        ...mediaData,
        id: nanoid(),
        createdAt: new Date(),
      })
      .returning();
    return media;
  }

  async deleteMediaFile(id: string): Promise<void> {
    await db.delete(cmsMedia).where(eq(cmsMedia.id, id));
  }

  // Settings operations
  async getCmsSettings(): Promise<CmsSettings | undefined> {
    const [settings] = await db.select().from(cmsSettings).limit(1);
    return settings;
  }

  async updateCmsSettings(settingsData: InsertCmsSettings): Promise<CmsSettings> {
    // First check if settings exist
    const existing = await this.getCmsSettings();
    
    if (existing) {
      // Update existing settings
      const [settings] = await db
        .update(cmsSettings)
        .set({
          ...settingsData,
          updatedAt: new Date(),
        })
        .where(eq(cmsSettings.id, existing.id))
        .returning();
      return settings;
    } else {
      // Create new settings
      const [settings] = await db
        .insert(cmsSettings)
        .values({
          ...settingsData,
          id: nanoid(),
          updatedAt: new Date(),
        })
        .returning();
      return settings;
    }
  }

  // System Certificate operations
  async getSystemCertificates(): Promise<SystemCertificate[]> {
    const certificates = await db.select().from(systemCertificates).orderBy(asc(systemCertificates.createdAt));
    return certificates;
  }

  async getSystemCertificate(id: string): Promise<SystemCertificate | undefined> {
    const [certificate] = await db.select().from(systemCertificates).where(eq(systemCertificates.id, id));
    return certificate;
  }

  async createSystemCertificate(certificate: InsertSystemCertificate): Promise<SystemCertificate> {
    const [newCertificate] = await db
      .insert(systemCertificates)
      .values({ ...certificate, id: nanoid() })
      .returning();
    return newCertificate;
  }

  async updateSystemCertificate(id: string, updates: Partial<InsertSystemCertificate>): Promise<SystemCertificate> {
    const [updatedCertificate] = await db
      .update(systemCertificates)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(systemCertificates.id, id))
      .returning();
    return updatedCertificate;
  }

  async deleteSystemCertificate(id: string): Promise<void> {
    await db.delete(systemCertificates).where(eq(systemCertificates.id, id));
  }

  // System Level operations
  async getSystemLevels(certificateId: string): Promise<SystemLevel[]> {
    const levels = await db.select()
      .from(systemLevels)
      .where(eq(systemLevels.certificateId, certificateId))
      .orderBy(asc(systemLevels.levelNumber));
    return levels;
  }

  async getSystemLevel(id: string): Promise<SystemLevel | undefined> {
    const [level] = await db.select().from(systemLevels).where(eq(systemLevels.id, id));
    return level;
  }

  async createSystemLevel(level: InsertSystemLevel): Promise<SystemLevel> {
    const [newLevel] = await db
      .insert(systemLevels)
      .values({ 
        ...level, 
        id: nanoid(),
        passingScore: level.passingScore || null
      })
      .returning();
    return newLevel;
  }

  async updateSystemLevel(id: string, updates: Partial<InsertSystemLevel>): Promise<SystemLevel> {
    const [updatedLevel] = await db
      .update(systemLevels)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(systemLevels.id, id))
      .returning();
    return updatedLevel;
  }

  async deleteSystemLevel(id: string): Promise<void> {
    await db.delete(systemLevels).where(eq(systemLevels.id, id));
  }

  // System Chapter operations
  async getSystemChapters(levelId: string): Promise<SystemChapter[]> {
    const chapters = await db.select().from(systemChapters).where(eq(systemChapters.levelId, levelId)).orderBy(asc(systemChapters.chapterNumber));
    return chapters;
  }

  async getSystemChapter(id: string): Promise<SystemChapter | undefined> {
    const [chapter] = await db.select().from(systemChapters).where(eq(systemChapters.id, id));
    return chapter;
  }

  async createSystemChapter(chapter: InsertSystemChapter): Promise<SystemChapter> {
    const [newChapter] = await db
      .insert(systemChapters)
      .values({ ...chapter, id: nanoid() })
      .returning();
    return newChapter;
  }

  async updateSystemChapter(id: string, updates: Partial<InsertSystemChapter>): Promise<SystemChapter> {
    const [updatedChapter] = await db
      .update(systemChapters)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(systemChapters.id, id))
      .returning();
    return updatedChapter;
  }

  async deleteSystemChapter(id: string): Promise<void> {
    await db.delete(systemChapters).where(eq(systemChapters.id, id));
  }

  // CMS Menu Items operations
  async getCmsMenuItems(): Promise<any[]> {
    const items = await db.select().from(cmsMenus).where(eq(cmsMenus.isActive, true));
    return items;
  }

  async getCmsMenuItem(id: string): Promise<any | undefined> {
    const [item] = await db.select().from(cmsMenus).where(eq(cmsMenus.id, id));
    return item;
  }

  async createCmsMenuItem(item: any): Promise<any> {
    const [newItem] = await db
      .insert(cmsMenus)
      .values({ ...item, id: nanoid() })
      .returning();
    return newItem;
  }

  async updateCmsMenuItem(id: string, updates: any): Promise<any> {
    const [updatedItem] = await db
      .update(cmsMenus)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(cmsMenus.id, id))
      .returning();
    return updatedItem;
  }

  async deleteCmsMenuItem(id: string): Promise<void> {
    await db.update(cmsMenus).set({ isActive: false }).where(eq(cmsMenus.id, id));
  }
}

export const storage = new DatabaseStorage();
