import type { Express } from "express";
import { createServer, type Server } from "http";
import { WebSocketServer, WebSocket } from "ws";
import { storage } from "./storage";
import { setupAuth, isAuthenticated } from "./auth";
import bcrypt from "bcryptjs";
import { z } from "zod";
import { db } from "./db";
import { sql } from "drizzle-orm";
import multer from "multer";
import path from "path";
import fs from "fs";
import express from "express";
import {
  insertCourseSchema,
  insertEnrollmentSchema,
  insertAssessmentSchema,
  insertAssessmentAttemptSchema,
  insertCertificateSchema,
  insertLearningAnalyticsSchema,
  insertExaminationSessionSchema,
  insertOrganizationSchema,
  insertSystemLevelSchema,
  insertSystemChapterSchema,
  accreditationApplications,
} from "@shared/schema";
import { ZodError } from "zod";

export async function registerRoutes(app: Express): Promise<Server> {
  // Ensure uploads directory exists
  const uploadsDir = path.join(process.cwd(), 'uploads');
  if (!fs.existsSync(uploadsDir)) {
    fs.mkdirSync(uploadsDir, { recursive: true });
  }

  // Configure multer for file uploads
  const multerStorage = multer.diskStorage({
    destination: function (req, file, cb) {
      cb(null, uploadsDir);
    },
    filename: function (req, file, cb) {
      // Generate unique filename with timestamp and original name
      const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
      const ext = path.extname(file.originalname);
      const name = path.basename(file.originalname, ext);
      cb(null, `${name}-${uniqueSuffix}${ext}`);
    }
  });

  const upload = multer({ 
    storage: multerStorage,
    limits: {
      fileSize: 10 * 1024 * 1024, // 10MB limit
    },
    fileFilter: function (req, file, cb) {
      // Accept image files only
      if (file.mimetype.startsWith('image/')) {
        cb(null, true);
      } else {
        cb(new Error('Only image files are allowed!'));
      }
    }
  });

  // Serve uploaded files statically
  app.use('/api/uploads', express.static(uploadsDir));

  // Auth middleware
  setupAuth(app);

  // Single image upload endpoint for CMS (must be after auth setup)
  app.post('/api/cms/upload-image', async (req: any, res) => {
    try {
      // Check authentication
      if (!req.session?.userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Check if user has website admin permissions
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      // Use multer to handle single image upload
      upload.single('image')(req, res, (err: any) => {
        if (err) {
          console.error('Upload error:', err);
          return res.status(400).json({ message: err.message });
        }
        
        if (!req.file) {
          return res.status(400).json({ message: "No image file provided" });
        }
        
        // Return the filename for storage in database
        res.json({ filename: req.file.filename });
      });
    } catch (error) {
      console.error('Image upload error:', error);
      res.status(500).json({ message: "Failed to upload image" });
    }
  });

  // Debug middleware to log all requests
  app.use((req: any, res: any, next: any) => {
    if (req.url.startsWith('/api/enrollments')) {
      console.log('DEBUG - Enrollment request:', {
        method: req.method,
        url: req.url,
        sessionId: req.sessionID,
        hasSession: !!req.session,
        userId: req.session?.userId
      });
    }
    next();
  });

  // Auth schemas
  const loginSchema = z.object({
    username: z.string().min(1),
    password: z.string().min(1),
  });

  const registerSchema = z.object({
    username: z.string().min(3),
    email: z.string().email(),
    password: z.string().min(6),
    firstName: z.string().min(1),
    lastName: z.string().min(1),
    userType: z.enum(['learner', 'corporate_admin', 'partner_admin', 'system_admin']),
    organizationId: z.string().optional(),
  });

  // Authentication Routes
  app.post('/api/auth/login', async (req: any, res) => {
    try {
      const { username, password } = loginSchema.parse(req.body);
      
      // Find user by username
      const user = await storage.getUserByUsername(username);
      if (!user) {
        return res.status(401).json({ message: "Invalid credentials" });
      }

      // Verify password
      const isValid = await bcrypt.compare(password, user.passwordHash || '');
      if (!isValid) {
        return res.status(401).json({ message: "Invalid credentials" });
      }

      // Create session
      req.session.userId = user.id;
      req.session.userType = user.userType;

      // Remove password from response
      const { passwordHash: _, ...userWithoutPassword } = user;
      
      res.json({ 
        user: userWithoutPassword,
        message: "Login successful" 
      });
    } catch (error) {
      console.error("Login error:", error);
      res.status(400).json({ message: "Invalid login request" });
    }
  });

  app.post('/api/auth/register', async (req: any, res) => {
    try {
      const userData = registerSchema.parse(req.body);
      
      // Check if username already exists
      const existingUser = await storage.getUserByUsername(userData.username);
      if (existingUser) {
        return res.status(400).json({ message: "Username already exists" });
      }

      // Hash password
      const hashedPassword = await bcrypt.hash(userData.password, 10);
      
      // Create user
      const user = await storage.createUser({
        ...userData,
        passwordHash: hashedPassword,
        id: `${userData.userType}-${Date.now()}`,
        createdAt: new Date(),
        updatedAt: new Date(),
      });

      // Create session
      req.session.userId = user.id;
      req.session.userType = user.userType;

      // Remove password from response
      const { passwordHash: _, ...userWithoutPassword } = user;
      
      res.json({ 
        user: userWithoutPassword,
        message: "Registration successful" 
      });
    } catch (error) {
      console.error("Registration error:", error);
      res.status(400).json({ message: "Registration failed" });
    }
  });

  app.get('/api/auth/user', async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      if (!userId) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const user = await storage.getUser(userId);
      if (!user) {
        return res.status(401).json({ message: "User not found" });
      }

      // Remove password hash from response
      const { passwordHash: _, ...userWithoutPassword } = user;
      res.json(userWithoutPassword);
    } catch (error) {
      console.error("Get user error:", error);
      res.status(500).json({ message: "Failed to get user" });
    }
  });

  // Logout handler function
  const handleLogout = (req: any, res: any, isPostRequest = false) => {
    console.log("Logout attempt for session:", req.sessionID, "user:", req.session?.userId, "method:", isPostRequest ? 'POST' : 'GET');
    req.session.destroy((err: any) => {
      if (err) {
        console.error("Logout error:", err);
        if (isPostRequest) {
          return res.status(500).json({ message: "Logout failed" });
        }
        return res.status(500).json({ message: "Logout failed" });
      }
      // Clear all possible session cookies
      res.clearCookie('connect.sid');
      res.clearCookie('connect.sid', { path: '/' });
      res.clearCookie('session');
      res.clearCookie('sessionId');
      
      console.log("Session destroyed successfully, redirecting to homepage");
      
      if (isPostRequest) {
        // For POST requests (DashboardLayout), return JSON success
        res.json({ message: "Logged out successfully" });
      } else {
        // For GET requests (Header), redirect to homepage
        res.redirect('/');
      }
    });
  };

  app.get('/api/logout', (req: any, res) => {
    handleLogout(req, res, false);
  });

  app.post('/api/logout', (req: any, res) => {
    handleLogout(req, res, true);
  });

  // Initialize demo users
  const initializeDemoUsers = async () => {
    try {
      const demoUsers = [
        {
          id: 'learner-1',
          username: 'learner1',
          email: 'learner@ibdl.com',
          password: 'password123',
          firstName: 'John',
          lastName: 'Learner',
          userType: 'learner' as const,
          organizationId: null,
        },
        {
          id: 'corporate-1',
          username: 'corp_admin',
          email: 'corporate@ibdl.com',
          password: 'password123',
          firstName: 'Jane',
          lastName: 'Corporate',
          userType: 'corporate_admin' as const,
          organizationId: 'org-techcorp',
        },
        {
          id: 'partner-1',
          username: 'partner_admin',
          email: 'partner@ibdl.com',
          password: 'password123',
          firstName: 'Mike',
          lastName: 'Partner',
          userType: 'partner_admin' as const,
          organizationId: 'org-training-center',
        },
        {
          id: 'system-1',
          username: 'sys_admin',
          email: 'admin@ibdl.com',
          password: 'password123',
          firstName: 'Sarah',
          lastName: 'Admin',
          userType: 'system_admin' as const,
          organizationId: null,
        },
        {
          id: 'qa-admin-1',
          username: 'qa_admin',
          email: 'qa@ibdl.com',
          password: 'password123',
          firstName: 'Quality',
          lastName: 'Administrator',
          userType: 'qa_admin' as const,
          organizationId: null,
        },
        {
          id: 'demo-1',
          username: 'demo',
          email: 'demo@ibdl.com',
          password: 'demo123',
          firstName: 'Demo',
          lastName: 'User',
          userType: 'system_admin' as const,
          organizationId: null,
        }
      ];

      for (const userData of demoUsers) {
        const existingUser = await storage.getUserByUsername(userData.username);
        if (!existingUser) {
          const hashedPassword = await bcrypt.hash(userData.password, 10);
          const { password, ...userDataWithoutPassword } = userData;
          await storage.createUser({
            ...userDataWithoutPassword,
            passwordHash: hashedPassword,
            createdAt: new Date(),
            updatedAt: new Date(),
          });
          console.log(`Created demo user: ${userData.username}`);
        }
      }
    } catch (error) {
      console.error('Error creating demo users:', error);
    }
  };

  // Initialize demo users on startup
  initializeDemoUsers();



  // Examination System API Routes
  app.get('/api/examinations', async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Sample examination sessions with comprehensive data
      const examinationSessions = [
        {
          id: 'exam-session-1',
          examTitle: 'IBDL Strategic Leadership Certification',
          examType: 'certification',
          scheduledDate: '2025-01-15T10:00:00Z',
          duration: 120, // minutes
          status: 'scheduled',
          questions: 75,
          passingScore: 80,
          attempts: 0,
          maxAttempts: 3,
          proctorSettings: {
            cameraRequired: true,
            microphoneMonitoring: true,
            screenRecording: true,
            tabSwitchingBlocked: true,
            identityVerification: true
          }
        },
        {
          id: 'exam-session-2',
          examTitle: 'Business Analytics Proficiency Assessment',
          examType: 'proctored',
          scheduledDate: '2025-01-10T14:00:00Z',
          duration: 90,
          status: 'in_progress',
          questions: 60,
          passingScore: 75,
          currentScore: 82,
          timeRemaining: 45,
          attempts: 1,
          maxAttempts: 2,
          proctorSettings: {
            cameraRequired: true,
            microphoneMonitoring: false,
            screenRecording: true,
            tabSwitchingBlocked: true,
            identityVerification: false
          }
        },
        {
          id: 'exam-session-3',
          examTitle: 'Digital Marketing Foundation Quiz',
          examType: 'practice',
          scheduledDate: '2024-12-20T09:00:00Z',
          duration: 45,
          status: 'completed',
          questions: 30,
          passingScore: 70,
          currentScore: 88,
          attempts: 1,
          maxAttempts: 5,
          proctorSettings: {
            cameraRequired: false,
            microphoneMonitoring: false,
            screenRecording: false,
            tabSwitchingBlocked: false,
            identityVerification: false
          }
        }
      ];

      res.json(examinationSessions);
    } catch (error) {
      console.error("Error fetching examination sessions:", error);
      res.status(500).json({ message: "Failed to fetch examination sessions" });
    }
  });

  app.post('/api/examinations/start', async (req: any, res) => {
    try {
      console.log('Examination start session check:', {
        method: req.method,
        url: req.url,
        sessionId: req.sessionID,
        userId: req.session?.userId,
        userType: req.session?.userType,
        hasSession: !!req.session,
        headers: req.get('Cookie')?.substring(0, 100) + '...'
      });

      const userId = req.session?.userId;
      const { sessionId } = req.body;

      if (!userId) {
        console.log('Auth failed - no userId in session');
        return res.status(401).json({ message: "Authentication required" });
      }

      // Create examination session record
      const sessionData = {
        id: sessionId,
        userId,
        status: 'in_progress',
        startTime: new Date(),
        proctorFlags: [],
        currentQuestion: 1
      };

      // Log examination start event
      await storage.logLearningEvent({
        userId,
        eventType: 'examination_started',
        eventData: { sessionId, examType: 'certification' }
      });

      res.json({ 
        sessionId,
        message: "Examination session started successfully",
        redirectUrl: `/examinations/${sessionId}/take`
      });
    } catch (error) {
      console.error("Error starting examination:", error);
      res.status(500).json({ message: "Failed to start examination" });
    }
  });

  app.get('/api/examinations/:sessionId/questions', async (req: any, res) => {
    try {
      const { sessionId } = req.params;
      const userId = req.session?.userId;

      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Comprehensive examination questions with multiple types
      const examQuestions = {
        'exam-session-1': [
          {
            id: 'q1',
            type: 'multiple_choice',
            question: 'What is the primary purpose of strategic leadership in organizational transformation?',
            options: [
              'To maintain current operational efficiency',
              'To guide organizational vision and long-term direction',
              'To manage day-to-day team activities',
              'To implement technical solutions'
            ],
            correctAnswer: 1,
            points: 2,
            timeLimit: 90
          },
          {
            id: 'q2',
            type: 'multiple_select',
            question: 'Which of the following are key characteristics of effective strategic leaders? (Select all that apply)',
            options: [
              'Visionary thinking',
              'Emotional intelligence',
              'Risk aversion',
              'Adaptive communication',
              'Micromanagement tendencies',
              'Systems thinking'
            ],
            correctAnswers: [0, 1, 3, 5],
            points: 3,
            timeLimit: 120
          },
          {
            id: 'q3',
            type: 'true_false',
            question: 'Strategic leaders should always maintain a consistent leadership style regardless of organizational context.',
            correctAnswer: false,
            points: 1,
            timeLimit: 60
          },
          {
            id: 'q4',
            type: 'short_answer',
            question: 'Describe a scenario where transformational leadership would be more effective than transactional leadership. (Maximum 150 words)',
            points: 5,
            timeLimit: 300,
            rubric: {
              criteria: [
                'Clear scenario identification',
                'Understanding of transformational vs transactional leadership',
                'Practical application',
                'Logical reasoning'
              ]
            }
          },
          {
            id: 'q5',
            type: 'case_study',
            question: 'Case Study: TechCorp is facing declining market share due to digital disruption. As the new CEO, outline your strategic leadership approach to navigate this transformation.',
            context: 'TechCorp is a 50-year-old manufacturing company with 5000 employees. Traditional products are becoming obsolete. Employee morale is low, and there is resistance to change.',
            points: 10,
            timeLimit: 600,
            rubric: {
              criteria: [
                'Strategic vision development',
                'Change management approach',
                'Stakeholder consideration',
                'Implementation planning',
                'Risk assessment'
              ]
            }
          }
        ]
      };

      const questions = examQuestions[sessionId as keyof typeof examQuestions] || [];
      res.json(questions);
    } catch (error) {
      console.error("Error fetching examination questions:", error);
      res.status(500).json({ message: "Failed to fetch examination questions" });
    }
  });

  app.post('/api/examinations/:sessionId/submit-answer', async (req: any, res) => {
    try {
      const { sessionId } = req.params;
      const { questionId, answer, timeSpent } = req.body;
      const userId = req.session?.userId;

      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Store answer and calculate score
      const answerData = {
        sessionId,
        questionId,
        answer,
        timeSpent,
        timestamp: new Date(),
        userId
      };

      // Log answer submission
      await storage.logLearningEvent({
        userId,
        eventType: 'examination_answer_submitted',
        eventData: { sessionId, questionId, timeSpent }
      });

      res.json({ 
        success: true,
        message: "Answer submitted successfully"
      });
    } catch (error) {
      console.error("Error submitting answer:", error);
      res.status(500).json({ message: "Failed to submit answer" });
    }
  });

  app.post('/api/examinations/:sessionId/complete', async (req: any, res) => {
    try {
      const { sessionId } = req.params;
      const userId = req.session?.userId;

      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Calculate final score and generate results
      const finalScore = 85; // Mock calculation
      const examPassed = finalScore >= 80;

      // Create examination attempt record
      const attemptData = {
        userId,
        sessionId,
        score: finalScore.toString(),
        status: examPassed ? 'passed' : 'failed',
        completedAt: new Date(),
        timeSpent: 110 // minutes
      };

      // Generate certificate if passed
      if (examPassed) {
        const certificateData = {
          userId,
          courseId: 'strategic-leadership-cert',
          type: 'certification',
          title: 'IBDL Strategic Leadership Certification',
          issueDate: new Date(),
          verificationCode: `IBDL-${Date.now()}`
        };

        await storage.createCertificate(certificateData);
      }

      // Log examination completion
      await storage.logLearningEvent({
        userId,
        eventType: 'examination_completed',
        eventData: { sessionId, score: finalScore, passed: examPassed }
      });

      res.json({
        success: true,
        results: {
          score: finalScore,
          passed: examPassed,
          certificateGenerated: examPassed,
          message: examPassed ? "Congratulations! You have passed the examination." : "Unfortunately, you did not meet the passing criteria. Please review the material and try again."
        }
      });
    } catch (error) {
      console.error("Error completing examination:", error);
      res.status(500).json({ message: "Failed to complete examination" });
    }
  });

  app.post('/api/examinations/:sessionId/proctor-flag', async (req: any, res) => {
    try {
      const { sessionId } = req.params;
      const { flagType, description, timestamp } = req.body;
      const userId = req.session.userId;

      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Log proctoring flag
      await storage.logLearningEvent({
        userId,
        eventType: 'examination_proctor_flag',
        eventData: { sessionId, flagType, description, timestamp }
      });

      res.json({ success: true });
    } catch (error) {
      console.error("Error logging proctor flag:", error);
      res.status(500).json({ message: "Failed to log proctor flag" });
    }
  });



  // Working enrollment endpoints that bypass middleware conflicts
  app.get('/api/user-enrollments', async (req: any, res) => {
    try {
      console.log('Get user enrollments request:', {
        sessionId: req.sessionID,
        userId: req.session?.userId,
        hasSession: !!req.session
      });

      if (!req.session?.userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      const enrollments = await storage.getUserEnrollments(req.session.userId);
      res.json(enrollments);
    } catch (error) {
      console.error("Error fetching enrollments:", error);
      res.status(500).json({ message: "Failed to fetch enrollments" });
    }
  });

  // Direct enrollment endpoint that bypasses middleware conflicts
  app.post('/api/enroll-course', async (req: any, res) => {
    try {
      console.log('Direct enroll request:', {
        sessionId: req.sessionID,
        userId: req.session?.userId,
        body: req.body,
        hasSession: !!req.session
      });

      if (!req.session?.userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      const { courseId } = req.body;
      const enrollmentData = {
        userId: req.session.userId,
        courseId,
        status: 'active',
        progress: '0'
      };

      const enrollment = await storage.enrollUser(enrollmentData);
      
      await storage.logLearningEvent({
        userId: req.session.userId,
        courseId,
        eventType: 'course_enrollment',
        eventData: { courseId }
      });

      res.status(201).json({ 
        message: "Successfully enrolled in course",
        enrollment 
      });
    } catch (error) {
      console.error("Enrollment error:", error);
      res.status(500).json({ message: "Failed to enroll in course" });
    }
  });

  // Username/Password Auth Routes
  app.post('/api/auth/login', async (req, res) => {
    try {
      const { username, password } = loginSchema.parse(req.body);
      
      const user = await storage.getUserByUsername(username);
      if (!user || !user.passwordHash) {
        return res.status(401).json({ message: "Invalid credentials" });
      }

      const isValidPassword = await bcrypt.compare(password, user.passwordHash);
      if (!isValidPassword) {
        return res.status(401).json({ message: "Invalid credentials" });
      }

      if (!user.isActive) {
        return res.status(401).json({ message: "Account is disabled" });
      }

      // Set session
      (req.session as any).userId = user.id;
      (req.session as any).userType = user.userType;
      
      res.json({ 
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
          firstName: user.firstName,
          lastName: user.lastName,
          userType: user.userType,
          organizationId: user.organizationId,
        },
        dashboardRoute: getDashboardRoute(user.userType)
      });
    } catch (error) {
      console.error("Login error:", error);
      res.status(400).json({ message: "Invalid request" });
    }
  });

  app.post('/api/auth/register', async (req, res) => {
    try {
      const userData = registerSchema.parse(req.body);
      
      // Check if username or email already exists
      const existingUser = await storage.getUserByUsername(userData.username);
      if (existingUser) {
        return res.status(400).json({ message: "Username already exists" });
      }

      const existingEmail = await storage.getUserByEmail(userData.email);
      if (existingEmail) {
        return res.status(400).json({ message: "Email already registered" });
      }

      // Hash password
      const passwordHash = await bcrypt.hash(userData.password, 12);

      // Create user
      const user = await storage.createUser({
        id: `${userData.userType}-${Date.now()}`,
        ...userData,
        passwordHash,
      });

      res.json({ 
        message: "User created successfully",
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
          userType: user.userType,
        }
      });
    } catch (error) {
      console.error("Registration error:", error);
      res.status(400).json({ message: "Invalid request" });
    }
  });



  // Session-based auth middleware
  const requireAuth = (req: any, res: any, next: any) => {
    console.log('Session check:', {
      method: req.method,
      url: req.url,
      sessionId: req.sessionID,
      userId: req.session?.userId,
      userType: req.session?.userType,
      hasSession: !!req.session,
      headers: req.headers['cookie']
    });
    
    if (!req.session.userId) {
      console.log('Auth failed - no userId in session');
      return res.status(401).json({ message: "Unauthorized" });
    }
    next();
  };

  // Examination results endpoint (needs to be after requireAuth is defined)
  app.get('/api/examinations/:sessionId/results', requireAuth, async (req: any, res) => {
    try {
      const { sessionId } = req.params;
      const userId = req.session.userId;

      // Mock detailed results
      const results = {
        sessionId,
        examTitle: 'IBDL Strategic Leadership Certification',
        completedAt: '2024-12-15T16:30:00Z',
        totalScore: 85,
        passingScore: 80,
        passed: true,
        timeSpent: 110,
        totalTime: 120,
        questionBreakdown: [
          { category: 'Strategic Vision', score: 90, total: 100 },
          { category: 'Leadership Psychology', score: 85, total: 100 },
          { category: 'Change Management', score: 80, total: 100 },
          { category: 'Team Dynamics', score: 85, total: 100 }
        ],
        certificateId: `IBDL-CERT-${Date.now()}`,
        feedback: 'Excellent performance in strategic vision and leadership psychology. Consider reviewing change management principles for future development.'
      };

      res.json(results);
    } catch (error) {
      console.error("Error fetching examination results:", error);
      res.status(500).json({ message: "Failed to fetch examination results" });
    }
  });

  // Dashboard APIs
  app.get('/api/dashboard/learner', requireAuth, async (req, res) => {
    try {
      const userId = req.session.userId;
      const stats = await storage.getLearnerDashboardStats(userId);
      res.json(stats);
    } catch (error) {
      console.error("Dashboard error:", error);
      res.status(500).json({ message: "Failed to fetch dashboard data" });
    }
  });

  app.get('/api/dashboard/corporate', requireAuth, async (req, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (!user?.organizationId) {
        return res.status(400).json({ message: "No organization associated" });
      }
      const stats = await storage.getCorporateDashboardStats(user.organizationId);
      res.json(stats);
    } catch (error) {
      console.error("Dashboard error:", error);
      res.status(500).json({ message: "Failed to fetch dashboard data" });
    }
  });

  app.get('/api/dashboard/partner', requireAuth, async (req, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (!user?.organizationId) {
        // Auto-assign organization for demo purposes
        const orgId = 'org-partner-1';
        await storage.updateUser(user.id, { organizationId: orgId });
        const stats = await storage.getPartnerDashboardStats(orgId);
        return res.json(stats);
      }
      const stats = await storage.getPartnerDashboardStats(user.organizationId);
      res.json(stats);
    } catch (error) {
      console.error("Dashboard error:", error);
      res.status(500).json({ message: "Failed to fetch dashboard data" });
    }
  });

  app.get('/api/dashboard/admin', requireAuth, async (req, res) => {
    try {
      const stats = await storage.getSystemDashboardStats();
      res.json(stats);
    } catch (error) {
      console.error("Dashboard error:", error);
      res.status(500).json({ message: "Failed to fetch dashboard data" });
    }
  });

  // Role-based auth middleware
  const requireRole = (roles: string[]) => {
    return (req: any, res: any, next: any) => {
      if (!req.session.userId) {
        return res.status(401).json({ message: "Authentication required" });
      }
      if (!roles.includes(req.session.userType)) {
        return res.status(403).json({ message: "Insufficient permissions" });
      }
      next();
    };
  };

  // Helper function to determine dashboard route
  function getDashboardRoute(userType: string): string {
    switch (userType) {
      case 'learner':
        return '/dashboard/learner';
      case 'corporate_admin':
        return '/dashboard/corporate';
      case 'partner_admin':
        return '/dashboard/partner';
      case 'system_admin':
        return '/dashboard/admin';
      default:
        return '/dashboard/learner';
    }
  }

  // Auth routes
  app.get('/api/auth/user', requireAuth, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (!user) {
        return res.status(404).json({ message: "User not found" });
      }
      res.json({
        id: user.id,
        username: user.username,
        email: user.email,
        firstName: user.firstName,
        lastName: user.lastName,
        userType: user.userType,
        organizationId: user.organizationId,
      });
    } catch (error) {
      console.error("Error fetching user:", error);
      res.status(500).json({ message: "Failed to fetch user" });
    }
  });

  // Original Replit auth route (keep for compatibility)
  app.get('/api/auth/user-replit', isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user.claims.sub;
      const user = await storage.getUser(userId);
      res.json(user);
    } catch (error) {
      console.error("Error fetching user:", error);
      res.status(500).json({ message: "Failed to fetch user" });
    }
  });

  // Dashboard analytics routes
  app.get('/api/dashboard/learner', isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user.claims.sub;
      const stats = await storage.getLearnerDashboardStats(userId);
      const enrollments = await storage.getUserEnrollments(userId);
      const certificates = await storage.getUserCertificates(userId);
      
      res.json({
        stats,
        recentEnrollments: enrollments.slice(0, 5),
        recentCertificates: certificates.slice(0, 3),
      });
    } catch (error) {
      console.error("Error fetching learner dashboard:", error);
      res.status(500).json({ message: "Failed to fetch dashboard data" });
    }
  });

  app.get('/api/dashboard/corporate', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (!user?.organizationId || user.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied" });
      }

      const stats = await storage.getCorporateDashboardStats(user.organizationId);
      const users = await storage.getUsersByOrganization(user.organizationId);
      const courses = await storage.getCoursesByOrganization(user.organizationId);
      
      res.json({
        stats,
        users: users.slice(0, 10),
        courses: courses.slice(0, 5),
        organization: await storage.getOrganization(user.organizationId),
      });
    } catch (error) {
      console.error("Error fetching corporate dashboard:", error);
      res.status(500).json({ message: "Failed to fetch dashboard data" });
    }
  });

  app.get('/api/dashboard/partner', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (!user?.organizationId || user.userType !== 'partner_admin') {
        return res.status(403).json({ message: "Access denied" });
      }

      const stats = await storage.getPartnerDashboardStats(user.organizationId);
      const students = await storage.getUsersByOrganization(user.organizationId);
      const courses = await storage.getCoursesByOrganization(user.organizationId);
      
      res.json({
        stats,
        students: students.slice(0, 10),
        topCourses: courses.slice(0, 5),
        organization: await storage.getOrganization(user.organizationId),
      });
    } catch (error) {
      console.error("Error fetching partner dashboard:", error);
      res.status(500).json({ message: "Failed to fetch dashboard data" });
    }
  });

  app.get('/api/dashboard/admin', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'system_admin') {
        return res.status(403).json({ message: "Access denied" });
      }

      const stats = await storage.getSystemDashboardStats();
      const activeSessions = await storage.getActiveExaminationSessions();
      
      res.json({
        stats,
        activeSessions: activeSessions.slice(0, 10),
        systemHealth: {
          database: "optimal",
          apiResponse: "145ms",
          storage: "78%",
          security: "all_clear"
        }
      });
    } catch (error) {
      console.error("Error fetching admin dashboard:", error);
      res.status(500).json({ message: "Failed to fetch dashboard data" });
    }
  });

  // Advanced course search with filters
  app.get('/api/courses/search', requireAuth, async (req: any, res) => {
    try {
      const { q, category, difficulty, duration, tags, sort } = req.query;
      
      // Get all courses first
      const allCourses = await getAllSampleCourses();
      let filteredCourses = [...allCourses];

      // Apply search query filter
      if (q && typeof q === 'string') {
        const searchTerm = q.toLowerCase();
        filteredCourses = filteredCourses.filter(course => 
          course.title.toLowerCase().includes(searchTerm) ||
          course.description.toLowerCase().includes(searchTerm) ||
          course.category.toLowerCase().includes(searchTerm) ||
          (course.tags && course.tags.some(tag => tag.toLowerCase().includes(searchTerm)))
        );
      }

      // Apply category filter
      if (category && category !== 'all' && typeof category === 'string') {
        filteredCourses = filteredCourses.filter(course => 
          course.category.toLowerCase().includes(category.toLowerCase())
        );
      }

      // Apply difficulty filter
      if (difficulty && difficulty !== 'all' && typeof difficulty === 'string') {
        filteredCourses = filteredCourses.filter(course => 
          course.difficulty.toLowerCase() === difficulty.toLowerCase()
        );
      }

      // Apply duration filter
      if (duration && duration !== 'all' && typeof duration === 'string') {
        filteredCourses = filteredCourses.filter(course => {
          switch (duration) {
            case 'short': return course.duration <= 240; // 4 hours or less
            case 'medium': return course.duration > 240 && course.duration <= 480; // 4-8 hours
            case 'long': return course.duration > 480; // More than 8 hours
            default: return true;
          }
        });
      }

      // Apply tags filter
      if (tags && typeof tags === 'string') {
        const searchTags = tags.toLowerCase().split(',').map(tag => tag.trim());
        filteredCourses = filteredCourses.filter(course => 
          course.tags && searchTags.some(searchTag => 
            course.tags.some(courseTag => courseTag.toLowerCase().includes(searchTag))
          )
        );
      }

      // Apply sorting
      if (sort && typeof sort === 'string') {
        switch (sort) {
          case 'title':
            filteredCourses.sort((a, b) => a.title.localeCompare(b.title));
            break;
          case 'rating':
            filteredCourses.sort((a, b) => b.rating - a.rating);
            break;
          case 'duration':
            filteredCourses.sort((a, b) => a.duration - b.duration);
            break;
          case 'students':
            filteredCourses.sort((a, b) => b.studentsEnrolled - a.studentsEnrolled);
            break;
          case 'newest':
            filteredCourses.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());
            break;
          default:
            // Default sort by rating
            filteredCourses.sort((a, b) => b.rating - a.rating);
        }
      }

      res.json(filteredCourses);
    } catch (error) {
      console.error("Error searching courses:", error);
      res.status(500).json({ message: "Failed to search courses" });
    }
  });

  // Helper function to get all sample courses
  async function getAllSampleCourses() {
    return [
        {
          id: 'course-1',
          title: 'Business Analytics Fundamentals',
          description: 'Learn the core principles of business analytics, data visualization, and decision-making frameworks used by leading organizations.',
          category: 'Business Analytics',
          difficulty: 'beginner',
          duration: 480,
          studentsEnrolled: 1247,
          rating: 4.8,
          price: 99,
          content: 'SCORM package with interactive modules',
          tags: ['analytics', 'data', 'business'],
          isActive: true,
          organizationId: null,
          createdBy: 'system',
          createdAt: new Date('2024-01-15'),
          updatedAt: new Date('2024-12-01')
        },
        {
          id: 'course-2',
          title: 'Strategic Leadership Mastery',
          description: 'Master the psychological principles of effective leadership, team dynamics, and organizational behavior.',
          category: 'Leadership Development',
          difficulty: 'intermediate',
          duration: 720,
          studentsEnrolled: 892,
          rating: 4.9,
          price: 149,
          content: 'Interactive video content with assessments',
          tags: ['leadership', 'psychology', 'management'],
          isActive: true,
          organizationId: null,
          createdBy: 'system',
          createdAt: new Date('2024-02-20'),
          updatedAt: new Date('2024-11-15')
        },
        {
          id: 'course-3',
          title: 'Financial Analysis & Modeling',
          description: 'Comprehensive financial analysis training with advanced modeling techniques and real-world case studies.',
          category: 'Finance',
          difficulty: 'advanced',
          duration: 960,
          studentsEnrolled: 634,
          rating: 4.7,
          price: 199,
          content: 'Excel-based modeling exercises and simulations',
          tags: ['finance', 'modeling', 'analysis'],
          isActive: true,
          organizationId: null,
          createdBy: 'system',
          createdAt: new Date('2024-03-10'),
          updatedAt: new Date('2024-12-15')
        },
        {
          id: 'course-4',
          title: 'Digital Marketing Excellence',
          description: 'Complete digital marketing strategy covering SEO, social media, content marketing, and analytics.',
          category: 'Marketing',
          difficulty: 'intermediate',
          duration: 600,
          studentsEnrolled: 1156,
          rating: 4.6,
          price: 129,
          content: 'Campaign simulations and practical exercises',
          tags: ['marketing', 'digital', 'seo', 'social media'],
          isActive: true,
          organizationId: null,
          createdBy: 'system',
          createdAt: new Date('2024-04-05'),
          updatedAt: new Date('2024-12-10')
        }
      ];
    }

  // Course management routes
  app.get('/api/courses', requireAuth, async (req: any, res) => {
    try {
      const sampleCourses = await getAllSampleCourses();
      res.json(sampleCourses);
    } catch (error) {
      console.error("Error fetching courses:", error);
      res.status(500).json({ message: "Failed to fetch courses" });
    }
  });

  app.get('/api/courses/search', requireAuth, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      const { q: search, category } = req.query;

      // Create sample courses with proper structure
      const sampleCourses = [
        {
          id: '1',
          title: 'Business Analytics Fundamentals',
          description: 'Learn the core principles of business analytics, data visualization, and decision-making frameworks used by leading organizations.',
          category: 'Business Analytics',
          difficulty: 'beginner',
          duration: 480, // 8 hours
          studentsEnrolled: 1247,
          rating: 4.8,
          price: 99,
          tags: ['analytics', 'data', 'business'],
          isActive: true,
          createdAt: new Date('2024-01-15'),
          updatedAt: new Date('2024-12-01')
        },
        {
          id: '2',
          title: 'Leadership Psychology & Development',
          description: 'Master the psychological principles of effective leadership, team dynamics, and organizational behavior.',
          category: 'Leadership Development',
          difficulty: 'intermediate',
          duration: 720, // 12 hours
          studentsEnrolled: 892,
          rating: 4.9,
          price: 149,
          tags: ['leadership', 'psychology', 'management'],
          isActive: true,
          createdAt: new Date('2024-02-20'),
          updatedAt: new Date('2024-11-15')
        },
        {
          id: '3',
          title: 'Advanced Project Management (PMP Prep)',
          description: 'Comprehensive project management training with PMP certification preparation and real-world case studies.',
          category: 'Project Management',
          difficulty: 'advanced',
          duration: 960, // 16 hours
          studentsEnrolled: 634,
          rating: 4.7,
          price: 199,
          tags: ['project management', 'pmp', 'certification'],
          isActive: true,
          createdAt: new Date('2024-03-10'),
          updatedAt: new Date('2024-12-20')
        },
        {
          id: '4',
          title: 'Digital Marketing Strategy',
          description: 'Modern digital marketing techniques including SEO, social media, content marketing, and analytics.',
          category: 'Digital Marketing',
          difficulty: 'intermediate',
          duration: 600, // 10 hours
          studentsEnrolled: 1543,
          rating: 4.6,
          price: 129,
          tags: ['marketing', 'digital', 'seo', 'social media'],
          isActive: true,
          createdAt: new Date('2024-04-05'),
          updatedAt: new Date('2024-12-10')
        },
        {
          id: '5',
          title: 'Data Science for Business Professionals',
          description: 'Learn data science fundamentals, statistical analysis, and machine learning applications for business.',
          category: 'Data Science',
          difficulty: 'advanced',
          duration: 840, // 14 hours
          studentsEnrolled: 756,
          rating: 4.9,
          price: 249,
          tags: ['data science', 'machine learning', 'statistics'],
          isActive: true,
          createdAt: new Date('2024-05-12'),
          updatedAt: new Date('2024-12-05')
        },
        {
          id: '6',
          title: 'Professional Communication Skills',
          description: 'Enhance your professional communication, presentation skills, and workplace collaboration.',
          category: 'Professional Development',
          difficulty: 'beginner',
          duration: 360, // 6 hours
          studentsEnrolled: 2156,
          rating: 4.5,
          price: 79,
          tags: ['communication', 'presentation', 'soft skills'],
          isActive: true,
          createdAt: new Date('2024-06-18'),
          updatedAt: new Date('2024-11-30')
        }
      ];

      let filteredCourses = sampleCourses;

      // Apply search filter
      if (search) {
        const searchLower = search.toLowerCase();
        filteredCourses = filteredCourses.filter(course => 
          course.title.toLowerCase().includes(searchLower) ||
          course.description.toLowerCase().includes(searchLower) ||
          course.category.toLowerCase().includes(searchLower)
        );
      }

      // Apply category filter
      if (category) {
        filteredCourses = filteredCourses.filter(course => 
          course.category === category
        );
      }

      res.json(filteredCourses);
    } catch (error) {
      console.error("Error searching courses:", error);
      res.status(500).json({ message: "Failed to search courses" });
    }
  });

  app.get('/api/courses/:courseId', requireAuth, async (req: any, res) => {
    try {
      const { courseId } = req.params;
      
      // Sample course detail data
      const courseDetails = {
        '1': {
          id: '1',
          title: 'Business Analytics Fundamentals',
          description: 'This comprehensive course covers the essential concepts and tools needed to make data-driven business decisions. You\'ll learn statistical analysis, data visualization techniques, and how to interpret business metrics effectively. The course includes hands-on exercises with real business scenarios and case studies from Fortune 500 companies.',
          category: 'Business Analytics',
          difficulty: 'beginner',
          duration: 480,
          studentsEnrolled: 1247,
          rating: 4.8,
          price: 99,
          content: 'SCORM package with interactive modules',
          tags: ['analytics', 'data', 'business'],
          isActive: true,
          createdAt: new Date('2024-01-15'),
          updatedAt: new Date('2024-12-01')
        },
        '2': {
          id: '2',
          title: 'Leadership Psychology & Development',
          description: 'Explore the psychological foundations of effective leadership through evidence-based approaches. This course combines theoretical knowledge with practical applications, covering emotional intelligence, team dynamics, conflict resolution, and organizational psychology.',
          category: 'Leadership Development',
          difficulty: 'intermediate',
          duration: 720,
          studentsEnrolled: 892,
          rating: 4.9,
          price: 149,
          content: 'Interactive video content with assessments',
          tags: ['leadership', 'psychology', 'management'],
          isActive: true,
          createdAt: new Date('2024-02-20'),
          updatedAt: new Date('2024-11-15')
        }
      };

      const course = courseDetails[courseId as keyof typeof courseDetails];
      if (!course) {
        return res.status(404).json({ message: "Course not found" });
      }

      res.json(course);
    } catch (error) {
      console.error("Error fetching course:", error);
      res.status(500).json({ message: "Failed to fetch course" });
    }
  });

  // Course modules endpoint with direct session check
  app.get('/api/courses/:courseId/modules', async (req: any, res) => {
    try {
      console.log('Course modules request:', {
        sessionId: req.sessionID,
        userId: req.session?.userId,
        hasSession: !!req.session,
        courseId: req.params.courseId
      });

      if (!req.session?.userId) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const { courseId } = req.params;
      
      // Comprehensive course modules with rich interactive content
      // Map course IDs to module data keys
      const courseIdMapping: Record<string, string> = {
        'course-1': '1',
        'course-2': '2',
        '1': '1',
        '2': '2'
      };

      const moduleData = {
        '1': [
          {
            id: 'module-1-1',
            title: 'Introduction to Business Analytics',
            description: 'Master the fundamentals of business analytics and its strategic applications in modern organizations',
            duration: 90,
            completed: false,
            order: 1,
            content: [
              {
                id: 'content-1-1-1',
                type: 'video',
                title: 'What is Business Analytics?',
                duration: 15,
                videoUrl: 'https://www.youtube.com/embed/VR_H9tBuGr8',
                description: 'Comprehensive introduction to business analytics concepts, frameworks, and real-world applications across industries',
                transcript: 'Welcome to Business Analytics Fundamentals. In this video, we explore how data-driven decision making transforms modern businesses. We\'ll cover the analytics value chain, from data collection through actionable insights, and examine case studies from companies like Netflix, Amazon, and Google who leverage analytics for competitive advantage. You\'ll learn the five-stage analytics process: data collection, processing, analysis, insight generation, and decision implementation.'
              },
              {
                id: 'content-1-1-2',
                type: 'reading',
                title: 'The Analytics Value Chain: From Data to Decisions',
                estimatedTime: 25,
                content: 'The Analytics Value Chain transforms raw data into competitive advantage through five critical stages. Stage 1: Data Collection & Acquisition includes internal sources like CRM systems and external sources like market research. Stage 2: Data Processing & Preparation involves cleaning, transformation, and integration. Stage 3: Advanced Analysis & Modeling covers descriptive, diagnostic, predictive, and prescriptive analytics. Stage 4: Insight Generation & Business Intelligence focuses on pattern recognition and visualization. Stage 5: Decision Making & Implementation translates insights into business action. Organizations that master this value chain achieve 23% faster revenue growth and 19% higher profitability than competitors.'
              },
              {
                id: 'content-1-1-3',
                type: 'quiz',
                title: 'Analytics Fundamentals Knowledge Check',
                questions: [
                  {
                    id: 'q1',
                    question: 'According to the Netflix case study, what percentage of watched content comes from their recommendation algorithm?',
                    options: ['65%', '75%', '80%', '90%'],
                    correctAnswer: 2,
                    explanation: 'Netflix\'s recommendation algorithm drives 80% of content consumption, demonstrating the power of personalized analytics.'
                  },
                  {
                    id: 'q2',
                    question: 'What percentage of time do data scientists typically spend on data preparation?',
                    options: ['20-30%', '40-50%', '60-80%', '90-95%'],
                    correctAnswer: 2,
                    explanation: 'Data preparation is the most time-consuming phase, requiring 60-80% of effort, highlighting the importance of data quality.'
                  },
                  {
                    id: 'q3',
                    question: 'Organizations that master analytics achieve what percentage faster revenue growth?',
                    options: ['15%', '23%', '35%', '50%'],
                    correctAnswer: 1,
                    explanation: 'Research shows analytics leaders achieve 23% faster revenue growth and 19% higher profitability than competitors.'
                  }
                ]
              },
              {
                id: 'content-1-1-4',
                type: 'simulation',
                title: 'Analytics Strategy Consultant Challenge',
                description: 'You are hired as an analytics consultant for TechCorp, a mid-size software company struggling with data-driven decision making.',
                scenario: 'TechCorp manufactures enterprise software with $200M annual revenue, 2,000 employees, and operates in 15 countries. They have limited analytics capabilities and want to compete with data-driven competitors. Current State: Scattered data across 20+ systems, manual reporting taking 2-3 weeks, no predictive capabilities, limited data governance, business decisions based on intuition. CEO Goal: "Become a data-driven organization within 12 months"',
                challenges: [
                  'Analytics Maturity Assessment: Evaluate current capabilities across data foundation, analytical capabilities, organizational readiness, and technology infrastructure',
                  'Quick Wins Identification: Identify 3-5 analytics initiatives that can deliver immediate value within 90 days',
                  'Technology Architecture Design: Design a scalable analytics architecture that integrates current systems',
                  'Change Management Strategy: Develop a plan to transform culture from intuition-based to data-driven decision making'
                ]
              }
            ]
          },
          {
            id: 'module-1-2',
            title: 'Data Visualization & Business Intelligence',
            description: 'Master advanced data visualization techniques and business intelligence implementation for executive decision making',
            duration: 120,
            completed: false,
            order: 2,
            content: [
              {
                id: 'content-1-2-1',
                type: 'video',
                title: 'Data Visualization Psychology: How the Brain Processes Visual Information',
                duration: 18,
                videoUrl: 'https://www.youtube.com/embed/AdSZJzb-aX8',
                description: 'Scientific principles of effective data visualization based on cognitive psychology and visual perception research',
                transcript: 'Our brains process visual information 60,000 times faster than text. Understanding how human visual perception works is crucial for creating effective data visualizations that drive business decisions. In this video, we explore the psychology behind data visualization, including pre-attentive processing, cognitive load theory, and visual hierarchy principles. You\'ll learn how to design dashboards that executives can understand in 23 seconds or less.'
              },
              {
                id: 'content-1-2-2',
                type: 'interactive',
                title: 'Chart Type Selection Mastery Tool',
                description: 'Master the art of selecting the most effective chart types for different business scenarios through hands-on practice',
                exercises: [
                  {
                    scenario: 'CEO wants to understand quarterly revenue performance across 4 product lines over the past 3 years to identify growth trends and seasonal patterns',
                    options: ['Stacked Bar Chart', 'Multi-line Chart', 'Area Chart', 'Pie Chart'],
                    correctAnswer: 1,
                    explanation: 'Multi-line charts excel at showing trends over time for multiple series, making it easy to compare product line performance and identify seasonal patterns. Each line represents a product, and time progression is clear.'
                  },
                  {
                    scenario: 'Marketing team needs to demonstrate the relationship between advertising spend and lead generation across 50 different campaigns',
                    options: ['Bar Chart', 'Scatter Plot', 'Line Chart', 'Histogram'],
                    correctAnswer: 1,
                    explanation: 'Scatter plots are perfect for revealing correlations between two continuous variables. Each point represents a campaign, with spend on X-axis and leads on Y-axis.'
                  },
                  {
                    scenario: 'Board presentation showing market share breakdown among 6 major competitors',
                    options: ['Line Chart', 'Pie Chart', 'Treemap', 'Radar Chart'],
                    correctAnswer: 1,
                    explanation: 'Pie charts effectively show parts of a whole when you have a manageable number of segments. Market share is a classic pie chart use case.'
                  }
                ]
              },
              {
                id: 'content-1-2-3',
                type: 'case_study',
                title: 'Tableau at Charles Schwab: $100M Analytics Investment Success Story',
                content: 'Charles Schwab invested $100M in analytics transformation, implementing Tableau Server with 10,000+ user licenses. The initiative addressed data silos across 50+ legacy systems, reporting delays of 3-4 weeks, and limited self-service capabilities. Key implementations included executive performance dashboards, financial advisor workstations, risk management command centers, and client self-service portals. Results: $120M annual cost savings, $85M additional revenue, 205% ROI over 24 months, 78% client portal utilization, and 35% reduction in call center volume. Success factors included executive sponsorship, user-centric design, data excellence, and comprehensive security compliance.'
              }
            ]
          },
          {
            id: 'module-1-3',
            title: 'Statistical Analysis for Business Decisions',
            description: 'Master statistical methods and their practical application to solve complex business problems and drive evidence-based decision making',
            duration: 150,
            completed: false,
            order: 3,
            content: [
              {
                id: 'content-1-3-1',
                type: 'reading',
                title: 'Business Statistics Mastery: From Data to Strategic Decisions',
                estimatedTime: 35,
                content: 'Statistical analysis forms the foundation of evidence-based business decisions. Companies that leverage statistical methods effectively achieve 126% higher profit margins and are 5x more likely to make faster decisions than competitors who rely on intuition alone. Key concepts include descriptive statistics (mean, median, mode, standard deviation), inferential statistics (hypothesis testing, confidence intervals), and advanced methods (regression analysis, time series forecasting). Applications span A/B testing for marketing optimization, quality control in manufacturing, risk assessment in finance, and customer analytics in retail.'
              },
              {
                id: 'content-1-3-2',
                type: 'video',
                title: 'Mastering A/B Testing for Business Growth',
                duration: 24,
                videoUrl: 'https://www.youtube.com/embed/BhY-un6LQSU',
                description: 'Comprehensive guide to designing, executing, and interpreting A/B tests that drive measurable business results',
                transcript: 'A/B testing is the gold standard for making data-driven business decisions. When implemented correctly, A/B testing can increase conversion rates by 20-30% and provide clear ROI measurement for business initiatives. This video covers the complete A/B testing process: hypothesis formation, sample size calculation, randomization strategies, statistical significance testing, and business interpretation of results. You\'ll learn how companies like Netflix, Amazon, and Google use A/B testing to optimize everything from user interfaces to pricing strategies.'
              },
              {
                id: 'content-1-3-3',
                type: 'practice',
                title: 'Business Statistics Mastery Workbook',
                description: 'Comprehensive practice problems covering real business scenarios with step-by-step solutions',
                exercises: [
                  {
                    title: 'Sales Performance Analysis',
                    scenario: 'Regional sales manager analyzing team performance across 12 representatives with monthly quotas of $150,000',
                    data: 'Actual sales: $165K, $142K, $178K, $159K, $148K, $171K, $138K, $182K, $156K, $163K, $149K, $175K',
                    questions: [
                      'Calculate mean, median, and standard deviation of sales performance',
                      'What percentage of representatives exceeded quota?',
                      'Identify top and bottom 25% performers',
                      'Calculate coefficient of variation to assess team consistency',
                      'Recommend performance improvement strategies based on analysis'
                    ],
                    solutions: [
                      'Mean: $160,500, Median: $159,500, Standard Deviation: $15,420',
                      '75% of representatives (9 out of 12) exceeded quota',
                      'Top 25%: >$171,000, Bottom 25%: <$148,000',
                      'Coefficient of Variation: 9.6% (moderate consistency)',
                      'Focus training on bottom quartile, share best practices from top performers'
                    ]
                  },
                  {
                    title: 'A/B Testing Email Campaign',
                    scenario: 'Marketing team testing two email subject line variations for open rate improvement',
                    data: 'Version A: 18.2% open rate (10,000 emails), Version B: 19.8% open rate (10,000 emails)',
                    questions: [
                      'Is the difference statistically significant at 95% confidence?',
                      'What is the practical business impact?',
                      'Calculate confidence interval for the difference',
                      'What sample size would be needed to detect a 1% improvement?'
                    ]
                  }
                ]
              }
            ]
          },
          {
            id: 'module-1-4',
            title: 'Advanced Analytics & Predictive Modeling',
            description: 'Master machine learning applications, predictive analytics, and AI-driven business intelligence for competitive advantage',
            duration: 180,
            completed: false,
            order: 4,
            content: [
              {
                id: 'content-1-4-1',
                type: 'video',
                title: 'Machine Learning for Business Leaders: Strategy and Implementation',
                duration: 28,
                videoUrl: 'https://www.youtube.com/embed/0ONr9ToYs_c',
                description: 'Executive guide to machine learning applications, ROI assessment, and organizational transformation for AI-driven businesses',
                transcript: 'Machine learning is transforming how businesses operate, compete, and create value. Companies implementing ML effectively see 15% increase in productivity and 25% improvement in decision-making speed. This comprehensive guide covers ML applications across industries: recommendation systems in e-commerce, predictive maintenance in manufacturing, fraud detection in finance, and customer segmentation in retail. You\'ll learn how to evaluate ML opportunities, calculate ROI, and build organizational capabilities for AI-driven success.'
              },
              {
                id: 'content-1-4-2',
                type: 'case_study',
                title: 'Amazon\'s $35 Billion Recommendation Engine: Architectural Deep Dive',
                content: 'Amazon\'s recommendation system generates over $35 billion in annual revenue, representing 35% of total sales. This comprehensive analysis examines the technical architecture, business strategy, and competitive advantages that make Amazon\'s recommendation engine the most successful commercial AI application in history. The system combines collaborative filtering, content-based filtering, and deep learning algorithms to process 20+ million events per second. Key components include multi-algorithm recommendation framework using matrix factorization and neural networks, real-time processing infrastructure with Apache Kafka and Spark, and comprehensive data management across petabytes of customer interaction data. Business impact includes 29% increase in conversion rates, 60% improvement in customer lifetime value, and creation of significant competitive moats through data network effects.'
              },
              {
                id: 'content-1-4-3',
                type: 'simulation',
                title: 'Predictive Analytics Implementation Challenge',
                description: 'Design and implement a predictive analytics solution for customer churn reduction',
                scenario: 'You are the Chief Data Officer at RetailCorp, a subscription-based e-commerce company with 1M customers and 5% monthly churn rate. The CEO wants to reduce churn by 25% using predictive analytics.',
                challenges: [
                  'Data Strategy: Identify relevant data sources and features for churn prediction',
                  'Model Development: Choose appropriate algorithms and evaluation metrics',
                  'Business Integration: Design intervention strategies for at-risk customers',
                  'ROI Calculation: Quantify business impact and return on investment'
                ],
                deliverables: [
                  'Technical architecture for churn prediction system',
                  'Model performance evaluation and comparison',
                  'Customer intervention workflow and automation',
                  'Business case with projected ROI and timeline'
                ]
              }
            ]
          }
        ],
        '2': [
          {
            id: 'module-2-1',
            title: 'Leadership Psychology Foundations',
            description: 'Master the psychological principles that drive effective leadership, team performance, and organizational success',
            duration: 120,
            completed: false,
            order: 1,
            content: [
              {
                id: 'content-2-1-1',
                type: 'video',
                title: 'The Neuroscience of Leadership: How the Brain Drives Success',
                duration: 22,
                videoUrl: 'https://www.youtube.com/embed/o8B1EfcUisY',
                description: 'Cutting-edge neuroscience research revealing how brain function impacts leadership effectiveness and decision-making',
                transcript: 'Recent neuroscience research has revolutionized our understanding of leadership. The prefrontal cortex, responsible for executive function, shows distinct patterns in effective leaders. Understanding these neural pathways helps develop better leadership capabilities. This video explores how the brain processes decisions under pressure, the neuroscience of emotional intelligence, and practical techniques for optimizing cognitive performance. You\'ll learn about mirror neurons and their role in empathy, the amygdala hijack phenomenon, and strategies for maintaining rational thinking during crisis situations.'
              },
              {
                id: 'content-2-1-2',
                type: 'reading',
                title: 'Emotional Intelligence Mastery: The Science of Leadership Excellence',
                estimatedTime: 40,
                content: 'Emotional Intelligence (EI) has emerged as the single most important predictor of leadership success, outweighing IQ and technical skills by a factor of 2:1. Research across 500,000 professionals reveals that 90% of top performers possess high emotional intelligence, while it accounts for $1,300 increase in annual salary for every point of EI improvement. The four domains of EI include: Self-Awareness (emotional self-awareness, accurate self-assessment, self-confidence), Self-Management (emotional self-control, adaptability, achievement orientation, positive outlook), Social Awareness (empathy, organizational awareness, service orientation), and Relationship Management (influence, coach and mentor, conflict management, team leadership, inspirational leadership). Companies with emotionally intelligent leaders achieve 20% better business results and experience 50% less team turnover.'
              },
              {
                id: 'content-2-1-3',
                type: 'assessment',
                title: 'Comprehensive Leadership Psychology Evaluation',
                description: 'Advanced assessment of leadership psychology principles, emotional intelligence competencies, and practical application scenarios',
                questions: [
                  {
                    question: 'According to neuroscience research, what percentage of top performers possess high emotional intelligence?',
                    options: ['70%', '80%', '90%', '95%'],
                    correctAnswer: 2,
                    explanation: 'Research across 500,000 professionals shows that 90% of top performers have high emotional intelligence, demonstrating its critical importance for success.'
                  },
                  {
                    question: 'Which brain region is primarily responsible for emotional regulation in leadership contexts?',
                    options: ['Amygdala', 'Hippocampus', 'Prefrontal Cortex', 'Limbic System'],
                    correctAnswer: 2,
                    explanation: 'The prefrontal cortex controls executive functions including emotional regulation, strategic thinking, and decision-making under pressure.'
                  },
                  {
                    question: 'What is the approximate duration of the initial neurochemical trigger of emotions?',
                    options: ['3 seconds', '6 seconds', '10 seconds', '15 seconds'],
                    correctAnswer: 1,
                    explanation: 'The 6-second rule is based on research showing that initial emotional neurochemical reactions last about 6 seconds, allowing time for rational response.'
                  }
                ]
              },
              {
                id: 'content-2-1-4',
                type: 'simulation',
                title: 'Executive Leadership Crisis Simulation',
                description: 'Navigate complex leadership scenarios requiring advanced emotional intelligence and psychological insight',
                scenario: 'You are the CEO of TechInnovate, a 5,000-employee software company facing a major crisis. A critical security breach has exposed customer data, media scrutiny is intense, employee morale is plummeting, and board members are questioning your leadership. You must navigate this crisis while maintaining team cohesion and stakeholder confidence.',
                challenges: [
                  'Immediate Response Strategy: Design your communication strategy for the first 24 hours, prioritize stakeholder communications (customers, employees, media, board), plan your personal emotional regulation strategy during high-pressure interactions',
                  'Team Leadership Under Pressure: Develop a plan to support team emotional wellness during the crisis, design decision-making processes that include emotional considerations, create strategies to maintain team collaboration under stress',
                  'Stakeholder Relationship Management: Craft emotionally intelligent responses to different stakeholder groups, balance transparency with legal considerations, design a long-term trust rebuilding strategy'
                ],
                evaluationCriteria: [
                  'Emotional self-awareness and regulation under extreme pressure',
                  'Empathy and understanding of diverse stakeholder emotional states',
                  'Strategic use of emotional intelligence in communications',
                  'Team support and leadership during crisis',
                  'Long-term relationship preservation and trust rebuilding'
                ]
              }
            ]
          },
          {
            id: 'module-2-2',
            title: 'Advanced Team Dynamics & Motivation',
            description: 'Build and lead high-performance teams using psychological principles and evidence-based practices',
            duration: 130,
            completed: false,
            order: 2,
            content: [
              {
                id: 'content-2-2-1',
                type: 'video',
                title: 'Building Psychological Safety in Teams',
                duration: 20,
                videoUrl: 'https://www.youtube.com/embed/LhoLuui9gX8',
                description: 'Google\'s Project Aristotle findings on team effectiveness and creating environments for innovation',
                transcript: 'Google\'s research across 180 teams revealed that psychological safety is the #1 factor in team effectiveness, more important than individual talent or resources. Psychological safety means team members feel safe to take risks, make mistakes, ask questions, and offer ideas without fear of negative consequences. This video explores the five key dynamics of effective teams: psychological safety, dependability, structure and clarity, meaning, and impact. You\'ll learn practical strategies for creating psychological safety, handling team conflicts constructively, and fostering innovation through trust-based leadership.'
              },
              {
                id: 'content-2-2-2',
                type: 'case_study',
                title: 'Microsoft\'s Cultural Transformation Under Satya Nadella',
                content: 'When Satya Nadella became CEO in 2014, Microsoft transformed from a "know-it-all" to "learn-it-all" culture, fundamentally changing how teams collaborate and innovate. Key strategies included empathy-driven leadership, growth mindset adoption, and collaborative decision-making. Nadella emphasized psychological safety, encouraging employees to admit mistakes and learn from failures. The transformation involved restructuring performance reviews to focus on learning and growth rather than competition, implementing inclusive hiring practices, and creating cross-functional teams that break down silos. Results included 600% stock price increase, dramatic improvement in employee satisfaction scores, renewed innovation leadership in cloud computing and AI, and successful cultural shift from internal competition to collaboration. The transformation demonstrates how leadership psychology principles can drive organizational change at scale.'
              },
              {
                id: 'content-2-2-3',
                type: 'interactive',
                title: 'Team Motivation Diagnostic Tool',
                description: 'Analyze team motivation factors using Herzberg\'s Two-Factor Theory and Self-Determination Theory',
                exercises: [
                  {
                    scenario: 'High-performing development team showing signs of burnout and decreased motivation despite recent project successes',
                    symptoms: 'Increased absenteeism, declining code quality, reduced participation in meetings, resistance to new projects',
                    analysisFramework: 'Self-Determination Theory (Autonomy, Mastery, Purpose) and Herzberg\'s Motivation-Hygiene Theory',
                    factors: ['Autonomy: Degree of control over work methods and decisions', 'Mastery: Opportunities for skill development and growth', 'Purpose: Connection to meaningful outcomes and impact', 'Recognition: Acknowledgment of contributions and achievements', 'Growth: Clear career progression and learning opportunities'],
                    tasks: [
                      'Identify which motivational factors are missing or weakened',
                      'Design specific intervention strategies for each identified gap',
                      'Create measurement plan to track motivation improvement',
                      'Develop communication strategy to engage team in solution design'
                    ]
                  }
                ]
              }
            ]
          },
          {
            id: 'module-2-3',
            title: 'Strategic Communication & Influence',
            description: 'Master persuasive communication, stakeholder management, and organizational influence using psychological principles',
            duration: 110,
            completed: false,
            order: 3,
            content: [
              {
                id: 'content-2-3-1',
                type: 'reading',
                title: 'The Psychology of Persuasion in Leadership',
                estimatedTime: 30,
                content: 'Robert Cialdini\'s principles of influence - reciprocity, commitment, social proof, authority, liking, and scarcity - form the foundation of ethical persuasion in leadership contexts. Leaders who master these principles are 5x more likely to achieve their objectives while maintaining trust and relationships. Reciprocity involves giving before asking and creating mutually beneficial exchanges. Commitment leverages the human tendency to align actions with public commitments. Social proof uses others\' behavior to guide decisions. Authority combines expertise with credible communication. Liking builds on similarity, compliments, and cooperation. Scarcity highlights unique value and limited opportunities. Effective leaders combine multiple influence techniques while maintaining ethical standards and authentic relationships.'
              },
              {
                id: 'content-2-3-2',
                type: 'video',
                title: 'Executive Presence and Authentic Authority',
                duration: 18,
                videoUrl: 'https://www.youtube.com/embed/8S0FDjFBj8o',
                description: 'Developing commanding presence that inspires confidence and drives results while maintaining authenticity',
                transcript: 'Executive presence is the ability to connect authentically with others in a way that motivates and inspires action. It combines gravitas (confidence and composure under pressure), communication excellence (clear and compelling messaging), and appearance (professional image that supports credibility). This video explores the psychology of authority and presence, including how body language, vocal tonality, and communication patterns influence perception. You\'ll learn practical techniques for developing executive presence: managing nervous energy, structuring persuasive messages, handling difficult questions, and projecting confidence without arrogance.'
              },
              {
                id: 'content-2-3-3',
                type: 'practice',
                title: 'Stakeholder Influence Scenarios',
                description: 'Practice influence techniques across different stakeholder groups and organizational contexts',
                scenarios: [
                  {
                    title: 'Board Presentation for Major Investment',
                    context: 'Convince skeptical board members to approve $50M technology investment with uncertain ROI timeline',
                    stakeholders: 'Conservative investors focused on short-term returns, tech-savvy members excited about innovation, risk-averse CFO',
                    challenges: 'Limited financial data, competitive pressure, internal resource constraints',
                    influenceTechniques: ['Rational persuasion with compelling business case', 'Social proof from industry leaders and competitors', 'Authority through expert advisors and technical validation', 'Reciprocity by addressing individual board member concerns'],
                    tasks: [
                      'Develop messaging strategy for each stakeholder type',
                      'Create visual presentation that supports rational persuasion',
                      'Prepare responses to likely objections and concerns',
                      'Design follow-up strategy to maintain momentum'
                    ]
                  },
                  {
                    title: 'Cross-functional Team Alignment',
                    context: 'Unite competing departments (Sales, Marketing, Product, Engineering) around shared strategic objective',
                    tension: 'Each department has different priorities, success metrics, and resource constraints',
                    objective: 'Launch integrated customer experience platform requiring collaboration across all functions',
                    influenceTechniques: ['Consultation through collaborative planning sessions', 'Inspirational appeals connecting to customer impact', 'Coalition building with influential team members', 'Commitment through public goal-setting'],
                    deliverables: [
                      'Stakeholder analysis with motivation and concern mapping',
                      'Influence strategy tailored to each department',
                      'Communication plan for ongoing alignment',
                      'Success metrics that incentivize collaboration'
                    ]
                  }
                ]
              }
            ]
          },
          {
            id: 'module-2-4',
            title: 'Organizational Change & Transformation',
            description: 'Lead successful organizational change using psychology principles and change management frameworks',
            duration: 140,
            completed: false,
            order: 4,
            content: [
              {
                id: 'content-2-4-1',
                type: 'video',
                title: 'Leading Through Uncertainty and Change',
                duration: 25,
                videoUrl: 'https://www.youtube.com/embed/f84n5oFoZBc',
                description: 'Navigate organizational change while maintaining performance and morale using psychological principles',
                transcript: 'Change is the only constant in business today. Leaders who excel at change management create organizations that thrive in uncertainty rather than merely survive it. This video explores the psychology of change resistance, including loss aversion, cognitive biases, and emotional responses to uncertainty. You\'ll learn Kotter\'s 8-step change process, the psychology of change adoption curves, and practical strategies for building change resilience. Key topics include creating urgency without panic, building coalition support, communicating vision effectively, empowering action, generating wins, sustaining momentum, and anchoring changes in culture.'
              },
              {
                id: 'content-2-4-2',
                type: 'case_study',
                title: 'IBM\'s Digital Transformation: From Hardware to Cloud',
                content: 'IBM\'s transformation from hardware company to cloud services leader required massive cultural and operational change under CEO Ginni Rometty\'s leadership. The transformation involved systematic workforce reskilling (100,000+ employees), cultural change programs emphasizing continuous learning, strategic acquisitions ($30B+ invested), and fundamental business model shift from products to services. Key psychological principles applied included creating urgency through competitive analysis, building coalitions with influential leaders, developing compelling vision of AI-powered future, empowering employees through new skills training, generating short-term wins through client successes, and sustaining change through performance metrics alignment. Results included successful pivot to hybrid cloud and AI leadership, $20B+ annual recurring revenue, and recognition as transformation leader. The case demonstrates how large-scale change requires both strategic vision and deep understanding of human psychology.'
              },
              {
                id: 'content-2-4-3',
                type: 'simulation',
                title: 'Comprehensive Change Leadership Challenge',
                description: 'Lead a complex organizational transformation in a simulated business environment with multiple stakeholders',
                scenario: 'You are the CEO of TraditionalRetail, a 50-year-old department store chain with 5,000 employees and 200 stores facing digital disruption. Online competitors are capturing market share, customer preferences are shifting rapidly, and your board is demanding transformation to e-commerce and omnichannel retail. You must lead comprehensive change while maintaining current operations and employee morale.',
                phases: [
                  'Creating Urgency: Build compelling case for change and overcome resistance from long-tenured employees and traditional-minded board members',
                  'Building Coalition: Assemble leadership team with change champions across stores, corporate functions, and union representatives',
                  'Vision Development: Create compelling future state vision that honors company heritage while embracing digital innovation',
                  'Communication Strategy: Engage all stakeholders (employees, customers, investors, suppliers) with tailored messaging',
                  'Empowering Action: Remove structural barriers, provide digital skills training, and establish new performance metrics',
                  'Generating Wins: Implement pilot programs, celebrate early successes, and build momentum for larger changes',
                  'Sustaining Change: Embed new behaviors in hiring, promotion, and reward systems while monitoring progress',
                  'Anchoring Changes: Institutionalize transformation through culture, processes, and leadership development'
                ],
                challenges: [
                  'Employee resistance from traditional retail workers fearful of job displacement',
                  'Customer confusion during transition from traditional to omnichannel experience',
                  'Investor skepticism about transformation timeline and ROI',
                  'Supplier relationships requiring renegotiation for digital capabilities',
                  'Technology integration complexity across legacy and modern systems',
                  'Competitive pressure requiring speed without sacrificing quality'
                ],
                evaluationCriteria: [
                  'Strategic change vision that balances innovation with heritage',
                  'Stakeholder engagement strategy addressing emotional and rational concerns',
                  'Implementation timeline with realistic milestones and risk management',
                  'Communication effectiveness across diverse audiences',
                  'Change management techniques demonstrating psychological insight',
                  'Success measurement framework with both quantitative and qualitative metrics'
                ]
              }
            ]
          }
        ]
      };

      // Get the mapped course ID
      const mappedCourseId = courseIdMapping[courseId] || courseId;
      const modules = moduleData[mappedCourseId as keyof typeof moduleData] || [];
      
      res.json(modules);
    } catch (error) {
      console.error("Error fetching course modules:", error);
      res.status(500).json({ message: "Failed to fetch course modules" });
    }
  });

  // Enrollment routes
  app.post('/api/enrollments', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      const { courseId } = req.body;

      if (!user) {
        return res.status(401).json({ message: "User not found" });
      }

      const enrollment = await storage.enrollUser({
        userId: user.id,
        courseId,
        progress: "0.00",
        status: 'active'
      });

      res.json(enrollment);
    } catch (error) {
      console.error("Error creating enrollment:", error);
      res.status(500).json({ message: "Failed to enroll in course" });
    }
  });

  app.get('/api/enrollments', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (!user) {
        return res.status(401).json({ message: "User not found" });
      }

      const enrollments = await storage.getUserEnrollments(user.id);
      res.json(enrollments);
    } catch (error) {
      console.error("Error fetching enrollments:", error);
      res.status(500).json({ message: "Failed to fetch enrollments" });
    }
  });

  app.get('/api/enrollments/course/:courseId', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      const { courseId } = req.params;

      if (!user) {
        return res.status(401).json({ message: "User not found" });
      }

      const enrollments = await storage.getUserEnrollments(user.id);
      const enrollment = enrollments.find(e => e.courseId === courseId);
      
      res.json(enrollment || null);
    } catch (error) {
      console.error("Error fetching enrollment:", error);
      res.status(500).json({ message: "Failed to fetch enrollment" });
    }
  });

  app.post('/api/courses', isAuthenticated, async (req, res) => {
    try {
      const courseData = insertCourseSchema.parse(req.body);
      const course = await storage.createCourse(courseData);
      res.status(201).json(course);
    } catch (error) {
      console.error("Error creating course:", error);
      res.status(500).json({ message: "Failed to create course" });
    }
  });

  app.get('/api/courses/:id', isAuthenticated, async (req, res) => {
    try {
      const course = await storage.getCourse(req.params.id);
      if (!course) {
        return res.status(404).json({ message: "Course not found" });
      }
      res.json(course);
    } catch (error) {
      console.error("Error fetching course:", error);
      res.status(500).json({ message: "Failed to fetch course" });
    }
  });

  // Enrollment routes - direct authentication check
  app.post('/api/enrollments', async (req: any, res) => {
    try {
      console.log('Direct enrollment check:', {
        sessionId: req.sessionID,
        session: req.session,
        userId: req.session?.userId,
        body: req.body
      });

      // Check authentication directly
      if (!req.session || !req.session.userId) {
        console.log('Enrollment failed: no valid session');
        return res.status(401).json({ message: "Authentication required" });
      }

      const enrollmentData = {
        userId: req.session.userId,
        courseId: req.body.courseId,
        status: 'active',
        progress: '0'
      };
      const enrollment = await storage.enrollUser(enrollmentData);
      
      // Log enrollment event
      await storage.logLearningEvent({
        userId: req.session.userId,
        courseId: req.body.courseId,
        eventType: 'course_enrollment',
        eventData: { courseId: req.body.courseId }
      });
      
      res.status(201).json(enrollment);
    } catch (error) {
      console.error("Error creating enrollment:", error);
      res.status(500).json({ message: "Failed to enroll in course" });
    }
  });

  app.get('/api/enrollments', async (req: any, res) => {
    try {
      console.log('Get enrollments request:', {
        sessionId: req.sessionID,
        userId: req.session?.userId,
        hasSession: !!req.session
      });

      if (!req.session?.userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      const enrollments = await storage.getUserEnrollments(req.session.userId);
      res.json(enrollments);
    } catch (error) {
      console.error("Error fetching enrollments:", error);
      res.status(500).json({ message: "Failed to fetch enrollments" });
    }
  });

  app.get('/api/enrollments/my', requireAuth, async (req: any, res) => {
    try {
      const enrollments = await storage.getUserEnrollments(req.session.userId);
      res.json(enrollments);
    } catch (error) {
      console.error("Error fetching enrollments:", error);
      res.status(500).json({ message: "Failed to fetch enrollments" });
    }
  });

  app.patch('/api/enrollments/:id/progress', isAuthenticated, async (req, res) => {
    try {
      const { progress } = req.body;
      const enrollment = await storage.updateEnrollmentProgress(req.params.id, progress);
      res.json(enrollment);
    } catch (error) {
      console.error("Error updating progress:", error);
      res.status(500).json({ message: "Failed to update progress" });
    }
  });

  // Assessment routes
  // Individual assessment endpoint (must come before the general route)
  app.get('/api/assessments/:id', requireAuth, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      const { id } = req.params;

      console.log('Assessment fetch request:', { 
        sessionId: req.sessionID,
        userId: userId,
        assessmentId: id,
        hasSession: !!req.session
      });

      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Different assessment types with appropriate structures
      let assessment;
      
      if (id === '2' || id === '4' || id === '5') {
        // Psychometric Assessment
        assessment = {
          id: id,
          title: id === '2' ? 'Leadership Psychometric Profile (PQP)' : 
                id === '4' ? 'CPAT Cognitive Performance Assessment' :
                'Big Five Personality Assessment (IBDL-BFI)',
          type: 'psychometric',
          settings: {
            timeLimit: id === '2' ? 45 : id === '4' ? 60 : 35,
            randomizeQuestions: false,
            allowReview: true,
            showResults: true,
            isAdaptive: false
          },
          dimensions: id === '2' ? [
            'Strategic Thinking',
            'Team Leadership', 
            'Decision Making',
            'Communication',
            'Adaptability',
            'Results Orientation'
          ] : id === '4' ? [
            'Logical Reasoning',
            'Pattern Recognition',
            'Working Memory',
            'Processing Speed',
            'Attention Control'
          ] : [
            'Openness',
            'Conscientiousness', 
            'Extraversion',
            'Agreeableness',
            'Neuroticism'
          ],
          questions: id === '2' ? [
            {
              id: 1,
              type: 'likert_scale',
              question: 'When faced with a complex organizational challenge, I prefer to analyze all possible variables before taking action.',
              scale: { min: 1, max: 5, labels: ['Strongly Disagree', 'Disagree', 'Neutral', 'Agree', 'Strongly Agree'] },
              category: 'Strategic Thinking',
              points: 1
            },
            {
              id: 2,
              type: 'likert_scale',
              question: 'I find it energizing to motivate and guide team members toward common objectives.',
              scale: { min: 1, max: 5, labels: ['Strongly Disagree', 'Disagree', 'Neutral', 'Agree', 'Strongly Agree'] },
              category: 'Team Leadership',
              points: 1
            },
            {
              id: 3,
              type: 'likert_scale',
              question: 'I consistently make effective decisions even when under significant time pressure and stress.',
              scale: { min: 1, max: 5, labels: ['Strongly Disagree', 'Disagree', 'Neutral', 'Agree', 'Strongly Agree'] },
              category: 'Decision Making',
              points: 1
            }
          ] : id === '4' ? [
            {
              id: 1,
              type: 'multiple_choice',
              question: 'If the pattern is: 2, 4, 8, 16, what comes next?',
              options: ['20', '24', '32', '36'],
              category: 'Pattern Recognition',
              points: 1
            },
            {
              id: 2,
              type: 'multiple_choice', 
              question: 'Which shape does not belong in this sequence: Circle, Square, Triangle, Rectangle, Pentagon?',
              options: ['Circle', 'Square', 'Triangle', 'Rectangle'],
              category: 'Logical Reasoning',
              points: 1
            }
          ] : [
            {
              id: 1,
              type: 'likert_scale',
              question: 'I am someone who is original, comes up with new ideas.',
              scale: { min: 1, max: 5, labels: ['Strongly Disagree', 'Disagree', 'Neutral', 'Agree', 'Strongly Agree'] },
              category: 'Openness',
              points: 1
            },
            {
              id: 2,
              type: 'likert_scale',
              question: 'I am someone who does a thorough job.',
              scale: { min: 1, max: 5, labels: ['Strongly Disagree', 'Disagree', 'Neutral', 'Agree', 'Strongly Agree'] },
              category: 'Conscientiousness',
              points: 1
            },
            {
              id: 3,
              type: 'likert_scale',
              question: 'I am someone who is talkative.',
              scale: { min: 1, max: 5, labels: ['Strongly Disagree', 'Disagree', 'Neutral', 'Agree', 'Strongly Agree'] },
              category: 'Extraversion',
              points: 1
            }
          ]
        };
      } else {
        // Certification Assessment
        assessment = {
          id: id,
          title: 'Business Analytics Certification',
          type: 'certification',
          settings: {
            timeLimit: 120,
            randomizeQuestions: false,
            allowReview: true,
            showResults: true,
            isAdaptive: false
          },
          passingScore: 75,
          questions: [
            {
              id: 1,
              type: 'multiple_choice',
              question: 'What is the primary purpose of data analytics in business?',
              options: [
                'To collect data',
                'To transform data into actionable insights',
                'To store data securely',
                'To delete old data'
              ],
              correct: 1,
              points: 2,
              category: 'Fundamentals'
            },
            {
              id: 2,
              type: 'multiple_choice',
              question: 'Which of the following is NOT a type of business analytics?',
              options: [
                'Descriptive Analytics',
                'Predictive Analytics',
                'Prescriptive Analytics',
                'Subjective Analytics'
              ],
              correct: 3,
              points: 2,
              category: 'Analytics Types'
            },
            {
              id: 3,
              type: 'true_false',
              question: 'Data visualization is only useful for presenting final results to stakeholders.',
              options: ['True', 'False'],
              correct: 1,
              points: 1,
              category: 'Data Visualization'
            }
          ]
        };
      }

      console.log('Returning assessment:', { id, title: assessment.title });
      res.json(assessment);
    } catch (error) {
      console.error("Error fetching assessment:", error);
      res.status(500).json({ message: "Failed to fetch assessment" });
    }
  });

  // Individual assessment attempt endpoint
  app.get('/api/assessment-attempts/:id', requireAuth, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      const { id } = req.params;

      console.log('Assessment attempt fetch request:', { 
        sessionId: req.sessionID,
        userId: userId,
        attemptId: id,
        hasSession: !!req.session
      });

      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Mock assessment attempt data
      const attempt = {
        id: id,
        assessmentId: '1',
        userId: userId,
        status: 'in_progress',
        startTime: new Date().toISOString(),
        answers: {},
        timeSpent: 0
      };

      console.log('Returning assessment attempt:', { id, status: attempt.status });
      res.json(attempt);
    } catch (error) {
      console.error("Error fetching assessment attempt:", error);
      res.status(500).json({ message: "Failed to fetch assessment attempt" });
    }
  });

  // Assessment submission endpoint
  app.post('/api/assessments/:id/submit', requireAuth, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      const { id: assessmentId } = req.params;
      const { answers, timeSpent } = req.body;

      console.log('Assessment submission request:', { 
        sessionId: req.sessionID,
        userId: userId,
        assessmentId: assessmentId,
        answersCount: Object.keys(answers || {}).length,
        timeSpent
      });

      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Different processing for psychometric vs certification assessments
      let results;

      if (assessmentId === '2' || assessmentId === '4' || assessmentId === '5') {
        // Psychometric Assessment - Calculate dimension scores
        const dimensions = assessmentId === '2' ? [
          'Strategic Thinking', 'Team Leadership', 'Decision Making', 
          'Communication', 'Adaptability', 'Results Orientation'
        ] : assessmentId === '4' ? [
          'Logical Reasoning', 'Pattern Recognition', 'Working Memory',
          'Processing Speed', 'Attention Control'
        ] : [
          'Openness', 'Conscientiousness', 'Extraversion', 
          'Agreeableness', 'Neuroticism'
        ];

        // Calculate scores for each dimension based on responses
        const dimensionScores = {};
        const radarData = [];

        if (assessmentId === '2') {
          // Leadership assessment scoring
          dimensionScores['Strategic Thinking'] = (answers[1] || 3) * 20; // Scale to 100
          dimensionScores['Team Leadership'] = (answers[2] || 3) * 20;
          dimensionScores['Decision Making'] = ((answers[3] || 1) + 1) * 20; // Multiple choice to scale
          dimensionScores['Communication'] = 70; // Mock for demo
          dimensionScores['Adaptability'] = 60;
          dimensionScores['Results Orientation'] = 85;
        } else if (assessmentId === '4') {
          // Cognitive assessment scoring
          dimensionScores['Logical Reasoning'] = (answers[2] === 2 ? 85 : 45); // Rectangle is correct
          dimensionScores['Pattern Recognition'] = (answers[1] === 2 ? 90 : 40); // 32 is correct
          dimensionScores['Working Memory'] = 75;
          dimensionScores['Processing Speed'] = 68;
          dimensionScores['Attention Control'] = 82;
        } else {
          // Big Five personality scoring
          dimensionScores['Openness'] = (answers[1] || 3) * 20;
          dimensionScores['Conscientiousness'] = (answers[2] || 3) * 20;
          dimensionScores['Extraversion'] = (answers[3] || 3) * 20;
          dimensionScores['Agreeableness'] = 75; // Mock
          dimensionScores['Neuroticism'] = 45; // Mock (lower is better)
        }

        // Create radar chart data
        dimensions.forEach(dim => {
          radarData.push({
            dimension: dim,
            score: dimensionScores[dim] || 60,
            percentile: Math.min(95, (dimensionScores[dim] || 60) + Math.random() * 20)
          });
        });

        results = {
          type: 'psychometric',
          assessmentTitle: assessmentId === '2' ? 'Leadership Profile' : 
                          assessmentId === '4' ? 'Cognitive Assessment' : 'Personality Profile',
          dimensionScores,
          radarData,
          timeSpent,
          assessorNotes: generateAssessorNotes(assessmentId, dimensionScores, answers),
          recommendations: generateRecommendations(assessmentId, dimensionScores)
        };
      } else {
        // Certification Assessment - Traditional scoring
        const assessmentQuestions = [
          { id: 1, correct: 1, points: 2 },
          { id: 2, correct: 3, points: 2 },
          { id: 3, correct: 1, points: 1 }
        ];

        let totalCorrect = 0;
        let totalPoints = 0;

        assessmentQuestions.forEach(question => {
          totalPoints += question.points;
          const userAnswer = answers[question.id];
          if (userAnswer === question.correct) {
            totalCorrect += question.points;
          }
        });

        const score = Math.round((totalCorrect / totalPoints) * 100);
        const passed = score >= 75;

        results = {
          type: 'certification',
          score,
          passed,
          totalQuestions: assessmentQuestions.length,
          correctAnswers: Math.round((totalCorrect / totalPoints) * assessmentQuestions.length),
          timeSpent,
          breakdown: assessmentQuestions.map(q => ({
            questionId: q.id,
            userAnswer: answers[q.id],
            correctAnswer: q.correct,
            isCorrect: answers[q.id] === q.correct,
            points: q.points
          }))
        };
      }

      // Helper functions for psychometric assessments
      function generateAssessorNotes(assessmentId, dimensionScores, answers) {
        if (assessmentId === '2') {
          return {
            summary: "The candidate demonstrates strong analytical thinking with a preference for data-driven decision making. Shows high potential for strategic leadership roles.",
            strengths: [
              "Strong strategic thinking capabilities with systematic approach to problem-solving",
              "High results orientation and goal achievement focus",
              "Good team leadership potential with people-focused mindset"
            ],
            developmentAreas: [
              "Consider enhancing adaptability in fast-changing environments",
              "Communication style could benefit from more assertive presentation",
              "Decision-making speed could be improved for high-pressure situations"
            ],
            overallAssessment: "Suitable for senior management roles with focus on strategic planning and team development."
          };
        } else if (assessmentId === '4') {
          return {
            summary: "Strong cognitive abilities with particular strength in pattern recognition and logical reasoning. Above-average performance across most cognitive dimensions.",
            strengths: [
              "Excellent pattern recognition skills",
              "Strong logical reasoning capabilities", 
              "Good attention control and focus"
            ],
            developmentAreas: [
              "Processing speed could be enhanced through practice",
              "Working memory capacity shows room for improvement"
            ],
            overallAssessment: "Cognitive profile suitable for complex analytical and problem-solving roles."
          };
        } else {
          return {
            summary: "Balanced personality profile with moderate to high scores across key dimensions. Shows good adaptability and conscientiousness.",
            strengths: [
              "High conscientiousness indicates reliability and attention to detail",
              "Balanced openness to experience suggests adaptability",
              "Moderate extraversion provides good interpersonal skills"
            ],
            developmentAreas: [
              "Could benefit from stress management techniques",
              "Assertiveness training may enhance leadership presence"
            ],
            overallAssessment: "Well-rounded personality suitable for diverse professional environments."
          };
        }
      }

      function generateRecommendations(assessmentId, dimensionScores) {
        if (assessmentId === '2') {
          return [
            "Consider roles in strategic planning or business development",
            "Leadership development program focusing on change management",
            "Cross-functional project management experience recommended"
          ];
        } else if (assessmentId === '4') {
          return [
            "Suitable for analytical and research positions",
            "Consider advanced problem-solving training",
            "Data analysis and pattern recognition roles recommended"
          ];
        } else {
          return [
            "Team-based collaborative environments would be ideal",
            "Professional development in areas of lower scores",
            "Consider personality-based career guidance"
          ];
        }
      }

      console.log('Assessment submission completed:', { 
        assessmentId, 
        userId, 
        type: results.type,
        completed: true
      });

      res.json(results);
    } catch (error) {
      console.error("Error submitting assessment:", error);
      res.status(500).json({ message: "Failed to submit assessment" });
    }
  });

  app.get('/api/assessments', requireAuth, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      const { q: search, type } = req.query;

      // Comprehensive assessment catalog
      const sampleAssessments = [
        {
          id: '1',
          title: 'Business Analytics Certification',
          description: 'Comprehensive assessment covering data analysis, statistical methods, and business intelligence tools.',
          type: 'certification',
          duration: 120,
          questionCount: 60,
          passingScore: 75,
          attemptsCount: 234,
          isProctored: true,
          allowRetakes: true,
          maxAttempts: 3,
          tags: ['analytics', 'certification', 'data'],
          isActive: true,
          createdAt: new Date('2024-01-15'),
          updatedAt: new Date('2024-12-01')
        },
        {
          id: '2',
          title: 'Leadership Psychometric Profile (PQP)',
          description: 'Advanced psychological assessment measuring leadership potential, decision-making styles, and team dynamics.',
          type: 'psychometric',
          duration: 45,
          questionCount: 150,
          passingScore: 60,
          attemptsCount: 156,
          isProctored: false,
          allowRetakes: false,
          maxAttempts: 1,
          tags: ['psychology', 'leadership', 'personality'],
          isActive: true,
          createdAt: new Date('2024-02-20'),
          updatedAt: new Date('2024-11-15')
        },
        {
          id: '3',
          title: 'Project Management Skills Assessment',
          description: 'Evaluate project management competencies including planning, execution, risk management, and stakeholder communication.',
          type: 'skills',
          duration: 90,
          questionCount: 45,
          passingScore: 70,
          attemptsCount: 412,
          isProctored: true,
          allowRetakes: true,
          maxAttempts: 2,
          tags: ['project management', 'skills', 'competency'],
          isActive: true,
          createdAt: new Date('2024-03-10'),
          updatedAt: new Date('2024-12-20')
        },
        {
          id: '4',
          title: 'CPAT Cognitive Performance Assessment',
          description: 'Cognitive ability test measuring problem-solving, analytical thinking, and decision-making under pressure.',
          type: 'psychometric',
          duration: 60,
          questionCount: 80,
          passingScore: 65,
          attemptsCount: 89,
          isProctored: true,
          allowRetakes: false,
          maxAttempts: 1,
          tags: ['cognitive', 'cpat', 'assessment'],
          isActive: true,
          createdAt: new Date('2024-04-05'),
          updatedAt: new Date('2024-11-30')
        },
        {
          id: '5',
          title: 'Big Five Personality Assessment (IBDL-BFI)',
          description: 'Comprehensive personality evaluation based on the Five-Factor Model measuring openness, conscientiousness, extraversion, agreeableness, and neuroticism.',
          type: 'personality',
          duration: 35,
          questionCount: 120,
          passingScore: null, // No pass/fail for personality assessments
          attemptsCount: 324,
          isProctored: false,
          allowRetakes: true,
          maxAttempts: 2,
          tags: ['personality', 'big five', 'psychology', 'behavioral'],
          isActive: true,
          createdAt: new Date('2024-05-12'),
          updatedAt: new Date('2024-12-10')
        },
        {
          id: '6',
          title: 'Digital Marketing Strategy Certification',
          description: 'Advanced certification covering digital marketing channels, analytics, customer journey mapping, and ROI optimization.',
          type: 'certification',
          duration: 150,
          questionCount: 75,
          passingScore: 80,
          attemptsCount: 187,
          isProctored: true,
          allowRetakes: true,
          maxAttempts: 3,
          tags: ['digital marketing', 'certification', 'strategy', 'analytics'],
          isActive: true,
          createdAt: new Date('2024-06-08'),
          updatedAt: new Date('2024-12-15')
        },
        {
          id: '7',
          title: 'Emotional Intelligence Quotient (EQ-i 2.0)',
          description: 'Scientifically validated assessment measuring emotional intelligence across five key competencies: self-perception, self-expression, interpersonal, decision making, and stress management.',
          type: 'psychometric',
          duration: 50,
          questionCount: 133,
          passingScore: 90,
          attemptsCount: 267,
          isProctored: false,
          allowRetakes: false,
          maxAttempts: 1,
          tags: ['emotional intelligence', 'eq', 'leadership', 'interpersonal'],
          isActive: true,
          createdAt: new Date('2024-07-20'),
          updatedAt: new Date('2024-12-05')
        },
        {
          id: '8',
          title: 'Myers-Briggs Type Indicator (MBTI) Professional',
          description: 'Professional-grade personality assessment identifying psychological preferences in how people perceive the world and make decisions.',
          type: 'personality',
          duration: 45,
          questionCount: 93,
          passingScore: null,
          attemptsCount: 445,
          isProctored: false,
          allowRetakes: true,
          maxAttempts: 1,
          tags: ['mbti', 'personality', 'psychology', 'preferences'],
          isActive: true,
          createdAt: new Date('2024-08-15'),
          updatedAt: new Date('2024-12-01')
        },
        {
          id: '9',
          title: 'Financial Analysis & Risk Management Skills',
          description: 'Comprehensive skills assessment covering financial modeling, risk analysis, investment evaluation, and regulatory compliance.',
          type: 'skills',
          duration: 110,
          questionCount: 55,
          passingScore: 75,
          attemptsCount: 156,
          isProctored: true,
          allowRetakes: true,
          maxAttempts: 2,
          tags: ['finance', 'risk management', 'skills', 'analysis'],
          isActive: true,
          createdAt: new Date('2024-09-10'),
          updatedAt: new Date('2024-12-18')
        },
        {
          id: '10',
          title: 'Critical Thinking & Problem Solving (Watson-Glaser)',
          description: 'Gold standard assessment measuring critical thinking ability including inference, recognition of assumptions, deduction, interpretation, and evaluation of arguments.',
          type: 'psychometric',
          duration: 40,
          questionCount: 40,
          passingScore: 70,
          attemptsCount: 123,
          isProctored: true,
          allowRetakes: false,
          maxAttempts: 1,
          tags: ['critical thinking', 'watson-glaser', 'problem solving', 'reasoning'],
          isActive: true,
          createdAt: new Date('2024-10-05'),
          updatedAt: new Date('2024-12-12')
        },
        {
          id: '11',
          title: 'Situational Judgment Test (SJT) - Leadership',
          description: 'Scenario-based assessment measuring leadership judgment, decision-making in complex situations, and ethical reasoning.',
          type: 'skills',
          duration: 75,
          questionCount: 30,
          passingScore: 65,
          attemptsCount: 234,
          isProctored: false,
          allowRetakes: true,
          maxAttempts: 2,
          tags: ['situational judgment', 'leadership', 'ethics', 'decision making'],
          isActive: true,
          createdAt: new Date('2024-11-01'),
          updatedAt: new Date('2024-12-10')
        },
        {
          id: '12',
          title: 'Digital Marketing Competency Exam',
          description: 'Assessment of digital marketing skills including SEO, social media strategy, analytics, and campaign management.',
          type: 'certification',
          duration: 100,
          questionCount: 50,
          passingScore: 80,
          attemptsCount: 298,
          isProctored: false,
          allowRetakes: true,
          maxAttempts: 3,
          tags: ['digital marketing', 'certification', 'competency'],
          isActive: true,
          createdAt: new Date('2024-05-12'),
          updatedAt: new Date('2024-12-05')
        },
        {
          id: '13',
          title: 'Workplace Personality Assessment',
          description: 'Comprehensive personality evaluation for team placement, role suitability, and professional development planning.',
          type: 'personality',
          duration: 30,
          questionCount: 120,
          passingScore: 50,
          attemptsCount: 567,
          isProctored: false,
          allowRetakes: true,
          maxAttempts: 2,
          tags: ['personality', 'workplace', 'assessment'],
          isActive: true,
          createdAt: new Date('2024-06-18'),
          updatedAt: new Date('2024-11-30')
        }
      ];

      let filteredAssessments = sampleAssessments;

      // Apply search filter
      if (search) {
        const searchLower = search.toLowerCase();
        filteredAssessments = filteredAssessments.filter(assessment => 
          assessment.title.toLowerCase().includes(searchLower) ||
          assessment.description.toLowerCase().includes(searchLower) ||
          assessment.type.toLowerCase().includes(searchLower)
        );
      }

      // Apply type filter
      if (type) {
        filteredAssessments = filteredAssessments.filter(assessment => 
          assessment.type === type
        );
      }

      res.json(filteredAssessments);
    } catch (error) {
      console.error("Error fetching assessments:", error);
      res.status(500).json({ message: "Failed to fetch assessments" });
    }
  });

  app.get('/api/assessment-attempts', requireAuth, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Sample user attempts - only show completed for assessment 1, leave psychometric assessments available
      const sampleAttempts = [
        {
          id: 'attempt-1',
          assessmentId: '1',
          userId: userId,
          status: 'completed',
          score: 85,
          startedAt: new Date('2024-12-20T10:00:00Z'),
          completedAt: new Date('2024-12-20T12:00:00Z'),
          timeSpent: 120,
          responses: {}
        }
      ];

      res.json(sampleAttempts);
    } catch (error) {
      console.error("Error fetching assessment attempts:", error);
      res.status(500).json({ message: "Failed to fetch attempts" });
    }
  });

  app.post('/api/assessment-attempts', requireAuth, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      const { assessmentId } = req.body;

      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Create a new assessment attempt
      const attempt = {
        id: `attempt-${Date.now()}`,
        assessmentId,
        userId: userId,
        status: 'in_progress',
        score: null,
        startedAt: new Date(),
        completedAt: null,
        timeSpent: 0,
        responses: {}
      };

      res.json(attempt);
    } catch (error) {
      console.error("Error creating assessment attempt:", error);
      res.status(500).json({ message: "Failed to start assessment" });
    }
  });

  app.get('/api/assessment-attempts/:id', requireAuth, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      const { id } = req.params;

      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Mock attempt data for now - in a real app this would fetch from database
      const attempt = {
        id: id,
        assessmentId: '1', // This would come from the actual attempt record
        userId: userId,
        status: 'in_progress',
        score: null,
        startedAt: new Date(),
        completedAt: null,
        timeSpent: 0,
        responses: {}
      };

      res.json(attempt);
    } catch (error) {
      console.error("Error fetching assessment attempt:", error);
      res.status(500).json({ message: "Failed to fetch attempt" });
    }
  });

  // Certificate management routes
  app.get('/api/certificates', requireAuth, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      if (!userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      // Sample certificates with blockchain verification
      const sampleCertificates = [
        {
          id: 'cert-1',
          userId: userId,
          title: 'Business Analytics Professional',
          description: 'Certification in advanced data analysis and business intelligence',
          courseId: '1',
          assessmentId: '1',
          verificationCode: 'IBDL-BA-2024-001234',
          issuedDate: new Date('2024-12-20'),
          validUntil: new Date('2026-12-20'),
          skills: ['Data Analysis', 'Business Intelligence', 'Statistical Modeling', 'Dashboard Creation'],
          blockchainTxId: '0x1234567890abcdef',
          metadata: {
            issuer: 'IBDL - International Board for Development and Learning',
            grade: 'A',
            score: 85,
            creditHours: 40
          }
        },
        {
          id: 'cert-2',
          userId: userId,
          title: 'Leadership Psychology Specialist',
          description: 'Advanced certification in leadership psychology and team dynamics',
          courseId: '2',
          assessmentId: '2',
          verificationCode: 'IBDL-LP-2024-005678',
          issuedDate: new Date('2024-12-25'),
          validUntil: new Date('2026-12-25'),
          skills: ['Leadership Psychology', 'Team Dynamics', 'Emotional Intelligence', 'Conflict Resolution'],
          blockchainTxId: '0xabcdef1234567890',
          metadata: {
            issuer: 'IBDL - International Board for Development and Learning',
            grade: 'A+',
            score: 92,
            creditHours: 30
          }
        }
      ];

      res.json(sampleCertificates);
    } catch (error) {
      console.error("Error fetching certificates:", error);
      res.status(500).json({ message: "Failed to fetch certificates" });
    }
  });

  app.get('/api/certificates/:id/download', isAuthenticated, async (req: any, res) => {
    try {
      const { id } = req.params;
      
      // Generate PDF certificate (in real implementation, this would create actual PDF)
      res.setHeader('Content-Type', 'application/pdf');
      res.setHeader('Content-Disposition', `attachment; filename="IBDL-Certificate-${id}.pdf"`);
      res.status(200).send("PDF certificate content would be generated here");
    } catch (error) {
      console.error("Error downloading certificate:", error);
      res.status(500).json({ message: "Failed to download certificate" });
    }
  });

  // System Setup API Routes
  app.get('/api/system-certificates', async (req: any, res) => {
    try {
      const certificates = await storage.getSystemCertificates();
      res.json(certificates);
    } catch (error) {
      console.error("Error fetching system certificates:", error);
      res.status(500).json({ message: "Failed to fetch certificates" });
    }
  });

  app.post('/api/system-certificates', async (req: any, res) => {
    try {
      const certificate = await storage.createSystemCertificate(req.body);
      res.status(201).json(certificate);
    } catch (error) {
      console.error("Error creating system certificate:", error);
      res.status(500).json({ message: "Failed to create certificate" });
    }
  });

  app.get('/api/system-levels', async (req: any, res) => {
    try {
      const { certificateId } = req.query;
      if (certificateId) {
        const levels = await storage.getSystemLevels(certificateId);
        res.json(levels);
      } else {
        // Return all levels if no certificate filter
        const levels = await storage.getSystemLevels();
        res.json(levels);
      }
    } catch (error) {
      console.error("Error fetching system levels:", error);
      res.status(500).json({ message: "Failed to fetch levels" });
    }
  });

  app.post('/api/system-levels', async (req: any, res) => {
    try {
      console.log("Creating system level with data:", req.body);
      
      // Validate the request body
      const validatedData = insertSystemLevelSchema.parse(req.body);
      console.log("Validated data:", validatedData);
      
      const level = await storage.createSystemLevel(validatedData);
      res.status(201).json(level);
    } catch (error) {
      console.error("Error creating system level:", error);
      if (error instanceof ZodError) {
        return res.status(400).json({ 
          message: "Validation error", 
          errors: error.errors.map(e => `${e.path.join('.')}: ${e.message}`)
        });
      }
      res.status(500).json({ message: "Failed to create level" });
    }
  });

  app.get('/api/system-chapters/:levelId', async (req: any, res) => {
    try {
      const { levelId } = req.params;
      console.log("Fetching chapters for level:", levelId);
      
      const chapters = await storage.getSystemChapters(levelId);
      console.log("Found chapters:", chapters);
      res.json(chapters);
    } catch (error) {
      console.error("Error fetching chapters:", error);
      res.status(500).json({ message: "Failed to fetch chapters" });
    }
  });

  app.post('/api/system-chapters', async (req: any, res) => {
    try {
      console.log("Creating system chapter with data:", req.body);
      const validatedData = insertSystemChapterSchema.parse(req.body);
      console.log("Validated data:", validatedData);
      
      const chapter = await storage.createSystemChapter(validatedData);
      console.log("Created chapter:", chapter);
      res.status(201).json(chapter);
    } catch (error) {
      console.error("Error creating system chapter:", error);
      res.status(500).json({ message: "Failed to create chapter", error: error.message });
    }
  });

  app.get('/api/exam-configurations', async (req: any, res) => {
    try {
      const configurations = [
        {
          id: 'config-1',
          certificateId: 'cert-1',
          examName: 'Strategic Leadership Certification Exam',
          examType: 'certification',
          questionCount: 75,
          timeLimit: 120,
          passingScore: 80,
          randomizeQuestions: true,
          allowReview: false,
          showResults: true,
          isProctored: true,
          maxAttempts: 3,
          retakeDelay: 24,
          isActive: true,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        }
      ];
      res.json(configurations);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch exam configurations" });
    }
  });

  app.post('/api/exam-configurations', async (req: any, res) => {
    try {
      const configuration = {
        id: `config-${Date.now()}`,
        ...req.body,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        isActive: true
      };
      res.status(201).json(configuration);
    } catch (error) {
      res.status(500).json({ message: "Failed to create exam configuration" });
    }
  });

  app.get('/api/elearning-content', async (req: any, res) => {
    try {
      const content = [
        {
          id: 'content-1',
          certificateId: 'cert-1',
          titleEn: 'Leadership Fundamentals Video',
          titleAr: 'فيديو أساسيات القيادة',
          titleFr: 'Vidéo Fondements du Leadership',
          contentType: 'video',
          duration: 30,
          isActive: true,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        },
        {
          id: 'content-2',
          certificateId: 'cert-1',
          titleEn: 'Strategic Decision Making Simulation',
          titleAr: 'محاكاة اتخاذ القرارات الاستراتيجية',
          titleFr: 'Simulation de Prise de Décision Stratégique',
          contentType: 'simulation',
          duration: 45,
          isActive: true,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        }
      ];
      res.json(content);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch eLearning content" });
    }
  });

  app.post('/api/elearning-content', async (req: any, res) => {
    try {
      const content = {
        id: `content-${Date.now()}`,
        ...req.body,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        isActive: true
      };
      res.status(201).json(content);
    } catch (error) {
      res.status(500).json({ message: "Failed to create eLearning content" });
    }
  });

  app.get('/api/ebooks', async (req: any, res) => {
    try {
      const ebooks = [
        {
          id: 'ebook-1',
          certificateId: 'cert-1',
          titleEn: 'Strategic Leadership Handbook',
          titleAr: 'دليل القيادة الاستراتيجية',
          titleFr: 'Manuel de Leadership Stratégique',
          author: 'Dr. Ahmed Hassan',
          description: 'Comprehensive guide to strategic leadership',
          fileSize: 2048000,
          downloadCount: 156,
          isActive: true,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        },
        {
          id: 'ebook-2',
          certificateId: 'cert-2',
          titleEn: 'Business Analytics Essentials',
          titleAr: 'أساسيات تحليل الأعمال',
          titleFr: 'Éléments Essentiels d\'Analyse d\'Affaires',
          author: 'Dr. Sarah Mohamed',
          description: 'Essential guide to business analytics',
          fileSize: 1536000,
          downloadCount: 89,
          isActive: true,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        }
      ];
      res.json(ebooks);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch eBooks" });
    }
  });

  app.post('/api/ebooks', async (req: any, res) => {
    try {
      const ebook = {
        id: `ebook-${Date.now()}`,
        ...req.body,
        downloadCount: 0,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        isActive: true
      };
      res.status(201).json(ebook);
    } catch (error) {
      res.status(500).json({ message: "Failed to create eBook" });
    }
  });

  app.get('/api/exam-results', async (req: any, res) => {
    try {
      const results = [
        {
          id: 'result-1',
          userId: 'user-1',
          assessmentId: 'assessment-1',
          score: 85,
          status: 'passed',
          completedAt: new Date().toISOString()
        },
        {
          id: 'result-2',
          userId: 'user-2',
          assessmentId: 'assessment-2',
          score: 72,
          status: 'passed',
          completedAt: new Date().toISOString()
        }
      ];
      res.json(results);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch exam results" });
    }
  });

  app.get('/api/examination-sessions', async (req: any, res) => {
    try {
      const sessions = [
        {
          id: 'session-1',
          userId: 'user-1',
          assessmentId: 'assessment-1',
          status: 'active',
          startTime: new Date().toISOString()
        },
        {
          id: 'session-2',
          userId: 'user-2',
          assessmentId: 'assessment-2',
          status: 'completed',
          startTime: new Date(Date.now() - 3600000).toISOString()
        }
      ];
      res.json(sessions);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch examination sessions" });
    }
  });

  app.patch('/api/examination-sessions/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      const { status } = req.body;
      
      const session = {
        id,
        status,
        updatedAt: new Date().toISOString()
      };
      
      res.json(session);
    } catch (error) {
      res.status(500).json({ message: "Failed to update examination session" });
    }
  });





  app.get('/api/certificates/verify/:code', async (req, res) => {
    try {
      const { code } = req.params;
      
      // Verify certificate using blockchain
      const verificationResult = {
        isValid: true,
        certificate: {
          title: 'Business Analytics Professional',
          recipientName: 'John Doe',
          issuedDate: '2024-12-20',
          issuer: 'IBDL - International Board for Development and Learning',
          verificationCode: code,
          blockchainTxId: '0x1234567890abcdef'
        },
        blockchainStatus: 'verified',
        verifiedAt: new Date()
      };

      res.json(verificationResult);
    } catch (error) {
      console.error("Error verifying certificate:", error);
      res.status(500).json({ message: "Failed to verify certificate" });
    }
  });

  app.post('/api/assessments', isAuthenticated, async (req, res) => {
    try {
      const assessmentData = insertAssessmentSchema.parse(req.body);
      const assessment = await storage.createAssessment(assessmentData);
      res.status(201).json(assessment);
    } catch (error) {
      console.error("Error creating assessment:", error);
      res.status(500).json({ message: "Failed to create assessment" });
    }
  });

  // Assessment attempt routes
  app.post('/api/assessments/:id/attempts', isAuthenticated, async (req: any, res) => {
    try {
      const attemptData = insertAssessmentAttemptSchema.parse({
        userId: req.user.claims.sub,
        assessmentId: req.params.id,
        ...req.body,
      });
      const attempt = await storage.createAssessmentAttempt(attemptData);
      
      // Log assessment start event
      await storage.logLearningEvent({
        userId: req.user.claims.sub,
        assessmentId: req.params.id,
        eventType: 'assessment_start',
        eventData: { assessmentId: req.params.id },
        sessionId: req.sessionID,
      });
      
      res.status(201).json(attempt);
    } catch (error) {
      console.error("Error creating assessment attempt:", error);
      res.status(500).json({ message: "Failed to start assessment" });
    }
  });

  app.patch('/api/assessment-attempts/:id', isAuthenticated, async (req: any, res) => {
    try {
      const updates = req.body;
      const attempt = await storage.updateAssessmentAttempt(req.params.id, updates);
      
      // If completing assessment, log completion event
      if (updates.status === 'completed') {
        await storage.logLearningEvent({
          userId: req.user.claims.sub,
          assessmentId: attempt.assessmentId,
          eventType: 'assessment_complete',
          eventData: { 
            assessmentId: attempt.assessmentId,
            score: updates.score,
            attemptId: req.params.id
          },
          sessionId: req.sessionID,
        });
        
        // Auto-generate certificate if passing score achieved
        if (updates.score && parseFloat(updates.score) >= 70) {
          const assessment = await storage.getAssessment(attempt.assessmentId);
          if (assessment) {
            await storage.createCertificate({
              userId: req.user.claims.sub,
              assessmentId: attempt.assessmentId,
              courseId: assessment.courseId,
              type: 'assessment',
              title: `${assessment.title} - Certificate of Completion`,
              metadata: { score: updates.score, attemptId: req.params.id }
            });
          }
        }
      }
      
      res.json(attempt);
    } catch (error) {
      console.error("Error updating assessment attempt:", error);
      res.status(500).json({ message: "Failed to update assessment attempt" });
    }
  });

  // Certificate routes
  app.get('/api/certificates/my', isAuthenticated, async (req: any, res) => {
    try {
      const certificates = await storage.getUserCertificates(req.user.claims.sub);
      res.json(certificates);
    } catch (error) {
      console.error("Error fetching certificates:", error);
      res.status(500).json({ message: "Failed to fetch certificates" });
    }
  });

  app.get('/api/certificates/verify/:code', async (req, res) => {
    try {
      const certificate = await storage.verifyCertificate(req.params.code);
      if (!certificate) {
        return res.status(404).json({ message: "Certificate not found" });
      }
      res.json(certificate);
    } catch (error) {
      console.error("Error verifying certificate:", error);
      res.status(500).json({ message: "Failed to verify certificate" });
    }
  });

  // Examination session routes (for live proctoring)
  app.post('/api/examination-sessions', isAuthenticated, async (req: any, res) => {
    try {
      const sessionData = insertExaminationSessionSchema.parse({
        ...req.body,
        userId: req.user.claims.sub,
      });
      const session = await storage.createExaminationSession(sessionData);
      res.status(201).json(session);
    } catch (error) {
      console.error("Error creating examination session:", error);
      res.status(500).json({ message: "Failed to create examination session" });
    }
  });

  app.get('/api/examination-sessions/my', isAuthenticated, async (req: any, res) => {
    try {
      const sessions = await storage.getUserExaminationSessions(req.user.claims.sub);
      res.json(sessions);
    } catch (error) {
      console.error("Error fetching examination sessions:", error);
      res.status(500).json({ message: "Failed to fetch examination sessions" });
    }
  });

  app.patch('/api/examination-sessions/:id', isAuthenticated, async (req, res) => {
    try {
      const updates = req.body;
      const session = await storage.updateExaminationSession(req.params.id, updates);
      res.json(session);
    } catch (error) {
      console.error("Error updating examination session:", error);
      res.status(500).json({ message: "Failed to update examination session" });
    }
  });

  // Analytics routes
  app.post('/api/analytics/events', isAuthenticated, async (req: any, res) => {
    try {
      const eventData = insertLearningAnalyticsSchema.parse({
        ...req.body,
        userId: req.user.claims.sub,
        sessionId: req.sessionID,
      });
      const event = await storage.logLearningEvent(eventData);
      res.status(201).json(event);
    } catch (error) {
      console.error("Error logging analytics event:", error);
      res.status(500).json({ message: "Failed to log event" });
    }
  });

  // Organization management routes (for system admins and corporate users)
  app.post('/api/organizations', isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      const user = await storage.getUser(userId);
      if (!user || user.userType !== 'system_admin') {
        return res.status(403).json({ message: "Access denied" });
      }

      const orgData = insertOrganizationSchema.parse(req.body);
      const organization = await storage.createOrganization(orgData);
      res.status(201).json(organization);
    } catch (error) {
      console.error("Error creating organization:", error);
      res.status(500).json({ message: "Failed to create organization" });
    }
  });

  // Accreditation Management - FRD Compliant Implementation
  // FR-AM-003: View All Accreditation Applications (IBDL Admin)
  app.get('/api/accreditation/applications', isAuthenticated, async (req: any, res) => {
    try {
      // Get user from session-based auth
      const userId = req.session?.userId;
      const user = await storage.getUser(userId);
      
      // Allow qa_admin and system_admin to access
      if (!user || !['system_admin', 'qa_admin'].includes(user.userType)) {
        return res.status(403).json({ message: "Access denied. Admin privileges required." });
      }

      const { status, type } = req.query;
      let applications;

      if (status && status !== 'all') {
        applications = await storage.getAccreditationApplicationsByStatus(status);
      } else if (type && type !== 'all') {
        applications = await storage.getAccreditationApplicationsByType(type);
      } else {
        applications = await storage.getAccreditationApplications();
      }

      res.json(applications);
    } catch (error) {
      console.error("Error fetching accreditation applications:", error);
      res.status(500).json({ message: "Failed to fetch accreditation applications" });
    }
  });

  // FR-AM-004: View Single Accreditation Application Details (IBDL Admin)
  app.get('/api/accreditation/applications/:id', isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.session?.userId;
      const user = await storage.getUser(userId);
      
      if (!user || !['system_admin', 'qa_admin'].includes(user.userType)) {
        return res.status(403).json({ message: "Access denied. Admin privileges required." });
      }

      const application = await storage.getAccreditationApplication(req.params.id);
      if (!application) {
        return res.status(404).json({ message: "Application not found" });
      }

      res.json(application);
    } catch (error) {
      console.error("Error fetching accreditation application:", error);
      res.status(500).json({ message: "Failed to fetch accreditation application" });
    }
  });

  // FR-AM-002: View Accreditation Application Status (Applicant)
  app.get('/api/accreditation/applications/user/:userId', isAuthenticated, async (req: any, res) => {
    try {
      const requestingUser = await storage.getUser(req.user.claims.sub);
      const targetUserId = req.params.userId;

      // Users can only view their own applications unless admin
      if (req.user.claims.sub !== targetUserId && requestingUser?.userType !== 'system_admin') {
        return res.status(403).json({ message: "Access denied. Can only view own applications." });
      }

      const applications = await storage.getUserAccreditationApplications(targetUserId);
      res.json(applications);
    } catch (error) {
      console.error("Error fetching user accreditation applications:", error);
      res.status(500).json({ message: "Failed to fetch user applications" });
    }
  });

  // FR-AM-001: Accreditation Application Submission (Updated for Training Center Applications)
  app.post('/api/accreditation/apply', async (req: any, res) => {
    try {
      const applicationData = req.body;
      console.log('New application submission:', { 
        organizationName: applicationData.organizationName,
        contactEmail: applicationData.contactEmail 
      });

      // Validate essential fields
      const requiredFields = [
        'applicationType', 'organizationName', 'contactPerson', 
        'contactEmail', 'contactPhone', 'address', 'city', 'country',
        'businessLicense', 'taxRegistration', 'organizationType',
        'establishedYear', 'numberOfEmployees', 'facilitiesDescription',
        'agreementToTerms', 'declarationOfAccuracy', 'password'
      ];

      for (const field of requiredFields) {
        if (applicationData[field] === undefined || applicationData[field] === null || applicationData[field] === '') {
          console.log(`Missing field: ${field}, value:`, applicationData[field]);
          return res.status(400).json({ message: `Missing required field: ${field}` });
        }
      }

      // Create user account for the applicant
      const userData = {
        username: applicationData.contactEmail,
        email: applicationData.contactEmail,
        password: applicationData.password, // Will be hashed in storage
        firstName: applicationData.contactPerson.split(' ')[0] || '',
        lastName: applicationData.contactPerson.split(' ').slice(1).join(' ') || '',
        userType: 'applicant',
        isActive: true
      };

      const user = await storage.createUser(userData);
      console.log('Created user account:', { id: user.id, email: user.email });

      // Create the application linked to the user
      const { password, confirmPassword, ...appData } = applicationData;
      const application = await storage.createAccreditationApplication({
        ...appData,
        userId: user.id
      });

      console.log('Created application:', { id: application.id, status: application.status });
      
      // TODO: Send automated email notification to applicant (FR-NS-001)
      // TODO: Notify IBDL Admins of new application
      
      res.json(application);
    } catch (error) {
      console.error("Error submitting accreditation application:", error);
      if (error.message?.includes('duplicate key')) {
        return res.status(400).json({ message: "An account with this email already exists" });
      }
      res.status(500).json({ message: "Failed to submit accreditation application" });
    }
  });

  // FR-AM-005: Approve Accreditation Application (IBDL Admin)
  app.post('/api/accreditation/approve', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'system_admin') {
        return res.status(403).json({ message: "Access denied. Admin privileges required." });
      }

      const { id, level, comments } = req.body;
      const reviewerId = req.user.claims.sub;

      if (!id || !level || level < 1 || level > 3) {
        return res.status(400).json({ message: "Invalid approval level or application ID" });
      }

      const current = await storage.getAccreditationApplication(id);
      if (!current) {
        return res.status(404).json({ message: "Application not found" });
      }

      // Determine target status based on approval level
      let targetStatus;
      if (level === 1) {
        targetStatus = 'pending_approval_2';
      } else if (level === 2) {
        targetStatus = 'pending_approval_3';
      } else if (level === 3) {
        targetStatus = 'approved';
      }

      const application = await storage.updateAccreditationStatus(id, targetStatus, comments || '', reviewerId);
      
      // TODO: Send automated email notification to applicant (FR-NS-001)
      
      res.json(application);
    } catch (error) {
      console.error("Error approving accreditation application:", error);
      res.status(500).json({ message: "Failed to approve application" });
    }
  });

  // FR-AM-006: Reject Accreditation Application (IBDL Admin)
  app.post('/api/accreditation/reject', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'system_admin') {
        return res.status(403).json({ message: "Access denied. Admin privileges required." });
      }

      const { id, reason } = req.body;
      const reviewerId = req.user.claims.sub;

      if (!id || !reason) {
        return res.status(400).json({ message: "Application ID and rejection reason are required" });
      }

      const application = await storage.updateAccreditationStatus(id, 'rejected', reason, reviewerId);
      
      // TODO: Send automated email notification to applicant with rejection reason (FR-NS-001)
      
      res.json(application);
    } catch (error) {
      console.error("Error rejecting accreditation application:", error);
      res.status(500).json({ message: "Failed to reject application" });
    }
  });



  // Get user's own application (for applicant dashboard)
  app.get('/api/accreditation/my-application', async (req: any, res) => {
    try {
      if (!req.session.userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      const application = await storage.getUserAccreditationApplication(req.session.userId);
      
      if (!application) {
        return res.status(404).json({ message: "No application found" });
      }

      res.json(application);
    } catch (error) {
      console.error('Error fetching user application:', error);
      res.status(500).json({ message: "Failed to fetch application" });
    }
  });

  // Update application status and QA checklist
  app.post('/api/accreditation/update-status', async (req: any, res) => {
    try {
      console.log('Update status request:', {
        sessionId: req.sessionID,
        userId: req.session?.userId,
        userType: req.session?.userType,
        body: req.body,
        hasSession: !!req.session
      });

      if (!req.session?.userId) {
        return res.status(401).json({ message: "Authentication required" });
      }

      const user = await storage.getUser(req.session.userId);
      // Allow both system_admin and qa_admin user types to update applications
      if (!['system_admin', 'qa_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied. Admin or QA privileges required." });
      }

      const { id, status, comments, qaCheckpoint1, qaCheckpoint2, qaCheckpoint3 } = req.body;
      
      console.log('QA Checklist values received:', {
        id,
        qaCheckpoint1,
        qaCheckpoint2,
        qaCheckpoint3,
        status,
        comments
      });
      
      const application = await storage.updateAccreditationStatus(
        id, 
        status, 
        comments, 
        req.session.userId, 
        { qaCheckpoint1, qaCheckpoint2, qaCheckpoint3 }
      );
      
      console.log('Updated application result:', {
        id: application.id,
        qaCheckpoint1: application.qaCheckpoint1,
        qaCheckpoint2: application.qaCheckpoint2, 
        qaCheckpoint3: application.qaCheckpoint3
      });
      
      res.json(application);
    } catch (error) {
      console.error("Error updating accreditation status:", error);
      res.status(500).json({ message: "Failed to update status" });
    }
  });

  // Question Bank Management API Routes
  app.get('/api/question-bank', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (!['system_admin', 'partner_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const questions = await storage.getQuestionBank();
      res.json(questions);
    } catch (error) {
      console.error("Error fetching question bank:", error);
      res.status(500).json({ message: "Failed to fetch questions" });
    }
  });

  app.post('/api/question-bank', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (!['system_admin', 'partner_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const questionData = { ...req.body, createdBy: req.user.claims.sub };
      const question = await storage.createQuestion(questionData);
      res.json(question);
    } catch (error) {
      console.error("Error creating question:", error);
      res.status(500).json({ message: "Failed to create question" });
    }
  });

  app.put('/api/question-bank/:id', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (!['system_admin', 'partner_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const { id } = req.params;
      const question = await storage.updateQuestion(id, req.body);
      res.json(question);
    } catch (error) {
      console.error("Error updating question:", error);
      res.status(500).json({ message: "Failed to update question" });
    }
  });

  app.delete('/api/question-bank/:id', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (!['system_admin', 'partner_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const { id } = req.params;
      await storage.deleteQuestion(id);
      res.json({ message: "Question deleted successfully" });
    } catch (error) {
      console.error("Error deleting question:", error);
      res.status(500).json({ message: "Failed to delete question" });
    }
  });

  // Testing Centers API Routes
  app.get('/api/testing-centers', isAuthenticated, async (req: any, res) => {
    try {
      const centers = await storage.getTestingCenters();
      res.json(centers);
    } catch (error) {
      console.error("Error fetching testing centers:", error);
      res.status(500).json({ message: "Failed to fetch testing centers" });
    }
  });

  // Exam Bookings API Routes
  app.get('/api/exam-bookings', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      let bookings;
      
      if (user?.userType === 'system_admin') {
        bookings = await storage.getAllExamBookings();
      } else {
        bookings = await storage.getUserExamBookings(req.user.claims.sub);
      }
      
      res.json(bookings);
    } catch (error) {
      console.error("Error fetching exam bookings:", error);
      res.status(500).json({ message: "Failed to fetch exam bookings" });
    }
  });

  app.post('/api/exam-bookings', isAuthenticated, async (req: any, res) => {
    try {
      const bookingData = { ...req.body, userId: req.user.claims.sub, bookedBy: req.user.claims.sub };
      const booking = await storage.createExamBooking(bookingData);
      res.json(booking);
    } catch (error) {
      console.error("Error creating exam booking:", error);
      res.status(500).json({ message: "Failed to create exam booking" });
    }
  });

  // Corporate Credits API Routes
  app.get('/api/corporate-credits', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (!user?.organizationId || !['corporate_admin', 'system_admin'].includes(user.userType)) {
        return res.status(403).json({ message: "Access denied" });
      }

      const credits = await storage.getCorporateCredits(user.organizationId);
      res.json(credits);
    } catch (error) {
      console.error("Error fetching corporate credits:", error);
      res.status(500).json({ message: "Failed to fetch corporate credits" });
    }
  });

  // Corporate Account Management API Routes - FRD Compliant Implementation
  // FR-CAM-001: View Corporate Account Dashboard
  app.get('/api/corporate/stats', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied. Corporate admin privileges required." });
      }

      const stats = await storage.getCorporateDashboardStats(user.organizationId || '');
      res.json(stats);
    } catch (error) {
      console.error("Error fetching corporate stats:", error);
      res.status(500).json({ message: "Failed to fetch corporate stats" });
    }
  });

  // FR-CAM-002: Add Single Employee (Trainee)
  app.post('/api/corporate/employees', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied. Corporate admin privileges required." });
      }

      const employeeData = {
        ...req.body,
        organizationId: user.organizationId,
        userType: 'learner',
        createdBy: req.user.claims.sub,
        createdAt: new Date(),
        updatedAt: new Date()
      };

      const employee = await storage.createUser(employeeData);
      res.json(employee);
    } catch (error) {
      console.error("Error creating employee:", error);
      res.status(500).json({ message: "Failed to create employee" });
    }
  });

  // FR-CAM-003: Bulk Add Employees (Trainees)
  app.post('/api/corporate/employees/bulk-upload', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied. Corporate admin privileges required." });
      }

      // TODO: Implement actual CSV file parsing and batch employee creation
      // For now, return success message
      res.json({ message: "Bulk employee upload successful", count: 0 });
    } catch (error) {
      console.error("Error bulk uploading employees:", error);
      res.status(500).json({ message: "Failed to bulk upload employees" });
    }
  });

  // FR-CAM-004: Manage Employee Profiles (Corporate Account)
  app.get('/api/corporate/employees', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied. Corporate admin privileges required." });
      }

      const employees = await storage.getUsersByOrganization(user.organizationId || '');
      res.json(employees);
    } catch (error) {
      console.error("Error fetching employees:", error);
      res.status(500).json({ message: "Failed to fetch employees" });
    }
  });

  app.put('/api/corporate/employees/:id', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied. Corporate admin privileges required." });
      }

      const { id } = req.params;
      const updates = {
        ...req.body,
        updatedAt: new Date()
      };

      const employee = await storage.updateUser(id, updates);
      res.json(employee);
    } catch (error) {
      console.error("Error updating employee:", error);
      res.status(500).json({ message: "Failed to update employee" });
    }
  });

  app.delete('/api/corporate/employees/:id', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied. Corporate admin privileges required." });
      }

      const { id } = req.params;
      // TODO: Implement soft delete or actual delete based on business requirements
      // For now, return success message
      res.json({ message: "Employee deleted successfully" });
    } catch (error) {
      console.error("Error deleting employee:", error);
      res.status(500).json({ message: "Failed to delete employee" });
    }
  });

  // FR-CAM-005: Book Exam for Employee
  app.post('/api/corporate/exam-bookings', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied. Corporate admin privileges required." });
      }

      const bookingData = {
        ...req.body,
        corporateId: user.organizationId,
        bookedBy: req.user.claims.sub,
        status: 'pending',
        createdAt: new Date(),
        updatedAt: new Date()
      };

      const booking = await storage.createExamBooking(bookingData);
      
      // TODO: Deduct credits from corporate account (FR-CAM-006)
      // TODO: Send notification to employee and testing center (FR-NS-001)
      
      res.json(booking);
    } catch (error) {
      console.error("Error booking exam:", error);
      res.status(500).json({ message: "Failed to book exam" });
    }
  });

  app.get('/api/corporate/exam-bookings', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied. Corporate admin privileges required." });
      }

      // TODO: Implement actual exam bookings retrieval with corporate filtering
      // For now, return sample data
      const bookings = [
        {
          id: 'booking-1',
          employeeName: 'John Doe',
          employeeId: 'emp-1',
          examTitle: 'IBDL Leadership Certification',
          examId: 'exam-1',
          testingCenterName: 'Downtown Testing Center',
          testingCenterId: 'center-1',
          examDate: '2025-01-20',
          examTime: '10:00',
          status: 'confirmed',
          createdAt: '2025-01-15T10:00:00Z'
        },
        {
          id: 'booking-2',
          employeeName: 'Jane Smith',
          employeeId: 'emp-2',
          examTitle: 'IBDL Project Management',
          examId: 'exam-2',
          testingCenterName: 'Business District Center',
          testingCenterId: 'center-2',
          examDate: '2025-01-25',
          examTime: '14:00',
          status: 'pending',
          createdAt: '2025-01-16T10:00:00Z'
        }
      ];

      res.json(bookings);
    } catch (error) {
      console.error("Error fetching exam bookings:", error);
      res.status(500).json({ message: "Failed to fetch exam bookings" });
    }
  });

  // Corporate Credits Management
  app.get('/api/corporate/credits', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.user.claims.sub);
      if (user?.userType !== 'corporate_admin') {
        return res.status(403).json({ message: "Access denied. Corporate admin privileges required." });
      }

      const credits = await storage.getCorporateCredits(user.organizationId || '');
      res.json(credits);
    } catch (error) {
      console.error("Error fetching corporate credits:", error);
      res.status(500).json({ message: "Failed to fetch corporate credits" });
    }
  });

  // CMS API Routes
  app.get('/api/cms/pages', async (req: any, res) => {
    try {
      const pages = await storage.getCmsPages();
      
      // Transform to expected format
      const formattedPages = pages.map(page => ({
        id: page.id,
        title: page.title,
        slug: page.slug,
        content: page.content,
        isPublished: page.status === 'published',
        metadata: {
          metaTitle: page.metaTitle,
          metaDescription: page.metaDescription
        },
        createdAt: page.createdAt,
        updatedAt: page.updatedAt
      }));
      
      res.json(formattedPages);
    } catch (error) {
      console.error("Error fetching CMS pages:", error);
      res.status(500).json({ message: "Failed to fetch CMS pages" });
    }
  });

  app.get('/api/cms/pages/:slug', async (req: any, res) => {
    try {
      const { slug } = req.params;
      
      // Mock pages data for now
      const pages = [
        {
          id: 'page-1',
          slug: 'about-us',
          title: 'About IBDL Learning Group',
          content: 'IBDL Learning Group is a leading provider of business education...',
          metaTitle: 'About IBDL Learning Group',
          metaDescription: 'Learn about our mission, vision, and commitment to excellence',
          status: 'published',
          pageType: 'about',
          featuredImage: '/images/about-hero.jpg',
          publishedAt: '2024-07-01T10:00:00Z',
          createdAt: '2024-06-15T10:00:00Z',
          updatedAt: '2024-07-01T10:00:00Z'
        },
        {
          id: 'page-2',
          slug: 'strategy',
          title: 'IBDL Strategic Framework',
          content: 'Our comprehensive strategic framework drives exponential value through revolutionary products and solutions, establishing IBDL as the global leader in learning-driven organizational excellence. We focus on four key strategic goals: Catalyze Exponential Value, Pioneer Learning-Driven Model, Ignite Digital Revolution, and Empower Universities.',
          metaTitle: 'IBDL Strategic Framework - Vision & Goals',
          metaDescription: 'Discover IBDL strategic framework driving exponential value through revolutionary learning solutions and digital transformation.',
          status: 'published',
          pageType: 'about',
          featuredImage: '/images/strategy.jpg',
          publishedAt: '2024-07-01T10:00:00Z',
          createdAt: '2024-06-15T10:00:00Z',
          updatedAt: '2024-07-01T10:00:00Z'
        },
        {
          id: 'page-3',
          slug: 'qualifications',
          title: 'IBDL Qualifications & Certifications',
          content: 'IBDL offers a comprehensive suite of professional qualifications designed to advance careers and drive organizational excellence across diverse industries. Our certification programs include IBDL Essential for fundamental business skills, IBDL Professional for advanced competencies, and IBDL Expert for master-level expertise.',
          metaTitle: 'IBDL Qualifications & Certifications - Professional Development',
          metaDescription: 'Explore IBDL comprehensive professional qualification programs designed to advance careers and drive organizational excellence.',
          status: 'published',
          pageType: 'product',
          featuredImage: '/images/qualifications.jpg',
          publishedAt: '2024-07-01T10:00:00Z',
          createdAt: '2024-06-15T10:00:00Z',
          updatedAt: '2024-07-01T10:00:00Z'
        },
        {
          id: 'page-4',
          slug: 'contact',
          title: 'Contact IBDL',
          content: 'Connect with IBDL to explore learning opportunities, partnerships, and professional development solutions. Our global offices are located in the United Kingdom, United States, and Middle East, providing comprehensive support worldwide.',
          metaTitle: 'Contact IBDL - Get in Touch',
          metaDescription: 'Contact IBDL for learning opportunities, partnerships, and professional development solutions. Global offices available.',
          status: 'published',
          pageType: 'about',
          featuredImage: '/images/contact.jpg',
          publishedAt: '2024-07-01T10:00:00Z',
          createdAt: '2024-06-15T10:00:00Z',
          updatedAt: '2024-07-01T10:00:00Z'
        }
      ];
      
      const page = pages.find(p => p.slug === slug);
      if (!page) {
        return res.status(404).json({ message: "Page not found" });
      }
      
      res.json(page);
    } catch (error) {
      console.error("Error fetching page:", error);
      res.status(500).json({ message: "Failed to fetch page" });
    }
  });

  app.post('/api/cms/pages', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const pageData = {
        title: req.body.title,
        slug: req.body.slug,
        content: req.body.content || '',
        metaTitle: req.body.metadata?.metaTitle || req.body.title,
        metaDescription: req.body.metadata?.metaDescription || '',
        status: req.body.isPublished ? 'published' : 'draft',
        pageType: 'page',
        createdBy: req.session.userId
      };
      
      const page = await storage.createCmsPage(pageData);
      
      // Transform response to expected format
      const formattedPage = {
        id: page.id,
        title: page.title,
        slug: page.slug,
        content: page.content,
        isPublished: page.status === 'published',
        metadata: {
          metaTitle: page.metaTitle,
          metaDescription: page.metaDescription
        },
        createdAt: page.createdAt,
        updatedAt: page.updatedAt
      };
      
      res.json(formattedPage);
    } catch (error) {
      console.error("Error creating CMS page:", error);
      res.status(500).json({ message: "Failed to create CMS page" });
    }
  });

  app.put('/api/cms/pages/:id', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const { id } = req.params;
      const pageData = {
        title: req.body.title,
        slug: req.body.slug,
        content: req.body.content,
        metaTitle: req.body.metadata?.metaTitle || req.body.title,
        metaDescription: req.body.metadata?.metaDescription || '',
        status: req.body.isPublished ? 'published' : 'draft',
        updatedBy: req.session.userId
      };
      
      const page = await storage.updateCmsPage(id, pageData);
      
      // Transform response to expected format
      const formattedPage = {
        id: page.id,
        title: page.title,
        slug: page.slug,
        content: page.content,
        isPublished: page.status === 'published',
        metadata: {
          metaTitle: page.metaTitle,
          metaDescription: page.metaDescription
        },
        createdAt: page.createdAt,
        updatedAt: page.updatedAt
      };
      
      res.json(formattedPage);
    } catch (error) {
      console.error("Error updating CMS page:", error);
      res.status(500).json({ message: "Failed to update CMS page" });
    }
  });

  app.delete('/api/cms/pages/:id', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const { id } = req.params;
      await storage.deleteCmsPage(id);
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting CMS page:", error);
      res.status(500).json({ message: "Failed to delete CMS page" });
    }
  });

  // CMS Menu Items Routes
  app.get('/api/cms/menu-items', async (req: any, res) => {
    try {
      const menus = await storage.getCmsMenuItems();
      res.json(menus);
    } catch (error) {
      console.error("Error fetching CMS menu items:", error);
      res.status(500).json({ message: "Failed to fetch CMS menu items" });
    }
  });

  app.post('/api/cms/menu-items', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const itemData = { ...req.body, createdBy: req.session.userId };
      const item = await storage.createCmsMenuItem(itemData);
      res.json(item);
    } catch (error) {
      console.error("Error creating CMS menu item:", error);
      res.status(500).json({ message: "Failed to create CMS menu item" });
    }
  });

  app.put('/api/cms/menu-items/:id', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const { id } = req.params;
      const itemData = { ...req.body, updatedBy: req.session.userId };
      const item = await storage.updateCmsMenuItem(id, itemData);
      res.json(item);
    } catch (error) {
      console.error("Error updating CMS menu item:", error);
      res.status(500).json({ message: "Failed to update CMS menu item" });
    }
  });

  app.delete('/api/cms/menu-items/:id', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }

      const { id } = req.params;
      await storage.deleteCmsMenuItem(id);
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting CMS menu item:", error);
      res.status(500).json({ message: "Failed to delete CMS menu item" });
    }
  });

  app.get('/api/cms/news', async (req: any, res) => {
    try {
      const articles = await storage.getNewsArticles();
      res.json(articles);
    } catch (error) {
      console.error("Error fetching news articles:", error);
      res.status(500).json({ message: "Failed to fetch news articles" });
    }
  });

  app.post('/api/cms/news', isAuthenticated, async (req: any, res) => {
    try {
      const user = await storage.getUser(req.session.userId);
      if (user?.userType !== 'system_admin') {
        return res.status(403).json({ message: "Access denied" });
      }

      const articleData = { ...req.body, createdBy: req.session.userId };
      const article = await storage.createNewsArticle(articleData);
      res.json(article);
    } catch (error) {
      console.error("Error creating news article:", error);
      res.status(500).json({ message: "Failed to create news article" });
    }
  });

  app.get('/api/cms/resources', async (req: any, res) => {
    try {
      const resources = [
        {
          id: 'resource-1',
          title: 'Business Leadership Guide',
          slug: 'business-leadership-guide',
          description: 'Comprehensive guide to modern business leadership...',
          content: 'Guide content...',
          resourceType: 'guide',
          fileUrl: '/downloads/leadership-guide.pdf',
          thumbnailUrl: '/images/resource-1.jpg',
          downloadCount: 1200,
          accessLevel: 'public',
          tags: ['leadership', 'management', 'guide'],
          createdBy: 'admin',
          status: 'published',
          publishedAt: '2024-06-15T10:00:00Z',
          createdAt: '2024-06-10T10:00:00Z',
          updatedAt: '2024-06-15T10:00:00Z'
        }
      ];
      res.json(resources);
    } catch (error) {
      console.error("Error fetching resources:", error);
      res.status(500).json({ message: "Failed to fetch resources" });
    }
  });

  app.post('/api/cms/resources', async (req: any, res) => {
    try {
      const resourceData = {
        id: `resource-${Date.now()}`,
        ...req.body,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString()
      };
      res.json(resourceData);
    } catch (error) {
      console.error("Error creating resource:", error);
      res.status(500).json({ message: "Failed to create resource" });
    }
  });

  app.get('/api/cms/store', async (req: any, res) => {
    try {
      const products = [
        {
          id: 'product-1',
          name: 'IBDL Essential Certification',
          slug: 'ibdl-essential-certification',
          description: 'Comprehensive business fundamentals certification program...',
          shortDescription: 'Business fundamentals certification',
          price: 299,
          salePrice: 249,
          sku: 'IBDL-ESSENTIAL-001',
          category: 'certification',
          tags: ['business', 'certification', 'fundamentals'],
          images: ['/images/product-1.jpg'],
          downloadable: true,
          stock: 1000,
          status: 'published',
          featured: true,
          createdBy: 'admin',
          publishedAt: '2024-06-01T10:00:00Z',
          createdAt: '2024-05-15T10:00:00Z',
          updatedAt: '2024-06-01T10:00:00Z'
        }
      ];
      res.json(products);
    } catch (error) {
      console.error("Error fetching products:", error);
      res.status(500).json({ message: "Failed to fetch products" });
    }
  });

  app.post('/api/cms/store', async (req: any, res) => {
    try {
      const productData = {
        id: `product-${Date.now()}`,
        ...req.body,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString()
      };
      res.json(productData);
    } catch (error) {
      console.error("Error creating product:", error);
      res.status(500).json({ message: "Failed to create product" });
    }
  });

  app.get('/api/cms/settings', async (req: any, res) => {
    try {
      const settings = {
        id: 'settings-1',
        siteName: 'IBDL Learning Group',
        siteDescription: 'We develop the learning journeys that enable every person and organization to move forward',
        logo: '/images/logo.png',
        favicon: '/images/favicon.ico',
        contactEmail: 'info@ibdl.net',
        contactPhone: '+1-555-0123',
        address: '123 Business District, Learning City, LC 12345',
        socialLinks: {
          facebook: 'https://facebook.com/ibdllearning',
          twitter: 'https://twitter.com/ibdllearning',
          linkedin: 'https://linkedin.com/company/ibdllearning'
        },
        seoSettings: {
          defaultTitle: 'IBDL Learning Group',
          defaultDescription: 'Business education and professional development',
          keywords: 'business education, professional development, certification'
        },
        maintenanceMode: false,
        updatedAt: '2024-07-01T10:00:00Z'
      };
      res.json(settings);
    } catch (error) {
      console.error("Error fetching settings:", error);
      res.status(500).json({ message: "Failed to fetch settings" });
    }
  });

  app.put('/api/cms/settings', async (req: any, res) => {
    try {
      const settingsData = {
        id: 'settings-1',
        ...req.body,
        updatedAt: new Date().toISOString()
      };
      res.json(settingsData);
    } catch (error) {
      console.error("Error updating settings:", error);
      res.status(500).json({ message: "Failed to update settings" });
    }
  });

  // Page Sections API Routes
  // Get all sections for a specific page
  app.get('/api/cms/pages/:pageId/sections', async (req: any, res) => {
    try {
      const { pageId } = req.params;
      const sections = await storage.getPageSections(pageId);
      res.json(sections);
    } catch (error) {
      console.error("Error fetching page sections:", error);
      res.status(500).json({ message: "Failed to fetch page sections" });
    }
  });

  // Get a specific section by ID
  app.get('/api/cms/sections/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      const section = await storage.getPageSection(id);
      if (!section) {
        return res.status(404).json({ message: "Section not found" });
      }
      res.json(section);
    } catch (error) {
      console.error("Error fetching page section:", error);
      res.status(500).json({ message: "Failed to fetch page section" });
    }
  });

  // Create a new section for a page
  app.post('/api/cms/pages/:pageId/sections', (req: any, res: any, next: any) => {
    // Check if this is a file upload request by looking at the content-type
    const contentType = req.headers['content-type'] || '';
    
    if (contentType.includes('multipart/form-data')) {
      // Use multer for file uploads (any() to support dynamic field names like trustee_image_*)
      upload.any()(req, res, next);
    } else {
      // Skip multer for JSON requests
      next();
    }
  }, async (req: any, res) => {
    try {
      // Check if user has website admin permissions
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }
      
      const { pageId } = req.params;
      
      // Parse the data based on request type
      let sectionData;
      const contentType = req.headers['content-type'] || '';
      
      if (contentType.includes('multipart/form-data') && req.body.data) {
        // FormData request - parse JSON from data field (data is a string)
        try {
          sectionData = JSON.parse(req.body.data);
        } catch (e) {
          console.error('JSON parse error:', e);
          return res.status(400).json({ message: "Invalid JSON data" });
        }
      } else {
        // Regular JSON request
        sectionData = req.body;
      }
      
      // Process uploaded files if any
      if (req.files) {
        const filesArray = req.files as Express.Multer.File[];
        
        // Convert array to lookup object by fieldname
        const filesByField: { [fieldname: string]: Express.Multer.File[] } = {};
        filesArray.forEach(file => {
          if (!filesByField[file.fieldname]) {
            filesByField[file.fieldname] = [];
          }
          filesByField[file.fieldname].push(file);
        });
        
        // Process standard image fields
        if (filesByField.image && filesByField.image[0]) {
          sectionData.image = filesByField.image[0].filename;
        }
        if (filesByField.background_image && filesByField.background_image[0]) {
          sectionData.backgroundImage = filesByField.background_image[0].filename;
        }
        if (filesByField.ceo_image && filesByField.ceo_image[0]) {
          // Save CEO image filename in the data JSON field where it belongs
          if (!sectionData.data) sectionData.data = {};
          sectionData.data.ceo_image_file = filesByField.ceo_image[0].filename;
        }
        if (filesByField.images && filesByField.images.length > 0) {
          sectionData.images = filesByField.images.map(file => file.filename);
        }
        
        // Process trustee image files
        const trusteeImageFiles: { [key: string]: string } = {};
        Object.keys(filesByField).forEach(fieldName => {
          if (fieldName.startsWith('trustee_image_')) {
            const key = fieldName.replace('trustee_image_', '');
            trusteeImageFiles[key] = filesByField[fieldName][0].filename;
          }
        });
        
        // Update trustee profiles with actual filenames
        if (Object.keys(trusteeImageFiles).length > 0 && sectionData.data?.trustee_profiles) {
          try {
            const trustees = JSON.parse(sectionData.data.trustee_profiles);
            const updatedTrustees = trustees.map((trustee: any) => {
              if (trustee.image && trusteeImageFiles[trustee.image]) {
                return { ...trustee, image: trusteeImageFiles[trustee.image] };
              }
              return trustee;
            });
            sectionData.data.trustee_profiles = JSON.stringify(updatedTrustees);
          } catch (error) {
            console.error('Error processing trustee profiles:', error);
          }
        }
      }
      
      // Add metadata
      sectionData.pageId = pageId;
      sectionData.createdBy = req.session.userId;
      
      console.log('Creating section with data:', JSON.stringify(sectionData, null, 2));
      const section = await storage.createPageSection(sectionData);
      res.status(201).json(section);
    } catch (error) {
      console.error("Error creating page section:", error);
      res.status(500).json({ message: "Failed to create page section" });
    }
  });

  // Update an existing section
  app.put('/api/cms/sections/:id', (req: any, res: any, next: any) => {
    // Check if this is a file upload request by looking at the content-type
    const contentType = req.headers['content-type'] || '';
    
    if (contentType.includes('multipart/form-data')) {
      // Use multer for file uploads
      upload.single('image')(req, res, next);
    } else {
      // Skip multer for JSON requests
      next();
    }
  }, async (req: any, res) => {
    try {
      // Check if user has website admin permissions
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }
      
      const { id } = req.params;
      
      // Parse the data based on request type
      let sectionData;
      const contentType = req.headers['content-type'] || '';
      
      if (contentType.includes('multipart/form-data') && req.body.data) {
        // FormData request - parse JSON from data field (data is a string)
        try {
          sectionData = JSON.parse(req.body.data);
        } catch (e) {
          console.error('JSON parse error:', e);
          return res.status(400).json({ message: "Invalid JSON data" });
        }
      } else {
        // JSON request - use body directly (already parsed by Express)
        sectionData = req.body;
      }
      
      // If a file was uploaded, update the image field
      if (req.file) {
        sectionData.image = req.file.filename;
      }
      
      console.log('Updating section with data:', JSON.stringify(sectionData, null, 2));
      const section = await storage.updatePageSection(id, sectionData);
      res.json(section);
    } catch (error) {
      console.error("Error updating page section:", error);
      res.status(500).json({ message: "Failed to update page section" });
    }
  });

  // Delete a section (soft delete)
  app.delete('/api/cms/sections/:id', async (req: any, res) => {
    try {
      // Check if user has website admin permissions
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }
      
      const { id } = req.params;
      await storage.deletePageSection(id);
      res.json({ message: 'Section deleted successfully', id });
    } catch (error) {
      console.error("Error deleting page section:", error);
      res.status(500).json({ message: "Failed to delete page section" });
    }
  });

  // Reorder sections for a page
  app.put('/api/cms/pages/:pageId/sections/reorder', async (req: any, res) => {
    try {
      // Check if user has website admin permissions
      const user = await storage.getUser(req.session.userId);
      if (!['system_admin', 'website_admin'].includes(user?.userType || '')) {
        return res.status(403).json({ message: "Access denied" });
      }
      
      const { pageId } = req.params;
      const { sections } = req.body;
      
      // Validate the request body
      if (!Array.isArray(sections)) {
        return res.status(400).json({ message: "Sections must be an array" });
      }
      
      // Update the sort order for each section
      await storage.reorderPageSections(pageId, sections);
      
      res.json({ message: 'Sections reordered successfully' });
    } catch (error) {
      console.error("Error reordering page sections:", error);
      res.status(500).json({ message: "Failed to reorder page sections" });
    }
  });

  // Delete endpoints for CMS
  app.delete('/api/cms/pages/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      // In a real implementation, this would delete from database
      res.json({ message: 'Page deleted successfully', id });
    } catch (error) {
      console.error("Error deleting page:", error);
      res.status(500).json({ message: "Failed to delete page" });
    }
  });

  app.delete('/api/cms/news/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      res.json({ message: 'Article deleted successfully', id });
    } catch (error) {
      console.error("Error deleting article:", error);
      res.status(500).json({ message: "Failed to delete article" });
    }
  });

  app.delete('/api/cms/resources/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      res.json({ message: 'Resource deleted successfully', id });
    } catch (error) {
      console.error("Error deleting resource:", error);
      res.status(500).json({ message: "Failed to delete resource" });
    }
  });

  app.delete('/api/cms/store/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      res.json({ message: 'Product deleted successfully', id });
    } catch (error) {
      console.error("Error deleting product:", error);
      res.status(500).json({ message: "Failed to delete product" });
    }
  });

  // Update endpoints for CMS
  app.put('/api/cms/pages/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      const updatedPage = {
        id,
        ...req.body,
        updatedAt: new Date().toISOString()
      };
      res.json(updatedPage);
    } catch (error) {
      console.error("Error updating page:", error);
      res.status(500).json({ message: "Failed to update page" });
    }
  });

  app.put('/api/cms/news/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      const updatedArticle = {
        id,
        ...req.body,
        updatedAt: new Date().toISOString()
      };
      res.json(updatedArticle);
    } catch (error) {
      console.error("Error updating article:", error);
      res.status(500).json({ message: "Failed to update article" });
    }
  });

  app.put('/api/cms/resources/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      const updatedResource = {
        id,
        ...req.body,
        updatedAt: new Date().toISOString()
      };
      res.json(updatedResource);
    } catch (error) {
      console.error("Error updating resource:", error);
      res.status(500).json({ message: "Failed to update resource" });
    }
  });

  app.put('/api/cms/store/:id', async (req: any, res) => {
    try {
      const { id } = req.params;
      const updatedProduct = {
        id,
        ...req.body,
        updatedAt: new Date().toISOString()
      };
      res.json(updatedProduct);
    } catch (error) {
      console.error("Error updating product:", error);
      res.status(500).json({ message: "Failed to update product" });
    }
  });

  const httpServer = createServer(app);

  // WebSocket server for real-time examination monitoring
  const wss = new WebSocketServer({ server: httpServer, path: '/ws' });

  wss.on('connection', (ws: WebSocket) => {
    console.log('New WebSocket connection established');

    ws.on('message', async (message: string) => {
      try {
        const data = JSON.parse(message);
        
        switch (data.type) {
          case 'join_examination':
            // Join examination session for real-time monitoring
            ws.send(JSON.stringify({
              type: 'examination_joined',
              sessionId: data.sessionId,
              status: 'connected'
            }));
            break;
            
          case 'examination_event':
            // Log examination events (screen capture, focus loss, etc.)
            await storage.logLearningEvent({
              userId: data.userId,
              assessmentId: data.assessmentId,
              eventType: 'examination_monitoring',
              eventData: data.eventData,
              sessionId: data.sessionId,
            });
            break;
            
          case 'proctor_message':
            // Handle proctor communications
            ws.send(JSON.stringify({
              type: 'proctor_response',
              message: data.message,
              timestamp: new Date().toISOString()
            }));
            break;
        }
      } catch (error) {
        console.error('WebSocket message handling error:', error);
        ws.send(JSON.stringify({
          type: 'error',
          message: 'Failed to process message'
        }));
      }
    });

    ws.on('close', () => {
      console.log('WebSocket connection closed');
    });

    // Send heartbeat every 30 seconds
    const heartbeat = setInterval(() => {
      if (ws.readyState === WebSocket.OPEN) {
        ws.send(JSON.stringify({ type: 'heartbeat', timestamp: new Date().toISOString() }));
      } else {
        clearInterval(heartbeat);
      }
    }, 30000);
  });

  return httpServer;
}
