import { ChevronRight, Target, Globe, Users, TrendingUp, Award, ArrowRight, BookOpen, Lightbulb, Heart, Shield, Star, CheckCircle, Compass, Eye, Handshake, Building, Zap, Settings, Crown, Rocket, Lock, Linkedin, Twitter, MessageSquare, BarChart3, Scale, DollarSign } from "lucide-react";

/**
 * Shared section rendering utility for all CMS pages
 * This provides a single source of truth for template rendering logic
 */
export function renderSection(
  section: any,
  index: number,
  visibleSections: Set<number>,
  expandedTrustees: Set<string>,
  toggleTrusteeExpansion: (key: string) => void,
  sectionRefs: React.MutableRefObject<(HTMLDivElement | null)[]>
) {
  const templateType = section.templateType || section.template_type || 'text_left_image_right';

  // Helper function to generate initials from name
  const getInitials = (name: string) => {
    if (!name) return 'CEO';
    return name.split(' ').map((n: string) => n[0]).join('').toUpperCase().slice(0, 2);
  };

  // Icon mapping for dynamic icon rendering
  const iconMap: Record<string, any> = {
    BarChart3, Shield, Award, Users, Heart, Target, Globe, Lock, Star,
    CheckCircle, Compass, Eye, Handshake, Lightbulb, TrendingUp,
    Building, BookOpen, Zap, Settings, Crown, Rocket, Scale, DollarSign
  };

  // Render different layouts based on template type
  if (templateType === 'centered_card') {
    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`bg-white dark:bg-slate-800 rounded-2xl shadow-md p-12 transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0 scale-100' 
            : 'opacity-0 translate-y-12 scale-95'
        }`}
      >
        <div className="text-start mb-12">
          <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white mb-8">{section.title}</h2>
          
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
            <div className="text-left">
              <h3 className="text-xl font-semibold text-[#ED1C24] mb-4">
                {section.data?.mission_label || "Our Mission:"}
              </h3>
              <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed">
                {section.data?.mission || "We empower individuals and organizations to achieve their highest potential through transformative learning experiences."}
              </p>
            </div>
            
            <div className="text-left">
              <h3 className="text-xl font-semibold text-[#ED1C24] mb-4">
                {section.data?.vision_label || "Our Vision:"}
              </h3>
              <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed">
                {section.data?.vision || "A world where every learner lives up to their full potential for positive impact through education."}
              </p>
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (templateType === 'gradient_centered') {
    const contentText = section.data?.card_content 
      || section.content 
      || (section.data?.content_items && Array.isArray(section.data.content_items) 
          ? section.data.content_items.join('\n\n') 
          : '');

    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`relative overflow-hidden rounded-2xl shadow-md transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0 scale-100' 
            : 'opacity-0 translate-y-12 scale-95'
        }`}
      >
        <div className="absolute inset-0 bg-gradient-to-r from-[#ED1C24] via-red-600 to-[#1A1A40]"></div>
        <div className="absolute inset-0 opacity-20" style={{
          backgroundImage: `url("data:image/svg+xml,%3Csvg width='120' height='120' viewBox='0 0 120 120' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ffffff' fill-opacity='0.3'%3E%3Cpolygon points='120,0 120,120 60,120 0,60 0,0 60,0'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`
        }}></div>
        <div className="relative py-16 px-12 text-center text-white">
          <h2 className="text-4xl font-bold mb-8">{section.title}</h2>
          <p className="text-xl leading-relaxed font-medium max-w-4xl mx-auto whitespace-pre-wrap">
            {contentText}
          </p>
        </div>
      </div>
    );
  }

  if (templateType === 'ceo_message') {
    // Content items is already an array of objects
    const contentItems = Array.isArray(section.data?.content_items) 
      ? section.data.content_items 
      : [];

    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        <div className="relative overflow-hidden rounded-2xl shadow-md">
          <div className="absolute inset-0 bg-gradient-to-r from-[#1A1A40] via-slate-800 to-[#ED1C24]"></div>
          <div className="absolute inset-0 bg-black/10"></div>
          
          <div className="relative py-16 px-12 text-white">
            <div className={`flex items-center justify-center mb-8 transition-all duration-700 delay-200 transform ${
              visibleSections.has(index) 
                ? 'opacity-100 translate-y-0' 
                : 'opacity-0 translate-y-8'
            }`}>
              <div className="p-4 bg-white/10 rounded-xl backdrop-blur-sm">
                <MessageSquare className="h-8 w-8 text-white" />
              </div>
              <h2 className="text-4xl font-bold ml-6">{section.title}</h2>
            </div>
            
            <div className={`max-w-4xl mx-auto transition-all duration-700 delay-400 transform ${
              visibleSections.has(index) 
                ? 'opacity-100 translate-y-0' 
                : 'opacity-0 translate-y-8'
            }`}>
              {section.data?.quote && (
                <blockquote className="text-2xl font-semibold text-center mb-8 italic">
                  "{section.data.quote}"
                </blockquote>
              )}
              
              {/* Dynamic Content Items */}
              {contentItems.length > 0 ? (
                <div className="space-y-6 text-lg leading-relaxed">
                  {contentItems.map((item: any, itemIndex: number) => (
                    <p key={itemIndex}>{typeof item === 'string' ? item : item.content}</p>
                  ))}
                </div>
              ) : (
                /* Fallback to regular content if no content items */
                section.content && (
                  <div className="space-y-6 text-lg leading-relaxed">
                    <p>{section.content}</p>
                  </div>
                )
              )}
              
              {/* CEO Profile Section */}
              <div className="mt-12 pt-8 border-t border-white/20">
                <div className="flex items-center justify-center space-x-6">
                  <div className="relative w-20 h-20 flex-shrink-0">
                    {section.data?.ceo_image_file ? (
                      <img 
                        src={`backend/api/uploads/${section.data.ceo_image_file}`} 
                        alt={section.data?.ceo_name || 'CEO'}
                        className="w-full h-full rounded-full object-cover shadow-md border-2 border-white/20"
                        onError={(e) => {
                          // Fallback to avatar with initials if image fails to load
                          e.currentTarget.style.display = 'none';
                          const parent = e.currentTarget.parentElement;
                          if (parent) {
                            parent.innerHTML = `
                              <div class="w-full h-full rounded-full bg-white/20 backdrop-blur-sm border-2 border-white/20 flex items-center justify-center">
                                <span class="text-2xl font-bold text-white">
                                  ${getInitials(section.data?.ceo_name || '')}
                                </span>
                              </div>
                            `;
                          }
                        }}
                      />
                    ) : (
                      <div className="w-full h-full rounded-full bg-white/20 backdrop-blur-sm border-2 border-white/20 flex items-center justify-center">
                        <span className="text-2xl font-bold text-white">
                          {getInitials(section.data?.ceo_name || '')}
                        </span>
                      </div>
                    )}
                  </div>
                  <div className="text-left">
                    <div className="text-xl font-semibold">
                      {section.data?.ceo_name || 'Chief Executive Officer'}
                    </div>
                    <div className="text-white/80 text-lg">
                      {section.data?.ceo_title || 'CEO'}
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (templateType === 'text_left_image_right') {
    // Extract content from different possible data structures
    const contentText = section.data?.card_content 
      || section.content 
      || (section.data?.content_items && Array.isArray(section.data.content_items) 
          ? section.data.content_items.join('\n\n') 
          : '');

    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        <div className="bg-white dark:bg-slate-800 rounded-2xl shadow-md p-12 border border-gray-100 dark:border-slate-700">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
            {/* Text Content */}
            <div className={`transition-all duration-700 delay-200 transform ${
              visibleSections.has(index) 
                ? 'opacity-100 translate-x-0' 
                : 'opacity-0 -translate-x-8'
            }`}>
              <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white mb-6 hover:text-[#ED1C24] transition-colors duration-300">
                {section.title}
              </h2>
              <div className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed whitespace-pre-wrap">
                {contentText}
              </div>
            </div>
            
            {/* Right Image */}
            <div className={`relative overflow-hidden rounded-2xl shadow-md transition-all duration-700 delay-200 transform ${
              visibleSections.has(index) 
                ? 'opacity-100 translate-x-0' 
                : 'opacity-0 translate-x-8'
            }`}>
              {section.image ? (
                <img 
                  src={`backend/api/uploads/${section.image}`} 
                  alt={section.title}
                  className="w-full h-80 object-cover hover:scale-105 transition-transform duration-500"
                />
              ) : (
                <div className="w-full h-80 bg-gradient-to-br from-[#ED1C24]/20 to-[#1A1A40]/20 flex items-center justify-center">
                  <BookOpen className="h-16 w-16 text-[#ED1C24]/50" />
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (templateType === 'image_left_text_right') {
    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`grid grid-cols-1 lg:grid-cols-2 gap-16 items-center transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        {/* Left Visual */}
        <div className={`relative overflow-hidden rounded-2xl shadow-md transition-all duration-700 delay-200 transform order-2 lg:order-1 ${
          visibleSections.has(index) 
            ? 'opacity-100 -translate-x-0 scale-100' 
            : 'opacity-0 -translate-x-8 scale-95'
        }`}>
          <img 
            src={section.image && section.image !== 'placeholder-for-vision-image.jpg' 
              ? `backend/api/uploads/${section.image}` 
              : `https://images.unsplash.com/photo-1524178232363-1fb2b075b655?w=600&h=400&fit=crop&q=80`
            }
            alt={section.title}
            className="w-full h-[25rem] object-cover rounded-2xl"
            onError={(e) => {
              // Fallback to random business image if upload fails
              e.currentTarget.src = `https://images.unsplash.com/photo-1524178232363-1fb2b075b655?w=600&h=400&fit=crop&q=80`;
            }}
          />
          <div className="absolute inset-0 bg-gradient-to-br from-black/10 to-black/30"></div>
        </div>

        {/* Right Content */}
        <div className={`group transition-all duration-700 delay-400 transform order-1 lg:order-2 ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-x-0' 
            : 'opacity-0 translate-x-8'
        }`}>
          <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white mb-6 leading-tight group-hover:text-[#ED1C24] transition-colors duration-300">
            {section.title}
          </h2>
          
          {/* Multiple content paragraphs with optional positioned elements */}
          {section.data?.content_items && Array.isArray(section.data.content_items) ? (
            <>
              {section.data.content_items.map((contentItem: string, contentIndex: number) => (
                <div key={contentIndex}>
                  <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed mb-6">
                    {contentItem}
                  </p>
                  
                  {/* Show card if it should be positioned after this content paragraph */}
                  {section.data?.card_content && 
                   Number(section.data?.card_position) === contentIndex && (
                    <div className="bg-gradient-to-r from-[#ED1C24] to-[#1A1A40] p-6 rounded-lg text-white text-center mb-6">
                      <p className="text-xl font-semibold italic">
                        "{section.data.card_content}"
                      </p>
                    </div>
                  )}

                  {/* Show bullet points if they should be positioned after this content paragraph */}
                  {section.data?.bullet_points && Array.isArray(section.data.bullet_points) && 
                   section.data.bullet_points.length > 0 &&
                   Number(section.data?.bullet_points_position) === contentIndex && (
                    <div className="mb-6">
                      <ul className="space-y-3">
                        {section.data.bullet_points.map((bullet: {title: string; description?: string}, bulletIndex: number) => (
                          <li key={bulletIndex} className="flex items-start">
                            <div className="flex-shrink-0 mt-1 mr-3">
                              <CheckCircle className="w-6 h-6 text-[#ED1C24]" />
                            </div>
                            <div>
                              <h4 className="text-lg font-semibold text-gray-700 dark:text-white mb-1">
                                {bullet.title}
                              </h4>
                              {bullet.description && (
                                <p className="text-gray-700 dark:text-gray-300 leading-relaxed">
                                  {bullet.description}
                                </p>
                              )}
                            </div>
                          </li>
                        ))}
                      </ul>
                    </div>
                  )}
                </div>
              ))}

              {/* Final content section */}
              {section.data?.final_content && (
                <div className="mt-6 p-4 bg-gray-50 dark:bg-slate-700 rounded-lg border-l-4 border-[#ED1C24]">
                  <p className="text-gray-700 dark:text-gray-300 leading-relaxed italic">
                    {section.data.final_content}
                  </p>
                </div>
              )}
            </>
          ) : (
            // Backward compatibility with single content field
            <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed">
              {section.content}
            </p>
          )}
        </div>
      </div>
    );
  }

  if (templateType === 'text_left_image_right_with_card') {
    const contentText = section.data?.card_content 
      || section.content 
      || (section.data?.content_items && Array.isArray(section.data.content_items) 
          ? section.data.content_items.join('\n\n') 
          : '');

    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`grid grid-cols-1 lg:grid-cols-2 gap-16 items-center transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        {/* Left Content */}
        <div className={`group transition-all duration-700 delay-200 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-x-0' 
            : 'opacity-0 -translate-x-8'
        }`}>
          <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white mb-6 leading-tight group-hover:text-[#ED1C24] transition-colors duration-300">
            {section.title}
          </h2>
          <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed mb-8 whitespace-pre-wrap">
            {contentText}
          </p>
          
          <div className="bg-white dark:bg-slate-700 p-6 rounded-xl border border-gray-200 dark:border-gray-600 shadow-md">
            <h3 className="text-lg font-semibold text-[#ED1C24] mb-4">
              {section.data?.card_title || "Our partnerships include esteemed names such as:"}
            </h3>
            <ul className="space-y-2 text-gray-700 dark:text-gray-300">
              {section.data?.card_items && Array.isArray(section.data.card_items) 
                ? section.data.card_items.map((item: string, itemIndex: number) => (
                    <li key={itemIndex} className="flex items-center">
                      <div className="w-2 h-2 bg-[#ED1C24] rounded-full mr-3"></div>
                      {item}
                    </li>
                  ))
                : (
                  <li className="flex items-center">
                    <div className="w-2 h-2 bg-[#ED1C24] rounded-full mr-3"></div>
                    Missouri State University
                  </li>
                )
              }
            </ul>
            <p className="text-gray-600 dark:text-gray-400 mt-4 text-sm">
              {section.data?.card_subtitle || "These collaborations ensure that our programs are academically rigorous, industry-relevant, and globally respected."}
            </p>
          </div>
        </div>
        
        {/* Right Visual */}
        <div className={`relative overflow-hidden rounded-2xl shadow-md transition-all duration-700 delay-400 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-x-0 scale-100' 
            : 'opacity-0 translate-x-8 scale-95'
        }`}>
          <img 
            src={section.image && section.image !== 'placeholder-for-global-image.jpg' 
              ? `backend/api/uploads/${section.image}` 
              : `https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=600&h=400&fit=crop&q=80`
            }
            alt={section.title}
            className="w-full h-[25rem] object-cover rounded-2xl"
            onError={(e) => {
              // Fallback to random global/network image if upload fails
              e.currentTarget.src = `https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=600&h=400&fit=crop&q=80`;
            }}
          />
          <div className="absolute inset-0 bg-gradient-to-br from-black/10 to-black/30"></div>
        </div>
      </div>
    );
  }

  if (templateType === 'board_trustees') {
    const trustees = (() => {
      try {
        return section.data?.trustee_profiles ? JSON.parse(section.data.trustee_profiles) : [];
      } catch {
        return [];
      }
    })();

    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        <div className={`text-center mb-12 transition-all duration-700 delay-200 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-8'
        }`}>
          <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white mb-6 hover:text-[#ED1C24] transition-colors duration-300">
            {section.title}
          </h2>
          {section.content && (
            <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed max-w-4xl mx-auto">
              {section.content}
            </p>
          )}
        </div>
        
        {/* Trustees Cards Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
          {trustees.map((trustee: any, trusteeIndex: number) => (
            <div 
              key={trusteeIndex}
              className={`bg-white dark:bg-slate-800 rounded-xl shadow-md hover:shadow-lg transition-all duration-500 p-6 group hover:scale-[1.02] border border-gray-100 dark:border-slate-700 h-full transition-all duration-700 transform ${
                visibleSections.has(index) 
                  ? 'opacity-100 translate-y-0' 
                  : 'opacity-0 translate-y-8'
              }`}
              style={{ 
                transitionDelay: `${400 + (trusteeIndex * 100)}ms` 
              }}
            >
              <div className="text-center h-full flex flex-col">
                <div className="relative mx-auto w-24 h-24 mb-4">
                  {trustee.image ? (
                    <img 
                      src={`backend/api/uploads/${trustee.image}`} 
                      alt={trustee.name}
                      className="w-full h-full rounded-full object-cover shadow-md group-hover:scale-110 transition-transform duration-300"
                      onError={(e) => {
                        // Fallback to placeholder
                        e.currentTarget.src = `https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=300&h=300&fit=crop&crop=face`;
                      }}
                    />
                  ) : (
                    <div className="w-full h-full rounded-full bg-gray-200 flex items-center justify-center">
                      <Users className="w-8 h-8 text-gray-400" />
                    </div>
                  )}
                  <div className="absolute inset-0 rounded-full bg-gradient-to-br from-[#ED1C24]/20 to-[#1A1A40]/20 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                </div>
                
                <h3 className="text-lg font-bold text-[#1A1A40] dark:text-white mb-2">
                  {trustee.name}
                </h3>
                
                <p className="text-sm font-semibold text-[#ED1C24] mb-3">
                  {trustee.title}
                </p>
                
                <div className="flex-grow">
                  {(() => {
                    const description = trustee.description || '';
                    const trusteeKey = `${index}-${trusteeIndex}`;
                    const isExpanded = expandedTrustees.has(trusteeKey);
                    const shouldTruncate = description.length > 120;
                    
                    return (
                      <>
                        <p className="text-xs text-gray-600 dark:text-gray-400 leading-relaxed mb-4">
                          {shouldTruncate && !isExpanded 
                            ? `${description.substring(0, 120)}...` 
                            : description
                          }
                        </p>
                        {shouldTruncate && (
                          <button
                            onClick={() => toggleTrusteeExpansion(trusteeKey)}
                            className="text-xs text-[#ED1C24] font-semibold hover:text-[#ED1C24]/80 transition-colors duration-200 mb-4"
                          >
                            {isExpanded ? 'Read Less' : 'Read More'}
                          </button>
                        )}
                      </>
                    );
                  })()}
                </div>
                
                {/* Social Media Icons */}
                {(trustee.linkedin || trustee.twitter) && (
                  <div className="flex justify-center space-x-2 pt-2 border-t border-gray-100 dark:border-slate-600 mt-auto">
                    {trustee.linkedin && (
                      <a 
                        href={trustee.linkedin} 
                        target="_blank"
                        rel="noopener noreferrer"
                        className="p-1.5 rounded-full bg-[#0077B5] text-white hover:bg-[#005885] transition-colors duration-200"
                        title="LinkedIn"
                      >
                        <Linkedin className="h-3 w-3" />
                      </a>
                    )}
                    {trustee.twitter && (
                      <a 
                        href={trustee.twitter} 
                        target="_blank"
                        rel="noopener noreferrer"
                        className="p-1.5 rounded-full bg-[#1DA1F2] text-white hover:bg-[#0d8bd9] transition-colors duration-200"
                        title="Twitter"
                      >
                        <Twitter className="h-3 w-3" />
                      </a>
                    )}
                  </div>
                )}
              </div>
            </div>
          ))}
        </div>
      </div>
    );
  }

  if (templateType === 'impact_statistics') {
    // Parse metrics from JSON data
    const metrics = (() => {
      try {
        return section.data?.metrics_data ? JSON.parse(section.data.metrics_data) : [];
      } catch {
        return [];
      }
    })();

    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        <div className="bg-white dark:bg-slate-800 rounded-2xl shadow-md p-12 border border-gray-100 dark:border-slate-700">
          <div className={`text-center mb-12 transition-all duration-700 delay-200 transform ${
            visibleSections.has(index) 
              ? 'opacity-100 translate-y-0' 
              : 'opacity-0 translate-y-8'
          }`}>
            <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white mb-6 hover:text-[#ED1C24] transition-colors duration-300">
              {section.title}
            </h2>
            {section.content && (
              <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed max-w-4xl mx-auto">
                {section.content}
              </p>
            )}
          </div>

          {/* Impact Metrics Section */}
          <div className={`transition-all duration-700 delay-400 transform ${
            visibleSections.has(index) 
              ? 'opacity-100 translate-y-0' 
              : 'opacity-0 translate-y-8'
          }`}>
            <div className="flex items-center justify-center mb-8">
              <div className="p-4 bg-gradient-to-br from-[#ED1C24] to-[#ED1C24]/80 rounded-xl shadow-md">
                {(() => {
                  const iconName = section.data?.table_icon || 'BarChart3';
                  const iconMap: Record<string, any> = {
                    BarChart3, Shield, Award, Users, Heart, Target, Globe, Lock, Star, 
                    CheckCircle, Compass, Eye, Handshake, Lightbulb, TrendingUp,
                    Building, BookOpen, Zap, Settings, Crown, Rocket
                  };
                  const IconComponent = iconMap[iconName] || BarChart3;
                  return <IconComponent className="h-8 w-8 text-white" />;
                })()}
              </div>
              <h3 className="text-3xl font-bold text-[#1A1A40] dark:text-white ml-6 hover:text-[#ED1C24] transition-colors duration-300">
                {section.data?.table_title || "Impact by the Numbers"}
              </h3>
            </div>
            
            {/* Metrics Display */}
            <div className="space-y-4">
              {metrics.length > 0 ? metrics.map((item: any, itemIndex: number) => (
                <div 
                  key={itemIndex}
                  className={`bg-gray-50 dark:bg-slate-700 rounded-xl p-6 hover:shadow-md transition-all duration-300 hover:scale-[1.01] transition-all duration-700 transform ${
                    visibleSections.has(index) 
                      ? 'opacity-100 translate-x-0' 
                      : 'opacity-0 translate-x-8'
                  }`}
                  style={{ 
                    transitionDelay: `${600 + (itemIndex * 100)}ms` 
                  }}
                >
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4 items-center">
                    <div className="font-semibold text-[#1A1A40] dark:text-white">
                      {item.category || item.metric}
                    </div>
                    <div className="text-2xl font-bold text-[#ED1C24]">
                      {item.metric}
                    </div>
                    <div className="text-gray-600 dark:text-gray-400 text-sm">
                      {item.description}
                    </div>
                  </div>
                </div>
              )) : (
                <div className="text-center py-8 text-gray-500 dark:text-gray-400">
                  No impact statistics configured yet.
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (templateType === 'image_left_text_card_right') {
    const contentText = section.data?.card_content 
      || section.content 
      || (section.data?.content_items && Array.isArray(section.data.content_items) 
          ? section.data.content_items.join('\n\n') 
          : '');

    const cardItems = section.data?.card_items ? 
      (typeof section.data.card_items === 'string' ? 
        section.data.card_items.split(',').map((item: string) => item.trim()) : 
        section.data.card_items) : [];

    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        <div className="bg-white dark:bg-slate-800 rounded-2xl shadow-md p-12 border border-gray-100 dark:border-slate-700">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-start">
            {/* Left side - Image */}
            <div className={`relative overflow-hidden rounded-2xl shadow-md transition-all duration-700 delay-200 transform order-2 lg:order-1 ${
              visibleSections.has(index) 
                ? 'opacity-100 translate-x-0' 
                : 'opacity-0 -translate-x-8'
            }`}>
              {section.image ? (
                <img 
                  src={`backend/api/uploads/${section.image}`} 
                  alt={section.title}
                  className="w-full h-80 object-cover hover:scale-105 transition-transform duration-500"
                />
              ) : (
                <div className="w-full h-80 bg-gradient-to-br from-[#ED1C24]/20 to-[#1A1A40]/20 flex items-center justify-center">
                  <Eye className="h-16 w-16 text-[#ED1C24]/50" />
                </div>
              )}
            </div>
            
            {/* Right side - Text with expandable card */}
            <div className={`space-y-6 transition-all duration-700 delay-400 transform order-1 lg:order-2 ${
              visibleSections.has(index) 
                ? 'opacity-100 translate-x-0' 
                : 'opacity-0 translate-x-8'
            }`}>
              <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white mb-6">
                {section.title}
              </h2>
              <div className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed whitespace-pre-wrap">
                {contentText}
              </div>
              
              {/* Expandable Card Details */}
              {(section.data?.card_title || section.data?.card_subtitle || cardItems.length > 0) && (
                <div className="bg-gray-50 dark:bg-slate-700 rounded-xl p-6">
                  {section.data?.card_title && (
                    <h3 className="text-xl font-semibold text-[#1A1A40] dark:text-white mb-2">
                      {section.data.card_title}
                    </h3>
                  )}
                  {section.data?.card_subtitle && (
                    <p className="text-gray-600 dark:text-gray-400 mb-4">
                      {section.data.card_subtitle}
                    </p>
                  )}
                  {cardItems.length > 0 && (
                    <ul className="space-y-2">
                      {cardItems.map((item: string, idx: number) => (
                        <li key={idx} className="flex items-start">
                          <span className="text-[#ED1C24] mr-2">•</span>
                          <span className="text-gray-700 dark:text-gray-300">{item}</span>
                        </li>
                      ))}
                    </ul>
                  )}
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (templateType === 'governance_principles') {
    const contentText = section.data?.card_content 
      || section.content 
      || (section.data?.content_items && Array.isArray(section.data.content_items) 
          ? section.data.content_items.join('\n\n') 
          : '');

    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        <div className="bg-white dark:bg-slate-800 rounded-2xl shadow-md p-12 border border-gray-100 dark:border-slate-700">
          <div className={`flex items-center mb-8 transition-all duration-700 delay-200 transform ${
            visibleSections.has(index) 
              ? 'opacity-100 translate-x-0' 
              : 'opacity-0 -translate-x-8'
          }`}>
            <div className="p-4 bg-gradient-to-br from-[#ED1C24] to-[#ED1C24]/80 rounded-xl shadow-md">
              {(() => {
                const iconName = section.data?.main_icon || 'Shield';
                const iconMap: Record<string, any> = {
                  Shield, Award, Users, Heart, Target, Globe, Lock, Star, 
                  CheckCircle, Compass, Eye, Handshake, Lightbulb, TrendingUp,
                  Building, BookOpen, Zap, Settings, Crown, Rocket
                };
                const IconComponent = iconMap[iconName] || Shield;
                return <IconComponent className="h-8 w-8 text-white" />;
              })()}
            </div>
            <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white ml-6 hover:text-[#ED1C24] transition-colors duration-300">
              {section.title}
            </h2>
          </div>
          
          <div className={`transition-all duration-700 delay-400 transform ${
            visibleSections.has(index) 
              ? 'opacity-100 translate-y-0' 
              : 'opacity-0 translate-y-8'
          }`}>
            <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed mb-8 whitespace-pre-wrap">
              {contentText}
            </p>
            
            {/* Principle Items (existing cards with icons) */}
            {section.data?.principle_items && Array.isArray(section.data.principle_items) && section.data.principle_items.length > 0 && (
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                {section.data.principle_items.map((principle: any, principleIndex: number) => (
                  <div key={principleIndex} className="bg-gray-50 dark:bg-slate-700 p-6 rounded-xl hover:shadow-md transition-all duration-300 hover:scale-[1.02]">
                    <div className="flex items-start space-x-4">
                      <div className="p-2 bg-[#ED1C24]/10 rounded-lg">
                        {(() => {
                          const iconName = principle.icon || 'Shield';
                          const iconMap: Record<string, any> = {
                            Shield, Award, Users, Heart, Target, Globe, Lock, Star, 
                            CheckCircle, Compass, Eye, Handshake, Lightbulb, TrendingUp,
                            Building, BookOpen, Zap, Settings, Crown, Rocket
                          };
                          const IconComponent = iconMap[iconName] || Shield;
                          return <IconComponent className="h-6 w-6 text-[#ED1C24]" />;
                        })()}
                      </div>
                      <div>
                        <h3 className="text-lg font-semibold text-[#1A1A40] dark:text-white mb-2">
                          {principle.title}
                        </h3>
                        <p className="text-gray-600 dark:text-gray-400 text-sm leading-relaxed">
                          {principle.description}
                        </p>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            )}
            
            {/* Numbered Cards (new Core Learning Modules style) */}
            {section.data?.numbered_cards && Array.isArray(section.data.numbered_cards) && section.data.numbered_cards.length > 0 && (
              <div className="mb-8">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  {section.data.numbered_cards.map((card: {title: string; content?: string}, cardIndex: number) => (
                    <div key={cardIndex} className="bg-gray-50 dark:bg-slate-700 p-4 rounded-xl hover:shadow-md transition-all duration-300 hover:scale-[1.02]">
                      <div className="flex items-start space-x-4">
                        <div className="w-8 h-8 bg-[#ED1C24] text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-1">
                          {cardIndex + 1}
                        </div>
                        <div>
                          <h3 className="text-lg font-semibold text-gray-700 dark:text-white mb-2">{card.title}</h3>
                          {card.content && (
                            <p className="text-gray-700 dark:text-gray-300 leading-relaxed">{card.content}</p>
                          )}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}
            
            {/* Governance Structure Box */}
            {section.data?.governance_structure && (
              <div className="bg-gradient-to-r from-[#ED1C24]/5 to-[#1A1A40]/5 p-6 rounded-xl border border-[#ED1C24]/20">
                <p className="text-gray-700 dark:text-gray-300 leading-relaxed">
                  {section.data.governance_structure}
                </p>
              </div>
            )}
          </div>
        </div>
      </div>
    );
  }

  if (templateType === 'why_numbers_matter') {
    // Get items from data (already parsed from database JSONB)
    const items = (() => {
      const matterItems = section.data?.matter_items;
      // If it's already an array, use it directly
      if (Array.isArray(matterItems)) {
        return matterItems;
      }
      // If it's a string, try to parse it
      if (typeof matterItems === 'string') {
        try {
          return JSON.parse(matterItems);
        } catch {
          return [];
        }
      }
      return [];
    })();

    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        <div className="bg-white dark:bg-slate-800 rounded-2xl shadow-md p-12 border border-gray-100 dark:border-slate-700">
          <div className={`text-center mb-8 transition-all duration-700 delay-200 transform ${
            visibleSections.has(index) 
              ? 'opacity-100 translate-y-0' 
              : 'opacity-0 translate-y-8'
          }`}>
            <div className="flex items-center justify-center mb-4">
              <div className="p-4 bg-gradient-to-br from-[#ED1C24] to-[#ED1C24]/80 rounded-xl shadow-md">
                {(() => {
                  const iconName = section.data?.main_icon || 'DollarSign';
                  const iconMap: Record<string, any> = {
                    Scale, Users, Building, Target, Award, Shield, Heart, Globe, Lock, Star, 
                    CheckCircle, Compass, Eye, Handshake, Lightbulb, TrendingUp,
                    BookOpen, Zap, Settings, Crown, Rocket, DollarSign
                  };
                  const IconComponent = iconMap[iconName] || Award;
                  return <IconComponent className="h-8 w-8 text-white" />;
                })()}
              </div>
              <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white ml-6 hover:text-[#ED1C24] transition-colors duration-300">
                {section.title}
              </h2>
            </div>
            {section.data?.subtitle && (
              <p className="text-lg text-gray-600 dark:text-gray-400 max-w-2xl mx-auto">
                {section.data.subtitle}
              </p>
            )}
          </div>
          

          {/* Items Grid - Dynamic layout based on item types */}
          <div className="grid gap-8 grid-cols-1 md:grid-cols-2">
            {items.length > 0 ? items.map((item: any, itemIndex: number) => (
              <div 
                key={itemIndex}
                className={`group ${item.type === 'pricing' ? 'bg-gray-50 dark:bg-slate-700 p-8 rounded-xl hover:shadow-md transition-all duration-300 hover:scale-[1.02] border border-gray-200 dark:border-slate-600' : 'bg-gradient-to-br from-gray-50 to-gray-100 dark:from-slate-700 dark:to-slate-600 p-8 rounded-xl shadow-md hover:shadow-lg transition-all duration-500 hover:scale-[1.02] hover:bg-gradient-to-br hover:from-white hover:to-gray-50 dark:hover:from-slate-600 dark:hover:to-slate-500 cursor-pointer'} transition-all duration-700 transform ${
                  visibleSections.has(index) 
                    ? 'opacity-100 translate-y-0' 
                    : 'opacity-0 translate-y-8'
                }`}
                style={{ 
                  transitionDelay: `${400 + (itemIndex * 150)}ms` 
                }}
              >
                {item.type === 'pricing' ? (
                  // Pricing item layout (center-aligned like IBDL Essential)
                  <div className="text-center">
                    {/* Icon */}
                    <div className="p-3 bg-[#ED1C24]/10 rounded-lg w-fit mx-auto mb-4">
                      {(() => {
                        const iconName = item.icon || 'DollarSign';
                        const iconMap: Record<string, any> = {
                          Scale, Users, Building, Target, Award, Shield, Heart, Globe, Lock, Star, 
                          CheckCircle, Compass, Eye, Handshake, Lightbulb, TrendingUp,
                          BookOpen, Zap, Settings, Crown, Rocket, DollarSign
                        };
                        const IconComponent = iconMap[iconName] || DollarSign;
                        return <IconComponent className="h-8 w-8 text-[#ED1C24]" />;
                      })()}
                    </div>
                    
                    {/* Title */}
                    <h3 className="text-xl font-semibold text-[#1A1A40] dark:text-white mb-3">
                      {item.title}
                    </h3>
                    
                    {/* Price */}
                    <p className="text-4xl font-bold text-[#ED1C24] mb-4">
                      {item.price}
                    </p>
                    
                    {/* Footer */}
                    <p className="text-gray-600 dark:text-gray-400 leading-relaxed">
                      {item.footer}
                    </p>
                  </div>
                ) : item.type === 'testimonial' ? (
                  // Testimonial layout (like IBDL Essential "What Learners Say")
                  <div className="flex items-start space-x-4">
                    <div className="p-2 bg-[#ED1C24]/10 rounded-lg">
                      <CheckCircle className="h-6 w-6 text-[#ED1C24]" />
                    </div>
                    <div>
                      <p className="text-gray-700 dark:text-gray-300 italic leading-relaxed mb-4">
                        "{item.quote}"
                      </p>
                      <p className="font-semibold text-[#1A1A40] dark:text-white">— {item.name}</p>
                    </div>
                  </div>
                ) : (
                  // Matter item layout (original layout)
                  <div className="flex items-start space-x-4">
                    <div className="p-3 bg-[#ED1C24]/10 rounded-lg flex-shrink-0 group-hover:bg-[#ED1C24]/20 group-hover:scale-110 transition-all duration-300">
                      {(() => {
                        const iconName = item.icon || 'Award';
                        const iconMap: Record<string, any> = {
                          Scale, Users, Building, Target, Award, Shield, Heart, Globe, Lock, Star, 
                          CheckCircle, Compass, Eye, Handshake, Lightbulb, TrendingUp,
                          BookOpen, Zap, Settings, Crown, Rocket, DollarSign
                        };
                        const IconComponent = iconMap[iconName] || Award;
                        return <IconComponent className="h-7 w-7 text-[#ED1C24] group-hover:scale-110 transition-transform duration-300" />;
                      })()}
                    </div>
                    <div>
                      <h3 className="text-xl font-bold text-[#1A1A40] dark:text-white mb-3 group-hover:text-[#ED1C24] transition-colors duration-300">
                        {item.title}
                      </h3>
                      <p className="text-gray-600 dark:text-gray-400 leading-relaxed group-hover:text-gray-700 dark:group-hover:text-gray-300 transition-colors duration-300">
                        {item.description}
                      </p>
                    </div>
                  </div>
                )}
              </div>
            )) : (
              <div className="col-span-full text-center py-8 text-gray-500 dark:text-gray-400">
                No items configured yet.
              </div>
            )}
          </div>
        </div>
      </div>
    );
  }

  if (templateType === 'narrative_summary') {
    return (
      <div 
        key={section.id}
        ref={(el) => sectionRefs.current[index] = el}
        data-section={index.toString()}
        className={`transition-all duration-1000 transform ${
          visibleSections.has(index) 
            ? 'opacity-100 translate-y-0' 
            : 'opacity-0 translate-y-12'
        }`}
      >
        <div className="relative overflow-hidden rounded-2xl shadow-md">
          <div className="absolute inset-0 bg-gradient-to-r from-[#ED1C24] via-red-600 to-[#1A1A40]"></div>
          <div className="absolute inset-0 opacity-20" style={{
            backgroundImage: `url("data:image/svg+xml,%3Csvg width='100' height='100' viewBox='0 0 100 100' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23ffffff' fill-opacity='0.3'%3E%3Ccircle cx='50' cy='50' r='20'/%3E%3Ccircle cx='50' cy='50' r='35'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`
          }}></div>
          
          <div className="relative py-16 px-12 text-white">
            <div className={`flex items-center justify-center mb-8 transition-all duration-700 delay-200 transform ${
              visibleSections.has(index) 
                ? 'opacity-100 translate-y-0' 
                : 'opacity-0 translate-y-8'
            }`}>
              <div className="p-4 bg-white/10 rounded-xl backdrop-blur-sm">
                {(() => {
                  const iconName = section.data?.main_icon || 'Globe';
                  const IconComponent = iconMap[iconName] || Globe;
                  return <IconComponent className="h-8 w-8 text-white" />;
                })()}
              </div>
              <h2 className="text-4xl font-bold ml-6">{section.title}</h2>
            </div>
            
            <div className={`text-center space-y-8 transition-all duration-700 delay-400 transform ${
              visibleSections.has(index) 
                ? 'opacity-100 translate-y-0' 
                : 'opacity-0 translate-y-8'
            }`}>
              {section.data?.subtitle && (
                <h3 className="text-3xl font-semibold">
                  {section.data.subtitle}
                </h3>
              )}
              
              {/* Render multiple content items */}
              {section.data?.content_items && Array.isArray(section.data.content_items) && section.data.content_items.length > 0 && (
                <div className="space-y-4 max-w-5xl mx-auto">
                  {section.data.content_items.map((contentItem: string, contentIndex: number) => (
                    <p key={contentIndex} className="text-xl leading-relaxed">
                      {contentItem}
                    </p>
                  ))}
                </div>
              )}
              
              {/* Fallback to single content field for backward compatibility */}
              {(!section.data?.content_items || !Array.isArray(section.data.content_items) || section.data.content_items.length === 0) && section.content && (
                <p className="text-xl leading-relaxed max-w-5xl mx-auto">
                  {section.content}
                </p>
              )}
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Default fallback for unknown template types
  return (
    <div 
      key={section.id}
      ref={(el) => sectionRefs.current[index] = el}
      data-section={index.toString()}
      className={`transition-all duration-1000 transform ${
        visibleSections.has(index) 
          ? 'opacity-100 translate-y-0' 
          : 'opacity-0 translate-y-12'
      }`}
    >
      <div className="bg-white dark:bg-slate-800 rounded-2xl shadow-md p-12 border border-gray-100 dark:border-slate-700">
        <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white mb-6">{section.title}</h2>
        {section.content && (
          <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed">
            {section.content}
          </p>
        )}
        {section.image && (
          <img 
            src={`backend/api/uploads/${section.image}`} 
            alt={section.title} 
            className="mt-6 rounded-lg w-full max-w-2xl mx-auto" 
          />
        )}
      </div>
    </div>
  );
}