import { useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { isUnauthorizedError } from "@/lib/authUtils";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { StatsCard } from "@/components/ui/stats-card";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { 
  Users, 
  DollarSign, 
  Star, 
  Award,
  TrendingUp,
  UserPlus,
  Calendar,
  BarChart3,
  BookOpen,
  GraduationCap,
  MessageSquare
} from "lucide-react";

export default function PartnerDashboard() {
  const { toast } = useToast();
  const { isAuthenticated, isLoading } = useAuth();

  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "backend/api/login";
      }, 500);
      return;
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: dashboardData, isLoading: dataLoading } = useQuery({
    queryKey: ["backend/api/dashboard/partner"],
    enabled: isAuthenticated,
    onError: (error: Error) => {
      if (isUnauthorizedError(error)) {
        toast({
          title: "Unauthorized",
          description: "You are logged out. Logging in again...",
          variant: "destructive",
        });
        setTimeout(() => {
          window.location.href = "backend/api/login";
        }, 500);
        return;
      }
    },
  });

  if (isLoading || dataLoading) {
    return (
      <DashboardLayout userType="partner">
        <div className="flex items-center justify-center h-64">
          <div className="text-center">
            <div className="w-12 h-12 bg-purple-600 rounded-lg flex items-center justify-center mx-auto mb-4 animate-pulse">
              <Users className="w-6 h-6 text-white" />
            </div>
            <p className="text-muted-foreground">Loading dashboard...</p>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  const stats = dashboardData?.stats || {};
  const students = dashboardData?.students || [];
  const topCourses = dashboardData?.topCourses || [];
  const organization = dashboardData?.organization;

  const revenueBreakdown = [
    {
      source: "Course Enrollments",
      amount: 28400,
      percentage: 60,
      students: 142,
      color: "bg-blue-100 text-blue-600"
    },
    {
      source: "Certification Fees", 
      amount: 13350,
      percentage: 28,
      students: 89,
      color: "bg-emerald-100 text-emerald-600"
    },
    {
      source: "Assessment Services",
      amount: 5480,
      percentage: 12,
      students: 67,
      color: "bg-purple-100 text-purple-600"
    }
  ];

  const topPerformingCourses = [
    {
      title: "Digital Marketing Excellence",
      students: 67,
      completion: 94,
      revenue: 8340,
      rating: 4.9
    },
    {
      title: "Project Management Certification",
      students: 52,
      completion: 89,
      revenue: 6760,
      rating: 4.7
    },
    {
      title: "Business Intelligence Fundamentals",
      students: 41,
      completion: 76,
      revenue: 4920,
      rating: 4.6
    },
    {
      title: "Advanced Leadership Skills",
      students: 38,
      completion: 82,
      revenue: 4560,
      rating: 4.8
    }
  ];

  const instructorPerformance = [
    {
      name: "Dr. Jane Doe",
      subject: "Project Management",
      rating: 4.9,
      reviews: 89,
      initials: "JD"
    },
    {
      name: "Mike Smith", 
      subject: "Digital Marketing",
      rating: 4.7,
      reviews: 67,
      initials: "MS"
    },
    {
      name: "Sarah Wilson",
      subject: "Business Intelligence", 
      rating: 4.6,
      reviews: 54,
      initials: "SW"
    }
  ];

  const recentActivities = [
    {
      message: "15 students enrolled in Digital Marketing Excellence this week",
      time: "2 hours ago",
      icon: UserPlus,
      color: "bg-blue-500"
    },
    {
      message: "23 students successfully completed Project Management Certification",
      time: "4 hours ago", 
      icon: Award,
      color: "bg-emerald-500"
    },
    {
      message: "New batch of Business Intelligence certifications processed",
      time: "1 day ago",
      icon: GraduationCap,
      color: "bg-purple-500"
    },
    {
      message: "Training center received 4.8/5.0 quality rating from recent audit",
      time: "2 days ago",
      icon: Star,
      color: "bg-amber-500"
    }
  ];

  return (
    <DashboardLayout userType="partner">
      <div className="space-y-8">
        {/* Business Performance Summary */}
        <div className="bg-gradient-to-r from-emerald-600 to-green-600 rounded-xl p-6 text-white">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold mb-2">Training Center Performance</h1>
              <p className="text-green-100">
                Real-time business intelligence and performance metrics for your training center operations
              </p>
            </div>
            <div className="flex items-center space-x-4">
              <div className="text-center">
                <div className="text-2xl font-bold">${stats.monthlyRevenue?.toLocaleString() || 0}</div>
                <div className="text-xs text-green-100">Monthly Revenue</div>
              </div>
              <div className="text-center">
                <div className="text-2xl font-bold">{stats.qualityScore || 0}★</div>
                <div className="text-xs text-green-100">Quality Rating</div>
              </div>
            </div>
          </div>
        </div>

        {/* Business Metrics */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <StatsCard
            title="Active Students"
            value={stats.activeStudents || 0}
            change="↗ +8.3% this month"
            changeType="positive"
            icon={Users}
            iconColor="bg-blue-100 text-blue-600"
          />
          <StatsCard
            title="Monthly Revenue"
            value={`$${stats.monthlyRevenue?.toLocaleString() || 0}`}
            change="↗ +12.7% vs last month"
            changeType="positive"
            icon={DollarSign}
            iconColor="bg-emerald-100 text-emerald-600"
          />
          <StatsCard
            title="Success Rate"
            value={`${stats.successRate || 0}%`}
            change="↗ +2.1% this quarter"
            changeType="positive"
            icon={TrendingUp}
            iconColor="bg-amber-100 text-amber-600"
          />
          <StatsCard
            title="Quality Score"
            value={`${stats.qualityScore || 0}/5.0`}
            change="Excellent rating"
            changeType="positive"
            icon={Star}
            iconColor="bg-purple-100 text-purple-600"
          />
        </div>

        {/* Revenue Analytics & Course Performance */}
        <div className="grid lg:grid-cols-2 gap-8">
          <Card className="dashboard-card">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg font-semibold">Revenue Breakdown</CardTitle>
                <select className="text-sm border border-slate-300 rounded-lg px-3 py-1 text-slate-600">
                  <option>This Month</option>
                  <option>Last Month</option>
                  <option>This Quarter</option>
                </select>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {revenueBreakdown.map((item, index) => (
                  <div key={index} className="flex items-center justify-between p-4 bg-slate-50 rounded-lg">
                    <div className="flex items-center space-x-3">
                      <div className={`w-10 h-10 ${item.color} rounded-lg flex items-center justify-center`}>
                        <DollarSign className="w-5 h-5" />
                      </div>
                      <div>
                        <p className="text-sm font-medium text-foreground">{item.source}</p>
                        <p className="text-xs text-muted-foreground">{item.students} students</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-semibold text-foreground">${item.amount.toLocaleString()}</p>
                      <p className="text-xs text-muted-foreground">{item.percentage}% of total</p>
                    </div>
                  </div>
                ))}
              </div>

              <div className="mt-6 p-4 bg-gradient-to-r from-emerald-50 to-blue-50 rounded-lg">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-foreground">Projected Monthly Growth</p>
                    <p className="text-2xl font-bold text-emerald-600">+15.4%</p>
                  </div>
                  <div className="text-right">
                    <p className="text-sm text-muted-foreground">Next Month Projection</p>
                    <p className="text-lg font-semibold text-foreground">$54,500</p>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="dashboard-card">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg font-semibold">Top Performing Courses</CardTitle>
                <Button variant="ghost" size="sm">
                  View All Courses
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {topPerformingCourses.map((course, index) => (
                  <div key={index} className="flex items-center space-x-4 p-4 bg-slate-50 rounded-lg">
                    <div className="w-16 h-12 bg-gradient-to-br from-blue-500 to-purple-600 rounded-lg flex items-center justify-center">
                      <BookOpen className="w-6 h-6 text-white" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <h4 className="text-sm font-medium text-foreground">{course.title}</h4>
                      <p className="text-xs text-muted-foreground">
                        {course.students} enrolled • {course.completion}% completion
                      </p>
                      <div className="mt-2 flex items-center space-x-4">
                        <div className="flex items-center text-emerald-600">
                          <span className="text-xs font-medium">${course.revenue.toLocaleString()} revenue</span>
                        </div>
                        <div className="flex items-center text-amber-600">
                          <Star className="w-3 h-3 mr-1 fill-current" />
                          <span className="text-xs">{course.rating}</span>
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Business Intelligence & Student Management */}
        <div className="grid lg:grid-cols-3 gap-8">
          <Card className="dashboard-card">
            <CardHeader>
              <CardTitle className="text-lg font-semibold">Business Actions</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <Button className="w-full justify-start bg-blue-50 text-blue-700 hover:bg-blue-100">
                  <UserPlus className="w-5 h-5 mr-3" />
                  Enroll New Students
                </Button>
                <Button className="w-full justify-start bg-emerald-50 text-emerald-700 hover:bg-emerald-100">
                  <Calendar className="w-5 h-5 mr-3" />
                  Schedule Exams
                </Button>
                <Button className="w-full justify-start bg-purple-50 text-purple-700 hover:bg-purple-100">
                  <BarChart3 className="w-5 h-5 mr-3" />
                  Revenue Reports
                </Button>
                <Button className="w-full justify-start bg-amber-50 text-amber-700 hover:bg-amber-100">
                  <MessageSquare className="w-5 h-5 mr-3" />
                  Student Support
                </Button>
              </div>
            </CardContent>
          </Card>

          <Card className="dashboard-card">
            <CardHeader>
              <CardTitle className="text-lg font-semibold">Instructor Performance</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {instructorPerformance.map((instructor, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-slate-50 rounded-lg">
                    <div className="flex items-center space-x-3">
                      <Avatar className="w-8 h-8">
                        <AvatarFallback className="text-xs font-medium text-primary bg-primary/10">
                          {instructor.initials}
                        </AvatarFallback>
                      </Avatar>
                      <div>
                        <div className="font-medium text-sm">{instructor.name}</div>
                        <div className="text-xs text-muted-foreground">{instructor.subject}</div>
                      </div>
                    </div>
                    <div className="text-right">
                      <div className="font-semibold text-amber-600 text-sm flex items-center">
                        <Star className="w-3 h-3 mr-1 fill-current" />
                        {instructor.rating}
                      </div>
                      <div className="text-xs text-muted-foreground">{instructor.reviews} reviews</div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          <Card className="dashboard-card">
            <CardHeader>
              <CardTitle className="text-lg font-semibold">Recent Student Activity</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {recentActivities.map((activity, index) => (
                  <div key={index} className="flex items-start space-x-3">
                    <div className={`w-8 h-8 ${activity.color} rounded-full flex items-center justify-center flex-shrink-0`}>
                      <activity.icon className="w-4 h-4 text-white" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <p className="text-sm text-foreground">{activity.message}</p>
                      <p className="text-xs text-muted-foreground">{activity.time}</p>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  );
}
