import { useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { isUnauthorizedError } from "@/lib/authUtils";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { 
  BookOpen, 
  Award, 
  Clock, 
  TrendingUp, 
  Calendar,
  Play,
  CheckCircle,
  AlertCircle,
  Star,
  Target
} from "lucide-react";

export default function LearnerDashboard() {
  const { toast } = useToast();
  const { isAuthenticated, isLoading } = useAuth();

  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "backend/api/login";
      }, 500);
      return;
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: dashboardData, isLoading: dataLoading } = useQuery({
    queryKey: ["backend/api/dashboard/learner"],
    enabled: isAuthenticated,
    onError: (error: Error) => {
      if (isUnauthorizedError(error)) {
        toast({
          title: "Unauthorized",
          description: "You are logged out. Logging in again...",
          variant: "destructive",
        });
        setTimeout(() => {
          window.location.href = "backend/api/login";
        }, 500);
        return;
      }
    },
  });

  if (isLoading || dataLoading) {
    return (
      <DashboardLayout userType="learner">
        <div className="flex items-center justify-center h-64">
          <div className="text-center">
            <div className="w-12 h-12 bg-primary rounded-lg flex items-center justify-center mx-auto mb-4 animate-pulse">
              <BookOpen className="w-6 h-6 text-white" />
            </div>
            <p className="text-muted-foreground">Loading your dashboard...</p>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  const stats = dashboardData?.stats || {};
  const recentEnrollments = dashboardData?.recentEnrollments || [];
  const recentCertificates = dashboardData?.recentCertificates || [];

  return (
    <DashboardLayout userType="learner">
      <div className="space-y-8">
        {/* Welcome Section */}
        <div className="bg-gradient-to-r from-primary to-blue-600 rounded-xl p-6 text-white">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold mb-2">Welcome back! 👋</h1>
              <p className="text-blue-100">You're making great progress. Here's your learning overview for today.</p>
            </div>
            <div className="text-right">
              <div className="text-3xl font-bold">{Math.round((stats.completedCourses / (stats.activeCourses + stats.completedCourses || 1)) * 100)}%</div>
              <div className="text-sm text-blue-100">Overall Progress</div>
            </div>
          </div>
        </div>

        {/* Quick Stats */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <Card className="dashboard-card">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Active Courses</p>
                  <p className="text-2xl font-bold text-foreground">{stats.activeCourses || 0}</p>
                </div>
                <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                  <BookOpen className="w-6 h-6 text-blue-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="dashboard-card">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Completed</p>
                  <p className="text-2xl font-bold text-foreground">{stats.completedCourses || 0}</p>
                </div>
                <div className="w-12 h-12 bg-emerald-100 rounded-lg flex items-center justify-center">
                  <CheckCircle className="w-6 h-6 text-emerald-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="dashboard-card">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Learning Time</p>
                  <p className="text-2xl font-bold text-foreground">{Math.round(stats.totalLearningTime || 0)}h</p>
                </div>
                <div className="w-12 h-12 bg-amber-100 rounded-lg flex items-center justify-center">
                  <Clock className="w-6 h-6 text-amber-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="dashboard-card">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Certificates</p>
                  <p className="text-2xl font-bold text-foreground">{stats.certificates || 0}</p>
                </div>
                <div className="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
                  <Award className="w-6 h-6 text-purple-600" />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Current Learning Activities */}
        <div className="grid lg:grid-cols-2 gap-8">
          <Card className="dashboard-card">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg font-semibold">Continue Learning</CardTitle>
                <Button variant="ghost" size="sm">
                  View All
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {recentEnrollments.length > 0 ? (
                <div className="space-y-4">
                  {recentEnrollments.slice(0, 3).map((enrollment: any, index: number) => (
                    <div key={index} className="flex items-center space-x-4 p-4 bg-slate-50 rounded-lg hover:bg-slate-100 transition-colors cursor-pointer">
                      <div className="w-16 h-12 bg-gradient-to-br from-blue-500 to-purple-600 rounded-lg flex items-center justify-center">
                        <BookOpen className="w-6 h-6 text-white" />
                      </div>
                      <div className="flex-1 min-w-0">
                        <h4 className="text-sm font-medium text-foreground truncate">
                          Course {index + 1}
                        </h4>
                        <p className="text-xs text-muted-foreground">
                          Progress: {enrollment.progress || 0}%
                        </p>
                        <div className="mt-2 w-full">
                          <Progress value={parseFloat(enrollment.progress || 0)} className="h-2" />
                        </div>
                      </div>
                      <Button size="sm" className="flex-shrink-0">
                        <Play className="w-4 h-4" />
                      </Button>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <BookOpen className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                  <p className="text-muted-foreground">No active courses yet</p>
                  <Button className="mt-4" size="sm">
                    Browse Catalog
                  </Button>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Upcoming Deadlines & Assessments */}
          <Card className="dashboard-card">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg font-semibold">Upcoming Deadlines</CardTitle>
                <Button variant="ghost" size="sm">
                  <Calendar className="w-4 h-4" />
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {stats.assessmentsPending > 0 ? (
                  <>
                    <div className="flex items-center space-x-4 p-4 border border-amber-200 bg-amber-50 rounded-lg">
                      <div className="flex-shrink-0 w-12 h-12 bg-amber-100 rounded-lg flex items-center justify-center">
                        <Clock className="w-6 h-6 text-amber-600" />
                      </div>
                      <div className="flex-1 min-w-0">
                        <h4 className="text-sm font-medium text-foreground">Assessment Due</h4>
                        <p className="text-xs text-amber-600 font-medium">Due in 3 days</p>
                        <p className="text-xs text-muted-foreground">Practice assessment available</p>
                      </div>
                      <Button size="sm" variant="outline" className="text-amber-600 border-amber-600">
                        Prepare
                      </Button>
                    </div>

                    <div className="flex items-center space-x-4 p-4 border border-blue-200 bg-blue-50 rounded-lg">
                      <div className="flex-shrink-0 w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                        <Target className="w-6 h-6 text-blue-600" />
                      </div>
                      <div className="flex-1 min-w-0">
                        <h4 className="text-sm font-medium text-foreground">PQP Assessment</h4>
                        <p className="text-xs text-blue-600 font-medium">Scheduled for next week</p>
                        <p className="text-xs text-muted-foreground">Personality evaluation</p>
                      </div>
                      <Button size="sm" variant="outline" className="text-blue-600 border-blue-600">
                        Schedule
                      </Button>
                    </div>
                  </>
                ) : (
                  <div className="text-center py-8">
                    <Calendar className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                    <p className="text-muted-foreground">No upcoming deadlines</p>
                    <p className="text-sm text-muted-foreground mt-2">You're all caught up!</p>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Recent Achievements & Recommendations */}
        <div className="grid lg:grid-cols-3 gap-8">
          <Card className="dashboard-card">
            <CardHeader>
              <CardTitle className="text-lg font-semibold">Recent Achievements</CardTitle>
            </CardHeader>
            <CardContent>
              {recentCertificates.length > 0 ? (
                <div className="space-y-4">
                  {recentCertificates.slice(0, 3).map((cert: any, index: number) => (
                    <div key={index} className="flex items-center space-x-3">
                      <div className="w-10 h-10 bg-yellow-100 rounded-full flex items-center justify-center">
                        <Star className="w-5 h-5 text-yellow-600" />
                      </div>
                      <div>
                        <p className="text-sm font-medium text-foreground">Certificate Earned!</p>
                        <p className="text-xs text-muted-foreground">{cert.title || 'Professional Course'}</p>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <Award className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                  <p className="text-muted-foreground">No achievements yet</p>
                  <p className="text-sm text-muted-foreground mt-2">Complete courses to earn certificates</p>
                </div>
              )}
            </CardContent>
          </Card>

          <div className="lg:col-span-2">
            <Card className="dashboard-card">
              <CardHeader>
                <CardTitle className="text-lg font-semibold">Recommended for You</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid md:grid-cols-2 gap-4">
                  <div className="border border-border rounded-lg p-4 hover:border-primary transition-colors cursor-pointer">
                    <div className="w-full h-32 bg-gradient-to-br from-blue-500 to-purple-600 rounded-lg mb-3 flex items-center justify-center">
                      <TrendingUp className="w-8 h-8 text-white" />
                    </div>
                    <div className="space-y-2">
                      <h4 className="text-sm font-medium text-foreground">Advanced Analytics</h4>
                      <p className="text-xs text-muted-foreground">Perfect follow-up to your current progress</p>
                      <div className="flex items-center justify-between">
                        <span className="text-xs text-muted-foreground">8 modules • 12 hours</span>
                        <Button size="sm" variant="outline">
                          Enroll
                        </Button>
                      </div>
                    </div>
                  </div>

                  <div className="border border-border rounded-lg p-4 hover:border-primary transition-colors cursor-pointer">
                    <div className="w-full h-32 bg-gradient-to-br from-emerald-500 to-teal-600 rounded-lg mb-3 flex items-center justify-center">
                      <Target className="w-8 h-8 text-white" />
                    </div>
                    <div className="space-y-2">
                      <h4 className="text-sm font-medium text-foreground">Strategic Management</h4>
                      <p className="text-xs text-muted-foreground">Build on your leadership skills</p>
                      <div className="flex items-center justify-between">
                        <span className="text-xs text-muted-foreground">6 modules • 15 hours</span>
                        <Button size="sm" variant="outline">
                          Enroll
                        </Button>
                      </div>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </DashboardLayout>
  );
}
