import { useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { isUnauthorizedError } from "@/lib/authUtils";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { StatsCard } from "@/components/ui/stats-card";
import { ProgressBar } from "@/components/ui/progress-bar";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { 
  Users, 
  BookOpen, 
  Award, 
  TrendingUp,
  UserPlus,
  FileCheck,
  BarChart3,
  Settings,
  Building,
  Shield,
  DollarSign,
  Zap
} from "lucide-react";

export default function CorporateDashboard() {
  const { toast } = useToast();
  const { isAuthenticated, isLoading } = useAuth();

  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "backend/api/login";
      }, 500);
      return;
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: dashboardData, isLoading: dataLoading } = useQuery({
    queryKey: ["backend/api/dashboard/corporate"],
    enabled: isAuthenticated,
    onError: (error: Error) => {
      if (isUnauthorizedError(error)) {
        toast({
          title: "Unauthorized",
          description: "You are logged out. Logging in again...",
          variant: "destructive",
        });
        setTimeout(() => {
          window.location.href = "backend/api/login";
        }, 500);
        return;
      }
    },
  });

  if (isLoading || dataLoading) {
    return (
      <DashboardLayout userType="corporate">
        <div className="flex items-center justify-center h-64">
          <div className="text-center">
            <div className="w-12 h-12 bg-emerald-600 rounded-lg flex items-center justify-center mx-auto mb-4 animate-pulse">
              <Building className="w-6 h-6 text-white" />
            </div>
            <p className="text-muted-foreground">Loading dashboard...</p>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  const stats = dashboardData?.stats || {};
  const users = dashboardData?.users || [];
  const courses = dashboardData?.courses || [];
  const organization = dashboardData?.organization;

  const departmentData = [
    { name: "Engineering", employees: 234, completion: 92, color: "bg-blue-500" },
    { name: "Sales", employees: 156, completion: 78, color: "bg-green-500" },
    { name: "Marketing", employees: 89, completion: 95, color: "bg-purple-500" },
    { name: "Operations", employees: 134, completion: 89, color: "bg-orange-500" },
  ];

  const recentActivities = [
    {
      type: "user_added",
      message: "24 new users added to Engineering Department",
      time: "2 hours ago",
      icon: UserPlus,
      color: "bg-green-500"
    },
    {
      type: "compliance",
      message: "Engineering team achieved 95% compliance rate",
      time: "5 hours ago",
      icon: Shield,
      color: "bg-emerald-500"
    },
    {
      type: "course",
      message: 'New certification program "Advanced Leadership" launched',
      time: "1 day ago",
      icon: Award,
      color: "bg-purple-500"
    },
    {
      type: "report",
      message: "Monthly learning analytics report generated",
      time: "2 days ago",
      icon: BarChart3,
      color: "bg-blue-500"
    }
  ];

  return (
    <DashboardLayout userType="corporate">
      <div className="space-y-8">
        {/* Executive Summary */}
        <div className="bg-gradient-to-r from-slate-850 to-slate-700 rounded-xl p-6 text-white">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold mb-2">Enterprise Learning Dashboard</h1>
              <p className="text-slate-300">
                Real-time insights into your organization's learning performance and compliance status
              </p>
            </div>
            <div className="flex items-center space-x-4 text-sm">
              <div className="flex items-center space-x-2">
                <div className="w-3 h-3 bg-green-400 rounded-full"></div>
                <span>All Systems Operational</span>
              </div>
            </div>
          </div>
        </div>

        {/* Key Metrics */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <StatsCard
            title="Active Employees"
            value={stats.totalUsers || 0}
            change="↗ +5.2% from last month"
            changeType="positive"
            icon={Users}
            iconColor="bg-blue-100 text-blue-600"
          />
          <StatsCard
            title="Course Completion"
            value={`${stats.completionRate || 0}%`}
            change="↗ +2.1% from last month"
            changeType="positive"
            icon={BookOpen}
            iconColor="bg-emerald-100 text-emerald-600"
          />
          <StatsCard
            title="Compliance Score"
            value={`${stats.compliance || 0}%`}
            change="↗ +0.8% from last month"
            changeType="positive"
            icon={Shield}
            iconColor="bg-amber-100 text-amber-600"
          />
          <StatsCard
            title="Training ROI"
            value={`${stats.trainingROI || 0}%`}
            change="↗ +12% from last quarter"
            changeType="positive"
            icon={DollarSign}
            iconColor="bg-purple-100 text-purple-600"
          />
        </div>

        {/* Department Performance & Learning Analytics */}
        <div className="grid lg:grid-cols-2 gap-8">
          <Card className="dashboard-card">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg font-semibold">Department Performance</CardTitle>
                <Button variant="ghost" size="sm">
                  View Details
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {departmentData.map((dept, index) => (
                  <div key={index} className="flex items-center justify-between p-4 bg-slate-50 rounded-lg">
                    <div className="flex items-center space-x-3">
                      <div className={`w-10 h-10 ${dept.color} rounded-lg flex items-center justify-center`}>
                        <Users className="w-5 h-5 text-white" />
                      </div>
                      <div>
                        <p className="text-sm font-medium text-foreground">{dept.name}</p>
                        <p className="text-xs text-muted-foreground">{dept.employees} employees</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-semibold text-foreground">{dept.completion}%</p>
                      <div className="w-20 bg-slate-200 rounded-full h-2 mt-1">
                        <div 
                          className={`${dept.color} h-2 rounded-full transition-all duration-500`}
                          style={{ width: `${dept.completion}%` }}
                        />
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          <Card className="dashboard-card">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg font-semibold">Learning Progress Trends</CardTitle>
                <select className="text-sm border border-slate-300 rounded-lg px-3 py-1 text-slate-600">
                  <option>Last 30 days</option>
                  <option>Last 90 days</option>
                  <option>Last 6 months</option>
                </select>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <ProgressBar value={87} label="Course Completions" color="primary" />
                <ProgressBar value={92} label="Assessment Scores" color="emerald" />
                <ProgressBar value={78} label="Engagement Rate" color="purple" />
                <ProgressBar value={84} label="Certification Rate" color="amber" />
              </div>

              <div className="mt-6 p-4 bg-emerald-50 rounded-lg">
                <div className="flex items-center space-x-2">
                  <TrendingUp className="w-5 h-5 text-emerald-600" />
                  <span className="text-sm font-medium text-emerald-800">
                    Overall performance up 12% this month
                  </span>
                </div>
                <p className="text-sm text-emerald-700 mt-1">
                  Best performing quarter with consistent growth across all departments.
                </p>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Quick Actions & Recent Activity */}
        <div className="grid lg:grid-cols-3 gap-8">
          <Card className="dashboard-card">
            <CardHeader>
              <CardTitle className="text-lg font-semibold">Quick Actions</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <Button className="w-full justify-start bg-blue-50 text-blue-700 hover:bg-blue-100">
                  <UserPlus className="w-5 h-5 mr-3" />
                  Add New Users
                </Button>
                <Button className="w-full justify-start bg-emerald-50 text-emerald-700 hover:bg-emerald-100">
                  <BookOpen className="w-5 h-5 mr-3" />
                  Assign Courses
                </Button>
                <Button className="w-full justify-start bg-purple-50 text-purple-700 hover:bg-purple-100">
                  <BarChart3 className="w-5 h-5 mr-3" />
                  Generate Reports
                </Button>
                <Button className="w-full justify-start bg-amber-50 text-amber-700 hover:bg-amber-100">
                  <Settings className="w-5 h-5 mr-3" />
                  Platform Settings
                </Button>
              </div>
            </CardContent>
          </Card>

          <div className="lg:col-span-2">
            <Card className="dashboard-card">
              <CardHeader>
                <CardTitle className="text-lg font-semibold">Recent Activity</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {recentActivities.map((activity, index) => (
                    <div key={index} className="flex items-start space-x-3">
                      <div className={`w-8 h-8 ${activity.color} rounded-full flex items-center justify-center flex-shrink-0`}>
                        <activity.icon className="w-4 h-4 text-white" />
                      </div>
                      <div className="flex-1 min-w-0">
                        <p className="text-sm text-foreground">{activity.message}</p>
                        <p className="text-xs text-muted-foreground">{activity.time}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </DashboardLayout>
  );
}
