import { useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { isUnauthorizedError } from "@/lib/authUtils";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { StatsCard } from "@/components/ui/stats-card";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { 
  Users, 
  Building, 
  Activity, 
  AlertTriangle,
  Monitor,
  Database,
  Shield,
  Zap,
  Server,
  Settings,
  UserCog,
  Archive,
  Award,
  BookOpen
} from "lucide-react";
import { Link } from "wouter";

export default function AdminDashboard() {
  const { toast } = useToast();
  const { isAuthenticated, isLoading } = useAuth();

  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "backend/api/login";
      }, 500);
      return;
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: dashboardData, isLoading: dataLoading } = useQuery({
    queryKey: ["backend/api/dashboard/admin"],
    enabled: isAuthenticated,
    onError: (error: Error) => {
      if (isUnauthorizedError(error)) {
        toast({
          title: "Unauthorized",
          description: "You are logged out. Logging in again...",
          variant: "destructive",
        });
        setTimeout(() => {
          window.location.href = "backend/api/login";
        }, 500);
        return;
      }
    },
  });

  if (isLoading || dataLoading) {
    return (
      <DashboardLayout userType="admin">
        <div className="flex items-center justify-center h-64">
          <div className="text-center">
            <div className="w-12 h-12 bg-slate-800 rounded-lg flex items-center justify-center mx-auto mb-4 animate-pulse">
              <Settings className="w-6 h-6 text-white" />
            </div>
            <p className="text-muted-foreground">Loading system dashboard...</p>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  const stats = dashboardData?.stats || {};
  const activeSessions = dashboardData?.activeSessions || [];
  const systemHealth = dashboardData?.systemHealth || {};

  const systemMetrics = [
    {
      title: "Total Users",
      value: stats.totalUsers || 0,
      change: "↗ +5.2%",
      changeType: "positive" as const,
      icon: Users,
      iconColor: "bg-blue-100 text-blue-600"
    },
    {
      title: "Active Sessions", 
      value: stats.activeSessions || 0,
      change: "Real-time",
      changeType: "neutral" as const,
      icon: Activity,
      iconColor: "bg-purple-100 text-purple-600"
    },
    {
      title: "System Uptime",
      value: `${stats.uptime || 0}%`,
      change: "30 days",
      changeType: "positive" as const,
      icon: Server,
      iconColor: "bg-emerald-100 text-emerald-600"
    },
    {
      title: "Active Alerts",
      value: stats.alerts || 0,
      change: "Needs attention",
      changeType: "negative" as const,
      icon: AlertTriangle,
      iconColor: "bg-red-100 text-red-600"
    }
  ];

  const systemHealthStatus = [
    {
      component: "Database Performance",
      status: systemHealth.database || "optimal",
      statusColor: "text-green-600",
      indicator: "bg-green-400"
    },
    {
      component: "API Response Time",
      status: systemHealth.apiResponse || "145ms avg",
      statusColor: "text-green-600", 
      indicator: "bg-green-400"
    },
    {
      component: "Storage Usage",
      status: systemHealth.storage || "78% capacity",
      statusColor: "text-yellow-600",
      indicator: "bg-yellow-400"
    },
    {
      component: "Security Scans",
      status: systemHealth.security || "all_clear",
      statusColor: "text-green-600",
      indicator: "bg-green-400"
    }
  ];

  const recentAdminActivities = [
    {
      message: "User role updated for Corporate Admin",
      time: "15 min ago",
      icon: UserCog,
      color: "bg-blue-500"
    },
    {
      message: "Security scan completed - No issues found",
      time: "1 hour ago",
      icon: Shield,
      color: "bg-green-500"
    },
    {
      message: "Database backup created (Automated backup)",
      time: "3 hours ago",
      icon: Database,
      color: "bg-purple-500"
    },
    {
      message: "System maintenance window scheduled",
      time: "6 hours ago",
      icon: Settings,
      color: "bg-orange-500"
    },
    {
      message: "New organization onboarded",
      time: "1 day ago",
      icon: Building,
      color: "bg-emerald-500"
    }
  ];

  return (
    <DashboardLayout userType="admin">
      <div className="space-y-8">
        {/* System Status Header */}
        <div className="bg-gradient-to-r from-slate-800 to-slate-700 rounded-xl p-6 text-white">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold mb-2">System Administration</h1>
              <p className="text-slate-300">Platform Operations & Management Console</p>
            </div>
            <div className="flex items-center space-x-4">
              <div className="flex items-center space-x-2">
                <div className="w-3 h-3 bg-green-400 rounded-full animate-pulse"></div>
                <span className="text-sm">All Systems Operational</span>
              </div>
            </div>
          </div>
        </div>

        {/* System Metrics */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {systemMetrics.map((metric, index) => (
            <StatsCard
              key={index}
              title={metric.title}
              value={metric.value}
              change={metric.change}
              changeType={metric.changeType}
              icon={metric.icon}
              iconColor={metric.iconColor}
            />
          ))}
        </div>

        {/* System Health & Live Sessions */}
        <div className="grid lg:grid-cols-2 gap-8">
          <Card className="dashboard-card">
            <CardHeader>
              <CardTitle className="text-lg font-semibold">System Health</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {systemHealthStatus.map((item, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-slate-50 rounded-lg">
                    <div className="flex items-center space-x-3">
                      <div className={`w-3 h-3 ${item.indicator} rounded-full`}></div>
                      <span className="font-medium text-sm">{item.component}</span>
                    </div>
                    <span className={`text-sm ${item.statusColor}`}>
                      {item.status === "optimal" ? "Optimal" : 
                       item.status === "all_clear" ? "All Clear" : item.status}
                    </span>
                  </div>
                ))}
              </div>

              <div className="mt-6 p-4 bg-emerald-50 rounded-lg">
                <div className="flex items-center space-x-2">
                  <Activity className="w-5 h-5 text-emerald-600" />
                  <span className="text-sm font-medium text-emerald-800">
                    System Performance: Excellent
                  </span>
                </div>
                <p className="text-sm text-emerald-700 mt-1">
                  All services running optimally with no critical issues detected.
                </p>
              </div>
            </CardContent>
          </Card>

          <Card className="dashboard-card">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg font-semibold">Live Examination Sessions</CardTitle>
                <Badge variant="secondary" className="bg-green-100 text-green-800">
                  {activeSessions.length} Active
                </Badge>
              </div>
            </CardHeader>
            <CardContent>
              {activeSessions.length > 0 ? (
                <div className="space-y-3">
                  {activeSessions.slice(0, 5).map((session: any, index: number) => (
                    <div key={index} className="flex items-center justify-between p-3 bg-slate-50 rounded-lg">
                      <div>
                        <p className="text-sm font-medium">Exam Session #{index + 1}</p>
                        <p className="text-xs text-muted-foreground">
                          Started: {new Date().toLocaleTimeString()}
                        </p>
                      </div>
                      <div className="flex items-center space-x-2">
                        <div className="w-2 h-2 bg-green-400 rounded-full animate-pulse"></div>
                        <span className="text-xs text-green-600 font-medium">Live</span>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <Monitor className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                  <p className="text-muted-foreground">No active examination sessions</p>
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        {/* Admin Controls & Recent Activities */}
        <div className="grid lg:grid-cols-3 gap-8">
          <Card className="dashboard-card">
            <CardHeader>
              <CardTitle className="text-lg font-semibold">System Controls</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <Link href="/accreditation">
                  <Button className="w-full justify-start bg-red-50 text-red-700 hover:bg-red-100">
                    <Award className="w-5 h-5 mr-3" />
                    Accreditation Management
                  </Button>
                </Link>
                <Link href="/question-bank">
                  <Button className="w-full justify-start bg-blue-50 text-blue-700 hover:bg-blue-100">
                    <BookOpen className="w-5 h-5 mr-3" />
                    Question Bank
                  </Button>
                </Link>
                <Link href="/cms">
                  <Button className="w-full justify-start bg-indigo-50 text-indigo-700 hover:bg-indigo-100">
                    <Database className="w-5 h-5 mr-3" />
                    CMS
                  </Button>
                </Link>
                <Button className="w-full justify-start bg-emerald-50 text-emerald-700 hover:bg-emerald-100">
                  <UserCog className="w-5 h-5 mr-3" />
                  Manage Users
                </Button>
                <Button className="w-full justify-start bg-purple-50 text-purple-700 hover:bg-purple-100">
                  <Building className="w-5 h-5 mr-3" />
                  Organizations
                </Button>
                <Button className="w-full justify-start bg-amber-50 text-amber-700 hover:bg-amber-100">
                  <Monitor className="w-5 h-5 mr-3" />
                  System Monitor
                </Button>
                <Button className="w-full justify-start bg-slate-50 text-slate-700 hover:bg-slate-100">
                  <Database className="w-5 h-5 mr-3" />
                  Database Admin
                </Button>
              </div>
            </CardContent>
          </Card>

          <div className="lg:col-span-2">
            <Card className="dashboard-card">
              <CardHeader>
                <CardTitle className="text-lg font-semibold">Recent Admin Activities</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {recentAdminActivities.map((activity, index) => (
                    <div key={index} className="flex items-start space-x-3">
                      <div className={`w-8 h-8 ${activity.color} rounded-full flex items-center justify-center flex-shrink-0`}>
                        <activity.icon className="w-4 h-4 text-white" />
                      </div>
                      <div className="flex-1 min-w-0">
                        <p className="text-sm text-foreground">{activity.message}</p>
                        <p className="text-xs text-muted-foreground">{activity.time}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Platform Statistics */}
        <div className="grid lg:grid-cols-2 gap-8">
          <Card className="dashboard-card">
            <CardHeader>
              <CardTitle className="text-lg font-semibold">Platform Usage</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-center justify-between">
                  <span className="text-sm text-muted-foreground">Daily Active Users</span>
                  <span className="font-semibold">2,847</span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm text-muted-foreground">Courses Delivered</span>
                  <span className="font-semibold">1,234</span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm text-muted-foreground">Assessments Completed</span>
                  <span className="font-semibold">5,689</span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm text-muted-foreground">Certificates Issued</span>
                  <span className="font-semibold">892</span>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="dashboard-card">
            <CardHeader>
              <CardTitle className="text-lg font-semibold">Resource Utilization</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <div className="flex justify-between text-sm mb-1">
                    <span>CPU Usage</span>
                    <span>34%</span>
                  </div>
                  <div className="w-full bg-slate-200 rounded-full h-2">
                    <div className="bg-blue-600 h-2 rounded-full" style={{ width: '34%' }}></div>
                  </div>
                </div>
                <div>
                  <div className="flex justify-between text-sm mb-1">
                    <span>Memory Usage</span>
                    <span>67%</span>
                  </div>
                  <div className="w-full bg-slate-200 rounded-full h-2">
                    <div className="bg-emerald-600 h-2 rounded-full" style={{ width: '67%' }}></div>
                  </div>
                </div>
                <div>
                  <div className="flex justify-between text-sm mb-1">
                    <span>Storage Usage</span>
                    <span>78%</span>
                  </div>
                  <div className="w-full bg-slate-200 rounded-full h-2">
                    <div className="bg-amber-600 h-2 rounded-full" style={{ width: '78%' }}></div>
                  </div>
                </div>
                <div>
                  <div className="flex justify-between text-sm mb-1">
                    <span>Network I/O</span>
                    <span>23%</span>
                  </div>
                  <div className="w-full bg-slate-200 rounded-full h-2">
                    <div className="bg-purple-600 h-2 rounded-full" style={{ width: '23%' }}></div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  );
}
