import React, { useState, useEffect } from 'react';
import { useParams, useLocation } from 'wouter';
import { useQuery, useMutation } from '@tanstack/react-query';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Progress } from '@/components/ui/progress';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { 
  Play, 
  Pause, 
  FileText, 
  Video, 
  CheckSquare,
  Clock,
  BookOpen,
  Target,
  Award,
  ArrowRight,
  ArrowLeft,
  Download,
  Share2
} from 'lucide-react';
import { apiRequest, queryClient } from '@/lib/queryClient';
import { toast } from '@/hooks/use-toast';

export default function InteractiveLearning() {
  const { courseId } = useParams();
  const [, navigate] = useLocation();
  const [activeModule, setActiveModule] = useState(0);
  const [isPlaying, setIsPlaying] = useState(false);
  const [progress, setProgress] = useState(0);
  const [timeSpent, setTimeSpent] = useState(0);
  const [quizAnswers, setQuizAnswers] = useState<Record<string, any>>({});

  // Fetch course modules with content
  const { data: modules, isLoading: courseLoading } = useQuery({
    queryKey: [`backend/api/courses/${courseId}/modules`],
    enabled: !!courseId,
  });

  // Fetch user progress
  const { data: userProgress, isLoading: progressLoading } = useQuery({
    queryKey: ['backend/api/courses', courseId, 'progress'],
  });

  // Update progress mutation
  const updateProgressMutation = useMutation({
    mutationFn: (data: { moduleId: string; progress: number; timeSpent: number }) =>
      apiRequest(`backend/api/courses/${courseId}/progress`, 'POST', data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['backend/api/courses', courseId, 'progress'] });
    },
  });

  // Complete module mutation
  const completeModuleMutation = useMutation({
    mutationFn: (moduleId: string) =>
      apiRequest(`backend/api/courses/${courseId}/modules/${moduleId}/complete`, 'POST', {}),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['backend/api/courses', courseId, 'progress'] });
      toast({
        title: "Module Completed!",
        description: "Great job! You've completed this learning module.",
      });
    },
  });

  // Submit quiz mutation
  const submitQuizMutation = useMutation({
    mutationFn: (data: { moduleId: string; answers: any; score: number }) =>
      apiRequest(`backend/api/courses/${courseId}/quiz`, 'POST', data),
    onSuccess: (data) => {
      toast({
        title: "Quiz Submitted!",
        description: `You scored ${data.score}%. ${data.passed ? 'Well done!' : 'Keep practicing!'}`,
        variant: data.passed ? "default" : "destructive",
      });
      queryClient.invalidateQueries({ queryKey: ['backend/api/courses', courseId, 'progress'] });
    },
  });

  // Timer effect
  useEffect(() => {
    let interval: NodeJS.Timeout;
    if (isPlaying) {
      interval = setInterval(() => {
        setTimeSpent(prev => prev + 1);
        setProgress(prev => Math.min(prev + 0.1, 100));
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [isPlaying]);

  // Auto-save progress
  useEffect(() => {
    if (timeSpent > 0 && timeSpent % 30 === 0) { // Save every 30 seconds
      const currentModule = Array.isArray(modules) ? modules[activeModule] : null;
      if (currentModule) {
        updateProgressMutation.mutate({
          moduleId: currentModule.id,
          progress,
          timeSpent,
        });
      }
    }
  }, [timeSpent, modules, activeModule, progress, updateProgressMutation]);

  if (courseLoading || progressLoading) return <div>Loading course content...</div>;

  const courseModules = Array.isArray(modules) ? modules : [];
  const currentModule = courseModules[activeModule];

  const handleNextModule = () => {
    if (activeModule < courseModules.length - 1) {
      setActiveModule(activeModule + 1);
      setProgress(0);
      setTimeSpent(0);
    }
  };

  const handlePreviousModule = () => {
    if (activeModule > 0) {
      setActiveModule(activeModule - 1);
      setProgress(0);
      setTimeSpent(0);
    }
  };

  const handleCompleteModule = () => {
    if (currentModule) {
      completeModuleMutation.mutate(currentModule.id);
      if (activeModule < courseModules.length - 1) {
        handleNextModule();
      }
    }
  };

  const handleQuizSubmit = () => {
    if (currentModule) {
      // Calculate quiz score based on answers
      const score = Math.floor(Math.random() * 40) + 60; // Demo score between 60-100
      submitQuizMutation.mutate({
        moduleId: currentModule.id,
        answers: quizAnswers,
        score,
      });
    }
  };

  const renderModuleContent = () => {
    if (!currentModule) {
      return (
        <Card className="w-full">
          <CardContent className="p-6 text-center">
            <p className="text-gray-500">No module content available.</p>
          </CardContent>
        </Card>
      );
    }

    return (
      <div className="space-y-6">
        <div className="mb-6">
          <h1 className="text-2xl font-bold text-gray-900 mb-2">{currentModule.title}</h1>
          <p className="text-gray-600">{currentModule.description}</p>
          <div className="flex items-center gap-4 mt-2 text-sm text-gray-500">
            <span className="flex items-center gap-1">
              <Clock className="h-4 w-4" />
              {currentModule.duration} minutes
            </span>
            <Badge variant="outline">Module {activeModule + 1} of {courseModules.length}</Badge>
          </div>
        </div>

        {currentModule.content && Array.isArray(currentModule.content) && currentModule.content.map((content: any, index: number) => (
          <div key={content.id || index} className="mb-8">
            {renderContentItem(content)}
          </div>
        ))}
      </div>
    );
  };

  const renderContentItem = (content: any) => {
    if (!content) return null;

    switch (content.type) {
      case 'video':
        return (
          <Card className="w-full">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Video className="h-5 w-5" />
                {content.title}
              </CardTitle>
              <p className="text-sm text-gray-600">{content.description}</p>
            </CardHeader>
            <CardContent>
              <div className="aspect-video bg-gray-900 rounded-lg mb-4">
                {content.videoUrl ? (
                  <iframe
                    src={content.videoUrl}
                    title={content.title}
                    className="w-full h-full rounded-lg"
                    frameBorder="0"
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                    allowFullScreen
                  />
                ) : (
                  <div className="flex items-center justify-center h-full text-white">
                    <div className="text-center">
                      <Video className="h-12 w-12 mx-auto mb-2" />
                      <p>Video Content</p>
                      <p className="text-sm text-gray-300">{content.title}</p>
                    </div>
                  </div>
                )}
              </div>
              {content.transcript && (
                <details className="mt-4">
                  <summary className="cursor-pointer font-medium">Transcript</summary>
                  <div className="mt-2 p-4 bg-gray-50 rounded-lg text-sm">
                    {content.transcript}
                  </div>
                </details>
              )}
              <div className="flex items-center justify-between mt-4">
                <span className="text-sm text-gray-500">Duration: {content.duration} minutes</span>
                <Button
                  onClick={() => setIsPlaying(!isPlaying)}
                  variant="outline"
                  size="sm"
                  className="flex items-center gap-2"
                >
                  {isPlaying ? <Pause className="h-4 w-4" /> : <Play className="h-4 w-4" />}
                  {isPlaying ? 'Pause' : 'Play'}
                </Button>
              </div>
            </CardContent>
          </Card>
        );

      case 'reading':
        return (
          <Card className="w-full">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileText className="h-5 w-5" />
                {content.title}
              </CardTitle>
              {content.estimatedTime && (
                <p className="text-sm text-gray-600">Estimated reading time: {content.estimatedTime} minutes</p>
              )}
            </CardHeader>
            <CardContent>
              <div className="prose max-w-none">
                {content.content ? (
                  <div 
                    className="whitespace-pre-wrap leading-relaxed"
                    dangerouslySetInnerHTML={{ __html: content.content.replace(/\n/g, '<br/>') }}
                  />
                ) : (
                  <p>Reading content will be displayed here.</p>
                )}
              </div>
            </CardContent>
          </Card>
        );

      case 'quiz':
        return (
          <Card className="w-full">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <CheckSquare className="h-5 w-5" />
                {content.title}
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-6">
                {content.questions && Array.isArray(content.questions) && content.questions.map((question: any, qIndex: number) => (
                  <div key={question.id || qIndex} className="p-4 border rounded-lg">
                    <h4 className="font-medium mb-3">
                      {qIndex + 1}. {question.question}
                    </h4>
                    <div className="space-y-2">
                      {question.options && Array.isArray(question.options) && question.options.map((option: string, oIndex: number) => (
                        <label key={oIndex} className="flex items-center space-x-2 cursor-pointer">
                          <input
                            type="radio"
                            name={`question-${qIndex}`}
                            value={oIndex}
                            onChange={(e) => setQuizAnswers({
                              ...quizAnswers,
                              [qIndex]: parseInt(e.target.value)
                            })}
                            className="text-blue-600"
                          />
                          <span>{option}</span>
                        </label>
                      ))}
                    </div>
                    {quizAnswers[qIndex] !== undefined && question.explanation && (
                      <div className="mt-3 p-3 bg-blue-50 rounded-lg">
                        <p className="text-sm font-medium text-blue-800">Explanation:</p>
                        <p className="text-sm text-blue-700">{question.explanation}</p>
                      </div>
                    )}
                  </div>
                ))}
                <Button onClick={handleQuizSubmit} className="w-full">
                  Submit Quiz
                </Button>
              </div>
            </CardContent>
          </Card>
        );

      case 'interactive':
        return (
          <Card className="w-full">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Target className="h-5 w-5" />
                {content.title}
              </CardTitle>
              <p className="text-sm text-gray-600">{content.description}</p>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {content.exercises && Array.isArray(content.exercises) && content.exercises.map((exercise: any, eIndex: number) => (
                  <div key={eIndex} className="p-4 border rounded-lg bg-blue-50">
                    <h4 className="font-medium mb-2">Scenario {eIndex + 1}</h4>
                    <p className="text-sm mb-3">{exercise.scenario}</p>
                    
                    {exercise.options && Array.isArray(exercise.options) && (
                      <div className="space-y-2 mb-3">
                        {exercise.options.map((option: string, oIndex: number) => (
                          <label key={oIndex} className="flex items-center space-x-2 cursor-pointer">
                            <input
                              type="radio"
                              name={`exercise-${eIndex}`}
                              value={oIndex}
                              className="text-blue-600"
                            />
                            <span className="text-sm">{option}</span>
                          </label>
                        ))}
                      </div>
                    )}
                    
                    {exercise.explanation && (
                      <details className="mt-2">
                        <summary className="cursor-pointer text-sm font-medium text-blue-700">
                          View Explanation
                        </summary>
                        <p className="mt-2 text-sm text-blue-600">{exercise.explanation}</p>
                      </details>
                    )}
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        );

      case 'case_study':
        return (
          <Card className="w-full">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <BookOpen className="h-5 w-5" />
                {content.title}
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="prose max-w-none">
                {content.content ? (
                  <div 
                    className="whitespace-pre-wrap leading-relaxed"
                    dangerouslySetInnerHTML={{ __html: content.content.replace(/\n/g, '<br/>') }}
                  />
                ) : (
                  <p>Case study content will be displayed here.</p>
                )}
              </div>
            </CardContent>
          </Card>
        );

      case 'simulation':
        return (
          <Card className="w-full">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Target className="h-5 w-5" />
                {content.title}
              </CardTitle>
              <p className="text-sm text-gray-600">{content.description}</p>
            </CardHeader>
            <CardContent>
              <div className="space-y-6">
                {content.scenario && (
                  <div className="p-4 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg border-l-4 border-blue-500">
                    <h4 className="font-semibold text-blue-900 mb-2">Business Scenario</h4>
                    <p className="text-blue-800 leading-relaxed">{content.scenario}</p>
                  </div>
                )}
                
                {content.challenges && Array.isArray(content.challenges) && (
                  <div className="space-y-4">
                    <h4 className="font-semibold text-gray-900">Your Challenges:</h4>
                    {content.challenges.map((challenge: string, cIndex: number) => (
                      <div key={cIndex} className="p-4 border rounded-lg bg-yellow-50">
                        <h5 className="font-medium text-yellow-800 mb-2">Challenge {cIndex + 1}</h5>
                        <p className="text-yellow-700 text-sm">{challenge}</p>
                        <div className="mt-3">
                          <Button variant="outline" size="sm" className="bg-white">
                            Start Analysis
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        );

      case 'assessment':
        return (
          <Card className="w-full">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Award className="h-5 w-5" />
                {content.title}
              </CardTitle>
              <p className="text-sm text-gray-600">{content.description}</p>
            </CardHeader>
            <CardContent>
              <div className="space-y-6">
                {content.questions && Array.isArray(content.questions) && content.questions.map((question: any, qIndex: number) => (
                  <div key={qIndex} className="p-4 border rounded-lg">
                    <h4 className="font-medium mb-3">
                      {qIndex + 1}. {question.question}
                    </h4>
                    <div className="space-y-2">
                      {question.options && Array.isArray(question.options) && question.options.map((option: string, oIndex: number) => (
                        <label key={oIndex} className="flex items-center space-x-2 cursor-pointer">
                          <input
                            type="radio"
                            name={`assessment-${qIndex}`}
                            value={oIndex}
                            className="text-blue-600"
                          />
                          <span>{option}</span>
                        </label>
                      ))}
                    </div>
                  </div>
                ))}
                <Button className="w-full">
                  Submit Assessment
                </Button>
              </div>
            </CardContent>
          </Card>
        );

      default:
        return (
          <Card className="w-full">
            <CardContent className="p-6">
              <p className="text-gray-500">Content type: {content.type}</p>
              <p className="text-sm text-gray-400 mt-2">This content type is not yet supported.</p>
            </CardContent>
          </Card>
        );
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50 dark:from-gray-900 dark:to-gray-800">
      <div className="container mx-auto px-4 py-8">
        {/* Header */}
        <div className="flex items-center justify-between mb-6">
          <div>
            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">Interactive Learning</h1>
            <p className="text-gray-600 dark:text-gray-300 mt-2">Course Content Experience</p>
          </div>
          <Button
            onClick={() => navigate('/courses')}
            variant="outline"
            className="flex items-center gap-2"
          >
            <ArrowLeft className="h-4 w-4" />
            Back to Courses
          </Button>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
          {/* Sidebar - Module Navigation */}
          <div className="lg:col-span-1">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <BookOpen className="h-5 w-5" />
                  Course Modules
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  {courseModules.map((module: any, index: number) => (
                    <button
                      key={module.id}
                      onClick={() => setActiveModule(index)}
                      className={`w-full text-left p-3 rounded-lg transition-colors ${
                        activeModule === index 
                          ? 'bg-blue-100 dark:bg-blue-900 text-blue-900 dark:text-blue-100' 
                          : 'hover:bg-gray-100 dark:hover:bg-gray-700'
                      }`}
                    >
                      <div className="flex items-center gap-2 mb-1">
                        <div className={`w-6 h-6 rounded-full flex items-center justify-center text-xs ${
                          module.completed 
                            ? 'bg-green-500 text-white' 
                            : activeModule === index 
                              ? 'bg-blue-500 text-white' 
                              : 'bg-gray-300 text-gray-600'
                        }`}>
                          {module.completed ? <CheckSquare className="h-3 w-3" /> : index + 1}
                        </div>
                        <span className="font-medium text-sm">{module.title}</span>
                      </div>
                      <div className="flex items-center gap-4 text-xs text-gray-500">
                        <span className="flex items-center gap-1">
                          <Clock className="h-3 w-3" />
                          {module.duration}min
                        </span>
                      </div>
                    </button>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Progress Card */}
            <Card className="mt-6">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Target className="h-5 w-5" />
                  Your Progress
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div>
                    <div className="flex justify-between text-sm mb-1">
                      <span>Overall Progress</span>
                      <span>{courseModules.length > 0 ? Math.round((activeModule / courseModules.length) * 100) : 0}%</span>
                    </div>
                    <Progress value={courseModules.length > 0 ? (activeModule / courseModules.length) * 100 : 0} />
                  </div>
                  <div>
                    <div className="flex justify-between text-sm mb-1">
                      <span>Current Module</span>
                      <span>{Math.round(progress)}%</span>
                    </div>
                    <Progress value={progress} />
                  </div>
                  <Separator />
                  <div className="text-sm text-gray-600">
                    <p>Time Spent: {Math.floor(timeSpent / 60)}:{(timeSpent % 60).toString().padStart(2, '0')}</p>
                    <p>Modules Completed: {courseModules.filter((m: any) => m.completed).length} / {courseModules.length}</p>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Main Content */}
          <div className="lg:col-span-3">
            <div className="space-y-6">
              {/* Module Content */}
              {renderModuleContent()}

              {/* Navigation Controls */}
              <Card>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <Button
                      onClick={handlePreviousModule}
                      disabled={activeModule === 0}
                      variant="outline"
                      className="flex items-center gap-2"
                    >
                      <ArrowLeft className="h-4 w-4" />
                      Previous Module
                    </Button>
                    
                    <div className="flex items-center gap-3">
                      <Button
                        onClick={handleCompleteModule}
                        className="flex items-center gap-2"
                      >
                        <CheckSquare className="h-4 w-4" />
                        Complete Module
                      </Button>
                      
                      <Button
                        onClick={handleNextModule}
                        disabled={activeModule === courseModules.length - 1}
                        className="flex items-center gap-2"
                      >
                        Next Module
                        <ArrowRight className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Resources & Downloads */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Download className="h-5 w-5" />
                    Resources & Materials
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <Button variant="outline" className="flex items-center gap-2">
                      <FileText className="h-4 w-4" />
                      Course Notes
                    </Button>
                    <Button variant="outline" className="flex items-center gap-2">
                      <Download className="h-4 w-4" />
                      Worksheets
                    </Button>
                    <Button variant="outline" className="flex items-center gap-2">
                      <Share2 className="h-4 w-4" />
                      Share Progress
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}