import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useRoute, useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Separator } from "@/components/ui/separator";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { 
  BookOpen, 
  Clock, 
  Users, 
  Star, 
  Play, 
  CheckCircle,
  FileText,
  Video,
  Download,
  Share,
  Bookmark,
  ArrowLeft,
  Calendar,
  Award,
  Target
} from "lucide-react";
import { Link } from "wouter";
import type { Course, Enrollment } from "@shared/schema";

export default function CourseDetail() {
  const [match, params] = useRoute("/courses/:courseId");
  const courseId = params?.courseId;
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [, setLocation] = useLocation();

  const { data: course, isLoading: courseLoading } = useQuery({
    queryKey: ["backend/api/courses", courseId],
    enabled: !!courseId,
  });

  const { data: enrollment } = useQuery({
    queryKey: ["backend/api/enrollments", "course", courseId],
    enabled: !!courseId,
  });

  const { data: modules } = useQuery({
    queryKey: ["backend/api/courses", courseId, "modules"],
    enabled: !!courseId,
  });

  const enrollMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("backend/api/enroll-course", "POST", { courseId });
    },
    onSuccess: () => {
      toast({
        title: "Enrollment Successful",
        description: "Starting your learning journey...",
      });
      queryClient.invalidateQueries({ queryKey: ["backend/api/user-enrollments"] });
      
      // Navigate to interactive learning after brief delay
      setTimeout(() => {
        setLocation(`/courses/${courseId}/learn`);
      }, 1500);
    },
    onError: (error) => {
      toast({
        title: "Enrollment Failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateProgressMutation = useMutation({
    mutationFn: async ({ moduleId, progress }: { moduleId: string; progress: number }) => {
      return await apiRequest(`backend/api/enrollments/${enrollment?.id}/progress`, {
        method: "PATCH",
        body: JSON.stringify({ moduleId, progress }),
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["backend/api/user-enrollments"] });
    },
  });

  if (courseLoading || !course) {
    return (
      <div className="container mx-auto px-4 py-8">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/3 mb-4"></div>
          <div className="h-4 bg-gray-200 rounded w-2/3 mb-8"></div>
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <div className="lg:col-span-2">
              <div className="h-64 bg-gray-200 rounded mb-6"></div>
              <div className="space-y-3">
                <div className="h-4 bg-gray-200 rounded"></div>
                <div className="h-4 bg-gray-200 rounded w-3/4"></div>
              </div>
            </div>
            <div>
              <div className="h-48 bg-gray-200 rounded"></div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  const formatDuration = (minutes: number) => {
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    return hours > 0 ? `${hours}h ${mins}m` : `${mins}m`;
  };

  const getDifficultyColor = (difficulty: string) => {
    switch (difficulty) {
      case "beginner": return "bg-green-100 text-green-800";
      case "intermediate": return "bg-yellow-100 text-yellow-800";
      case "advanced": return "bg-red-100 text-red-800";
      default: return "bg-gray-100 text-gray-800";
    }
  };

  const isEnrolled = !!enrollment;
  const progress = enrollment?.progress || 0;

  return (
    <div className="container mx-auto px-4 py-8">
      {/* Back Navigation */}
      <div className="mb-6">
        <Link href="/courses">
          <Button variant="ghost" size="sm" className="text-muted-foreground hover:text-foreground">
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Catalog
          </Button>
        </Link>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        {/* Main Content */}
        <div className="lg:col-span-2">
          {/* Course Header */}
          <div className="mb-8">
            <div className="flex flex-wrap items-center gap-2 mb-4">
              <Badge className={getDifficultyColor(course.difficulty || "")}>
                {course.difficulty}
              </Badge>
              {course.category && (
                <Badge variant="secondary">{course.category}</Badge>
              )}
              {isEnrolled && (
                <Badge className="bg-green-100 text-green-800">
                  <CheckCircle className="w-3 h-3 mr-1" />
                  Enrolled
                </Badge>
              )}
            </div>
            
            <h1 className="text-3xl font-bold text-foreground mb-4">{course.title}</h1>
            
            <div className="flex items-center gap-6 text-sm text-muted-foreground mb-4">
              <div className="flex items-center gap-1">
                <Clock className="w-4 h-4" />
                {formatDuration(course.duration || 0)}
              </div>
              <div className="flex items-center gap-1">
                <Users className="w-4 h-4" />
                {course.studentsEnrolled || 0} students
              </div>
              <div className="flex items-center gap-1">
                <Star className="w-4 h-4 fill-yellow-400 text-yellow-400" />
                {course.rating?.toFixed(1) || "N/A"}
              </div>
              <div className="flex items-center gap-1">
                <Calendar className="w-4 h-4" />
                Updated {new Date(course.updatedAt).toLocaleDateString()}
              </div>
            </div>

            {isEnrolled && (
              <div className="mb-6">
                <div className="flex justify-between items-center mb-2">
                  <span className="text-sm font-medium">Your Progress</span>
                  <span className="text-sm text-muted-foreground">{progress}%</span>
                </div>
                <Progress value={progress} className="h-3" />
              </div>
            )}
          </div>

          {/* Course Content Tabs */}
          <Tabs defaultValue="overview" className="w-full">
            <TabsList className="grid w-full grid-cols-4">
              <TabsTrigger value="overview">Overview</TabsTrigger>
              <TabsTrigger value="curriculum">Curriculum</TabsTrigger>
              <TabsTrigger value="reviews">Reviews</TabsTrigger>
              <TabsTrigger value="resources">Resources</TabsTrigger>
            </TabsList>
            
            <TabsContent value="overview" className="mt-6">
              <Card>
                <CardHeader>
                  <CardTitle>Course Description</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-muted-foreground leading-relaxed mb-6">
                    {course.description}
                  </p>

                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <h4 className="font-semibold mb-3 flex items-center">
                        <Target className="w-4 h-4 mr-2" />
                        Learning Objectives
                      </h4>
                      <ul className="space-y-2 text-sm text-muted-foreground">
                        <li>• Master fundamental concepts and principles</li>
                        <li>• Apply practical skills in real-world scenarios</li>
                        <li>• Develop critical thinking and problem-solving abilities</li>
                        <li>• Prepare for professional certification</li>
                      </ul>
                    </div>
                    
                    <div>
                      <h4 className="font-semibold mb-3 flex items-center">
                        <Award className="w-4 h-4 mr-2" />
                        What You'll Get
                      </h4>
                      <ul className="space-y-2 text-sm text-muted-foreground">
                        <li>• Comprehensive video lectures</li>
                        <li>• Interactive assignments and quizzes</li>
                        <li>• Downloadable resources</li>
                        <li>• Certificate of completion</li>
                      </ul>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="curriculum" className="mt-6">
              <Card>
                <CardHeader>
                  <CardTitle>Course Modules</CardTitle>
                </CardHeader>
                <CardContent>
                  {modules?.length > 0 ? (
                    <div className="space-y-4">
                      {modules.map((module: any, index: number) => (
                        <div key={module.id} className="border rounded-lg p-4">
                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-3">
                              <div className="w-8 h-8 bg-primary rounded-full flex items-center justify-center text-white text-sm font-medium">
                                {index + 1}
                              </div>
                              <div>
                                <h5 className="font-medium">{module.title}</h5>
                                <p className="text-sm text-muted-foreground">{module.description}</p>
                              </div>
                            </div>
                            <div className="flex items-center gap-2">
                              <span className="text-xs text-muted-foreground">{module.duration}min</span>
                              {isEnrolled && (
                                <Button
                                  size="sm"
                                  variant={module.completed ? "default" : "outline"}
                                  onClick={() => updateProgressMutation.mutate({ 
                                    moduleId: module.id, 
                                    progress: module.completed ? 0 : 100 
                                  })}
                                >
                                  {module.completed ? <CheckCircle className="w-4 h-4" /> : <Play className="w-4 h-4" />}
                                </Button>
                              )}
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="text-center py-8">
                      <BookOpen className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                      <p className="text-muted-foreground">Course modules will be available after enrollment</p>
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="reviews" className="mt-6">
              <Card>
                <CardHeader>
                  <CardTitle>Student Reviews</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="text-center py-8">
                    <Star className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                    <p className="text-muted-foreground">No reviews yet. Be the first to review this course!</p>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="resources" className="mt-6">
              <Card>
                <CardHeader>
                  <CardTitle>Course Resources</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between p-3 bg-muted rounded-lg">
                      <div className="flex items-center gap-3">
                        <FileText className="w-5 h-5 text-muted-foreground" />
                        <div>
                          <p className="font-medium">Course Handbook</p>
                          <p className="text-sm text-muted-foreground">Complete reference guide (PDF)</p>
                        </div>
                      </div>
                      <Button size="sm" variant="outline" disabled={!isEnrolled}>
                        <Download className="w-4 h-4" />
                      </Button>
                    </div>
                    
                    <div className="flex items-center justify-between p-3 bg-muted rounded-lg">
                      <div className="flex items-center gap-3">
                        <Video className="w-5 h-5 text-muted-foreground" />
                        <div>
                          <p className="font-medium">Video Transcripts</p>
                          <p className="text-sm text-muted-foreground">Text version of all lectures</p>
                        </div>
                      </div>
                      <Button size="sm" variant="outline" disabled={!isEnrolled}>
                        <Download className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>

        {/* Sidebar */}
        <div className="space-y-6">
          {/* Enrollment Card */}
          <Card>
            <CardContent className="p-6">
              {!isEnrolled ? (
                <div className="text-center">
                  <div className="text-2xl font-bold text-foreground mb-2">
                    {course.price ? `$${course.price}` : 'Free'}
                  </div>
                  <Button 
                    onClick={() => enrollMutation.mutate()}
                    disabled={enrollMutation.isPending}
                    className="w-full mb-4"
                    size="lg"
                  >
                    <BookOpen className="w-4 h-4 mr-2" />
                    {enrollMutation.isPending ? "Enrolling..." : "Enroll Now"}
                  </Button>
                  <div className="space-y-2">
                    <Button variant="outline" size="sm" className="w-full">
                      <Bookmark className="w-4 h-4 mr-2" />
                      Save for Later
                    </Button>
                    <Button variant="outline" size="sm" className="w-full">
                      <Share className="w-4 h-4 mr-2" />
                      Share Course
                    </Button>
                  </div>
                </div>
              ) : (
                <div className="text-center">
                  <div className="text-lg font-semibold text-green-600 mb-4">
                    ✓ You're Enrolled!
                  </div>
                  <Button 
                    className="w-full mb-4" 
                    size="lg"
                    onClick={() => setLocation(`/courses/${courseId}/learn`)}
                  >
                    <Play className="w-4 h-4 mr-2" />
                    Start Learning
                  </Button>
                  <div className="text-sm text-muted-foreground">
                    {progress}% complete
                  </div>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Course Info */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Course Information</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex justify-between">
                <span className="text-muted-foreground">Duration</span>
                <span className="font-medium">{formatDuration(course.duration || 0)}</span>
              </div>
              <Separator />
              <div className="flex justify-between">
                <span className="text-muted-foreground">Students</span>
                <span className="font-medium">{course.studentsEnrolled || 0}</span>
              </div>
              <Separator />
              <div className="flex justify-between">
                <span className="text-muted-foreground">Language</span>
                <span className="font-medium">English</span>
              </div>
              <Separator />
              <div className="flex justify-between">
                <span className="text-muted-foreground">Certificate</span>
                <span className="font-medium">Yes</span>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}