import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { 
  BookOpen, 
  Clock, 
  Users, 
  Star, 
  Play, 
  Search,
  Filter,
  Grid,
  List,
  ChevronRight
} from "lucide-react";
import { Link } from "wouter";
import type { Course } from "@shared/schema";

export default function CourseCatalog() {
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCategory, setSelectedCategory] = useState<string>("");
  const [viewMode, setViewMode] = useState<"grid" | "list">("grid");
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [, setLocation] = useLocation();

  const { data: courses, isLoading } = useQuery({
    queryKey: ["backend/api/courses", searchQuery, selectedCategory],
    queryFn: async () => {
      if (searchQuery || (selectedCategory && selectedCategory !== "all")) {
        const params = new URLSearchParams();
        if (searchQuery) params.append("q", searchQuery);
        if (selectedCategory && selectedCategory !== "all") params.append("category", selectedCategory);
        return await apiRequest(`backend/api/courses/search?${params.toString()}`, "GET");
      }
      return await apiRequest("backend/api/courses", "GET");
    },
  });

  const { data: userEnrollments } = useQuery({
    queryKey: ["backend/api/user-enrollments"],
  });

  // Helper function to check if user is enrolled in a course
  const isEnrolled = (courseId: string) => {
    return Array.isArray(userEnrollments) && userEnrollments.some((enrollment: any) => enrollment.courseId === courseId);
  };

  const enrollMutation = useMutation({
    mutationFn: async (courseId: string) => {
      return await apiRequest("backend/api/enroll-course", "POST", { courseId });
    },
    onSuccess: (data, courseId) => {
      toast({
        title: "Enrollment Successful",
        description: "Redirecting to course content...",
      });
      queryClient.invalidateQueries({ queryKey: ["backend/api/courses/search"] });
      queryClient.invalidateQueries({ queryKey: ["backend/api/user-enrollments"] });
      
      // Navigate to course detail page after brief delay
      setTimeout(() => {
        setLocation(`/courses/${courseId}`);
      }, 1500);
    },
    onError: (error) => {
      toast({
        title: "Enrollment Failed",
        description: error.message || "Failed to enroll in course",
        variant: "destructive",
      });
    },
  });

  const categories = [
    "Business Analytics",
    "Leadership Development", 
    "Project Management",
    "Digital Marketing",
    "Data Science",
    "Professional Development",
    "Certification Prep"
  ];

  const getDifficultyColor = (difficulty: string) => {
    switch (difficulty) {
      case "beginner":
        return "bg-green-100 text-green-800";
      case "intermediate":
        return "bg-yellow-100 text-yellow-800";
      case "advanced":
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const formatDuration = (minutes: number) => {
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    return hours > 0 ? `${hours}h ${mins}m` : `${mins}m`;
  };

  if (isLoading) {
    return (
      <div className="container mx-auto px-4 py-8">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {Array.from({ length: 6 }).map((_, i) => (
            <Card key={i}>
              <CardHeader>
                <Skeleton className="h-6 w-3/4" />
                <Skeleton className="h-4 w-full" />
              </CardHeader>
              <CardContent>
                <Skeleton className="h-32 w-full mb-4" />
                <Skeleton className="h-4 w-1/2" />
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      {/* Header */}
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-foreground mb-2">Course Catalog</h1>
        <p className="text-muted-foreground">
          Discover and enroll in courses to advance your professional development
        </p>
      </div>

      {/* Search and Filters */}
      <div className="mb-8 space-y-4">
        <div className="flex flex-col md:flex-row gap-4">
          <div className="relative flex-1">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
            <Input
              placeholder="Search courses..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10"
            />
          </div>
          <Select value={selectedCategory} onValueChange={setSelectedCategory}>
            <SelectTrigger className="w-full md:w-48">
              <SelectValue placeholder="All Categories" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Categories</SelectItem>
              {categories.map((category) => (
                <SelectItem key={category} value={category}>
                  {category}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
          <div className="flex gap-2">
            <Button
              variant={viewMode === "grid" ? "default" : "outline"}
              size="sm"
              onClick={() => setViewMode("grid")}
            >
              <Grid className="w-4 h-4" />
            </Button>
            <Button
              variant={viewMode === "list" ? "default" : "outline"}
              size="sm"
              onClick={() => setViewMode("list")}
            >
              <List className="w-4 h-4" />
            </Button>
          </div>
        </div>
      </div>

      {/* Course Grid */}
      <div className={`grid gap-6 ${
        viewMode === "grid" 
          ? "grid-cols-1 md:grid-cols-2 lg:grid-cols-3" 
          : "grid-cols-1"
      }`}>
        {courses?.map((course: Course) => (
          <Card key={course.id} className="hover:shadow-lg transition-shadow">
            <CardHeader className="pb-3">
              <div className="flex justify-between items-start gap-2">
                <CardTitle className="text-lg leading-tight">{course.title}</CardTitle>
                <Badge className={getDifficultyColor(course.difficulty || "")}>
                  {course.difficulty}
                </Badge>
              </div>
              <p className="text-sm text-muted-foreground line-clamp-2">
                {course.description}
              </p>
            </CardHeader>
            <CardContent className="pt-0">
              <div className="space-y-4">
                {/* Course Stats */}
                <div className="flex items-center gap-4 text-sm text-muted-foreground">
                  <div className="flex items-center gap-1">
                    <Clock className="w-4 h-4" />
                    {formatDuration(course.duration || 0)}
                  </div>
                  <div className="flex items-center gap-1">
                    <Users className="w-4 h-4" />
                    {course.studentsEnrolled || 0}
                  </div>
                  <div className="flex items-center gap-1">
                    <Star className="w-4 h-4 fill-yellow-400 text-yellow-400" />
                    {course.rating?.toFixed(1) || "N/A"}
                  </div>
                </div>

                {/* Course Category */}
                <div className="flex gap-2 flex-wrap">
                  {course.category && (
                    <Badge variant="secondary" className="text-xs">
                      {course.category}
                    </Badge>
                  )}
                  {isEnrolled(course.id) && (
                    <Badge variant="default" className="text-xs bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-100">
                      ✓ Enrolled
                    </Badge>
                  )}
                </div>

                {/* Action Buttons */}
                <div className="flex gap-2">
                  {isEnrolled(course.id) ? (
                    <>
                      <Button
                        onClick={() => setLocation(`/courses/${course.id}/learn`)}
                        className="flex-1"
                      >
                        <Play className="w-4 h-4 mr-2" />
                        Start Learning
                      </Button>
                      <Link href={`/courses/${course.id}`}>
                        <Button variant="outline" size="sm">
                          <BookOpen className="w-4 h-4" />
                        </Button>
                      </Link>
                    </>
                  ) : (
                    <>
                      <Button
                        onClick={() => enrollMutation.mutate(course.id)}
                        disabled={enrollMutation.isPending}
                        className="flex-1"
                      >
                        <BookOpen className="w-4 h-4 mr-2" />
                        {enrollMutation.isPending ? "Enrolling..." : "Enroll Now"}
                      </Button>
                      <Link href={`/courses/${course.id}`}>
                        <Button variant="outline" size="sm">
                          <Play className="w-4 h-4" />
                        </Button>
                      </Link>
                    </>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Empty State */}
      {courses?.length === 0 && (
        <div className="text-center py-12">
          <BookOpen className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
          <h3 className="text-lg font-medium text-foreground mb-2">No courses found</h3>
          <p className="text-muted-foreground">
            Try adjusting your search criteria or browse different categories
          </p>
        </div>
      )}
    </div>
  );
}