import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Skeleton } from "@/components/ui/skeleton";
import { 
  Award, 
  Download, 
  Share2, 
  ExternalLink, 
  Calendar,
  CheckCircle,
  Search,
  Shield,
  Printer,
  Globe
} from "lucide-react";
import { Link } from "wouter";

export default function CertificateList() {
  const [searchQuery, setSearchQuery] = useState("");

  const { data: certificates, isLoading } = useQuery({
    queryKey: ["backend/api/certificates", searchQuery],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (searchQuery) params.append("q", searchQuery);
      return await fetch(`backend/api/certificates?${params.toString()}`).then(res => res.json());
    },
  });

  const handleDownload = (certificateId: string) => {
    window.open(`backend/api/certificates/${certificateId}/download`, '_blank');
  };

  const handleVerify = (verificationCode: string) => {
    window.open(`/certificates/verify/${verificationCode}`, '_blank');
  };

  const handleShare = (certificate: any) => {
    if (navigator.share) {
      navigator.share({
        title: `${certificate.title} Certificate`,
        text: `I've earned a certificate in ${certificate.title}`,
        url: `${window.location.origin}/certificates/verify/${certificate.verificationCode}`
      });
    } else {
      navigator.clipboard.writeText(`${window.location.origin}/certificates/verify/${certificate.verificationCode}`);
    }
  };

  if (isLoading) {
    return (
      <div className="container mx-auto px-4 py-8">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {Array.from({ length: 6 }).map((_, i) => (
            <Card key={i} className="animate-pulse">
              <CardHeader>
                <Skeleton className="h-6 w-3/4 mb-2" />
                <Skeleton className="h-4 w-full" />
              </CardHeader>
              <CardContent>
                <Skeleton className="h-32 w-full mb-4" />
                <Skeleton className="h-4 w-1/2" />
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      {/* Header */}
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-foreground mb-2">My Certificates</h1>
        <p className="text-muted-foreground">
          Digital certificates with blockchain verification for completed courses and assessments
        </p>
      </div>

      {/* Search */}
      <div className="mb-8">
        <div className="relative max-w-md">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
          <Input
            placeholder="Search certificates..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-10"
          />
        </div>
      </div>

      {/* Certificates Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {certificates?.map((certificate: any) => (
          <Card key={certificate.id} className="hover:shadow-lg transition-shadow relative overflow-hidden">
            {/* Certificate Header Strip */}
            <div className="h-2 bg-gradient-to-r from-primary to-secondary"></div>
            
            <CardHeader className="pb-3">
              <div className="flex items-start justify-between gap-2 mb-2">
                <div className="flex items-center gap-2">
                  <div className="w-10 h-10 bg-primary rounded-lg flex items-center justify-center">
                    <Award className="w-6 h-6 text-white" />
                  </div>
                  <Badge className="bg-green-100 text-green-800">
                    <CheckCircle className="w-3 h-3 mr-1" />
                    Verified
                  </Badge>
                </div>
                <Badge variant="outline" className="text-xs">
                  <Shield className="w-3 h-3 mr-1" />
                  Blockchain
                </Badge>
              </div>
              
              <CardTitle className="text-lg leading-tight">{certificate.title}</CardTitle>
              <p className="text-sm text-muted-foreground">
                {certificate.description}
              </p>
            </CardHeader>
            
            <CardContent className="pt-0">
              <div className="space-y-4">
                {/* Certificate Details */}
                <div className="grid grid-cols-2 gap-4 text-sm">
                  <div>
                    <div className="text-muted-foreground">Issued Date</div>
                    <div className="font-medium flex items-center gap-1">
                      <Calendar className="w-3 h-3" />
                      {new Date(certificate.issuedDate).toLocaleDateString()}
                    </div>
                  </div>
                  <div>
                    <div className="text-muted-foreground">Certificate ID</div>
                    <div className="font-medium text-xs">
                      {certificate.verificationCode}
                    </div>
                  </div>
                </div>

                {/* Skills/Competencies */}
                {certificate.skills && (
                  <div>
                    <div className="text-sm font-medium mb-2">Skills Demonstrated</div>
                    <div className="flex flex-wrap gap-1">
                      {certificate.skills.slice(0, 3).map((skill: string, index: number) => (
                        <Badge key={index} variant="secondary" className="text-xs">
                          {skill}
                        </Badge>
                      ))}
                      {certificate.skills.length > 3 && (
                        <Badge variant="secondary" className="text-xs">
                          +{certificate.skills.length - 3} more
                        </Badge>
                      )}
                    </div>
                  </div>
                )}

                {/* Actions */}
                <div className="grid grid-cols-2 gap-2">
                  <Button
                    onClick={() => handleDownload(certificate.id)}
                    variant="outline"
                    size="sm"
                    className="w-full"
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Download
                  </Button>
                  <Button
                    onClick={() => handleShare(certificate)}
                    variant="outline"
                    size="sm"
                    className="w-full"
                  >
                    <Share2 className="w-4 h-4 mr-2" />
                    Share
                  </Button>
                </div>

                <div className="grid grid-cols-2 gap-2">
                  <Button
                    onClick={() => handleVerify(certificate.verificationCode)}
                    size="sm"
                    className="w-full"
                  >
                    <Globe className="w-4 h-4 mr-2" />
                    Verify Online
                  </Button>
                  <Button
                    onClick={() => window.print()}
                    variant="outline"
                    size="sm"
                    className="w-full"
                  >
                    <Printer className="w-4 h-4 mr-2" />
                    Print
                  </Button>
                </div>

                {/* Additional Info */}
                <div className="pt-2 border-t text-xs text-muted-foreground">
                  <div className="flex items-center gap-2">
                    <Shield className="w-3 h-3" />
                    Blockchain verified • Employer accepted • Tamper-proof
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Empty State */}
      {certificates?.length === 0 && (
        <div className="text-center py-12">
          <Award className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
          <h3 className="text-lg font-medium text-foreground mb-2">No certificates yet</h3>
          <p className="text-muted-foreground mb-6">
            Complete courses and assessments to earn digital certificates
          </p>
          <div className="flex gap-4 justify-center">
            <Link href="/courses">
              <Button>Browse Courses</Button>
            </Link>
            <Link href="/assessments">
              <Button variant="outline">Take Assessment</Button>
            </Link>
          </div>
        </div>
      )}

      {/* Info Banner */}
      <div className="mt-12 p-6 bg-blue-50 border border-blue-200 rounded-lg">
        <div className="flex items-start gap-4">
          <div className="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center flex-shrink-0">
            <Shield className="w-6 h-6 text-blue-600" />
          </div>
          <div>
            <h3 className="font-semibold text-blue-900 mb-2">Blockchain-Verified Certificates</h3>
            <p className="text-blue-800 text-sm mb-3">
              All IBDL certificates are secured using blockchain technology, ensuring they cannot be forged or tampered with. 
              Each certificate includes a unique verification code that can be independently verified by employers and institutions.
            </p>
            <div className="flex items-center gap-4 text-xs text-blue-700">
              <div className="flex items-center gap-1">
                <CheckCircle className="w-3 h-3" />
                Tamper-proof
              </div>
              <div className="flex items-center gap-1">
                <Globe className="w-3 h-3" />
                Globally verifiable
              </div>
              <div className="flex items-center gap-1">
                <Award className="w-3 h-3" />
                Industry recognized
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}