import { useState, useEffect, useRef, useCallback } from "react";
import { useParams, useLocation } from "wouter";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Progress } from "@/components/ui/progress";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { Checkbox } from "@/components/ui/checkbox";
import { Textarea } from "@/components/ui/textarea";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { 
  Camera, 
  Mic, 
  Shield, 
  Clock, 
  AlertTriangle, 
  Monitor,
  Flag,
  ChevronLeft,
  ChevronRight,
  Save,
  Send,
  Eye,
  Volume2,
  Pause,
  Play
} from "lucide-react";

interface Question {
  id: string;
  type: 'multiple_choice' | 'multiple_select' | 'true_false' | 'short_answer' | 'case_study';
  question: string;
  options?: string[];
  correctAnswer?: number | boolean;
  correctAnswers?: number[];
  context?: string;
  points: number;
  timeLimit: number;
  rubric?: {
    criteria: string[];
  };
}

interface ExamSession {
  id: string;
  examTitle: string;
  duration: number;
  questions: Question[];
  proctorSettings: {
    cameraRequired: boolean;
    microphoneMonitoring: boolean;
    screenRecording: boolean;
    tabSwitchingBlocked: boolean;
  };
}

export default function ExaminationTaking() {
  const { sessionId } = useParams();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const queryClient = useQueryClient();
  
  const [currentQuestionIndex, setCurrentQuestionIndex] = useState(0);
  const [answers, setAnswers] = useState<Record<string, any>>({});
  const [timeRemaining, setTimeRemaining] = useState(7200); // 2 hours in seconds
  const [questionTimeSpent, setQuestionTimeSpent] = useState<Record<string, number>>({});
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [proctorStatus, setProctorStatus] = useState({
    camera: false,
    microphone: false,
    screen: false,
    tabSwitches: 0,
    suspiciousActivity: 0
  });
  const [examStarted, setExamStarted] = useState(false);
  
  const videoRef = useRef<HTMLVideoElement>(null);
  const startTimeRef = useRef<number>(Date.now());
  const questionStartTimeRef = useRef<number>(Date.now());
  const visibilityRef = useRef<boolean>(true);

  // Fetch examination questions
  const { data: questions, isLoading } = useQuery({
    queryKey: [`backend/api/examinations/${sessionId}/questions`],
    queryFn: async () => {
      return await apiRequest(`backend/api/examinations/${sessionId}/questions`, "GET");
    },
    enabled: !!sessionId
  });

  // Submit answer mutation
  const submitAnswerMutation = useMutation({
    mutationFn: async ({ questionId, answer, timeSpent }: { questionId: string; answer: any; timeSpent: number }) => {
      return await apiRequest(`backend/api/examinations/${sessionId}/submit-answer`, "POST", { questionId, answer, timeSpent });
    },
    onError: (error: any) => {
      toast({
        title: "Failed to Save Answer",
        description: error.message || "Your answer could not be saved",
        variant: "destructive",
      });
    },
  });

  // Complete examination mutation
  const completeExamMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest(`backend/api/examinations/${sessionId}/complete`, "POST", {});
    },
    onSuccess: (data) => {
      toast({
        title: "Examination Completed",
        description: "Your examination has been submitted successfully",
      });
      setLocation(`/examinations/${sessionId}/results`);
    },
    onError: (error: any) => {
      toast({
        title: "Submission Failed",
        description: error.message || "Failed to submit examination",
        variant: "destructive",
      });
    },
  });

  // Flag proctor event mutation
  const flagProctorMutation = useMutation({
    mutationFn: async ({ flagType, description }: { flagType: string; description: string }) => {
      return await apiRequest(`backend/api/examinations/${sessionId}/proctor-flag`, "POST", { flagType, description, timestamp: new Date().toISOString() });
    },
  });

  // Initialize proctoring
  const initializeProctoring = useCallback(async () => {
    try {
      // Request camera access
      const cameraStream = await navigator.mediaDevices.getUserMedia({ 
        video: { facingMode: 'user' },
        audio: true 
      });
      
      if (videoRef.current) {
        videoRef.current.srcObject = cameraStream;
      }
      
      setProctorStatus(prev => ({ 
        ...prev, 
        camera: true, 
        microphone: true 
      }));

      // Monitor tab visibility
      const handleVisibilityChange = () => {
        if (document.hidden && examStarted) {
          setProctorStatus(prev => ({ 
            ...prev, 
            tabSwitches: prev.tabSwitches + 1 
          }));
          
          flagProctorMutation.mutate({
            flagType: "tab_switch",
            description: "User switched away from examination tab"
          });
          
          toast({
            title: "Tab Switch Detected",
            description: "Switching tabs during examination is not allowed",
            variant: "destructive",
          });
        }
        visibilityRef.current = !document.hidden;
      };

      document.addEventListener('visibilitychange', handleVisibilityChange);
      
      // Prevent context menu
      const preventContextMenu = (e: MouseEvent) => {
        e.preventDefault();
        flagProctorMutation.mutate({
          flagType: "context_menu",
          description: "User attempted to access context menu"
        });
      };
      
      document.addEventListener('contextmenu', preventContextMenu);
      
      // Monitor suspicious keyboard activity
      const handleKeyDown = (e: KeyboardEvent) => {
        const suspiciousKeys = ['F12', 'F5', 'PrintScreen'];
        if (suspiciousKeys.includes(e.key) || 
            (e.ctrlKey && ['c', 'v', 'x', 'a', 's'].includes(e.key.toLowerCase()))) {
          e.preventDefault();
          setProctorStatus(prev => ({ 
            ...prev, 
            suspiciousActivity: prev.suspiciousActivity + 1 
          }));
          
          flagProctorMutation.mutate({
            flagType: "suspicious_key",
            description: `Suspicious key combination: ${e.key}`
          });
        }
      };
      
      document.addEventListener('keydown', handleKeyDown);
      
      return () => {
        document.removeEventListener('visibilitychange', handleVisibilityChange);
        document.removeEventListener('contextmenu', preventContextMenu);
        document.removeEventListener('keydown', handleKeyDown);
        cameraStream.getTracks().forEach(track => track.stop());
      };
      
    } catch (error) {
      console.error("Failed to initialize proctoring:", error);
      toast({
        title: "Proctoring Setup Failed",
        description: "Camera and microphone access required for examination",
        variant: "destructive",
      });
    }
  }, [sessionId, examStarted, flagProctorMutation, toast]);

  // Timer effect
  useEffect(() => {
    if (!examStarted) return;
    
    const timer = setInterval(() => {
      setTimeRemaining(prev => {
        if (prev <= 1) {
          completeExamMutation.mutate();
          return 0;
        }
        return prev - 1;
      });
    }, 1000);

    return () => clearInterval(timer);
  }, [examStarted, completeExamMutation]);

  // Track question time
  useEffect(() => {
    if (!questions || !examStarted) return;
    
    const currentQuestion = questions[currentQuestionIndex];
    if (currentQuestion) {
      questionStartTimeRef.current = Date.now();
    }
    
    return () => {
      if (currentQuestion) {
        const timeSpent = Math.round((Date.now() - questionStartTimeRef.current) / 1000);
        setQuestionTimeSpent(prev => ({
          ...prev,
          [currentQuestion.id]: (prev[currentQuestion.id] || 0) + timeSpent
        }));
      }
    };
  }, [currentQuestionIndex, questions, examStarted]);

  // Initialize proctoring on component mount
  useEffect(() => {
    const cleanup = initializeProctoring();
    return () => {
      if (cleanup instanceof Promise) {
        cleanup.then(fn => fn && fn());
      }
    };
  }, [initializeProctoring]);

  const formatTime = (seconds: number) => {
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    return `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
  };

  const handleAnswerChange = (questionId: string, answer: any) => {
    setAnswers(prev => ({ ...prev, [questionId]: answer }));
    
    // Auto-save answer
    const timeSpent = questionTimeSpent[questionId] || 0;
    submitAnswerMutation.mutate({ questionId, answer, timeSpent });
  };

  const goToQuestion = (index: number) => {
    if (index >= 0 && index < (questions?.length || 0)) {
      setCurrentQuestionIndex(index);
    }
  };

  const handleSubmitExam = () => {
    setIsSubmitting(true);
    completeExamMutation.mutate();
  };

  const startExamination = () => {
    setExamStarted(true);
    startTimeRef.current = Date.now();
    questionStartTimeRef.current = Date.now();
  };

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-red-600"></div>
      </div>
    );
  }

  if (!questions || questions.length === 0) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <Card className="max-w-md">
          <CardContent className="p-6 text-center">
            <AlertTriangle className="w-12 h-12 text-yellow-500 mx-auto mb-4" />
            <h2 className="text-xl font-semibold mb-2">Examination Not Found</h2>
            <p className="text-gray-600 mb-4">The requested examination session could not be loaded.</p>
            <Button onClick={() => setLocation('/examinations')}>
              Return to Examination Center
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  if (!examStarted) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <Card className="max-w-2xl w-full mx-4">
          <CardHeader>
            <CardTitle className="text-center">Ready to Begin Examination</CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            {/* Proctoring Status */}
            <div className="border rounded-lg p-4">
              <h3 className="font-medium mb-3 flex items-center gap-2">
                <Shield className="w-4 h-4" />
                Proctoring Status
              </h3>
              <div className="grid grid-cols-2 gap-3">
                <div className={`flex items-center gap-2 p-2 rounded ${proctorStatus.camera ? 'bg-green-50 text-green-700' : 'bg-red-50 text-red-700'}`}>
                  <Camera className="w-4 h-4" />
                  <span className="text-sm">Camera: {proctorStatus.camera ? 'Ready' : 'Not Connected'}</span>
                </div>
                <div className={`flex items-center gap-2 p-2 rounded ${proctorStatus.microphone ? 'bg-green-50 text-green-700' : 'bg-red-50 text-red-700'}`}>
                  <Mic className="w-4 h-4" />
                  <span className="text-sm">Microphone: {proctorStatus.microphone ? 'Ready' : 'Not Connected'}</span>
                </div>
              </div>
            </div>

            {/* Camera Preview */}
            <div className="border rounded-lg p-4">
              <h3 className="font-medium mb-3">Camera Preview</h3>
              <video
                ref={videoRef}
                autoPlay
                muted
                className="w-full max-w-sm mx-auto rounded-lg bg-gray-900"
                style={{ height: '200px', objectFit: 'cover' }}
              />
            </div>

            {/* Examination Info */}
            <div className="border rounded-lg p-4">
              <h3 className="font-medium mb-3">Examination Details</h3>
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-gray-600">Questions:</span>
                  <span className="ml-2 font-medium">{questions.length}</span>
                </div>
                <div>
                  <span className="text-gray-600">Duration:</span>
                  <span className="ml-2 font-medium">{formatTime(timeRemaining)}</span>
                </div>
              </div>
            </div>

            <div className="text-center">
              <Button
                onClick={startExamination}
                disabled={!proctorStatus.camera || !proctorStatus.microphone}
                className="px-8 py-3 text-lg"
              >
                <Play className="w-5 h-5 mr-2" />
                Begin Examination
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  const currentQuestion = questions[currentQuestionIndex];
  const progress = ((currentQuestionIndex + 1) / questions.length) * 100;

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Fixed Header */}
      <div className="fixed top-0 left-0 right-0 bg-white border-b shadow-sm z-50">
        <div className="container mx-auto px-4 py-3">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <Badge variant="outline" className="bg-red-50 text-red-700 border-red-200">
                IBDL Certification Examination
              </Badge>
              <div className="text-sm text-gray-600">
                Question {currentQuestionIndex + 1} of {questions.length}
              </div>
            </div>
            
            <div className="flex items-center gap-4">
              {/* Proctoring Indicators */}
              <div className="flex items-center gap-2">
                <div className={`w-2 h-2 rounded-full ${proctorStatus.camera ? 'bg-green-500' : 'bg-red-500'}`} />
                <Camera className="w-4 h-4 text-gray-500" />
                <div className={`w-2 h-2 rounded-full ${proctorStatus.microphone ? 'bg-green-500' : 'bg-red-500'}`} />
                <Mic className="w-4 h-4 text-gray-500" />
              </div>
              
              {/* Time Remaining */}
              <div className={`text-lg font-mono ${timeRemaining < 600 ? 'text-red-600' : 'text-gray-900'}`}>
                {formatTime(timeRemaining)}
              </div>
              
              <Button
                variant="destructive"
                onClick={handleSubmitExam}
                disabled={isSubmitting}
              >
                <Send className="w-4 h-4 mr-2" />
                Submit Exam
              </Button>
            </div>
          </div>
          
          <Progress value={progress} className="mt-2" />
        </div>
      </div>

      {/* Main Content */}
      <div className="pt-24 pb-8">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-12 gap-6">
            {/* Question Panel */}
            <div className="col-span-12 lg:col-span-8">
              <Card>
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <CardTitle className="text-lg">
                      Question {currentQuestionIndex + 1}
                    </CardTitle>
                    <Badge variant="secondary">
                      {currentQuestion.points} {currentQuestion.points === 1 ? 'point' : 'points'}
                    </Badge>
                  </div>
                </CardHeader>
                <CardContent className="space-y-6">
                  {/* Question Context */}
                  {currentQuestion.context && (
                    <Alert>
                      <AlertDescription>
                        <strong>Context:</strong> {currentQuestion.context}
                      </AlertDescription>
                    </Alert>
                  )}

                  {/* Question Text */}
                  <div className="text-lg leading-relaxed">
                    {currentQuestion.question}
                  </div>

                  {/* Answer Input Based on Question Type */}
                  <div className="space-y-4">
                    {currentQuestion.type === 'multiple_choice' && (
                      <RadioGroup
                        value={answers[currentQuestion.id]?.toString() || ""}
                        onValueChange={(value) => handleAnswerChange(currentQuestion.id, parseInt(value))}
                      >
                        {currentQuestion.options?.map((option, index) => (
                          <div key={index} className="flex items-center space-x-2 p-3 rounded border hover:bg-gray-50">
                            <RadioGroupItem value={index.toString()} id={`option-${index}`} />
                            <label htmlFor={`option-${index}`} className="flex-1 cursor-pointer">
                              {option}
                            </label>
                          </div>
                        ))}
                      </RadioGroup>
                    )}

                    {currentQuestion.type === 'multiple_select' && (
                      <div className="space-y-2">
                        {currentQuestion.options?.map((option, index) => (
                          <div key={index} className="flex items-center space-x-2 p-3 rounded border hover:bg-gray-50">
                            <Checkbox
                              id={`checkbox-${index}`}
                              checked={(answers[currentQuestion.id] || []).includes(index)}
                              onCheckedChange={(checked) => {
                                const currentAnswers = answers[currentQuestion.id] || [];
                                const newAnswers = checked
                                  ? [...currentAnswers, index]
                                  : currentAnswers.filter((i: number) => i !== index);
                                handleAnswerChange(currentQuestion.id, newAnswers);
                              }}
                            />
                            <label htmlFor={`checkbox-${index}`} className="flex-1 cursor-pointer">
                              {option}
                            </label>
                          </div>
                        ))}
                      </div>
                    )}

                    {currentQuestion.type === 'true_false' && (
                      <RadioGroup
                        value={answers[currentQuestion.id]?.toString() || ""}
                        onValueChange={(value) => handleAnswerChange(currentQuestion.id, value === 'true')}
                      >
                        <div className="flex items-center space-x-2 p-3 rounded border hover:bg-gray-50">
                          <RadioGroupItem value="true" id="true" />
                          <label htmlFor="true" className="cursor-pointer">True</label>
                        </div>
                        <div className="flex items-center space-x-2 p-3 rounded border hover:bg-gray-50">
                          <RadioGroupItem value="false" id="false" />
                          <label htmlFor="false" className="cursor-pointer">False</label>
                        </div>
                      </RadioGroup>
                    )}

                    {(currentQuestion.type === 'short_answer' || currentQuestion.type === 'case_study') && (
                      <div>
                        <Textarea
                          placeholder="Enter your answer here..."
                          value={answers[currentQuestion.id] || ""}
                          onChange={(e) => handleAnswerChange(currentQuestion.id, e.target.value)}
                          className="min-h-[200px]"
                        />
                        {currentQuestion.rubric && (
                          <div className="mt-4 p-3 bg-blue-50 rounded">
                            <h4 className="font-medium text-sm mb-2">Evaluation Criteria:</h4>
                            <ul className="text-sm space-y-1">
                              {currentQuestion.rubric.criteria.map((criterion, index) => (
                                <li key={index} className="flex items-center gap-2">
                                  <div className="w-1.5 h-1.5 bg-blue-500 rounded-full" />
                                  {criterion}
                                </li>
                              ))}
                            </ul>
                          </div>
                        )}
                      </div>
                    )}
                  </div>

                  {/* Navigation */}
                  <div className="flex items-center justify-between pt-6 border-t">
                    <Button
                      variant="outline"
                      onClick={() => goToQuestion(currentQuestionIndex - 1)}
                      disabled={currentQuestionIndex === 0}
                    >
                      <ChevronLeft className="w-4 h-4 mr-2" />
                      Previous
                    </Button>
                    
                    <div className="flex items-center gap-2">
                      <Save className="w-4 h-4 text-green-600" />
                      <span className="text-sm text-green-600">Auto-saved</span>
                    </div>
                    
                    <Button
                      onClick={() => goToQuestion(currentQuestionIndex + 1)}
                      disabled={currentQuestionIndex === questions.length - 1}
                    >
                      Next
                      <ChevronRight className="w-4 h-4 ml-2" />
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Sidebar */}
            <div className="col-span-12 lg:col-span-4 space-y-6">
              {/* Camera Feed */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-sm">Proctoring Monitor</CardTitle>
                </CardHeader>
                <CardContent>
                  <video
                    ref={videoRef}
                    autoPlay
                    muted
                    className="w-full rounded bg-gray-900"
                    style={{ aspectRatio: '4/3' }}
                  />
                  <div className="mt-3 grid grid-cols-2 gap-2 text-xs">
                    <div className="flex items-center justify-between">
                      <span>Tab Switches:</span>
                      <span className={proctorStatus.tabSwitches > 3 ? 'text-red-600 font-medium' : ''}>
                        {proctorStatus.tabSwitches}
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span>Flags:</span>
                      <span className={proctorStatus.suspiciousActivity > 5 ? 'text-red-600 font-medium' : ''}>
                        {proctorStatus.suspiciousActivity}
                      </span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Question Navigator */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-sm">Question Navigator</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-5 gap-2">
                    {questions.map((_, index) => (
                      <Button
                        key={index}
                        variant={index === currentQuestionIndex ? "default" : "outline"}
                        size="sm"
                        className={`h-8 text-xs ${
                          answers[questions[index].id] 
                            ? 'bg-green-100 border-green-300 text-green-800' 
                            : index === currentQuestionIndex 
                              ? '' 
                              : 'bg-gray-50'
                        }`}
                        onClick={() => goToQuestion(index)}
                      >
                        {index + 1}
                      </Button>
                    ))}
                  </div>
                  <div className="mt-4 text-xs space-y-2">
                    <div className="flex items-center gap-2">
                      <div className="w-3 h-3 bg-green-100 border border-green-300 rounded"></div>
                      <span>Answered</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <div className="w-3 h-3 bg-blue-600 rounded"></div>
                      <span>Current</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <div className="w-3 h-3 bg-gray-50 border rounded"></div>
                      <span>Not answered</span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Examination Progress */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-sm">Progress Summary</CardTitle>
                </CardHeader>
                <CardContent className="space-y-3">
                  <div className="flex justify-between text-sm">
                    <span>Questions Answered:</span>
                    <span>{Object.keys(answers).length} / {questions.length}</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>Time Elapsed:</span>
                    <span>{formatTime(7200 - timeRemaining)}</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>Average per Question:</span>
                    <span>
                      {Object.keys(answers).length > 0 
                        ? formatTime(Math.round((7200 - timeRemaining) / Object.keys(answers).length))
                        : "0:00:00"
                      }
                    </span>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}