import { useParams, useLocation } from "wouter";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { apiRequest } from "@/lib/queryClient";
import { 
  CheckCircle, 
  XCircle, 
  Clock, 
  Award, 
  Download, 
  Share, 
  BarChart,
  TrendingUp,
  Calendar,
  User,
  FileText,
  ArrowLeft
} from "lucide-react";

interface ExaminationResults {
  sessionId: string;
  examTitle: string;
  completedAt: string;
  totalScore: number;
  passingScore: number;
  passed: boolean;
  timeSpent: number;
  totalTime: number;
  questionBreakdown: Array<{
    category: string;
    score: number;
    total: number;
  }>;
  certificateId?: string;
  feedback: string;
}

export default function ExaminationResults() {
  const { sessionId } = useParams();
  const [, setLocation] = useLocation();

  const { data: results, isLoading } = useQuery({
    queryKey: [`backend/api/examinations/${sessionId}/results`],
    queryFn: async () => {
      return await apiRequest(`backend/api/examinations/${sessionId}/results`, "GET");
    },
    enabled: !!sessionId
  });

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-red-600"></div>
      </div>
    );
  }

  if (!results) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <Card className="max-w-md">
          <CardContent className="p-6 text-center">
            <XCircle className="w-12 h-12 text-red-500 mx-auto mb-4" />
            <h2 className="text-xl font-semibold mb-2">Results Not Found</h2>
            <p className="text-gray-600 mb-4">The examination results could not be loaded.</p>
            <Button onClick={() => setLocation('/examinations')}>
              Return to Examination Center
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  const formatTime = (minutes: number) => {
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    return hours > 0 ? `${hours}h ${mins}m` : `${mins}m`;
  };

  const getScoreColor = (score: number, passing: number) => {
    if (score >= passing) return 'text-green-600';
    if (score >= passing * 0.8) return 'text-yellow-600';
    return 'text-red-600';
  };

  const getPerformanceLevel = (score: number) => {
    if (score >= 90) return { label: 'Excellent', color: 'bg-green-500' };
    if (score >= 80) return { label: 'Good', color: 'bg-blue-500' };
    if (score >= 70) return { label: 'Satisfactory', color: 'bg-yellow-500' };
    return { label: 'Needs Improvement', color: 'bg-red-500' };
  };

  const performance = getPerformanceLevel(results.totalScore);

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="mb-8">
          <Button
            variant="ghost"
            onClick={() => setLocation('/examinations')}
            className="mb-4"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Examination Center
          </Button>
          
          <div className="text-center">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Examination Results</h1>
            <p className="text-gray-600">{results.examTitle}</p>
          </div>
        </div>

        {/* Overall Result Card */}
        <Card className="mb-8">
          <CardContent className="p-8">
            <div className="text-center">
              {results.passed ? (
                <CheckCircle className="w-16 h-16 text-green-500 mx-auto mb-4" />
              ) : (
                <XCircle className="w-16 h-16 text-red-500 mx-auto mb-4" />
              )}
              
              <h2 className="text-2xl font-bold mb-2">
                {results.passed ? 'Congratulations!' : 'Better Luck Next Time'}
              </h2>
              
              <p className="text-gray-600 mb-6">
                {results.passed 
                  ? 'You have successfully passed the examination'
                  : 'You did not meet the passing criteria'}
              </p>

              <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                <div className="text-center">
                  <div className={`text-3xl font-bold ${getScoreColor(results.totalScore, results.passingScore)}`}>
                    {results.totalScore}%
                  </div>
                  <div className="text-sm text-gray-600">Your Score</div>
                </div>
                
                <div className="text-center">
                  <div className="text-3xl font-bold text-gray-700">
                    {results.passingScore}%
                  </div>
                  <div className="text-sm text-gray-600">Passing Score</div>
                </div>
                
                <div className="text-center">
                  <Badge className={`${performance.color} text-white px-3 py-1`}>
                    {performance.label}
                  </Badge>
                  <div className="text-sm text-gray-600 mt-1">Performance Level</div>
                </div>
              </div>

              {results.certificateId && results.passed && (
                <Alert className="bg-green-50 border-green-200 mb-6">
                  <Award className="w-4 h-4 text-green-600" />
                  <AlertDescription className="text-green-800">
                    <strong>Certificate Generated!</strong> Your IBDL certification is ready for download.
                  </AlertDescription>
                </Alert>
              )}
            </div>
          </CardContent>
        </Card>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* Performance Breakdown */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <BarChart className="w-5 h-5" />
                Performance Breakdown
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              {results.questionBreakdown.map((category, index) => (
                <div key={index} className="space-y-2">
                  <div className="flex justify-between text-sm">
                    <span className="font-medium">{category.category}</span>
                    <span className={getScoreColor(category.score, results.passingScore)}>
                      {category.score}%
                    </span>
                  </div>
                  <Progress 
                    value={category.score} 
                    className="h-2"
                  />
                </div>
              ))}
            </CardContent>
          </Card>

          {/* Examination Details */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileText className="w-5 h-5" />
                Examination Details
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div className="flex items-center gap-2">
                  <Calendar className="w-4 h-4 text-gray-500" />
                  <span className="text-gray-600">Completed:</span>
                </div>
                <span>{new Date(results.completedAt).toLocaleDateString()}</span>
                
                <div className="flex items-center gap-2">
                  <Clock className="w-4 h-4 text-gray-500" />
                  <span className="text-gray-600">Time Spent:</span>
                </div>
                <span>{formatTime(results.timeSpent)}</span>
                
                <div className="flex items-center gap-2">
                  <TrendingUp className="w-4 h-4 text-gray-500" />
                  <span className="text-gray-600">Total Time:</span>
                </div>
                <span>{formatTime(results.totalTime)}</span>
                
                <div className="flex items-center gap-2">
                  <User className="w-4 h-4 text-gray-500" />
                  <span className="text-gray-600">Session ID:</span>
                </div>
                <span className="font-mono text-xs">{results.sessionId}</span>
              </div>

              {results.certificateId && (
                <div className="border-t pt-4">
                  <div className="flex items-center gap-2 mb-2">
                    <Award className="w-4 h-4 text-gray-500" />
                    <span className="text-gray-600 text-sm">Certificate ID:</span>
                  </div>
                  <span className="font-mono text-xs bg-gray-100 px-2 py-1 rounded">
                    {results.certificateId}
                  </span>
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        {/* Feedback Section */}
        <Card className="mt-8">
          <CardHeader>
            <CardTitle>Personalized Feedback</CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-gray-700 leading-relaxed">{results.feedback}</p>
          </CardContent>
        </Card>

        {/* Action Buttons */}
        <div className="mt-8 flex flex-wrap gap-4 justify-center">
          {results.passed && results.certificateId && (
            <>
              <Button className="flex items-center gap-2">
                <Download className="w-4 h-4" />
                Download Certificate
              </Button>
              <Button variant="outline" className="flex items-center gap-2">
                <Share className="w-4 h-4" />
                Share Achievement
              </Button>
            </>
          )}
          
          {!results.passed && (
            <Button 
              onClick={() => setLocation('/examinations')}
              className="flex items-center gap-2"
            >
              <Clock className="w-4 h-4" />
              Schedule Retake
            </Button>
          )}
          
          <Button 
            variant="outline"
            onClick={() => setLocation('/courses')}
            className="flex items-center gap-2"
          >
            <TrendingUp className="w-4 h-4" />
            Continue Learning
          </Button>
        </div>

        {/* Next Steps */}
        <Card className="mt-8">
          <CardHeader>
            <CardTitle>Next Steps</CardTitle>
          </CardHeader>
          <CardContent>
            {results.passed ? (
              <div className="space-y-3">
                <h4 className="font-medium text-green-800">Congratulations on passing!</h4>
                <ul className="space-y-2 text-sm text-gray-600">
                  <li>• Download and save your certificate for professional use</li>
                  <li>• Add your new certification to your LinkedIn profile</li>
                  <li>• Explore advanced courses to further develop your skills</li>
                  <li>• Consider specialized certification tracks in your field</li>
                </ul>
              </div>
            ) : (
              <div className="space-y-3">
                <h4 className="font-medium text-red-800">Areas for Improvement</h4>
                <ul className="space-y-2 text-sm text-gray-600">
                  <li>• Review the areas where you scored below the passing threshold</li>
                  <li>• Take additional practice assessments to strengthen weak areas</li>
                  <li>• Schedule a retake when you feel confident in your preparation</li>
                  <li>• Consider enrolling in preparatory courses for better results</li>
                </ul>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}