import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { 
  Shield, 
  Clock, 
  Users, 
  FileText, 
  Play, 
  Pause,
  AlertTriangle,
  CheckCircle,
  Calendar,
  Monitor,
  Camera,
  Mic,
  Eye,
  BookOpen,
  Award
} from "lucide-react";

interface ExaminationSession {
  id: string;
  examTitle: string;
  examType: 'certification' | 'proctored' | 'practice';
  scheduledDate: string;
  duration: number;
  status: 'scheduled' | 'in_progress' | 'paused' | 'completed' | 'cancelled';
  questions: number;
  passingScore: number;
  currentScore?: number;
  timeRemaining?: number;
  proctorSettings: {
    cameraRequired: boolean;
    microphoneMonitoring: boolean;
    screenRecording: boolean;
    tabSwitchingBlocked: boolean;
    identityVerification: boolean;
  };
  attempts: number;
  maxAttempts: number;
}

export default function ExaminationCenter() {
  const [selectedSession, setSelectedSession] = useState<ExaminationSession | null>(null);
  const [proctorChecksComplete, setProctorChecksComplete] = useState(false);
  const [systemChecks, setSystemChecks] = useState({
    camera: false,
    microphone: false,
    screen: false,
    browser: false,
    network: false
  });
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [, setLocation] = useLocation();

  const { data: examinationSessions, isLoading } = useQuery({
    queryKey: ["backend/api/examinations"],
    queryFn: async () => {
      return await apiRequest("backend/api/examinations", "GET");
    },
  });

  const startExaminationMutation = useMutation({
    mutationFn: async ({ sessionId }: { sessionId: string }) => {
      return await apiRequest("backend/api/examinations/start", "POST", { sessionId });
    },
    onSuccess: (data) => {
      toast({
        title: "Examination Started",
        description: "Your proctored examination has begun. Good luck!",
      });
      setLocation(`/examinations/${data.sessionId}/take`);
      queryClient.invalidateQueries({ queryKey: ["backend/api/examinations"] });
    },
    onError: (error: any) => {
      toast({
        title: "Failed to Start Examination",
        description: error.message || "Unable to begin examination session",
        variant: "destructive",
      });
    },
  });

  // System check functions
  const performSystemChecks = async () => {
    try {
      // Camera check
      const cameraStream = await navigator.mediaDevices.getUserMedia({ video: true });
      setSystemChecks(prev => ({ ...prev, camera: true }));
      cameraStream.getTracks().forEach(track => track.stop());

      // Microphone check
      const micStream = await navigator.mediaDevices.getUserMedia({ audio: true });
      setSystemChecks(prev => ({ ...prev, microphone: true }));
      micStream.getTracks().forEach(track => track.stop());

      // Screen sharing check
      if (navigator.mediaDevices.getDisplayMedia) {
        setSystemChecks(prev => ({ ...prev, screen: true }));
      }

      // Browser compatibility
      const isChrome = /Chrome/.test(navigator.userAgent);
      const isFirefox = /Firefox/.test(navigator.userAgent);
      setSystemChecks(prev => ({ ...prev, browser: isChrome || isFirefox }));

      // Network check
      setSystemChecks(prev => ({ ...prev, network: navigator.onLine }));

    } catch (error) {
      console.error("System check failed:", error);
      toast({
        title: "System Check Failed",
        description: "Please ensure camera and microphone permissions are granted.",
        variant: "destructive",
      });
    }
  };

  useEffect(() => {
    performSystemChecks();
  }, []);

  const allSystemChecksPass = Object.values(systemChecks).every(check => check);

  const formatDuration = (minutes: number) => {
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    return hours > 0 ? `${hours}h ${mins}m` : `${mins}m`;
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'scheduled': return 'bg-blue-100 text-blue-800';
      case 'in_progress': return 'bg-yellow-100 text-yellow-800';
      case 'completed': return 'bg-green-100 text-green-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  if (isLoading) {
    return (
      <div className="container mx-auto px-4 py-8">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-1/3"></div>
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {[1, 2, 3, 4].map(i => (
              <div key={i} className="h-64 bg-gray-200 rounded"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900 mb-2">IBDL Examination Center</h1>
        <p className="text-gray-600">Secure proctored examinations with comprehensive monitoring</p>
      </div>

      {/* System Status Check */}
      <Card className="mb-8">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Shield className="w-5 h-5" />
            System Readiness Check
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-2 md:grid-cols-5 gap-4">
            <div className={`flex items-center gap-2 p-3 rounded-lg ${systemChecks.camera ? 'bg-green-50' : 'bg-red-50'}`}>
              <Camera className={`w-4 h-4 ${systemChecks.camera ? 'text-green-600' : 'text-red-600'}`} />
              <span className={`text-sm font-medium ${systemChecks.camera ? 'text-green-700' : 'text-red-700'}`}>
                Camera
              </span>
            </div>
            <div className={`flex items-center gap-2 p-3 rounded-lg ${systemChecks.microphone ? 'bg-green-50' : 'bg-red-50'}`}>
              <Mic className={`w-4 h-4 ${systemChecks.microphone ? 'text-green-600' : 'text-red-600'}`} />
              <span className={`text-sm font-medium ${systemChecks.microphone ? 'text-green-700' : 'text-red-700'}`}>
                Microphone
              </span>
            </div>
            <div className={`flex items-center gap-2 p-3 rounded-lg ${systemChecks.screen ? 'bg-green-50' : 'bg-red-50'}`}>
              <Monitor className={`w-4 h-4 ${systemChecks.screen ? 'text-green-600' : 'text-red-600'}`} />
              <span className={`text-sm font-medium ${systemChecks.screen ? 'text-green-700' : 'text-red-700'}`}>
                Screen Share
              </span>
            </div>
            <div className={`flex items-center gap-2 p-3 rounded-lg ${systemChecks.browser ? 'bg-green-50' : 'bg-red-50'}`}>
              <Eye className={`w-4 h-4 ${systemChecks.browser ? 'text-green-600' : 'text-red-600'}`} />
              <span className={`text-sm font-medium ${systemChecks.browser ? 'text-green-700' : 'text-red-700'}`}>
                Browser
              </span>
            </div>
            <div className={`flex items-center gap-2 p-3 rounded-lg ${systemChecks.network ? 'bg-green-50' : 'bg-red-50'}`}>
              <Shield className={`w-4 h-4 ${systemChecks.network ? 'text-green-600' : 'text-red-600'}`} />
              <span className={`text-sm font-medium ${systemChecks.network ? 'text-green-700' : 'text-red-700'}`}>
                Network
              </span>
            </div>
          </div>
          {!allSystemChecksPass && (
            <Alert className="mt-4">
              <AlertTriangle className="w-4 h-4" />
              <AlertDescription>
                System checks incomplete. Please ensure all requirements are met before starting examinations.
              </AlertDescription>
            </Alert>
          )}
        </CardContent>
      </Card>

      {/* Examination Sessions */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {examinationSessions?.map((session: ExaminationSession) => (
          <Card key={session.id} className="hover:shadow-lg transition-shadow">
            <CardHeader>
              <div className="flex justify-between items-start">
                <div>
                  <CardTitle className="text-xl mb-2">{session.examTitle}</CardTitle>
                  <Badge className={getStatusColor(session.status)}>
                    {session.status.replace('_', ' ')}
                  </Badge>
                </div>
                <div className="text-right">
                  <Badge variant="outline" className="mb-2">
                    {session.examType.toUpperCase()}
                  </Badge>
                </div>
              </div>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div className="flex items-center gap-2">
                  <Calendar className="w-4 h-4 text-gray-500" />
                  <span>{new Date(session.scheduledDate).toLocaleDateString()}</span>
                </div>
                <div className="flex items-center gap-2">
                  <Clock className="w-4 h-4 text-gray-500" />
                  <span>{formatDuration(session.duration)}</span>
                </div>
                <div className="flex items-center gap-2">
                  <FileText className="w-4 h-4 text-gray-500" />
                  <span>{session.questions} Questions</span>
                </div>
                <div className="flex items-center gap-2">
                  <Award className="w-4 h-4 text-gray-500" />
                  <span>{session.passingScore}% Required</span>
                </div>
              </div>

              {/* Proctoring Requirements */}
              <div className="border-t pt-4">
                <h4 className="font-medium text-sm mb-2">Proctoring Requirements:</h4>
                <div className="grid grid-cols-2 gap-2 text-xs">
                  {session.proctorSettings.cameraRequired && (
                    <div className="flex items-center gap-1">
                      <Camera className="w-3 h-3" />
                      <span>Camera Required</span>
                    </div>
                  )}
                  {session.proctorSettings.microphoneMonitoring && (
                    <div className="flex items-center gap-1">
                      <Mic className="w-3 h-3" />
                      <span>Audio Monitoring</span>
                    </div>
                  )}
                  {session.proctorSettings.screenRecording && (
                    <div className="flex items-center gap-1">
                      <Monitor className="w-3 h-3" />
                      <span>Screen Recording</span>
                    </div>
                  )}
                  {session.proctorSettings.identityVerification && (
                    <div className="flex items-center gap-1">
                      <Shield className="w-3 h-3" />
                      <span>ID Verification</span>
                    </div>
                  )}
                </div>
              </div>

              {/* Progress for in-progress exams */}
              {session.status === 'in_progress' && session.timeRemaining && (
                <div className="border-t pt-4">
                  <div className="flex justify-between text-sm mb-2">
                    <span>Time Remaining</span>
                    <span>{formatDuration(session.timeRemaining)}</span>
                  </div>
                  <Progress value={(session.timeRemaining / session.duration) * 100} className="mb-2" />
                  {session.currentScore && (
                    <div className="flex justify-between text-sm">
                      <span>Current Score</span>
                      <span>{session.currentScore}%</span>
                    </div>
                  )}
                </div>
              )}

              {/* Attempt Information */}
              <div className="flex justify-between text-sm text-gray-600">
                <span>Attempts: {session.attempts}/{session.maxAttempts}</span>
                {session.status === 'completed' && session.currentScore && (
                  <span className={`font-medium ${session.currentScore >= session.passingScore ? 'text-green-600' : 'text-red-600'}`}>
                    Final Score: {session.currentScore}%
                  </span>
                )}
              </div>

              {/* Action Buttons */}
              <div className="flex gap-2 pt-4">
                {session.status === 'scheduled' && (
                  <Button
                    className="flex-1"
                    onClick={() => setSelectedSession(session)}
                    disabled={!allSystemChecksPass || session.attempts >= session.maxAttempts}
                  >
                    <Play className="w-4 h-4 mr-2" />
                    Start Examination
                  </Button>
                )}
                {session.status === 'in_progress' && (
                  <Button
                    className="flex-1"
                    onClick={() => setLocation(`/examinations/${session.id}/take`)}
                  >
                    <Play className="w-4 h-4 mr-2" />
                    Continue Examination
                  </Button>
                )}
                {session.status === 'completed' && (
                  <Button
                    variant="outline"
                    className="flex-1"
                    onClick={() => setLocation(`/examinations/${session.id}/results`)}
                  >
                    <FileText className="w-4 h-4 mr-2" />
                    View Results
                  </Button>
                )}
                <Button
                  variant="outline"
                  onClick={() => setLocation(`/examinations/${session.id}/details`)}
                >
                  <BookOpen className="w-4 h-4 mr-2" />
                  Details
                </Button>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Pre-Examination Modal */}
      {selectedSession && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
          <Card className="w-full max-w-2xl max-h-[90vh] overflow-y-auto">
            <CardHeader>
              <CardTitle>Pre-Examination Setup</CardTitle>
              <p className="text-gray-600">Please complete all requirements before starting</p>
            </CardHeader>
            <CardContent className="space-y-6">
              {/* Identity Verification */}
              {selectedSession.proctorSettings.identityVerification && (
                <div className="border rounded-lg p-4">
                  <h3 className="font-medium mb-2 flex items-center gap-2">
                    <Shield className="w-4 h-4" />
                    Identity Verification Required
                  </h3>
                  <p className="text-sm text-gray-600 mb-3">
                    Please have your government-issued ID ready for verification
                  </p>
                  <Button variant="outline" size="sm">
                    <Camera className="w-4 h-4 mr-2" />
                    Verify Identity
                  </Button>
                </div>
              )}

              {/* System Requirements */}
              <div className="border rounded-lg p-4">
                <h3 className="font-medium mb-2">System Requirements</h3>
                <div className="space-y-2 text-sm">
                  <div className="flex items-center justify-between">
                    <span>Stable internet connection</span>
                    <CheckCircle className="w-4 h-4 text-green-600" />
                  </div>
                  <div className="flex items-center justify-between">
                    <span>Camera and microphone access</span>
                    <CheckCircle className="w-4 h-4 text-green-600" />
                  </div>
                  <div className="flex items-center justify-between">
                    <span>Supported browser (Chrome/Firefox)</span>
                    <CheckCircle className="w-4 h-4 text-green-600" />
                  </div>
                </div>
              </div>

              {/* Examination Rules */}
              <div className="border rounded-lg p-4">
                <h3 className="font-medium mb-2">Examination Rules</h3>
                <ul className="text-sm space-y-1 text-gray-600">
                  <li>• No external materials or assistance allowed</li>
                  <li>• Keep your face visible to the camera at all times</li>
                  <li>• Do not switch tabs or leave the examination window</li>
                  <li>• Ensure quiet environment without interruptions</li>
                  <li>• Any suspicious activity may result in disqualification</li>
                </ul>
              </div>

              {/* Agreement Checkbox */}
              <div className="flex items-start gap-2">
                <input
                  type="checkbox"
                  id="agreement"
                  className="mt-1"
                  onChange={(e) => setProctorChecksComplete(e.target.checked)}
                />
                <label htmlFor="agreement" className="text-sm">
                  I have read and agree to the examination rules and understand that this session will be monitored and recorded
                </label>
              </div>

              <div className="flex gap-2 pt-4">
                <Button
                  variant="outline"
                  className="flex-1"
                  onClick={() => setSelectedSession(null)}
                >
                  Cancel
                </Button>
                <Button
                  className="flex-1"
                  disabled={!proctorChecksComplete || !allSystemChecksPass}
                  onClick={() => {
                    startExaminationMutation.mutate({ sessionId: selectedSession.id });
                    setSelectedSession(null);
                  }}
                >
                  {startExaminationMutation.isPending ? "Starting..." : "Begin Examination"}
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>
      )}
    </div>
  );
}