import React, { useState, useEffect } from 'react';
import { useParams, useLocation } from 'wouter';
import { useQuery, useMutation } from '@tanstack/react-query';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Progress } from '@/components/ui/progress';
import { Badge } from '@/components/ui/badge';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import { Textarea } from '@/components/ui/textarea';
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from '@/components/ui/alert-dialog';
import { 
  Clock, 
  AlertTriangle, 
  CheckCircle, 
  ArrowRight, 
  ArrowLeft,
  FileText,
  Award,
  Timer,
  Eye,
  EyeOff,
  Trophy,
  Target,
  TrendingUp
} from 'lucide-react';
import { apiRequest, queryClient } from '@/lib/queryClient';
import { toast } from '@/hooks/use-toast';
import { 
  RadarChart, 
  Radar, 
  PolarGrid, 
  PolarAngleAxis, 
  PolarRadiusAxis, 
  ResponsiveContainer,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip
} from 'recharts';

interface Question {
  id: number;
  type: 'multiple_choice' | 'true_false' | 'multiple_select' | 'short_answer' | 'essay' | 'likert_scale' | 'scenario_based' | 'cognitive_reasoning';
  question: string;
  options?: string[];
  scenario?: string;
  scale?: {
    min: number;
    max: number;
    labels: string[];
  };
  correct?: any;
  points: number;
  category?: string;
  competency?: string;
}

interface Assessment {
  id: string;
  title: string;
  type: 'certification' | 'skills' | 'psychometric' | 'personality';
  questions: Question[];
  settings: {
    timeLimit: number;
    randomizeQuestions: boolean;
    allowReview: boolean;
    showResults: boolean;
    isAdaptive: boolean;
  };
  passingScore?: number;
  competencyFramework?: string[];
}

export default function AssessmentTaking() {
  const { assessmentId, attemptId } = useParams();
  const [, navigate] = useLocation();

  // Debug logging - only log once on mount
  useEffect(() => {
    console.log('AssessmentTaking component mounted:', { assessmentId, attemptId });
  }, [assessmentId, attemptId]);
  const [currentQuestion, setCurrentQuestion] = useState(0);
  const [answers, setAnswers] = useState<Record<number, any>>({});
  const [timeRemaining, setTimeRemaining] = useState(0);
  const [isSubmitted, setIsSubmitted] = useState(false);
  const [showReview, setShowReview] = useState(false);
  const [proctoring, setProctoring] = useState(false);

  // Fetch assessment and attempt data
  const { data: assessment, isLoading, error } = useQuery({
    queryKey: [`backend/api/assessments/${assessmentId}`],
    enabled: !!assessmentId,
  });

  const { data: attempt } = useQuery({
    queryKey: [`backend/api/assessment-attempts/${attemptId}`],
    enabled: !!attemptId,
  });

  // Submit assessment mutation
  const submitAssessmentMutation = useMutation({
    mutationFn: (data: { answers: Record<number, any>; timeSpent: number }) =>
      apiRequest(`backend/api/assessments/${assessmentId}/submit`, 'POST', data),
    onSuccess: (result) => {
      setIsSubmitted(true);
      toast({
        title: "Assessment Submitted!",
        description: `You scored ${result.score}%. ${result.passed ? 'Congratulations!' : 'Please try again.'}`,
        variant: result.passed ? "default" : "destructive",
      });
      queryClient.invalidateQueries({ queryKey: ['backend/api/assessments'] });
    },
  });

  // Timer effect
  useEffect(() => {
    if (assessment && typeof assessment === 'object' && 'settings' in assessment && !isSubmitted) {
      const settings = (assessment as Assessment).settings;
      setTimeRemaining(settings.timeLimit * 60); // Convert minutes to seconds
    }
  }, [assessment, isSubmitted]);

  useEffect(() => {
    let interval: NodeJS.Timeout;
    if (timeRemaining > 0 && !isSubmitted) {
      interval = setInterval(() => {
        setTimeRemaining(prev => {
          if (prev <= 1) {
            handleSubmitAssessment(true); // Auto-submit when time runs out
            return 0;
          }
          return prev - 1;
        });
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [timeRemaining, isSubmitted]);

  // Proctoring simulation
  useEffect(() => {
    if (assessment && typeof assessment === 'object' && 'type' in assessment && (assessment as Assessment).type === 'certification') {
      setProctoring(true);
      // Simulate proctoring alerts
      const handleVisibilityChange = () => {
        if (document.hidden && !isSubmitted) {
          toast({
            title: "Proctoring Alert",
            description: "Tab switching detected during examination.",
            variant: "destructive",
          });
        }
      };
      document.addEventListener('visibilitychange', handleVisibilityChange);
      return () => document.removeEventListener('visibilitychange', handleVisibilityChange);
    }
  }, [assessment, isSubmitted]);

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary mx-auto mb-4"></div>
          <p>Loading assessment...</p>
          <p className="text-sm text-gray-500 mt-2">Assessment ID: {assessmentId}</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <p className="text-red-600 mb-2">Error loading assessment</p>
          <p className="text-sm text-gray-500 mb-4">
            {error instanceof Error ? error.message : 'Unknown error'}
          </p>
          <p className="text-xs text-gray-400 mb-4">Assessment ID: {assessmentId}</p>
          <Button onClick={() => navigate('/assessments')} className="mt-4">
            Back to Assessments
          </Button>
        </div>
      </div>
    );
  }

  if (!assessment || typeof assessment !== 'object') {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <p>Assessment not found</p>
          <p className="text-xs text-gray-400 mb-4">
            Assessment ID: {assessmentId}, Data: {JSON.stringify(assessment)}
          </p>
          <Button onClick={() => navigate('/assessments')} className="mt-4">
            Back to Assessments
          </Button>
        </div>
      </div>
    );
  }

  // Type cast for assessment data
  const assessmentData = assessment as Assessment;

  const questions: Question[] = assessmentData.questions || [];
  const totalQuestions = questions.length;
  const progress = ((currentQuestion + 1) / totalQuestions) * 100;

  const formatTime = (seconds: number) => {
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    if (hours > 0) {
      return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    }
    return `${minutes}:${secs.toString().padStart(2, '0')}`;
  };

  const handleAnswerChange = (questionId: number, answer: any) => {
    setAnswers(prev => ({
      ...prev,
      [questionId]: answer
    }));
  };

  const handleSubmitAssessment = (autoSubmit = false) => {
    const timeSpent = (assessmentData.settings.timeLimit * 60) - timeRemaining;
    submitAssessmentMutation.mutate({
      answers,
      timeSpent
    });
  };

  const getQuestionExplanation = (question: Question, userAnswer: any) => {
    // Provide educational explanations based on question content
    switch (question.id) {
      case 1:
        return "Data analytics transforms raw data into actionable insights that drive business decisions. While collecting and storing data are important, the core purpose is analysis and insight generation.";
      case 2:
        return "Subjective Analytics is not a recognized type of business analytics. The three main types are Descriptive (what happened), Predictive (what might happen), and Prescriptive (what should happen).";
      case 3:
        return "Data visualization is useful throughout the analytics process, not just for final presentations. It helps analysts explore data, identify patterns, and communicate findings at every stage.";
      default:
        return userAnswer === question.correct 
          ? "Correct! You demonstrated good understanding of this concept." 
          : "This answer was incorrect. Consider reviewing the relevant material to strengthen your understanding.";
    }
  };

  const calculateScore = () => {
    let correct = 0;
    let total = 0;
    
    questions.forEach((q) => {
      total += q.points || 1;
      const userAnswer = answers[q.id];
      
      if (q.type === 'multiple_choice' || q.type === 'true_false') {
        if (userAnswer === q.correct) {
          correct += q.points || 1;
        }
      } else if (q.type === 'multiple_select') {
        const correctAnswers = q.correct || [];
        const userAnswers = userAnswer || [];
        if (JSON.stringify(correctAnswers.sort()) === JSON.stringify(userAnswers.sort())) {
          correct += q.points || 1;
        }
      }
    });
    
    return total > 0 ? (correct / total) * 100 : 0;
  };

  const renderQuestion = (question: Question) => {
    const userAnswer = answers[question.id];

    switch (question.type) {
      case 'multiple_choice':
        return (
          <RadioGroup
            value={userAnswer || ''}
            onValueChange={(value) => handleAnswerChange(question.id, value)}
          >
            {question.options?.map((option, index) => (
              <div key={index} className="flex items-center space-x-2">
                <RadioGroupItem value={option} id={`q${question.id}-${index}`} />
                <Label htmlFor={`q${question.id}-${index}`} className="cursor-pointer">
                  {option}
                </Label>
              </div>
            ))}
          </RadioGroup>
        );

      case 'true_false':
        return (
          <RadioGroup
            value={userAnswer?.toString() || ''}
            onValueChange={(value) => handleAnswerChange(question.id, value === 'true')}
          >
            <div className="flex items-center space-x-2">
              <RadioGroupItem value="true" id={`q${question.id}-true`} />
              <Label htmlFor={`q${question.id}-true`} className="cursor-pointer">True</Label>
            </div>
            <div className="flex items-center space-x-2">
              <RadioGroupItem value="false" id={`q${question.id}-false`} />
              <Label htmlFor={`q${question.id}-false`} className="cursor-pointer">False</Label>
            </div>
          </RadioGroup>
        );

      case 'multiple_select':
        return (
          <div className="space-y-2">
            {question.options?.map((option, index) => (
              <div key={index} className="flex items-center space-x-2">
                <Checkbox
                  id={`q${question.id}-${index}`}
                  checked={(userAnswer || []).includes(option)}
                  onCheckedChange={(checked) => {
                    const currentAnswers = userAnswer || [];
                    if (checked) {
                      handleAnswerChange(question.id, [...currentAnswers, option]);
                    } else {
                      handleAnswerChange(question.id, currentAnswers.filter((a: string) => a !== option));
                    }
                  }}
                />
                <Label htmlFor={`q${question.id}-${index}`} className="cursor-pointer">
                  {option}
                </Label>
              </div>
            ))}
          </div>
        );

      case 'likert_scale':
        return (
          <div className="space-y-4">
            <div className="grid grid-cols-5 gap-2 text-center text-sm">
              {question.scale?.labels.map((label, index) => (
                <div key={index} className="text-muted-foreground">{label}</div>
              ))}
            </div>
            <RadioGroup
              value={userAnswer?.toString() || ''}
              onValueChange={(value) => handleAnswerChange(question.id, parseInt(value))}
              className="grid grid-cols-5 gap-2"
            >
              {Array.from({ length: question.scale?.max || 5 }, (_, index) => {
                const value = (question.scale?.min || 1) + index;
                return (
                  <div key={value} className="flex flex-col items-center space-y-1">
                    <RadioGroupItem value={value.toString()} id={`q${question.id}-${value}`} />
                    <Label htmlFor={`q${question.id}-${value}`} className="text-xs">{value}</Label>
                  </div>
                );
              })}
            </RadioGroup>
          </div>
        );

      case 'scenario_based':
        return (
          <div className="space-y-4">
            {question.scenario && (
              <div className="p-4 bg-blue-50 border-l-4 border-blue-500 rounded-r-lg">
                <h4 className="font-semibold text-blue-900 mb-2">Scenario</h4>
                <p className="text-blue-800 text-sm leading-relaxed">{question.scenario}</p>
              </div>
            )}
            <RadioGroup
              value={userAnswer || ''}
              onValueChange={(value) => handleAnswerChange(question.id, value)}
            >
              {question.options?.map((option, index) => (
                <div key={index} className="flex items-start space-x-2 p-3 border rounded-lg hover:bg-gray-50">
                  <RadioGroupItem value={option} id={`q${question.id}-${index}`} className="mt-1" />
                  <Label htmlFor={`q${question.id}-${index}`} className="cursor-pointer text-sm leading-relaxed">
                    {option}
                  </Label>
                </div>
              ))}
            </RadioGroup>
          </div>
        );

      case 'cognitive_reasoning':
        return (
          <div className="space-y-4">
            <div className="p-4 bg-purple-50 border-l-4 border-purple-500 rounded-r-lg">
              <div className="flex items-center gap-2 mb-2">
                <Award className="h-4 w-4 text-purple-600" />
                <span className="font-semibold text-purple-900">Critical Thinking Assessment</span>
              </div>
              <p className="text-purple-800 text-sm">
                Analyze the information carefully and select the most logical conclusion.
              </p>
            </div>
            <RadioGroup
              value={userAnswer || ''}
              onValueChange={(value) => handleAnswerChange(question.id, value)}
            >
              {question.options?.map((option, index) => (
                <div key={index} className="flex items-start space-x-2 p-3 border rounded-lg hover:bg-gray-50">
                  <RadioGroupItem value={option} id={`q${question.id}-${index}`} className="mt-1" />
                  <Label htmlFor={`q${question.id}-${index}`} className="cursor-pointer text-sm leading-relaxed">
                    <span className="font-medium">Option {String.fromCharCode(65 + index)}:</span> {option}
                  </Label>
                </div>
              ))}
            </RadioGroup>
          </div>
        );

      case 'short_answer':
        return (
          <Textarea
            placeholder="Enter your answer..."
            value={userAnswer || ''}
            onChange={(e) => handleAnswerChange(question.id, e.target.value)}
            rows={3}
          />
        );

      case 'essay':
        return (
          <Textarea
            placeholder="Write your essay response..."
            value={userAnswer || ''}
            onChange={(e) => handleAnswerChange(question.id, e.target.value)}
            rows={8}
          />
        );

      default:
        return <div>Unsupported question type</div>;
    }
  };

  if (isSubmitted) {
    // Check if this is a psychometric assessment based on submissionResults
    const submissionResults = submitAssessmentMutation.data;
    const isPsychometric = submissionResults?.type === 'psychometric';

    if (isPsychometric) {
      return (
        <div className="min-h-screen bg-gray-50 dark:bg-gray-900 p-6">
          <div className="max-w-6xl mx-auto space-y-6">
            {/* Header */}
            <Card className="text-center">
              <CardHeader>
                <div className="mx-auto mb-4">
                  <CheckCircle className="h-16 w-16 text-blue-500" />
                </div>
                <CardTitle className="text-2xl">
                  {submissionResults.assessmentTitle} Complete
                </CardTitle>
                <p className="text-muted-foreground mt-2">
                  Your assessment has been completed and analyzed by our expert assessors.
                </p>
              </CardHeader>
            </Card>

            {/* Performance Summary Dashboard */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
              {submissionResults.radarData?.slice(0, 4).map((item, index) => {
                const icons = [Trophy, Target, TrendingUp, CheckCircle];
                const Icon = icons[index % icons.length];
                const colors = ['bg-blue-500', 'bg-purple-500', 'bg-green-500', 'bg-red-500'];
                const bgColor = colors[index % colors.length];
                
                return (
                  <Card key={item.dimension} className="relative overflow-hidden">
                    <CardContent className="p-4">
                      <div className="flex items-center justify-between">
                        <div>
                          <p className="text-sm font-medium text-muted-foreground">{item.dimension}</p>
                          <p className="text-2xl font-bold">{item.score}</p>
                          <p className="text-xs text-muted-foreground">out of 100</p>
                        </div>
                        <div className={`${bgColor} p-2 rounded-full`}>
                          <Icon className="h-6 w-6 text-white" />
                        </div>
                      </div>
                      <div className="mt-2">
                        <div className="w-full bg-gray-200 rounded-full h-1.5">
                          <div 
                            className={`${bgColor} h-1.5 rounded-full transition-all duration-1000`}
                            style={{ width: `${item.score}%` }}
                          ></div>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                );
              })}
            </div>

            {/* Main Radar Chart - Full Width */}
            <Card className="mb-6">
              <CardHeader>
                <CardTitle>Radar Chart Analysis</CardTitle>
                <p className="text-sm text-muted-foreground">
                  Your dimensional profile visualization
                </p>
              </CardHeader>
              <CardContent>
                <div className="flex justify-center">
                  <div className="relative w-96 h-96">
                    <svg viewBox="0 0 400 400" className="w-full h-full">
                      <defs>
                        {/* Gradient for radar area */}
                        <linearGradient id="radarGradient" x1="0%" y1="0%" x2="100%" y2="100%">
                          <stop offset="0%" stopColor="#3b82f6" stopOpacity="0.3"/>
                          <stop offset="100%" stopColor="#8b5cf6" stopOpacity="0.1"/>
                        </linearGradient>
                        {/* Drop shadow filter */}
                        <filter id="dropshadow" x="-20%" y="-20%" width="140%" height="140%">
                          <feDropShadow dx="2" dy="2" stdDeviation="3" floodColor="#000" floodOpacity="0.1"/>
                        </filter>
                      </defs>
                      
                      {/* Background grid circles */}
                      {[20, 40, 60, 80, 100].map((radius) => (
                        <circle
                          key={radius}
                          cx="200"
                          cy="200"
                          r={radius * 1.5}
                          fill="none"
                          stroke="#e5e7eb"
                          strokeWidth="1"
                          opacity="0.6"
                        />
                      ))}
                      
                      {/* Grid lines from center */}
                      {submissionResults.radarData?.map((_, index) => {
                        const angle = (index * 360) / submissionResults.radarData.length;
                        const radian = (angle - 90) * (Math.PI / 180);
                        const x = 200 + Math.cos(radian) * 150;
                        const y = 200 + Math.sin(radian) * 150;
                        
                        return (
                          <line
                            key={index}
                            x1="200"
                            y1="200"
                            x2={x}
                            y2={y}
                            stroke="#e5e7eb"
                            strokeWidth="1"
                            opacity="0.6"
                          />
                        );
                      })}
                      
                      {/* Radar polygon */}
                      {submissionResults.radarData && (
                        <polygon
                          points={submissionResults.radarData.map((item, index) => {
                            const angle = (index * 360) / submissionResults.radarData.length;
                            const radian = (angle - 90) * (Math.PI / 180);
                            const radius = (item.score / 100) * 150;
                            const x = 200 + Math.cos(radian) * radius;
                            const y = 200 + Math.sin(radian) * radius;
                            return `${x},${y}`;
                          }).join(' ')}
                          fill="url(#radarGradient)"
                          stroke="#3b82f6"
                          strokeWidth="2"
                          filter="url(#dropshadow)"
                        />
                      )}
                      
                      {/* Data points */}
                      {submissionResults.radarData?.map((item, index) => {
                        const angle = (index * 360) / submissionResults.radarData.length;
                        const radian = (angle - 90) * (Math.PI / 180);
                        const radius = (item.score / 100) * 150;
                        const x = 200 + Math.cos(radian) * radius;
                        const y = 200 + Math.sin(radian) * radius;
                        
                        return (
                          <g key={index}>
                            {/* Data point */}
                            <circle
                              cx={x}
                              cy={y}
                              r="5"
                              fill="#1d4ed8"
                              stroke="#ffffff"
                              strokeWidth="2"
                              filter="url(#dropshadow)"
                            />
                            
                            {/* Score label */}
                            <text
                              x={x + (Math.cos(radian) * 20)}
                              y={y + (Math.sin(radian) * 20)}
                              textAnchor="middle"
                              dominantBaseline="middle"
                              className="text-xs font-bold fill-blue-600"
                            >
                              {item.score}
                            </text>
                          </g>
                        );
                      })}
                      
                      {/* Dimension labels */}
                      {submissionResults.radarData?.map((item, index) => {
                        const angle = (index * 360) / submissionResults.radarData.length;
                        const radian = (angle - 90) * (Math.PI / 180);
                        const labelRadius = 170;
                        const x = 200 + Math.cos(radian) * labelRadius;
                        const y = 200 + Math.sin(radian) * labelRadius;
                        
                        return (
                          <text
                            key={index}
                            x={x}
                            y={y}
                            textAnchor="middle"
                            dominantBaseline="middle"
                            className="text-sm font-semibold fill-gray-700"
                          >
                            {item.dimension}
                          </text>
                        );
                      })}
                      
                      {/* Scale labels */}
                      {[20, 40, 60, 80, 100].map((value) => (
                        <text
                          key={value}
                          x="205"
                          y={200 - (value * 1.5)}
                          className="text-xs fill-gray-500"
                          dominantBaseline="middle"
                        >
                          {value}
                        </text>
                      ))}
                    </svg>
                  </div>
                </div>
                
                {/* Legend */}
                <div className="mt-6 grid grid-cols-3 gap-4 text-sm">
                  {submissionResults.radarData?.map((item) => (
                    <div key={item.dimension} className="flex items-center justify-between">
                      <span className="text-muted-foreground">{item.dimension}:</span>
                      <span className="font-semibold text-blue-600">{item.score}/100</span>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Secondary Charts Section */}
            <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
              {/* Circular Progress Chart */}
              <Card>
                <CardHeader>
                  <CardTitle>Dimensional Breakdown</CardTitle>
                  <p className="text-sm text-muted-foreground">
                    Individual dimension scores
                  </p>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-2 gap-4">
                    {submissionResults.radarData?.map((item, index) => {
                      const colors = [
                        'from-blue-500 to-blue-600',
                        'from-purple-500 to-purple-600', 
                        'from-green-500 to-green-600',
                        'from-red-500 to-red-600',
                        'from-yellow-500 to-yellow-600',
                        'from-indigo-500 to-indigo-600'
                      ];
                      const color = colors[index % colors.length];
                      
                      return (
                        <div key={item.dimension} className="text-center space-y-2">
                          <div className="relative w-20 h-20 mx-auto">
                            <svg className="w-20 h-20 transform -rotate-90" viewBox="0 0 80 80">
                              {/* Background circle */}
                              <circle
                                cx="40"
                                cy="40"
                                r="30"
                                stroke="currentColor"
                                strokeWidth="8"
                                fill="transparent"
                                className="text-gray-200"
                              />
                              {/* Progress circle */}
                              <circle
                                cx="40"
                                cy="40"
                                r="30"
                                stroke="url(#gradient-${index})"
                                strokeWidth="8"
                                fill="transparent"
                                strokeDasharray={`${(item.score / 100) * 188.4} 188.4`}
                                strokeLinecap="round"
                                className="transition-all duration-1000 ease-out"
                              />
                              {/* Gradient definition */}
                              <defs>
                                <linearGradient id={`gradient-${index}`}>
                                  <stop offset="0%" stopColor="#3b82f6" />
                                  <stop offset="100%" stopColor="#1d4ed8" />
                                </linearGradient>
                              </defs>
                            </svg>
                            <div className="absolute inset-0 flex items-center justify-center">
                              <span className="text-lg font-bold">{item.score}</span>
                            </div>
                          </div>
                          <div>
                            <h4 className="font-medium text-sm">{item.dimension}</h4>
                            <p className="text-xs text-muted-foreground">{item.score}/100</p>
                          </div>
                        </div>
                      );
                    })}
                  </div>
                </CardContent>
              </Card>

              {/* Comparative Analysis */}
              <Card>
                <CardHeader>
                  <CardTitle>Performance Analysis</CardTitle>
                  <p className="text-sm text-muted-foreground">
                    Benchmarked against industry standards
                  </p>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {submissionResults.radarData?.map((item) => {
                      const benchmarkScore = 70; // Industry benchmark
                      const isAboveBenchmark = item.score >= benchmarkScore;
                      
                      return (
                        <div key={item.dimension} className="space-y-2">
                          <div className="flex justify-between items-center">
                            <span className="font-medium text-sm">{item.dimension}</span>
                            <div className="flex items-center gap-2">
                              <span className={`text-sm font-bold ${isAboveBenchmark ? 'text-green-600' : 'text-orange-600'}`}>
                                {item.score}/100
                              </span>
                              {isAboveBenchmark ? (
                                <Badge variant="default" className="bg-green-100 text-green-800">Above Average</Badge>
                              ) : (
                                <Badge variant="secondary" className="bg-orange-100 text-orange-800">Below Average</Badge>
                              )}
                            </div>
                          </div>
                          
                          {/* Stacked progress bar showing your score vs benchmark */}
                          <div className="relative w-full bg-gray-200 rounded-full h-3">
                            {/* Benchmark line */}
                            <div 
                              className="absolute top-0 w-0.5 h-3 bg-gray-400 z-10"
                              style={{ left: `${benchmarkScore}%` }}
                              title={`Industry Average: ${benchmarkScore}`}
                            ></div>
                            {/* Your score */}
                            <div 
                              className={`h-3 rounded-full transition-all duration-700 ${
                                isAboveBenchmark 
                                  ? 'bg-gradient-to-r from-green-400 to-green-600' 
                                  : 'bg-gradient-to-r from-orange-400 to-orange-600'
                              }`}
                              style={{ width: `${item.score}%` }}
                            ></div>
                          </div>
                          
                          <div className="flex justify-between text-xs text-muted-foreground">
                            <span>0</span>
                            <span className="text-gray-600">Industry Avg: {benchmarkScore}</span>
                            <span>100</span>
                          </div>
                        </div>
                      );
                    })}
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Assessor Notes */}
            <Card>
              <CardHeader>
                <CardTitle>Professional Assessment Report</CardTitle>
                <p className="text-sm text-muted-foreground">
                  Detailed analysis by certified assessors
                </p>
              </CardHeader>
              <CardContent className="space-y-6">
                <div>
                  <h3 className="font-semibold mb-2">Executive Summary</h3>
                  <p className="text-sm text-muted-foreground leading-relaxed">
                    {submissionResults.assessorNotes?.summary}
                  </p>
                </div>

                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <h3 className="font-semibold mb-3 text-green-700">Key Strengths</h3>
                    <ul className="space-y-2">
                      {submissionResults.assessorNotes?.strengths?.map((strength, index) => (
                        <li key={index} className="flex items-start gap-2 text-sm">
                          <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
                          <span>{strength}</span>
                        </li>
                      ))}
                    </ul>
                  </div>

                  <div>
                    <h3 className="font-semibold mb-3 text-blue-700">Development Areas</h3>
                    <ul className="space-y-2">
                      {submissionResults.assessorNotes?.developmentAreas?.map((area, index) => (
                        <li key={index} className="flex items-start gap-2 text-sm">
                          <ArrowRight className="h-4 w-4 text-blue-500 mt-0.5 flex-shrink-0" />
                          <span>{area}</span>
                        </li>
                      ))}
                    </ul>
                  </div>
                </div>

                <div className="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                  <h4 className="font-semibold mb-2">Overall Assessment</h4>
                  <p className="text-sm text-blue-800 dark:text-blue-200">
                    {submissionResults.assessorNotes?.overallAssessment}
                  </p>
                </div>

                <div>
                  <h3 className="font-semibold mb-3">Professional Recommendations</h3>
                  <ul className="space-y-2">
                    {submissionResults.recommendations?.map((rec, index) => (
                      <li key={index} className="flex items-start gap-2 text-sm">
                        <Award className="h-4 w-4 text-purple-500 mt-0.5 flex-shrink-0" />
                        <span>{rec}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              </CardContent>
            </Card>

            {/* Actions */}
            <div className="text-center space-x-4">
              <Button onClick={() => navigate('/assessments')}>
                Back to Assessments
              </Button>
              <Button variant="outline" onClick={() => window.print()}>
                <FileText className="h-4 w-4 mr-2" />
                Download Report
              </Button>
            </div>
          </div>
        </div>
      );
    } else {
      // Traditional certification assessment results
      const score = calculateScore();
      const passed = score >= (assessmentData.passingScore || 70);

      return (
        <div className="min-h-screen bg-gray-50 dark:bg-gray-900 p-6">
          <div className="max-w-4xl mx-auto">
            <Card className="text-center">
              <CardHeader>
                <div className="mx-auto mb-4">
                  {passed ? (
                    <CheckCircle className="h-16 w-16 text-green-500" />
                  ) : (
                    <AlertTriangle className="h-16 w-16 text-red-500" />
                  )}
                </div>
                <CardTitle className="text-2xl">
                  Assessment {passed ? 'Completed Successfully!' : 'Completed'}
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="text-center">
                    <div className="text-3xl font-bold text-primary">{Math.round(score)}%</div>
                    <div className="text-sm text-muted-foreground">Your Score</div>
                  </div>
                  <div className="text-center">
                    <div className="text-3xl font-bold">{assessmentData.passingScore}%</div>
                    <div className="text-sm text-muted-foreground">Passing Score</div>
                  </div>
                  <div className="text-center">
                    <div className="text-3xl font-bold">{totalQuestions}</div>
                    <div className="text-sm text-muted-foreground">Total Questions</div>
                  </div>
                </div>

                <div className="space-y-4">
                  {passed && (
                    <Badge variant="secondary" className="text-green-700 bg-green-100">
                      <Award className="h-4 w-4 mr-2" />
                      Certificate Eligible
                    </Badge>
                  )}
                  
                  {assessmentData.settings.showResults && (
                    <Button
                      variant="outline"
                      onClick={() => setShowReview(true)}
                      className="mr-4"
                    >
                      <Eye className="h-4 w-4 mr-2" />
                      Review Answers
                    </Button>
                  )}
                  
                  <Button onClick={() => navigate('/assessments')}>
                    Back to Assessments
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      );
    }
  }

  // Show review screen
  if (showReview) {
    return (
      <div className="min-h-screen bg-gray-50 dark:bg-gray-900 p-6">
        <div className="max-w-4xl mx-auto">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center justify-between">
                <span>Assessment Review</span>
                <Button variant="outline" onClick={() => setShowReview(false)}>
                  <ArrowLeft className="h-4 w-4 mr-2" />
                  Back to Results
                </Button>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              {questions.map((question, index) => {
                const userAnswer = answers[question.id];
                const isCorrect = userAnswer === question.correct;
                const isPsychometric = assessmentData.type === 'psychometric' || assessmentData.type === 'personality';
                
                return (
                  <div key={question.id} className="border rounded-lg p-4 space-y-3">
                    <div className="flex items-start justify-between">
                      <h3 className="font-medium">Question {index + 1}</h3>
                      {!isPsychometric ? (
                        <Badge variant={isCorrect ? "default" : "destructive"}>
                          {isCorrect ? "Correct" : "Incorrect"} ({question.points || 1} pts)
                        </Badge>
                      ) : (
                        <Badge variant="secondary">
                          {question.category || 'Response'} ({question.points || 1} pts)
                        </Badge>
                      )}
                    </div>
                    
                    <p className="text-gray-700 dark:text-gray-300">{question.question}</p>
                    
                    {question.type === 'likert_scale' && question.scale ? (
                      <div className="space-y-2">
                        <p className="text-sm text-muted-foreground">Your response: <strong>{question.scale.labels[userAnswer - 1] || 'Not answered'}</strong></p>
                        <div className="flex items-center gap-2 text-sm">
                          <span>{question.scale.labels[0]}</span>
                          <div className="flex-1 bg-gray-200 rounded-full h-2 relative">
                            <div 
                              className="bg-primary h-2 rounded-full" 
                              style={{ width: `${((userAnswer || 1) / question.scale.max) * 100}%` }}
                            ></div>
                          </div>
                          <span>{question.scale.labels[question.scale.labels.length - 1]}</span>
                        </div>
                      </div>
                    ) : question.options && !isPsychometric ? (
                      <div className="space-y-2">
                        {question.options.map((option, optionIndex) => {
                          const isUserAnswer = userAnswer === optionIndex;
                          const isCorrectAnswer = question.correct === optionIndex;
                          
                          return (
                            <div 
                              key={optionIndex}
                              className={`p-2 rounded border ${
                                isCorrectAnswer 
                                  ? 'bg-green-50 border-green-200 text-green-800' 
                                  : isUserAnswer 
                                  ? 'bg-red-50 border-red-200 text-red-800'
                                  : 'bg-gray-50 border-gray-200'
                              }`}
                            >
                              <div className="flex items-center gap-2">
                                {isCorrectAnswer && <CheckCircle className="h-4 w-4 text-green-600" />}
                                {isUserAnswer && !isCorrectAnswer && <AlertTriangle className="h-4 w-4 text-red-600" />}
                                <span>{option}</span>
                                {isUserAnswer && <span className="ml-auto text-sm">(Your answer)</span>}
                                {isCorrectAnswer && <span className="ml-auto text-sm">(Correct answer)</span>}
                              </div>
                            </div>
                          );
                        })}
                      </div>
                    ) : question.options && isPsychometric ? (
                      <div className="space-y-2">
                        {question.options.map((option, optionIndex) => {
                          const isUserAnswer = userAnswer === optionIndex;
                          
                          return (
                            <div 
                              key={optionIndex}
                              className={`p-2 rounded border ${
                                isUserAnswer 
                                  ? 'bg-blue-50 border-blue-200 text-blue-800'
                                  : 'bg-gray-50 border-gray-200'
                              }`}
                            >
                              <div className="flex items-center gap-2">
                                {isUserAnswer && <CheckCircle className="h-4 w-4 text-blue-600" />}
                                <span>{option}</span>
                                {isUserAnswer && <span className="ml-auto text-sm">(Your response)</span>}
                              </div>
                            </div>
                          );
                        })}
                      </div>
                    ) : null}
                    
                    {/* Show explanation for educational value */}
                    <div className="mt-3 p-3 bg-blue-50 dark:bg-blue-900/20 rounded">
                      <p className="text-sm text-blue-800 dark:text-blue-200">
                        <strong>Explanation:</strong> {getQuestionExplanation(question, userAnswer)}
                      </p>
                    </div>
                  </div>
                );
              })}
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  const currentQ = questions[currentQuestion];

  return (
    <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
      {/* Header */}
      <div className="bg-white dark:bg-gray-800 border-b shadow-sm">
        <div className="max-w-7xl mx-auto p-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-xl font-bold">{assessmentData.title}</h1>
              <p className="text-sm text-muted-foreground">
                Question {currentQuestion + 1} of {totalQuestions}
              </p>
            </div>
            <div className="flex items-center gap-4">
              {proctoring && (
                <Badge variant="destructive">
                  <Eye className="h-3 w-3 mr-1" />
                  Proctored
                </Badge>
              )}
              <div className="flex items-center gap-2 text-sm">
                <Timer className="h-4 w-4" />
                <span className={timeRemaining < 300 ? 'text-red-600 font-bold' : ''}>
                  {formatTime(timeRemaining)}
                </span>
              </div>
              <Badge variant="outline">
                {assessmentData.type}
              </Badge>
            </div>
          </div>
          <div className="mt-4">
            <Progress value={progress} className="h-2" />
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-4xl mx-auto p-6">
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              Question {currentQuestion + 1}
              <Badge variant="secondary" className="ml-auto">
                {currentQ?.points || 1} point{(currentQ?.points || 1) !== 1 ? 's' : ''}
              </Badge>
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="text-lg">{currentQ?.question}</div>
            
            {currentQ && renderQuestion(currentQ)}

            <div className="flex justify-between pt-6 border-t">
              <Button
                variant="outline"
                onClick={() => setCurrentQuestion(Math.max(0, currentQuestion - 1))}
                disabled={currentQuestion === 0}
              >
                <ArrowLeft className="h-4 w-4 mr-2" />
                Previous
              </Button>

              <div className="flex gap-2">
                {assessmentData.settings.allowReview && (
                  <Button
                    variant="outline"
                    onClick={() => setShowReview(true)}
                  >
                    Review
                  </Button>
                )}

                {currentQuestion === totalQuestions - 1 ? (
                  <AlertDialog>
                    <AlertDialogTrigger asChild>
                      <Button className="bg-red-600 hover:bg-red-700">
                        Submit Assessment
                      </Button>
                    </AlertDialogTrigger>
                    <AlertDialogContent>
                      <AlertDialogHeader>
                        <AlertDialogTitle>Submit Assessment?</AlertDialogTitle>
                        <AlertDialogDescription>
                          Are you sure you want to submit your assessment? You cannot change your answers after submission.
                        </AlertDialogDescription>
                      </AlertDialogHeader>
                      <AlertDialogFooter>
                        <AlertDialogCancel>Cancel</AlertDialogCancel>
                        <AlertDialogAction
                          onClick={() => handleSubmitAssessment()}
                          disabled={submitAssessmentMutation.isPending}
                        >
                          {submitAssessmentMutation.isPending ? 'Submitting...' : 'Submit'}
                        </AlertDialogAction>
                      </AlertDialogFooter>
                    </AlertDialogContent>
                  </AlertDialog>
                ) : (
                  <Button
                    onClick={() => setCurrentQuestion(Math.min(totalQuestions - 1, currentQuestion + 1))}
                  >
                    Next
                    <ArrowRight className="h-4 w-4 ml-2" />
                  </Button>
                )}
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Question Navigator */}
        <Card className="mt-6">
          <CardHeader>
            <CardTitle className="text-lg">Question Navigator</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-10 gap-2">
              {questions.map((_, index) => (
                <Button
                  key={index}
                  variant={index === currentQuestion ? "default" : answers[questions[index].id] ? "secondary" : "outline"}
                  size="sm"
                  onClick={() => setCurrentQuestion(index)}
                  className="aspect-square"
                >
                  {index + 1}
                </Button>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}