import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { 
  FileCheck, 
  Clock, 
  Users, 
  Star, 
  Play, 
  Search,
  Brain,
  Award,
  AlertCircle,
  Calendar,
  Timer,
  CheckCircle2
} from "lucide-react";
import { Link } from "wouter";

export default function AssessmentList() {
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedType, setSelectedType] = useState<string>("all");
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: assessments, isLoading } = useQuery({
    queryKey: ["backend/api/assessments", searchQuery, selectedType],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (searchQuery) params.append("q", searchQuery);
      if (selectedType && selectedType !== "all") params.append("type", selectedType);
      const url = params.toString() ? `backend/api/assessments?${params.toString()}` : "backend/api/assessments";
      return await apiRequest(url);
    },
  });

  const { data: userAttempts } = useQuery({
    queryKey: ["backend/api/assessment-attempts"],
  });

  const startAssessmentMutation = useMutation({
    mutationFn: async (assessmentId: string) => {
      return await apiRequest("backend/api/assessment-attempts", "POST", { assessmentId });
    },
    onSuccess: (data, assessmentId) => {
      toast({
        title: "Assessment Started",
        description: "You can now begin the assessment.",
      });
      // Redirect to assessment attempt page
      window.location.href = `/assessments/${assessmentId}/attempt/${data.id}`;
    },
    onError: (error) => {
      toast({
        title: "Failed to Start",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const assessmentTypes = [
    { value: "", label: "All Assessment Types" },
    { value: "certification", label: "Certification Exams" },
    { value: "psychometric", label: "Psychometric Assessments" },
    { value: "skills", label: "Skills Evaluations" },
    { value: "personality", label: "Personality Tests" }
  ];

  const getTypeColor = (type: string) => {
    switch (type) {
      case "certification":
        return "bg-blue-100 text-blue-800 border-blue-200";
      case "psychometric":
        return "bg-purple-100 text-purple-800 border-purple-200";
      case "skills":
        return "bg-green-100 text-green-800 border-green-200";
      case "personality":
        return "bg-orange-100 text-orange-800 border-orange-200";
      default:
        return "bg-gray-100 text-gray-800 border-gray-200";
    }
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case "certification":
        return <Award className="h-4 w-4" />;
      case "psychometric":
        return <Brain className="h-4 w-4" />;
      case "skills":
        return <FileCheck className="h-4 w-4" />;
      case "personality":
        return <Users className="h-4 w-4" />;
      default:
        return <FileCheck className="h-4 w-4" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "completed":
        return "bg-green-100 text-green-800";
      case "in_progress":
        return "bg-yellow-100 text-yellow-800";
      case "failed":
        return "bg-red-100 text-red-800";
      case "pending":
        return "bg-gray-100 text-gray-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const formatDuration = (minutes: number) => {
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    return hours > 0 ? `${hours}h ${mins}m` : `${mins}m`;
  };

  const getUserAttempt = (assessmentId: string) => {
    return Array.isArray(userAttempts) ? userAttempts.find((attempt: any) => attempt.assessmentId === assessmentId) : undefined;
  };

  if (isLoading) {
    return (
      <div className="container mx-auto px-4 py-8">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {Array.from({ length: 6 }).map((_, i) => (
            <Card key={i} className="animate-pulse">
              <CardHeader>
                <div className="h-6 bg-gray-200 rounded w-3/4 mb-2"></div>
                <div className="h-4 bg-gray-200 rounded w-full"></div>
              </CardHeader>
              <CardContent>
                <div className="h-32 bg-gray-200 rounded mb-4"></div>
                <div className="h-4 bg-gray-200 rounded w-1/2"></div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      {/* Header */}
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-foreground mb-2">Assessments & Examinations</h1>
        <p className="text-muted-foreground">
          Take assessments to validate your skills and earn certifications
        </p>
      </div>

      {/* Search and Filters */}
      <div className="mb-8 space-y-4">
        <div className="flex flex-col md:flex-row gap-4">
          <div className="relative flex-1">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
            <Input
              placeholder="Search assessments..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10"
            />
          </div>
          <Select value={selectedType} onValueChange={setSelectedType}>
            <SelectTrigger className="w-full md:w-48">
              <SelectValue placeholder="All Types" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Types</SelectItem>
              {assessmentTypes.filter(type => type.value !== "").map((type) => (
                <SelectItem key={type.value} value={type.value}>
                  {type.label}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Assessment Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {assessments?.map((assessment: any) => {
          const userAttempt = getUserAttempt(assessment.id);
          const canRetake = !userAttempt || userAttempt.status === 'failed' || assessment.allowRetakes;
          
          return (
            <Card key={assessment.id} className="hover:shadow-lg transition-shadow">
              <CardHeader className="pb-3">
                <div className="flex justify-between items-start gap-2 mb-2">
                  <Badge className={getTypeColor(assessment.type)}>
                    {assessment.type === 'psychometric' ? (
                      <Brain className="w-3 h-3 mr-1" />
                    ) : assessment.type === 'certification' ? (
                      <Award className="w-3 h-3 mr-1" />
                    ) : (
                      <FileCheck className="w-3 h-3 mr-1" />
                    )}
                    {assessmentTypes.find(t => t.value === assessment.type)?.label}
                  </Badge>
                  {userAttempt && (
                    <Badge className={getStatusColor(userAttempt.status)}>
                      {userAttempt.status === 'completed' && <CheckCircle2 className="w-3 h-3 mr-1" />}
                      {userAttempt.status}
                    </Badge>
                  )}
                </div>
                
                <CardTitle className="text-lg leading-tight">{assessment.title}</CardTitle>
                <p className="text-sm text-muted-foreground line-clamp-2">
                  {assessment.description}
                </p>
              </CardHeader>
              
              <CardContent className="pt-0">
                <div className="space-y-4">
                  {/* Assessment Info */}
                  <div className="grid grid-cols-2 gap-4 text-sm text-muted-foreground">
                    <div className="flex items-center gap-1">
                      <Timer className="w-4 h-4" />
                      {formatDuration(assessment.duration)}
                    </div>
                    <div className="flex items-center gap-1">
                      <FileCheck className="w-4 h-4" />
                      {assessment.questionCount} questions
                    </div>
                    <div className="flex items-center gap-1">
                      <Users className="w-4 h-4" />
                      {assessment.attemptsCount || 0} attempts
                    </div>
                    <div className="flex items-center gap-1">
                      <Star className="w-4 h-4" />
                      {assessment.passingScore ? `${assessment.passingScore}% to pass` : 'No pass/fail'}
                    </div>
                  </div>

                  {/* User's Previous Attempt Result */}
                  {userAttempt && (
                    <div className="p-3 bg-muted rounded-lg">
                      <div className="flex items-center justify-between">
                        <span className="text-sm font-medium">Your Result</span>
                        <span className={`text-sm font-bold ${
                          userAttempt.score >= assessment.passingScore 
                            ? 'text-green-600' 
                            : 'text-red-600'
                        }`}>
                          {userAttempt.score}%
                        </span>
                      </div>
                      <div className="text-xs text-muted-foreground mt-1">
                        Completed {new Date(userAttempt.completedAt).toLocaleDateString()}
                      </div>
                    </div>
                  )}

                  {/* Security Notice for Proctored Exams */}
                  {assessment.isProctored && (
                    <div className="flex items-start gap-2 p-3 bg-orange-50 border border-orange-200 rounded-lg">
                      <AlertCircle className="w-4 h-4 text-orange-600 mt-0.5 flex-shrink-0" />
                      <div className="text-xs text-orange-800">
                        <div className="font-medium">Proctored Assessment</div>
                        <div>Camera and screen monitoring required</div>
                      </div>
                    </div>
                  )}

                  {/* Action Buttons */}
                  <div className="space-y-2">
                    {!userAttempt ? (
                      <Button
                        onClick={() => startAssessmentMutation.mutate(assessment.id)}
                        disabled={startAssessmentMutation.isPending}
                        className="w-full"
                      >
                        <Play className="w-4 h-4 mr-2" />
                        {startAssessmentMutation.isPending ? "Starting..." : "Start Assessment"}
                      </Button>
                    ) : userAttempt.status === 'in_progress' ? (
                      <Link href={`/assessments/${assessment.id}/attempt/${userAttempt.id}`}>
                        <Button className="w-full">
                          <Play className="w-4 h-4 mr-2" />
                          Continue Assessment
                        </Button>
                      </Link>
                    ) : canRetake ? (
                      <Button
                        onClick={() => startAssessmentMutation.mutate(assessment.id)}
                        disabled={startAssessmentMutation.isPending}
                        variant="outline"
                        className="w-full"
                      >
                        <Play className="w-4 h-4 mr-2" />
                        Retake Assessment
                      </Button>
                    ) : (
                      <Button disabled className="w-full">
                        <CheckCircle2 className="w-4 h-4 mr-2" />
                        Completed
                      </Button>
                    )}
                    
                    <Link href={`/assessments/${assessment.id}`}>
                      <Button variant="ghost" size="sm" className="w-full">
                        View Details
                      </Button>
                    </Link>
                  </div>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>

      {/* Empty State */}
      {assessments?.length === 0 && (
        <div className="text-center py-12">
          <FileCheck className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
          <h3 className="text-lg font-medium text-foreground mb-2">No assessments found</h3>
          <p className="text-muted-foreground">
            Try adjusting your search criteria or check back later for new assessments
          </p>
        </div>
      )}
    </div>
  );
}