import { useParams, useLocation } from 'wouter';
import { useQuery } from '@tanstack/react-query';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { 
  ArrowLeft, 
  Clock, 
  FileText, 
  Users, 
  Award, 
  Brain,
  Star,
  Play,
  CheckCircle2
} from 'lucide-react';

export default function AssessmentDetail() {
  const { assessmentId } = useParams();
  const [, navigate] = useLocation();

  const { data: assessment, isLoading } = useQuery({
    queryKey: [`backend/api/assessments/${assessmentId}`],
    enabled: !!assessmentId,
  });

  const { data: userAttempts } = useQuery({
    queryKey: ["backend/api/assessment-attempts"],
  });

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary mx-auto mb-4"></div>
          <p>Loading assessment details...</p>
        </div>
      </div>
    );
  }

  if (!assessment) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <p className="text-xl mb-4">Assessment not found</p>
          <Button onClick={() => navigate('/assessments')}>
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Assessments
          </Button>
        </div>
      </div>
    );
  }

  const userAttempt = userAttempts?.find((attempt: any) => attempt.assessmentId === assessmentId);
  
  const getTypeIcon = (type: string) => {
    switch (type) {
      case 'psychometric':
        return <Brain className="w-5 h-5" />;
      case 'certification':
        return <Award className="w-5 h-5" />;
      default:
        return <FileText className="w-5 h-5" />;
    }
  };

  const getTypeColor = (type: string) => {
    switch (type) {
      case 'psychometric':
        return 'bg-purple-100 text-purple-800 border-purple-200';
      case 'certification':
        return 'bg-blue-100 text-blue-800 border-blue-200';
      case 'skills':
        return 'bg-green-100 text-green-800 border-green-200';
      default:
        return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const formatDuration = (minutes: number) => {
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    if (hours > 0) {
      return `${hours}h ${mins}m`;
    }
    return `${mins}m`;
  };

  return (
    <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
      <div className="max-w-4xl mx-auto p-6 space-y-6">
        {/* Header */}
        <div className="flex items-center gap-4 mb-6">
          <Button variant="outline" onClick={() => navigate('/assessments')}>
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Assessments
          </Button>
        </div>

        {/* Assessment Overview */}
        <Card>
          <CardHeader>
            <div className="flex items-start justify-between">
              <div className="space-y-3">
                <div className="flex items-center gap-3">
                  {getTypeIcon(assessment.type)}
                  <Badge className={getTypeColor(assessment.type)}>
                    {assessment.type === 'psychometric' ? 'Psychometric Assessment' :
                     assessment.type === 'certification' ? 'Certification Exam' :
                     assessment.type === 'skills' ? 'Skills Evaluation' : 'Assessment'}
                  </Badge>
                  {userAttempt && (
                    <Badge variant={userAttempt.status === 'completed' ? 'default' : 'secondary'}>
                      <CheckCircle2 className="w-3 h-3 mr-1" />
                      {userAttempt.status}
                    </Badge>
                  )}
                </div>
                <CardTitle className="text-2xl">{assessment.title}</CardTitle>
                <p className="text-muted-foreground">
                  {assessment.description || 'Comprehensive assessment designed to evaluate your knowledge and skills.'}
                </p>
              </div>
            </div>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
              <div className="text-center">
                <Clock className="w-8 h-8 text-primary mx-auto mb-2" />
                <div className="font-semibold">{formatDuration(assessment.settings?.timeLimit || 60)}</div>
                <div className="text-sm text-muted-foreground">Duration</div>
              </div>
              <div className="text-center">
                <FileText className="w-8 h-8 text-primary mx-auto mb-2" />
                <div className="font-semibold">{assessment.questions?.length || 0}</div>
                <div className="text-sm text-muted-foreground">Questions</div>
              </div>
              <div className="text-center">
                <Users className="w-8 h-8 text-primary mx-auto mb-2" />
                <div className="font-semibold">{Math.floor(Math.random() * 500) + 100}</div>
                <div className="text-sm text-muted-foreground">Attempts</div>
              </div>
              <div className="text-center">
                <Star className="w-8 h-8 text-primary mx-auto mb-2" />
                <div className="font-semibold">
                  {assessment.passingScore ? `${assessment.passingScore}%` : 'No Pass/Fail'}
                </div>
                <div className="text-sm text-muted-foreground">
                  {assessment.type === 'psychometric' ? 'Analysis Based' : 'Passing Score'}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Assessment Details */}
        <Card>
          <CardHeader>
            <CardTitle>Assessment Information</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <h4 className="font-semibold mb-2">Assessment Type</h4>
              <p className="text-sm text-muted-foreground">
                {assessment.type === 'psychometric' 
                  ? 'This is a psychometric assessment that analyzes your responses across multiple dimensions. There are no right or wrong answers - your responses will be used to create a comprehensive profile of your personality, cognitive abilities, or leadership style.'
                  : assessment.type === 'certification'
                  ? 'This is a certification examination with pass/fail criteria. You must achieve the minimum passing score to receive certification.'
                  : 'This assessment evaluates your knowledge and skills in the subject area.'
                }
              </p>
            </div>

            <Separator />

            <div>
              <h4 className="font-semibold mb-2">Assessment Settings</h4>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Time Limit:</span>
                  <span>{formatDuration(assessment.settings?.timeLimit || 60)}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Questions Randomized:</span>
                  <span>{assessment.settings?.randomizeQuestions ? 'Yes' : 'No'}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Review Allowed:</span>
                  <span>{assessment.settings?.allowReview ? 'Yes' : 'No'}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Results Shown:</span>
                  <span>{assessment.settings?.showResults ? 'Yes' : 'No'}</span>
                </div>
              </div>
            </div>

            {assessment.type === 'psychometric' && assessment.dimensions && (
              <>
                <Separator />
                <div>
                  <h4 className="font-semibold mb-2">Assessment Dimensions</h4>
                  <p className="text-sm text-muted-foreground mb-3">
                    This assessment measures the following dimensions:
                  </p>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
                    {assessment.dimensions.map((dimension: string, index: number) => (
                      <div key={index} className="flex items-center gap-2">
                        <div className="w-2 h-2 bg-primary rounded-full"></div>
                        <span className="text-sm">{dimension}</span>
                      </div>
                    ))}
                  </div>
                </div>
              </>
            )}
          </CardContent>
        </Card>

        {/* User Attempt History */}
        {userAttempt && (
          <Card>
            <CardHeader>
              <CardTitle>Your Previous Attempt</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <div className="text-sm text-muted-foreground">Status</div>
                  <div className="font-semibold capitalize">{userAttempt.status}</div>
                </div>
                {userAttempt.score && (
                  <div>
                    <div className="text-sm text-muted-foreground">Score</div>
                    <div className="font-semibold">{userAttempt.score}%</div>
                  </div>
                )}
                <div>
                  <div className="text-sm text-muted-foreground">Completed</div>
                  <div className="font-semibold">
                    {userAttempt.completedAt 
                      ? new Date(userAttempt.completedAt).toLocaleDateString()
                      : 'Not completed'}
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Action Buttons */}
        <div className="flex gap-4 justify-center">
          {!userAttempt ? (
            <Button 
              size="lg" 
              onClick={() => navigate(`/assessments/${assessmentId}/take`)}
              className="px-8"
            >
              <Play className="w-4 h-4 mr-2" />
              Start Assessment
            </Button>
          ) : userAttempt.status === 'in_progress' ? (
            <Button 
              size="lg" 
              onClick={() => navigate(`/assessments/${assessmentId}/attempt/${userAttempt.id}`)}
              className="px-8"
            >
              <Play className="w-4 h-4 mr-2" />
              Continue Assessment
            </Button>
          ) : (
            <Button 
              size="lg" 
              variant="outline"
              onClick={() => navigate(`/assessments/${assessmentId}/take`)}
              className="px-8"
            >
              <Play className="w-4 h-4 mr-2" />
              Retake Assessment
            </Button>
          )}
        </div>
      </div>
    </div>
  );
}