import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { Shield, ArrowRight, Building, Users } from "lucide-react";
import { apiRequest } from "@/lib/queryClient";

export default function ZeroAdminLogin() {
  const [, setLocation] = useLocation();
  const [username, setUsername] = useState("");
  const [password, setPassword] = useState("");
  const { toast } = useToast();

  const loginMutation = useMutation({
    mutationFn: async (credentials: { username: string; password: string }) => {
      return await apiRequest("backend/api/auth/login", "POST", credentials);
    },
    onSuccess: (data) => {
      // Check if user is an admin type (not learner or website_admin)
      const adminTypes = ['system_admin', 'corporate_admin', 'partner_admin', 'qa_admin', 'assessment_admin', 'exam_admin', 'elearning_admin'];
      if (!adminTypes.includes(data.user.userType)) {
        toast({
          title: "Access denied",
          description: "This portal is for system administrators and specialized admin roles only",
          variant: "destructive",
        });
        return;
      }

      toast({
        title: "Login successful",
        description: `Welcome back, ${data.user.firstName}!`,
      });

      // Redirect based on user type
      if (data.user.userType === "system_admin") {
        window.location.href = "/dashboard/admin";
      } else if (data.user.userType === "corporate_admin") {
        window.location.href = "/dashboard/corporate";
      } else if (data.user.userType === "partner_admin") {
        window.location.href = "/dashboard/partner";
      } else if (data.user.userType === "qa_admin") {
        window.location.href = "/qa";
      } else {
        // For other admin types, redirect to main dashboard
        window.location.href = "/";
      }
    },
    onError: (error: any) => {
      toast({
        title: "Login failed",
        description: error.message || "Invalid username or password",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!username || !password) {
      toast({
        title: "Missing credentials",
        description: "Please enter both username and password",
        variant: "destructive",
      });
      return;
    }
    loginMutation.mutate({ username, password });
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-[#1A1A40] via-[#2A2A50] to-[#ED1C24] flex items-center justify-center p-4">
      <div className="w-full max-w-md">
        <Card className="shadow-2xl border-0">
          <CardHeader className="space-y-1 text-center">
            <div className="flex items-center justify-center w-16 h-16 bg-[#ED1C24] rounded-full mx-auto mb-4">
              <Shield className="h-8 w-8 text-white" />
            </div>
            <CardTitle className="text-2xl font-bold text-[#1A1A40]">
              Zero Admin Access
            </CardTitle>
            <p className="text-gray-600">
              System administration, partner management, and specialized admin portal
            </p>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="username" className="text-[#1A1A40] font-medium">
                  Username
                </Label>
                <Input
                  id="username"
                  type="text"
                  placeholder="Enter your admin username"
                  value={username}
                  onChange={(e) => setUsername(e.target.value)}
                  className="border-gray-300 focus:border-[#ED1C24] focus:ring-[#ED1C24]"
                  disabled={loginMutation.isPending}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="password" className="text-[#1A1A40] font-medium">
                  Password
                </Label>
                <Input
                  id="password"
                  type="password"
                  placeholder="Enter your admin password"
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  className="border-gray-300 focus:border-[#ED1C24] focus:ring-[#ED1C24]"
                  disabled={loginMutation.isPending}
                />
              </div>
              <Button
                type="submit"
                className="w-full bg-[#ED1C24] hover:bg-red-700 text-white"
                disabled={loginMutation.isPending}
              >
                {loginMutation.isPending ? "Signing In..." : "Access Zero Admin"}
                <ArrowRight className="ml-2 h-4 w-4" />
              </Button>
            </form>

            {/* Demo Credentials */}
            <div className="mt-6 border-t pt-6">
              <h4 className="text-sm font-medium text-gray-700 mb-3">Demo Admin Credentials (System & Specialized Admins)</h4>
              <div className="space-y-2">
                {[
                  { type: "System Admin", username: "sys_admin", password: "password123" },
                  { type: "QA Admin", username: "qa_admin", password: "password123" },
                  { type: "Corporate Admin", username: "corp_admin", password: "password123" },
                  { type: "Partner Admin", username: "partner_admin", password: "password123" },
                  { type: "Demo Account", username: "demo", password: "demo123" }
                ].map((demo, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <div className="flex items-center space-x-3">
                      {demo.type === "System Admin" ? (
                        <Shield className="h-4 w-4 text-gray-500" />
                      ) : demo.type === "QA Admin" ? (
                        <Shield className="h-4 w-4 text-blue-500" />
                      ) : demo.type === "Corporate Admin" ? (
                        <Building className="h-4 w-4 text-gray-500" />
                      ) : demo.type === "Demo Account" ? (
                        <Users className="h-4 w-4 text-green-500" />
                      ) : (
                        <Users className="h-4 w-4 text-gray-500" />
                      )}
                      <div>
                        <p className="text-sm font-medium text-gray-700">{demo.type}</p>
                        <p className="text-xs text-gray-500">{demo.username} / {demo.password}</p>
                      </div>
                    </div>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => {
                        setUsername(demo.username);
                        setPassword(demo.password);
                      }}
                      className="border-[#ED1C24] text-[#ED1C24] hover:bg-[#ED1C24] hover:text-white"
                    >
                      Use
                    </Button>
                  </div>
                ))}
              </div>
            </div>

            {/* Navigation Links */}
            <div className="mt-6 pt-4 border-t border-gray-200 space-y-3">
              <p className="text-xs text-gray-600 text-center">Need a different portal?</p>
              <div className="flex justify-center space-x-4 text-xs">
                <a href="/admin" className="text-[#4A90E2] hover:underline">Website Admin</a>
                <span className="text-gray-400">|</span>
                <a href="/login" className="text-[#ED1C24] hover:underline">Learner Login</a>
                <span className="text-gray-400">|</span>
                <a href="/" className="text-gray-600 hover:underline">Main Site</a>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}