import React, { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { toast } from "@/hooks/use-toast";
import { Plus, Edit, Trash2, Globe, Menu, FileText, Settings } from "lucide-react";
import { apiRequest } from "@/lib/queryClient";

// Form schemas
const pageSchema = z.object({
  title: z.string().min(1, "Title is required"),
  slug: z.string().min(1, "Slug is required").regex(/^[a-z0-9-]+$/, "Slug must contain only lowercase letters, numbers, and hyphens"),
  content: z.string().min(1, "Content is required"),
  metaTitle: z.string().optional(),
  metaDescription: z.string().optional(),
  isPublished: z.boolean().default(false)
});

const menuSchema = z.object({
  name: z.string().min(1, "Name is required"),
  location: z.string().min(1, "Location is required"),
  items: z.array(z.object({
    title: z.string(),
    url: z.string(),
    target: z.enum(["_self", "_blank"]).default("_self")
  })).default([])
});

type PageFormData = z.infer<typeof pageSchema>;
type MenuFormData = z.infer<typeof menuSchema>;

export default function WebsiteAdmin() {
  const [selectedPage, setSelectedPage] = useState<any>(null);
  const [selectedMenu, setSelectedMenu] = useState<any>(null);
  const [isPageDialogOpen, setIsPageDialogOpen] = useState(false);
  const [isMenuDialogOpen, setIsMenuDialogOpen] = useState(false);
  
  const queryClient = useQueryClient();

  // Fetch pages
  const { data: pages, isLoading: pagesLoading } = useQuery({
    queryKey: ["backend/api/cms/pages"],
    staleTime: 30 * 1000,
  });

  // Fetch menus
  const { data: menus, isLoading: menuLoading } = useQuery({
    queryKey: ["backend/api/cms/menu-items"],
    staleTime: 30 * 1000,
  });

  // Page form
  const pageForm = useForm<PageFormData>({
    resolver: zodResolver(pageSchema),
    defaultValues: {
      title: "",
      slug: "",
      content: "",
      metaTitle: "",
      metaDescription: "",
      isPublished: false
    }
  });

  // Menu form
  const menuForm = useForm<MenuFormData>({
    resolver: zodResolver(menuSchema),
    defaultValues: {
      name: "",
      location: "header",
      items: []
    }
  });

  // Page mutations
  const createPageMutation = useMutation({
    mutationFn: (data: PageFormData) => apiRequest("backend/api/cms/pages", { method: "POST", body: data }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["backend/api/cms/pages"] });
      toast({ title: "Success", description: "Page created successfully" });
      setIsPageDialogOpen(false);
      pageForm.reset();
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to create page", variant: "destructive" });
    }
  });

  const updatePageMutation = useMutation({
    mutationFn: ({ id, data }: { id: string; data: PageFormData }) => 
      apiRequest(`backend/api/cms/pages/${id}`, { method: "PUT", body: data }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["backend/api/cms/pages"] });
      toast({ title: "Success", description: "Page updated successfully" });
      setIsPageDialogOpen(false);
      setSelectedPage(null);
      pageForm.reset();
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to update page", variant: "destructive" });
    }
  });

  const deletePageMutation = useMutation({
    mutationFn: (id: string) => apiRequest(`backend/api/cms/pages/${id}`, { method: "DELETE" }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["backend/api/cms/pages"] });
      toast({ title: "Success", description: "Page deleted successfully" });
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to delete page", variant: "destructive" });
    }
  });

  // Menu mutations
  const createMenuMutation = useMutation({
    mutationFn: (data: MenuFormData) => apiRequest("backend/api/cms/menu-items", { method: "POST", body: data }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["backend/api/cms/menu-items"] });
      toast({ title: "Success", description: "Menu created successfully" });
      setIsMenuDialogOpen(false);
      menuForm.reset();
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to create menu", variant: "destructive" });
    }
  });

  const updateMenuMutation = useMutation({
    mutationFn: ({ id, data }: { id: string; data: MenuFormData }) => 
      apiRequest(`backend/api/cms/menu-items/${id}`, { method: "PUT", body: data }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["backend/api/cms/menu-items"] });
      toast({ title: "Success", description: "Menu updated successfully" });
      setIsMenuDialogOpen(false);
      setSelectedMenu(null);
      menuForm.reset();
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to update menu", variant: "destructive" });
    }
  });

  const deleteMenuMutation = useMutation({
    mutationFn: (id: string) => apiRequest(`backend/api/cms/menu-items/${id}`, { method: "DELETE" }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["backend/api/cms/menu-items"] });
      toast({ title: "Success", description: "Menu deleted successfully" });
    },
    onError: () => {
      toast({ title: "Error", description: "Failed to delete menu", variant: "destructive" });
    }
  });

  const handleEditPage = (page: any) => {
    setSelectedPage(page);
    pageForm.reset({
      title: page.title,
      slug: page.slug,
      content: page.content || "",
      metaTitle: page.metadata?.metaTitle || "",
      metaDescription: page.metadata?.metaDescription || "",
      isPublished: page.isPublished || false
    });
    setIsPageDialogOpen(true);
  };

  const handleEditMenu = (menu: any) => {
    setSelectedMenu(menu);
    menuForm.reset({
      name: menu.name,
      location: menu.location,
      items: menu.items || []
    });
    setIsMenuDialogOpen(true);
  };

  const onPageSubmit = (data: PageFormData) => {
    if (selectedPage) {
      updatePageMutation.mutate({ id: selectedPage.id, data });
    } else {
      createPageMutation.mutate(data);
    }
  };

  const onMenuSubmit = (data: MenuFormData) => {
    if (selectedMenu) {
      updateMenuMutation.mutate({ id: selectedMenu.id, data });
    } else {
      createMenuMutation.mutate(data);
    }
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <Settings className="h-8 w-8 text-[#ED1C24] mr-3" />
              <h1 className="text-2xl font-bold text-gray-900">Website Admin</h1>
            </div>
            <div className="flex space-x-4">
              <Button 
                onClick={() => window.location.href = '/'}
                variant="outline"
                className="border-[#1A1A40] text-[#1A1A40] hover:bg-[#1A1A40] hover:text-white"
              >
                <Globe className="mr-2 h-4 w-4" />
                View Website
              </Button>
              <Button 
                onClick={() => window.location.href = '/sign-in'}
                className="bg-[#ED1C24] hover:bg-red-700 text-white"
              >
                Sign Out
              </Button>
            </div>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <Tabs defaultValue="pages" className="space-y-6">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="pages" className="flex items-center">
              <FileText className="mr-2 h-4 w-4" />
              Pages Management
            </TabsTrigger>
            <TabsTrigger value="menus" className="flex items-center">
              <Menu className="mr-2 h-4 w-4" />
              Menus Management
            </TabsTrigger>
          </TabsList>

          {/* Pages Management */}
          <TabsContent value="pages">
            <Card>
              <CardHeader>
                <div className="flex justify-between items-center">
                  <CardTitle>Website Pages</CardTitle>
                  <Dialog open={isPageDialogOpen} onOpenChange={setIsPageDialogOpen}>
                    <DialogTrigger asChild>
                      <Button 
                        onClick={() => {
                          setSelectedPage(null);
                          pageForm.reset();
                        }}
                        className="bg-[#ED1C24] hover:bg-red-700 text-white"
                      >
                        <Plus className="mr-2 h-4 w-4" />
                        Add Page
                      </Button>
                    </DialogTrigger>
                    <DialogContent className="max-w-2xl">
                      <DialogHeader>
                        <DialogTitle>
                          {selectedPage ? "Edit Page" : "Create New Page"}
                        </DialogTitle>
                      </DialogHeader>
                      <Form {...pageForm}>
                        <form onSubmit={pageForm.handleSubmit(onPageSubmit)} className="space-y-4">
                          <FormField
                            control={pageForm.control}
                            name="title"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Title</FormLabel>
                                <FormControl>
                                  <Input placeholder="Page title" {...field} />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                          <FormField
                            control={pageForm.control}
                            name="slug"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Slug (URL)</FormLabel>
                                <FormControl>
                                  <Input placeholder="page-url-slug" {...field} />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                          <FormField
                            control={pageForm.control}
                            name="content"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Content</FormLabel>
                                <FormControl>
                                  <Textarea 
                                    placeholder="Page content (HTML supported)" 
                                    className="min-h-32"
                                    {...field} 
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                          <div className="grid grid-cols-2 gap-4">
                            <FormField
                              control={pageForm.control}
                              name="metaTitle"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>Meta Title</FormLabel>
                                  <FormControl>
                                    <Input placeholder="SEO title" {...field} />
                                  </FormControl>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                            <FormField
                              control={pageForm.control}
                              name="isPublished"
                              render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3">
                                  <div className="space-y-0.5">
                                    <FormLabel>Published</FormLabel>
                                    <div className="text-sm text-muted-foreground">
                                      Make this page visible to visitors
                                    </div>
                                  </div>
                                  <FormControl>
                                    <input
                                      type="checkbox"
                                      checked={field.value}
                                      onChange={field.onChange}
                                      className="h-4 w-4"
                                    />
                                  </FormControl>
                                </FormItem>
                              )}
                            />
                          </div>
                          <FormField
                            control={pageForm.control}
                            name="metaDescription"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Meta Description</FormLabel>
                                <FormControl>
                                  <Textarea placeholder="SEO description" {...field} />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                          <div className="flex justify-end space-x-2">
                            <Button 
                              type="button" 
                              variant="outline" 
                              onClick={() => setIsPageDialogOpen(false)}
                            >
                              Cancel
                            </Button>
                            <Button 
                              type="submit" 
                              className="bg-[#ED1C24] hover:bg-red-700 text-white"
                              disabled={createPageMutation.isPending || updatePageMutation.isPending}
                            >
                              {selectedPage ? "Update" : "Create"} Page
                            </Button>
                          </div>
                        </form>
                      </Form>
                    </DialogContent>
                  </Dialog>
                </div>
              </CardHeader>
              <CardContent>
                {pagesLoading ? (
                  <div className="text-center py-8">Loading pages...</div>
                ) : (
                  <div className="space-y-4">
                    {pages && Array.isArray(pages) && pages.length > 0 ? (
                      pages.map((page: any) => (
                        <div key={page.id} className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                          <div>
                            <h3 className="font-semibold text-gray-900">{page.title}</h3>
                            <p className="text-sm text-gray-600">/{page.slug}</p>
                            <div className="flex items-center mt-2 space-x-2">
                              <Badge variant={page.isPublished ? 'default' : 'secondary'}>
                                {page.isPublished ? 'Published' : 'Draft'}
                              </Badge>
                              {page.updatedAt && (
                                <span className="text-xs text-gray-500">
                                  Updated: {new Date(page.updatedAt).toLocaleDateString()}
                                </span>
                              )}
                            </div>
                          </div>
                          <div className="flex space-x-2">
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => handleEditPage(page)}
                            >
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => deletePageMutation.mutate(page.id)}
                              className="text-red-600 hover:text-red-700"
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      ))
                    ) : (
                      <div className="text-center py-8 text-gray-500">
                        No pages found. Create your first page to get started.
                      </div>
                    )}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Menus Management */}
          <TabsContent value="menus">
            <Card>
              <CardHeader>
                <div className="flex justify-between items-center">
                  <CardTitle>Navigation Menus</CardTitle>
                  <Dialog open={isMenuDialogOpen} onOpenChange={setIsMenuDialogOpen}>
                    <DialogTrigger asChild>
                      <Button 
                        onClick={() => {
                          setSelectedMenu(null);
                          menuForm.reset();
                        }}
                        className="bg-[#ED1C24] hover:bg-red-700 text-white"
                      >
                        <Plus className="mr-2 h-4 w-4" />
                        Add Menu
                      </Button>
                    </DialogTrigger>
                    <DialogContent>
                      <DialogHeader>
                        <DialogTitle>
                          {selectedMenu ? "Edit Menu" : "Create Menu"}
                        </DialogTitle>
                      </DialogHeader>
                      <Form {...menuForm}>
                        <form onSubmit={menuForm.handleSubmit(onMenuSubmit)} className="space-y-4">
                          <FormField
                            control={menuForm.control}
                            name="name"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Menu Name</FormLabel>
                                <FormControl>
                                  <Input placeholder="Main Navigation" {...field} />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                          <FormField
                            control={menuForm.control}
                            name="location"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Location</FormLabel>
                                <Select onValueChange={field.onChange} defaultValue={field.value}>
                                  <FormControl>
                                    <SelectTrigger>
                                      <SelectValue placeholder="Select location" />
                                    </SelectTrigger>
                                  </FormControl>
                                  <SelectContent>
                                    <SelectItem value="header">Header</SelectItem>
                                    <SelectItem value="footer">Footer</SelectItem>
                                    <SelectItem value="sidebar">Sidebar</SelectItem>
                                  </SelectContent>
                                </Select>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                          <div className="flex justify-end space-x-2">
                            <Button 
                              type="button" 
                              variant="outline" 
                              onClick={() => setIsMenuDialogOpen(false)}
                            >
                              Cancel
                            </Button>
                            <Button 
                              type="submit" 
                              className="bg-[#ED1C24] hover:bg-red-700 text-white"
                              disabled={createMenuMutation.isPending || updateMenuMutation.isPending}
                            >
                              {selectedMenu ? "Update" : "Create"} Menu
                            </Button>
                          </div>
                        </form>
                      </Form>
                    </DialogContent>
                  </Dialog>
                </div>
              </CardHeader>
              <CardContent>
                {menuLoading ? (
                  <div className="text-center py-8">Loading menus...</div>
                ) : (
                  <div className="space-y-4">
                    {menus && Array.isArray(menus) && menus.length > 0 ? (
                      menus.map((menu: any) => (
                        <div key={menu.id} className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                          <div>
                            <h3 className="font-semibold text-gray-900">{menu.name}</h3>
                            <p className="text-sm text-gray-600">Location: {menu.location}</p>
                            <div className="flex items-center mt-2 space-x-2">
                              <Badge variant="secondary">
                                {Array.isArray(menu.items) ? menu.items.length : 0} items
                              </Badge>
                              <Badge variant={menu.isActive ? 'default' : 'secondary'}>
                                {menu.isActive ? 'Active' : 'Inactive'}
                              </Badge>
                            </div>
                          </div>
                          <div className="flex space-x-2">
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => handleEditMenu(menu)}
                            >
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => deleteMenuMutation.mutate(menu.id)}
                              className="text-red-600 hover:text-red-700"
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      ))
                    ) : (
                      <div className="text-center py-8 text-gray-500">
                        No menus found. Create your first menu to get started.
                      </div>
                    )}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}