import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { z } from "zod";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { CheckCircle } from "lucide-react";
import { WebsiteLayout } from "@/components/WebsiteLayout";

// Form validation schema
const applicationSchema = z.object({
  // Basic Information
  applicationType: z.string().min(1, "Application type is required"),
  organizationType: z.string().min(1, "Organization type is required"),
  organizationName: z.string().min(1, "Organization name is required"),
  establishedYear: z.number().min(1900, "Invalid year").max(new Date().getFullYear()),
  numberOfEmployees: z.number().min(1, "Number of employees is required"),
  businessLicense: z.string().min(1, "Business license number is required"),
  taxRegistration: z.string().min(1, "Tax registration number is required"),
  
  // Contact Details
  contactPerson: z.string().min(1, "Contact person is required"),
  contactEmail: z.string().email("Valid email is required"),
  contactPhone: z.string().min(1, "Phone number is required"),
  address: z.string().min(1, "Address is required"),
  city: z.string().min(1, "City is required"),
  country: z.string().min(1, "Country is required"),
  
  // Account Creation
  password: z.string().min(8, "Password must be at least 8 characters"),
  confirmPassword: z.string().min(8, "Please confirm your password"),
  
  // Facilities
  facilitiesDescription: z.string().min(1, "Facilities description is required"),
  
  // Personnel & Programs
  certificationPrograms: z.string().min(1, "Certification programs are required"),
  
  // Documents & Declaration
  agreementToTerms: z.boolean().refine(val => val === true, "You must agree to terms"),
  declarationOfAccuracy: z.boolean().refine(val => val === true, "You must declare accuracy"),
}).refine((data) => data.password === data.confirmPassword, {
  message: "Passwords don't match",
  path: ["confirmPassword"],
});

type ApplicationFormData = z.infer<typeof applicationSchema>;

const steps = [
  { id: 1, title: "Basic Information", description: "Tell us about your organization" },
  { id: 2, title: "Contact Details", description: "How can we reach you?" },
  { id: 3, title: "Facilities", description: "Describe your training facilities" },
  { id: 4, title: "Personnel & Programs", description: "Your team and programs" },
  { id: 5, title: "Documents & Declaration", description: "Final documentation" },
];

export default function TrainingCenterApplication() {
  const [currentStep, setCurrentStep] = useState(1);
  const [isSubmitted, setIsSubmitted] = useState(false);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const form = useForm<ApplicationFormData>({
    resolver: zodResolver(applicationSchema),
    defaultValues: {
      applicationType: "training_center",
      organizationType: "private",
      establishedYear: new Date().getFullYear(),
      numberOfEmployees: 1,
      agreementToTerms: false,
      declarationOfAccuracy: false,
      organizationName: "",
      businessLicense: "",
      taxRegistration: "",
      contactPerson: "",
      contactEmail: "",
      contactPhone: "",
      address: "",
      city: "",
      country: "",
      password: "",
      confirmPassword: "",
      facilitiesDescription: "",
      certificationPrograms: "",
    },
  });

  const submitMutation = useMutation({
    mutationFn: (data: ApplicationFormData) => {
      // Transform form data to match API expectations
      const apiData = {
        ...data,
        technicalInfrastructure: {
          computers: 10,
          projectors: 2,
          wifi: "standard",
          lms: "basic"
        },
        qualifiedPersonnel: {
          trainers: 5,
          experience: "3+ years",
          certifications: ["Basic Training"]
        },
        certificationPrograms: {
          programs: data.certificationPrograms.split(',').map(p => p.trim()),
          accreditation: "pending"
        },
        operatingHours: {
          weekdays: "9:00-17:00",
          weekends: "closed",
          holidays: "closed"
        },
        capacityInformation: {
          classrooms: 3,
          labs: 1,
          max_students: 50
        },
        qualityAssurance: {
          iso_certified: false,
          annual_audit: false,
          feedback_system: true
        },
        supportingDocuments: {
          business_license: "pending_upload",
          tax_cert: "pending_upload",
          insurance: "pending_upload"
        },
        facilityPhotos: {
          facility_photos: ["pending_upload"]
        }
      };
      
      return apiRequest("backend/api/accreditation/apply", "POST", apiData);
    },
    onSuccess: (data) => {
      setIsSubmitted(true);
      toast({ 
        title: "Application Submitted Successfully!", 
        description: `Your application ID is ${data.id}. You can now log in with your email and password.`
      });
      queryClient.invalidateQueries({ queryKey: ["backend/api/accreditation/applications"] });
    },
    onError: (error) => {
      toast({ 
        title: "Error submitting application", 
        description: error.message, 
        variant: "destructive" 
      });
    },
  });

  const nextStep = () => {
    if (currentStep < 5) setCurrentStep(currentStep + 1);
  };

  const prevStep = () => {
    if (currentStep > 1) setCurrentStep(currentStep - 1);
  };

  const onSubmit = (data: ApplicationFormData) => {
    console.log('onSubmit called with data:', data);
    submitMutation.mutate(data);
  };

  if (isSubmitted) {
    return (
      <WebsiteLayout>
        <div className="min-h-screen bg-gray-50 py-12">
          <div className="max-w-2xl mx-auto px-4">
            <Card className="text-center">
              <CardContent className="pt-6">
                <CheckCircle className="h-16 w-16 text-green-500 mx-auto mb-4" />
                <h1 className="text-2xl font-bold text-gray-900 mb-2">Application Submitted!</h1>
                <p className="text-gray-600 mb-6">
                  Thank you for your application. You will receive a confirmation email shortly.
                  You can now log in to track your application status.
                </p>
                <div className="space-y-4">
                  <Button 
                    onClick={() => window.location.href = '/'}
                    className="bg-[#ED1C24] hover:bg-[#d91b22] w-full"
                  >
                    Go to Home Page
                  </Button>
                  <p className="text-sm text-gray-600 text-center">
                    Click "Sign In" on the home page to access your applicant dashboard
                  </p>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </WebsiteLayout>
    );
  }

  return (
    <WebsiteLayout>
      <div className="min-h-screen bg-gray-50 py-12">
        <div className="max-w-4xl mx-auto px-4">
          <div className="text-center mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">
              Training Center Accreditation Application
            </h1>
            <p className="text-gray-600">Apply for IBDL accreditation to become an authorized training center</p>
          </div>

          {/* Progress Indicator */}
          <div className="mb-8">
            <div className="flex items-center justify-between mb-4">
              {steps.map((step) => (
                <div key={step.id} className="flex flex-col items-center">
                  <div className={`w-10 h-10 rounded-full flex items-center justify-center text-sm font-medium
                    ${currentStep >= step.id 
                      ? 'bg-[#ED1C24] text-white' 
                      : 'bg-gray-200 text-gray-500'
                    }`}>
                    {step.id}
                  </div>
                  <div className="text-xs text-center mt-2 max-w-20">
                    <div className="font-medium">{step.title}</div>
                  </div>
                </div>
              ))}
            </div>
            <Progress value={(currentStep / 5) * 100} className="h-2" />
          </div>

          <form onSubmit={form.handleSubmit(onSubmit)}>
            <Card>
              <CardHeader>
                <CardTitle>{steps[currentStep - 1].title}</CardTitle>
                <p className="text-gray-600">{steps[currentStep - 1].description}</p>
              </CardHeader>
              <CardContent className="space-y-4">
                {/* Step 1: Basic Information */}
                {currentStep === 1 && (
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="applicationType">Application Type</Label>
                      <Select 
                        value={form.watch("applicationType")} 
                        onValueChange={(value) => form.setValue("applicationType", value)}
                      >
                        <SelectTrigger>
                          <SelectValue placeholder="Select application type" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="training_center">Training Center</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="organizationType">Organization Type</Label>
                      <Select 
                        value={form.watch("organizationType")} 
                        onValueChange={(value) => form.setValue("organizationType", value)}
                      >
                        <SelectTrigger>
                          <SelectValue placeholder="Select organization type" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="private">Private</SelectItem>
                          <SelectItem value="public">Public</SelectItem>
                          <SelectItem value="non_profit">Non-Profit</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="organizationName">Organization Name</Label>
                      <Input 
                        {...form.register("organizationName")}
                        placeholder="Enter organization name"
                      />
                      {form.formState.errors.organizationName && (
                        <p className="text-sm text-red-500">{form.formState.errors.organizationName.message}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="establishedYear">Established Year</Label>
                      <Input 
                        type="number"
                        {...form.register("establishedYear", { valueAsNumber: true })}
                        placeholder="2025"
                      />
                      {form.formState.errors.establishedYear && (
                        <p className="text-sm text-red-500">{form.formState.errors.establishedYear.message}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="numberOfEmployees">Number of Employees</Label>
                      <Input 
                        type="number"
                        {...form.register("numberOfEmployees", { valueAsNumber: true })}
                        placeholder="1"
                      />
                      {form.formState.errors.numberOfEmployees && (
                        <p className="text-sm text-red-500">{form.formState.errors.numberOfEmployees.message}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="businessLicense">Business License Number</Label>
                      <Input 
                        {...form.register("businessLicense")}
                        placeholder="Enter business license number"
                      />
                      {form.formState.errors.businessLicense && (
                        <p className="text-sm text-red-500">{form.formState.errors.businessLicense.message}</p>
                      )}
                    </div>

                    <div className="md:col-span-2">
                      <Label htmlFor="taxRegistration">Tax Registration Number</Label>
                      <Input 
                        {...form.register("taxRegistration")}
                        placeholder="Enter tax registration number"
                      />
                      {form.formState.errors.taxRegistration && (
                        <p className="text-sm text-red-500">{form.formState.errors.taxRegistration.message}</p>
                      )}
                    </div>
                  </div>
                )}

                {/* Step 2: Contact Details */}
                {currentStep === 2 && (
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="contactPerson">Contact Person</Label>
                      <Input 
                        {...form.register("contactPerson")}
                        placeholder="Enter contact person name"
                      />
                      {form.formState.errors.contactPerson && (
                        <p className="text-sm text-red-500">{form.formState.errors.contactPerson.message}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="contactEmail">Contact Email</Label>
                      <Input 
                        type="email"
                        {...form.register("contactEmail")}
                        placeholder="Enter email address"
                      />
                      {form.formState.errors.contactEmail && (
                        <p className="text-sm text-red-500">{form.formState.errors.contactEmail.message}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="contactPhone">Phone Number</Label>
                      <Input 
                        {...form.register("contactPhone")}
                        placeholder="Enter phone number"
                      />
                      {form.formState.errors.contactPhone && (
                        <p className="text-sm text-red-500">{form.formState.errors.contactPhone.message}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="city">City</Label>
                      <Input 
                        {...form.register("city")}
                        placeholder="Enter city"
                      />
                      {form.formState.errors.city && (
                        <p className="text-sm text-red-500">{form.formState.errors.city.message}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="country">Country</Label>
                      <Input 
                        {...form.register("country")}
                        placeholder="Enter country"
                      />
                      {form.formState.errors.country && (
                        <p className="text-sm text-red-500">{form.formState.errors.country.message}</p>
                      )}
                    </div>

                    <div className="md:col-span-2">
                      <Label htmlFor="address">Address</Label>
                      <Textarea 
                        {...form.register("address")}
                        placeholder="Enter complete address"
                        rows={3}
                      />
                      {form.formState.errors.address && (
                        <p className="text-sm text-red-500">{form.formState.errors.address.message}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="password">Password</Label>
                      <Input 
                        type="password"
                        {...form.register("password")}
                        placeholder="Enter password for your account"
                      />
                      {form.formState.errors.password && (
                        <p className="text-sm text-red-500">{form.formState.errors.password.message}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="confirmPassword">Confirm Password</Label>
                      <Input 
                        type="password"
                        {...form.register("confirmPassword")}
                        placeholder="Confirm your password"
                      />
                      {form.formState.errors.confirmPassword && (
                        <p className="text-sm text-red-500">{form.formState.errors.confirmPassword.message}</p>
                      )}
                    </div>
                  </div>
                )}

                {/* Step 3: Facilities */}
                {currentStep === 3 && (
                  <div className="space-y-4">
                    <div>
                      <Label htmlFor="facilitiesDescription">Facilities Description</Label>
                      <Textarea 
                        {...form.register("facilitiesDescription")}
                        placeholder="Describe your training facilities, equipment, and infrastructure"
                        rows={6}
                      />
                      {form.formState.errors.facilitiesDescription && (
                        <p className="text-sm text-red-500">{form.formState.errors.facilitiesDescription.message}</p>
                      )}
                    </div>
                  </div>
                )}

                {/* Step 4: Personnel & Programs */}
                {currentStep === 4 && (
                  <div className="space-y-4">
                    <div>
                      <Label htmlFor="certificationPrograms">Certification Programs</Label>
                      <Textarea 
                        {...form.register("certificationPrograms")}
                        placeholder="List the certification programs you plan to offer (separate with commas)"
                        rows={4}
                      />
                      {form.formState.errors.certificationPrograms && (
                        <p className="text-sm text-red-500">{form.formState.errors.certificationPrograms.message}</p>
                      )}
                    </div>
                  </div>
                )}

                {/* Step 5: Documents & Declaration */}
                {currentStep === 5 && (
                  <div className="space-y-6">
                    <div className="space-y-4">
                      <div className="flex items-start space-x-3">
                        <input
                          type="checkbox"
                          {...form.register("agreementToTerms")}
                          className="mt-1 h-4 w-4"
                          id="agreementToTerms"
                        />
                        <div>
                          <Label htmlFor="agreementToTerms" className="text-sm cursor-pointer">Agreement to Terms</Label>
                          <p className="text-sm text-gray-600">
                            I agree to comply with IBDL standards and regulations for accredited training centers.
                          </p>
                        </div>
                      </div>
                      {form.formState.errors.agreementToTerms && (
                        <p className="text-sm text-red-500">{form.formState.errors.agreementToTerms.message}</p>
                      )}

                      <div className="flex items-start space-x-3">
                        <input
                          type="checkbox"
                          {...form.register("declarationOfAccuracy")}
                          className="mt-1 h-4 w-4"
                          id="declarationOfAccuracy"
                        />
                        <div>
                          <Label htmlFor="declarationOfAccuracy" className="text-sm cursor-pointer">Declaration of Accuracy</Label>
                          <p className="text-sm text-gray-600">
                            I declare that all information provided in this application is accurate and complete.
                          </p>
                        </div>
                      </div>
                      {form.formState.errors.declarationOfAccuracy && (
                        <p className="text-sm text-red-500">{form.formState.errors.declarationOfAccuracy.message}</p>
                      )}
                    </div>
                  </div>
                )}

                {/* Navigation Buttons */}
                <div className="flex justify-between pt-6">
                  <Button 
                    type="button" 
                    variant="outline" 
                    onClick={prevStep}
                    disabled={currentStep === 1}
                  >
                    Previous
                  </Button>
                  
                  {currentStep < 5 ? (
                    <Button 
                      type="button" 
                      onClick={nextStep}
                      className="bg-[#ED1C24] hover:bg-[#d91b22]"
                    >
                      Next
                    </Button>
                  ) : (
                    <Button 
                      type="button"
                      disabled={submitMutation.isPending}
                      className="bg-[#ED1C24] hover:bg-[#d91b22] disabled:opacity-50"
                      onClick={(e) => {
                        const values = form.getValues();
                        console.log('Submit clicked, form values:', values);
                        console.log('Checkboxes:', {
                          agreementToTerms: values.agreementToTerms,
                          declarationOfAccuracy: values.declarationOfAccuracy
                        });
                        
                        // Manual validation for required checkboxes
                        if (!values.agreementToTerms || !values.declarationOfAccuracy) {
                          toast({
                            title: "Please complete all required fields",
                            description: "You must agree to the terms and declare accuracy before submitting.",
                            variant: "destructive"
                          });
                          return;
                        }
                        
                        // Check for required fields
                        const requiredFields = [
                          { field: 'organizationName', name: 'Organization Name', step: 1 },
                          { field: 'contactPerson', name: 'Contact Person', step: 2 },
                          { field: 'contactEmail', name: 'Contact Email', step: 2 },
                          { field: 'contactPhone', name: 'Contact Phone', step: 2 },
                          { field: 'password', name: 'Password', step: 2 },
                          { field: 'facilitiesDescription', name: 'Facilities Description', step: 3 },
                          { field: 'certificationPrograms', name: 'Certification Programs', step: 4 }
                        ];
                        
                        for (const { field, name, step } of requiredFields) {
                          if (!values[field]?.toString().trim()) {
                            toast({
                              title: `${name} is required`,
                              description: `Please complete ${name} in step ${step}.`,
                              variant: "destructive"
                            });
                            setCurrentStep(step);
                            return;
                          }
                        }
                        
                        console.log('All validation passed, submitting application...');
                        onSubmit(values);
                      }}
                    >
                      {submitMutation.isPending ? "Submitting..." : "Submit Application"}
                    </Button>
                  )}
                </div>
              </CardContent>
            </Card>
          </form>
        </div>
      </div>
    </WebsiteLayout>
  );
}