import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Plus, Edit, Trash2, BookOpen, Award, FileText, Languages } from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { 
  insertSystemCertificateSchema, 
  insertSystemLevelSchema, 
  insertSystemChapterSchema,
  type SystemCertificate,
  type SystemLevel,
  type SystemChapter
} from "@shared/schema";

export default function SystemSetup() {
  const [activeTab, setActiveTab] = useState("certificates");
  const [selectedCertificate, setSelectedCertificate] = useState<string | null>(null);
  const [selectedLevel, setSelectedLevel] = useState<string | null>(null);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch certificates
  const { data: certificates = [], isLoading: loadingCertificates } = useQuery({
    queryKey: ["backend/api/system-certificates"],
  });

  // Fetch levels for selected certificate
  const { data: levels = [], isLoading: loadingLevels } = useQuery({
    queryKey: ["backend/api/system-levels", selectedCertificate],
    enabled: !!selectedCertificate,
  });

  // Fetch chapters for selected level
  const { data: chapters = [], isLoading: loadingChapters } = useQuery({
    queryKey: ["backend/api/system-chapters", selectedLevel],
    enabled: !!selectedLevel,
  });

  // Certificate form
  const certificateForm = useForm({
    resolver: zodResolver(insertSystemCertificateSchema),
    defaultValues: {
      name: "",
      nameEn: "",
      nameAr: "",
      nameFr: "",
      description: "",
      descriptionEn: "",
      descriptionAr: "",
      descriptionFr: "",
      certificateType: "professional",
      duration: 40,
      validityPeriod: 24,
      prerequisites: {},
      createdBy: "sys-admin-1",
    },
  });

  // Level form
  const levelForm = useForm({
    resolver: zodResolver(insertSystemLevelSchema),
    defaultValues: {
      certificateId: "",
      levelNumber: 1,
      nameEn: "",
      nameAr: "",
      nameFr: "",
      description: "",
      requirements: {},

      timeLimit: 120,
      createdBy: "sys-admin-1",
    },
  });

  // Chapter form
  const chapterForm = useForm({
    resolver: zodResolver(insertSystemChapterSchema),
    defaultValues: {
      levelId: "",
      chapterNumber: 1,
      title: "",
      titleEn: "",
      titleAr: "",
      titleFr: "",
      content: {},
      objectives: {},
      duration: 60,
      createdBy: "sys-admin-1",
    },
  });

  // Mutations
  const createCertificateMutation = useMutation({
    mutationFn: (data: any) => apiRequest("backend/api/system-certificates", "POST", data),
    onSuccess: () => {
      toast({ title: "Certificate created successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/system-certificates"] });
      certificateForm.reset();
    },
    onError: (error) => {
      toast({ title: "Error creating certificate", description: error.message, variant: "destructive" });
    },
  });

  const createLevelMutation = useMutation({
    mutationFn: (data: any) => apiRequest("backend/api/system-levels", "POST", data),
    onSuccess: () => {
      toast({ title: "Level created successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/system-levels", selectedCertificate] });
      // Form reset handled by dialog close
    },
    onError: (error) => {
      toast({ title: "Error creating level", description: error.message, variant: "destructive" });
    },
  });

  const createChapterMutation = useMutation({
    mutationFn: (data: any) => apiRequest("backend/api/system-chapters", "POST", data),
    onSuccess: () => {
      toast({ title: "Chapter created successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/system-chapters", selectedLevel] });
      chapterForm.reset();
    },
    onError: (error) => {
      toast({ title: "Error creating chapter", description: error.message, variant: "destructive" });
    },
  });

  return (
    <DashboardLayout userType="admin">
      <div className="space-y-8">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">System Setup</h1>
          <p className="text-muted-foreground">
            Configure certificates, levels, and chapters for the learning platform
          </p>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="certificates" className="flex items-center gap-2">
              <Award className="w-4 h-4" />
              Certificates
            </TabsTrigger>
            <TabsTrigger value="levels" className="flex items-center gap-2">
              <BookOpen className="w-4 h-4" />
              Levels
            </TabsTrigger>
            <TabsTrigger value="chapters" className="flex items-center gap-2">
              <FileText className="w-4 h-4" />
              Chapters
            </TabsTrigger>
          </TabsList>

          <TabsContent value="certificates" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Certificates Management</h2>
              <Dialog>
                <DialogTrigger asChild>
                  <Button>
                    <Plus className="w-4 h-4 mr-2" />
                    Add Certificate
                  </Button>
                </DialogTrigger>
                <DialogContent className="max-w-4xl">
                  <DialogHeader>
                    <DialogTitle>Create New Certificate</DialogTitle>
                  </DialogHeader>
                  <Form {...certificateForm}>
                    <form onSubmit={certificateForm.handleSubmit((data) => createCertificateMutation.mutate(data))} className="space-y-4">
                      <div className="grid grid-cols-2 gap-4">
                        <FormField
                          control={certificateForm.control}
                          name="name"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Certificate Name *</FormLabel>
                              <FormControl>
                                <Input placeholder="Strategic Leadership Certificate" {...field} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={certificateForm.control}
                          name="description"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Description *</FormLabel>
                              <FormControl>
                                <Textarea placeholder="Certificate description" {...field} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="grid grid-cols-2 gap-4">
                        <FormField
                          control={certificateForm.control}
                          name="nameEn"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Name (English)</FormLabel>
                              <FormControl>
                                <Input placeholder="Strategic Leadership Certificate" {...field} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={certificateForm.control}
                          name="nameAr"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Name (Arabic)</FormLabel>
                              <FormControl>
                                <Input placeholder="شهادة القيادة الاستراتيجية" {...field} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={certificateForm.control}
                          name="nameFr"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Name (French)</FormLabel>
                              <FormControl>
                                <Input placeholder="Certificat de Leadership Stratégique" {...field} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={certificateForm.control}
                          name="certificateType"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Certificate Type</FormLabel>
                              <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl>
                                  <SelectTrigger>
                                    <SelectValue placeholder="Select type" />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  <SelectItem value="professional">Professional</SelectItem>
                                  <SelectItem value="executive">Executive</SelectItem>
                                  <SelectItem value="academic">Academic</SelectItem>
                                </SelectContent>
                              </Select>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="grid grid-cols-3 gap-4">
                        <FormField
                          control={certificateForm.control}
                          name="descriptionEn"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Description (English)</FormLabel>
                              <FormControl>
                                <Textarea placeholder="Certificate description in English" {...field} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={certificateForm.control}
                          name="descriptionAr"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Description (Arabic)</FormLabel>
                              <FormControl>
                                <Textarea placeholder="وصف الشهادة باللغة العربية" {...field} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={certificateForm.control}
                          name="descriptionFr"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Description (French)</FormLabel>
                              <FormControl>
                                <Textarea placeholder="Description du certificat en français" {...field} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="grid grid-cols-2 gap-4">
                        <FormField
                          control={certificateForm.control}
                          name="duration"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Duration (Hours)</FormLabel>
                              <FormControl>
                                <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={certificateForm.control}
                          name="validityPeriod"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Validity Period (Months)</FormLabel>
                              <FormControl>
                                <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <Button type="submit" disabled={createCertificateMutation.isPending}>
                        {createCertificateMutation.isPending ? "Creating..." : "Create Certificate"}
                      </Button>
                    </form>
                  </Form>
                </DialogContent>
              </Dialog>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {certificates.map((certificate: SystemCertificate) => (
                <Card key={certificate.id} className="cursor-pointer hover:shadow-lg transition-shadow" onClick={() => setSelectedCertificate(certificate.id)}>
                  <CardHeader>
                    <div className="flex justify-between items-start">
                      <div>
                        <CardTitle className="text-lg">{certificate.nameEn}</CardTitle>
                        <CardDescription>
                          <div className="flex items-center gap-2 mt-1">
                            <Languages className="w-4 h-4" />
                            <span className="text-sm">{certificate.nameAr}</span>
                          </div>
                        </CardDescription>
                      </div>
                      <Badge variant={certificate.certificateType === "professional" ? "default" : "secondary"}>
                        {certificate.certificateType}
                      </Badge>
                    </div>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>Duration:</span>
                        <span>{certificate.duration}h</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span>Validity:</span>
                        <span>{certificate.validityPeriod} months</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span>Status:</span>
                        <Badge variant={certificate.isActive ? "default" : "secondary"}>
                          {certificate.isActive ? "Active" : "Inactive"}
                        </Badge>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>

          <TabsContent value="levels" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Levels Management</h2>
              <div className="flex gap-2">
                <Select value={selectedCertificate || ""} onValueChange={setSelectedCertificate}>
                  <SelectTrigger className="w-64">
                    <SelectValue placeholder="Select Certificate" />
                  </SelectTrigger>
                  <SelectContent>
                    {certificates.map((cert: SystemCertificate) => (
                      <SelectItem key={cert.id} value={cert.id}>{cert.nameEn}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                {selectedCertificate && (
                  <Dialog>
                    <DialogTrigger asChild>
                      <Button>
                        <Plus className="w-4 h-4 mr-2" />
                        Add Level
                      </Button>
                    </DialogTrigger>
                    <DialogContent className="max-w-4xl">
                      <DialogHeader>
                        <DialogTitle>Create New Level</DialogTitle>
                      </DialogHeader>
                      <form onSubmit={(e) => {
                        e.preventDefault();
                        const formData = new FormData(e.currentTarget);
                        const data = {
                          name: formData.get('nameEn') as string,
                          nameEn: formData.get('nameEn') as string,
                          nameAr: formData.get('nameAr') as string,
                          nameFr: formData.get('nameFr') as string,
                          description: formData.get('description') as string,
                          levelNumber: Number(formData.get('levelNumber')),
                          timeLimit: Number(formData.get('timeLimit')),
                          certificateId: selectedCertificate,
                          requirements: {},
                          createdBy: "sys-admin-1"
                        };
                        console.log("Submitting level data:", data);
                        createLevelMutation.mutate(data);
                      }} className="space-y-4">
                        <div className="grid grid-cols-2 gap-4">
                          <div>
                            <label htmlFor="levelNumber" className="block text-sm font-medium mb-2">Level Number</label>
                            <Input 
                              id="levelNumber"
                              name="levelNumber"
                              type="number" 
                              defaultValue="1"
                              required
                            />
                          </div>
                          <div>
                            <label htmlFor="nameEn" className="block text-sm font-medium mb-2">Name (English)</label>
                            <Input 
                              id="nameEn"
                              name="nameEn"
                              placeholder="Foundation Level"
                              required
                            />
                          </div>
                        </div>
                        <div className="grid grid-cols-2 gap-4">
                          <div>
                            <label htmlFor="nameAr" className="block text-sm font-medium mb-2">Name (Arabic)</label>
                            <Input 
                              id="nameAr"
                              name="nameAr"
                              placeholder="المستوى الأساسي"
                              required
                            />
                          </div>
                          <div>
                            <label htmlFor="nameFr" className="block text-sm font-medium mb-2">Name (French)</label>
                            <Input 
                              id="nameFr"
                              name="nameFr"
                              placeholder="Niveau Fondamental"
                              required
                            />
                          </div>
                        </div>
                        <div>
                          <label htmlFor="description" className="block text-sm font-medium mb-2">Description</label>
                          <textarea
                            id="description"
                            name="description"
                            className="w-full p-2 border rounded-md"
                            rows={3}
                            placeholder="Level description and requirements"
                            required
                          />
                        </div>
                        <div>
                          <label htmlFor="timeLimit" className="block text-sm font-medium mb-2">Time Limit (Minutes)</label>
                          <Input 
                            id="timeLimit"
                            name="timeLimit"
                            type="number" 
                            defaultValue="120"
                            min="1"
                            required
                          />
                        </div>
                        <Button type="submit" disabled={createLevelMutation.isPending}>
                          {createLevelMutation.isPending ? "Creating..." : "Create Level"}
                        </Button>
                      </form>
                    </DialogContent>
                  </Dialog>
                )}
              </div>
            </div>

            {selectedCertificate && (
              <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                {levels.map((level: SystemLevel) => (
                  <Card key={level.id} className="cursor-pointer hover:shadow-lg transition-shadow" onClick={() => setSelectedLevel(level.id)}>
                    <CardHeader>
                      <div className="flex justify-between items-start">
                        <div>
                          <CardTitle className="text-lg">Level {level.levelNumber}</CardTitle>
                          <CardDescription>{level.nameEn}</CardDescription>
                        </div>
                        <Badge variant="outline">{level.timeLimit}min</Badge>
                      </div>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-2">
                        <div className="flex justify-between text-sm">
                          <span>Time Limit:</span>
                          <span>{level.timeLimit} min</span>
                        </div>
                        <div className="flex justify-between text-sm">
                          <span>Status:</span>
                          <Badge variant={level.isActive ? "default" : "secondary"}>
                            {level.isActive ? "Active" : "Inactive"}
                          </Badge>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          <TabsContent value="chapters" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Chapters Management</h2>
              <div className="flex gap-2">
                <Select value={selectedLevel || ""} onValueChange={setSelectedLevel}>
                  <SelectTrigger className="w-64">
                    <SelectValue placeholder="Select Level" />
                  </SelectTrigger>
                  <SelectContent>
                    {levels.map((level: SystemLevel) => (
                      <SelectItem key={level.id} value={level.id}>Level {level.levelNumber} - {level.nameEn}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                {selectedLevel && (
                  <Dialog>
                    <DialogTrigger asChild>
                      <Button>
                        <Plus className="w-4 h-4 mr-2" />
                        Add Chapter
                      </Button>
                    </DialogTrigger>
                    <DialogContent className="max-w-4xl">
                      <DialogHeader>
                        <DialogTitle>Create New Chapter</DialogTitle>
                      </DialogHeader>
                      <Form {...chapterForm}>
                        <form onSubmit={chapterForm.handleSubmit((data) => createChapterMutation.mutate({ ...data, levelId: selectedLevel, title: data.titleEn }))} className="space-y-4">
                          <div className="grid grid-cols-2 gap-4">
                            <FormField
                              control={chapterForm.control}
                              name="chapterNumber"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>Chapter Number</FormLabel>
                                  <FormControl>
                                    <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                                  </FormControl>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                            <FormField
                              control={chapterForm.control}
                              name="titleEn"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>Title (English)</FormLabel>
                                  <FormControl>
                                    <Input placeholder="Introduction to Leadership" {...field} />
                                  </FormControl>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                          </div>
                          <div className="grid grid-cols-3 gap-4">
                            <FormField
                              control={chapterForm.control}
                              name="titleAr"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>Title (Arabic)</FormLabel>
                                  <FormControl>
                                    <Input placeholder="مقدمة في القيادة" {...field} />
                                  </FormControl>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                            <FormField
                              control={chapterForm.control}
                              name="titleFr"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>Title (French)</FormLabel>
                                  <FormControl>
                                    <Input placeholder="Introduction au Leadership" {...field} />
                                  </FormControl>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                            <FormField
                              control={chapterForm.control}
                              name="duration"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>Duration (Minutes)</FormLabel>
                                  <FormControl>
                                    <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                                  </FormControl>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                          </div>
                          <Button type="submit" disabled={createChapterMutation.isPending}>
                            {createChapterMutation.isPending ? "Creating..." : "Create Chapter"}
                          </Button>
                        </form>
                      </Form>
                    </DialogContent>
                  </Dialog>
                )}
              </div>
            </div>

            {selectedLevel && (
              <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                {chapters.map((chapter: SystemChapter) => (
                  <Card key={chapter.id} className="hover:shadow-lg transition-shadow">
                    <CardHeader>
                      <div className="flex justify-between items-start">
                        <div>
                          <CardTitle className="text-lg">Chapter {chapter.chapterNumber}</CardTitle>
                          <CardDescription>{chapter.titleEn}</CardDescription>
                        </div>
                        <Badge variant="outline">{chapter.duration}min</Badge>
                      </div>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-2">
                        <div className="flex justify-between text-sm">
                          <span>Arabic Title:</span>
                          <span>{chapter.titleAr}</span>
                        </div>
                        <div className="flex justify-between text-sm">
                          <span>French Title:</span>
                          <span>{chapter.titleFr}</span>
                        </div>
                        <div className="flex justify-between text-sm">
                          <span>Status:</span>
                          <Badge variant={chapter.isActive ? "default" : "secondary"}>
                            {chapter.isActive ? "Active" : "Inactive"}
                          </Badge>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>
        </Tabs>
      </div>
    </DashboardLayout>
  );
}