import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import WebsiteLayout from "@/components/WebsiteLayout";
import { Target, TrendingUp, Globe, Users, Zap, BookOpen, Award, Building } from "lucide-react";
import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";

export default function Strategy() {
  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/strategy"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <WebsiteLayout>
        <div className="min-h-screen bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="space-y-6">
            <Skeleton className="h-12 w-3/4 mx-auto" />
            <Skeleton className="h-6 w-1/2 mx-auto" />
            <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mt-12">
              {[1, 2, 3, 4].map((i) => (
                <Skeleton key={i} className="h-48 w-full" />
              ))}
            </div>
          </div>
        </div>
      </WebsiteLayout>
    );
  }

  const strategicGoals = [
    {
      number: "01",
      title: "Catalyze Exponential Value",
      description: "Generate exponential value through disruptive and revolutionary products and solutions",
      icon: Zap,
      color: "bg-red-50 text-red-700",
      borderColor: "border-red-200"
    },
    {
      number: "02", 
      title: "Pioneer Learning-Driven Model",
      description: "Establish an unparalleled learning-driven organization model for unleashing limitless potential",
      icon: BookOpen,
      color: "bg-blue-50 text-blue-700",
      borderColor: "border-blue-200"
    },
    {
      number: "03",
      title: "Ignite Digital Revolution",
      description: "Lead a digital revolution with revolutionary and disruptive digital solutions",
      icon: TrendingUp,
      color: "bg-purple-50 text-purple-700",
      borderColor: "border-purple-200"
    },
    {
      number: "04",
      title: "Empower Universities",
      description: "Enable universities to forge the path of unprecedented excellence",
      icon: Building,
      color: "bg-emerald-50 text-emerald-700",
      borderColor: "border-emerald-200"
    }
  ];

  const strategicPillars = [
    {
      title: "Innovation Leadership",
      description: "Driving continuous innovation in learning methodologies and digital solutions",
      icon: Target,
      metrics: ["50+ innovative products", "15+ patents pending", "100% digital transformation"]
    },
    {
      title: "Global Expansion",
      description: "Expanding our reach to serve learners and organizations worldwide",
      icon: Globe,
      metrics: ["39 countries", "120+ strategic partners", "222,000+ learners"]
    },
    {
      title: "Partnership Excellence",
      description: "Building strategic partnerships with leading educational institutions",
      icon: Users,
      metrics: ["4 university partnerships", "McGraw-Hill collaboration", "735 exam providers"]
    },
    {
      title: "Quality Assurance",
      description: "Maintaining the highest standards of educational excellence and accreditation",
      icon: Award,
      metrics: ["ISO 9001 certified", "99.9% uptime", "4.7/5 trust score"]
    }
  ];

  return (
    <WebsiteLayout>
      <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="text-center">
            <Badge variant="secondary" className="mb-4 bg-red-100 text-red-700">
              Strategic Vision
            </Badge>
            <h1 className="text-4xl md:text-5xl font-bold mb-6">
              {pageData?.title || "IBDL Strategic Framework"}
            </h1>
            <p className="text-xl text-gray-300 max-w-3xl mx-auto">
              {pageData?.content || ""}
              Our strategic roadmap guides us toward becoming the world's leading learning-driven organization, 
              empowering individuals and institutions to achieve unprecedented excellence.
            </p>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        {/* Mission and Vision */}
        <div className="grid md:grid-cols-2 gap-8 mb-16">
          <Card className="border-l-4 border-l-red-500">
            <CardHeader>
              <CardTitle className="text-2xl font-bold text-[#1A1A40]">Our Mission</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-gray-700 text-lg leading-relaxed">
                We strive to create innovative and digitalized learning solutions that empower 
                individuals and organizations to progress and thrive in an ever-evolving global landscape.
              </p>
            </CardContent>
          </Card>

          <Card className="border-l-4 border-l-blue-500">
            <CardHeader>
              <CardTitle className="text-2xl font-bold text-[#1A1A40]">Our Vision</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-gray-700 text-lg leading-relaxed">
                To be the world's most trusted and innovative learning partner, developing the learning 
                journeys that enable every person and organization to move forward.
              </p>
            </CardContent>
          </Card>
        </div>

        {/* Strategic Goals */}
        <div className="mb-16">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Strategic Goals</h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              Four key strategic goals that drive our mission to transform learning and development globally
            </p>
          </div>

          <div className="grid md:grid-cols-2 gap-8">
            {strategicGoals.map((goal, index) => (
              <Card key={index} className={`${goal.borderColor} border-2 hover:shadow-lg transition-shadow`}>
                <CardHeader>
                  <div className="flex items-center space-x-4">
                    <div className={`w-12 h-12 rounded-full ${goal.color} flex items-center justify-center`}>
                      <goal.icon className="w-6 h-6" />
                    </div>
                    <div>
                      <Badge variant="outline" className="mb-2">{goal.number}</Badge>
                      <CardTitle className="text-xl font-bold text-[#1A1A40]">{goal.title}</CardTitle>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-700">{goal.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Strategic Pillars */}
        <div className="mb-16">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Strategic Pillars</h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              The foundation of our strategic approach built on four core pillars of excellence
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {strategicPillars.map((pillar, index) => (
              <Card key={index} className="text-center hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="w-16 h-16 bg-gradient-to-br from-[#1A1A40] to-[#ED1C24] rounded-full flex items-center justify-center mx-auto mb-4">
                    <pillar.icon className="w-8 h-8 text-white" />
                  </div>
                  <CardTitle className="text-lg font-bold text-[#1A1A40]">{pillar.title}</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-600 mb-4">{pillar.description}</p>
                  <div className="space-y-2">
                    {pillar.metrics.map((metric, metricIndex) => (
                      <div key={metricIndex} className="text-sm text-gray-500 bg-gray-50 p-2 rounded">
                        {metric}
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Implementation Timeline */}
        <Card className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <CardHeader>
            <CardTitle className="text-2xl font-bold text-center">Strategic Implementation Timeline</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid md:grid-cols-3 gap-8 text-center">
              <div>
                <div className="text-3xl font-bold text-[#ED1C24] mb-2">2024-2025</div>
                <div className="text-lg font-semibold mb-2">Foundation Phase</div>
                <p className="text-gray-300">Strengthen core capabilities and expand digital infrastructure</p>
              </div>
              <div>
                <div className="text-3xl font-bold text-[#ED1C24] mb-2">2025-2027</div>
                <div className="text-lg font-semibold mb-2">Growth Phase</div>
                <p className="text-gray-300">Scale operations globally and launch revolutionary products</p>
              </div>
              <div>
                <div className="text-3xl font-bold text-[#ED1C24] mb-2">2027-2030</div>
                <div className="text-lg font-semibold mb-2">Leadership Phase</div>
                <p className="text-gray-300">Establish market leadership and shape industry standards</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
    </WebsiteLayout>
  );
}