import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { 
  ShoppingBag, 
  BookOpen, 
  Download, 
  Star, 
  Award,
  GraduationCap,
  Brain,
  Target,
  Users,
  CheckCircle,
  ArrowRight,
  Globe,
  Clock,
  Building2
} from "lucide-react";
import WebsiteLayout from "@/components/WebsiteLayout";

export default function Store() {
  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/store"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <WebsiteLayout>
        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mt-12">
                {[1, 2, 3, 4, 5, 6].map((i) => (
                  <Skeleton key={i} className="h-64 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
      </WebsiteLayout>
    );
  }

  const storeCategories = [
    {
      title: "Professional Qualifications",
      description: "Internationally recognized IBDL certifications for career advancement",
      icon: Award,
      color: "from-blue-500 to-indigo-600",
      products: [
        {
          name: "IBDL Essential",
          description: "Understanding Business - Foundation level certification covering core business principles",
          price: "$1,299",
          duration: "6 months",
          level: "Foundation",
          features: ["8 comprehensive modules", "International examination", "Digital certificate", "Global recognition"],
          bestseller: true
        },
        {
          name: "IBDL Professional", 
          description: "Business Management and Planning - Advanced certification for business professionals",
          price: "$2,499",
          duration: "8 months",
          level: "Advanced",
          features: ["Advanced business strategy", "Leadership development", "Project management", "International certification"]
        },
        {
          name: "IBDL Master",
          description: "Business Leadership and Strategies - Executive level certification for senior leaders",
          price: "$4,999",
          duration: "12 months", 
          level: "Executive",
          features: ["Strategic leadership", "Executive decision making", "Global business perspective", "Master-level certification"]
        }
      ]
    },
    {
      title: "Business Simulations",
      description: "Interactive learning experiences through gamified business scenarios",
      icon: Target,
      color: "from-green-500 to-emerald-600",
      products: [
        {
          name: "Win vs. War",
          description: "Strategic business simulation focusing on competitive advantage and market positioning",
          price: "$599",
          duration: "3 months",
          level: "Intermediate",
          features: ["Real-time simulation", "Team collaboration", "Strategic thinking", "Performance analytics"]
        },
        {
          name: "Master Board Game",
          description: "Executive-level simulation for complex business decision making",
          price: "$899",
          duration: "4 months",
          level: "Advanced", 
          features: ["Executive scenarios", "Multi-dimensional challenges", "Leadership skills", "Strategic planning"]
        },
        {
          name: "Sparta",
          description: "Leadership and team management simulation in challenging environments",
          price: "$699",
          duration: "3 months",
          level: "Intermediate",
          features: ["Leadership challenges", "Team dynamics", "Crisis management", "Decision making"]
        }
      ]
    },
    {
      title: "Assessment Services",
      description: "Professional psychometric and cognitive assessment tools",
      icon: Brain,
      color: "from-purple-500 to-violet-600",
      products: [
        {
          name: "PQP Assessment",
          description: "Professional Qualification Profiler for comprehensive skills evaluation",
          price: "$299",
          duration: "1 month",
          level: "All Levels",
          features: ["Psychometric testing", "Skills profiling", "Career guidance", "Development recommendations"]
        },
        {
          name: "CPAT Assessment",
          description: "Cognitive Performance Assessment Tool for intellectual capacity evaluation",
          price: "$399",
          duration: "1 month", 
          level: "All Levels",
          features: ["Cognitive assessment", "Intelligence profiling", "Aptitude testing", "Performance insights"]
        }
      ]
    },
    {
      title: "Executive Education",
      description: "Elite programs for senior executives and emerging leaders",
      icon: GraduationCap,
      color: "from-red-500 to-rose-600",
      products: [
        {
          name: "Executive Leadership Program",
          description: "Comprehensive leadership development for senior executives",
          price: "$7,999",
          duration: "6 months",
          level: "Executive",
          features: ["Executive coaching", "Strategic leadership", "Global perspective", "Networking opportunities"]
        },
        {
          name: "Women Leadership Program",
          description: "Specialized leadership development program for women executives",
          price: "$6,999",
          duration: "5 months",
          level: "Executive",
          features: ["Gender-specific challenges", "Leadership skills", "Mentorship program", "Professional network"]
        },
        {
          name: "Economic Diplomacy Program",
          description: "Advanced program on international economic relations and diplomacy",
          price: "$8,999",
          duration: "8 months",
          level: "Executive",
          features: ["Economic strategy", "International relations", "Diplomatic skills", "Global economics"]
        }
      ]
    }
  ];

  return (
    <WebsiteLayout>
      <div className="min-h-screen bg-gray-50">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="text-center">
              <Badge variant="secondary" className="mb-4 bg-[#ffffff] text-[#1d1d43]">
                Digital Learning Store
              </Badge>
              <h1 className="text-4xl md:text-5xl font-bold mb-6">
                {(pageData as any)?.title || "IBDL Digital Store"}
              </h1>
              <p className="text-xl text-gray-300 max-w-4xl mx-auto leading-relaxed">
                Explore our comprehensive collection of professional development resources, qualifications, and learning solutions designed to accelerate your career and business success
              </p>
              <div className="mt-8 flex flex-col sm:flex-row gap-4 justify-center">
                <Button 
                  size="lg"
                  className="bg-[#ED1C24] hover:bg-[#CC1820] text-white px-8 py-3"
                >
                  <ShoppingBag className="mr-2 h-5 w-5" />
                  Browse All Products
                </Button>
                <Button 
                  size="lg"
                  variant="outline" 
                  className="border-white text-white bg-transparent hover:bg-white hover:text-[#1A1A40] px-8 py-3"
                >
                  <Download className="mr-2 h-5 w-5" />
                  Download Catalog
                </Button>
              </div>
            </div>
          </div>
        </div>

        {/* Store Categories */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          {storeCategories.map((category, categoryIndex) => (
            <div key={categoryIndex} className="mb-16">
              {/* Category Header */}
              <div className="text-center mb-12">
                <div className={`inline-flex items-center justify-center w-16 h-16 rounded-full bg-gradient-to-r ${category.color} text-white mb-4`}>
                  <category.icon className="h-8 w-8" />
                </div>
                <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">{category.title}</h2>
                <p className="text-xl text-gray-600 max-w-3xl mx-auto">{category.description}</p>
              </div>

              {/* Products Grid */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                {category.products.map((product, productIndex) => (
                  <Card key={productIndex} className="relative overflow-hidden border-0 shadow-lg hover:shadow-xl transition-all duration-300 group">
                    {product.bestseller && (
                      <div className="absolute top-4 right-4 z-10">
                        <Badge className="bg-[#ED1C24] text-white">
                          <Star className="w-3 h-3 mr-1" />
                          Bestseller
                        </Badge>
                      </div>
                    )}
                    
                    <div className={`h-2 bg-gradient-to-r ${category.color}`} />
                    
                    <CardHeader className="pb-4">
                      <div className="flex items-start justify-between">
                        <div>
                          <CardTitle className="text-xl font-bold text-[#1A1A40] mb-2 group-hover:text-[#ED1C24] transition-colors">
                            {product.name}
                          </CardTitle>
                          <Badge variant="outline" className="text-sm">
                            {product.level}
                          </Badge>
                        </div>
                        <div className="text-right">
                          <div className="text-2xl font-bold text-[#ED1C24]">{product.price}</div>
                          <div className="text-sm text-gray-500 flex items-center">
                            <Clock className="w-4 h-4 mr-1" />
                            {product.duration}
                          </div>
                        </div>
                      </div>
                    </CardHeader>

                    <CardContent className="pt-0">
                      <p className="text-gray-600 mb-6 leading-relaxed">
                        {product.description}
                      </p>

                      <div className="space-y-3 mb-6">
                        <h4 className="font-semibold text-[#1A1A40] text-sm uppercase tracking-wide">Key Features</h4>
                        <div className="space-y-2">
                          {product.features.map((feature, featureIndex) => (
                            <div key={featureIndex} className="flex items-center text-sm text-gray-600">
                              <CheckCircle className="w-4 h-4 text-green-500 mr-2 flex-shrink-0" />
                              {feature}
                            </div>
                          ))}
                        </div>
                      </div>

                      <div className="flex gap-3">
                        <Button className="flex-1 bg-[#1A1A40] hover:bg-[#2A2A50] text-white">
                          <ShoppingBag className="w-4 h-4 mr-2" />
                          Add to Cart
                        </Button>
                        <Button variant="outline" size="sm" className="border-[#1A1A40] text-[#1A1A40] hover:bg-[#1A1A40] hover:text-white">
                          <ArrowRight className="w-4 h-4" />
                        </Button>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>
          ))}
        </div>

        {/* Why Choose IBDL Store */}
        <div className="bg-white py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Why Choose IBDL Store?</h2>
              <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                Your trusted partner for professional development and business excellence
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
              {[
                {
                  icon: Globe,
                  title: "Global Recognition",
                  description: "Internationally recognized certifications accepted worldwide"
                },
                {
                  icon: Award,
                  title: "University Partnerships",
                  description: "Collaborations with Missouri State University and leading institutions"
                },
                {
                  icon: Users,
                  title: "Expert Faculty",
                  description: "Learn from industry experts and certified professionals"
                },
                {
                  icon: Building2,
                  title: "Corporate Solutions",
                  description: "Customized learning solutions for organizations and teams"
                }
              ].map((benefit, index) => (
                <Card key={index} className="text-center border-0 shadow-md hover:shadow-lg transition-shadow">
                  <CardContent className="pt-8 pb-6">
                    <div className="inline-flex items-center justify-center w-12 h-12 rounded-full bg-[#ED1C24] text-white mb-4">
                      <benefit.icon className="h-6 w-6" />
                    </div>
                    <h3 className="text-lg font-semibold text-[#1A1A40] mb-2">{benefit.title}</h3>
                    <p className="text-gray-600 text-sm">{benefit.description}</p>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>

        {/* CMS Integration Notice */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 text-center">
            <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
            <p className="text-blue-700">
              This page content is managed through the CMS and can be updated by administrators.
              {pageData && (
                <span className="block mt-2 text-sm">
                  Page ID: {(pageData as any).id} | Status: {(pageData as any).status}
                </span>
              )}
            </p>
          </div>
        </div>
      </div>
    </WebsiteLayout>
  );
}