import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { 
  BookOpen, 
  Download, 
  FileText, 
  BarChart3,
  TrendingUp,
  Users,
  Globe,
  Award,
  Building2,
  Brain,
  Target,
  ArrowRight,
  Star,
  Clock,
  Search
} from "lucide-react";
import WebsiteLayout from "@/components/WebsiteLayout";

export default function Resources() {
  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/resources"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <WebsiteLayout>
        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mt-12">
                {[1, 2, 3, 4, 5, 6].map((i) => (
                  <Skeleton key={i} className="h-64 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
      </WebsiteLayout>
    );
  }

  const resourceCategories = [
    {
      title: "Research Papers",
      description: "In-depth academic research on business education and professional development",
      icon: FileText,
      color: "from-blue-500 to-indigo-600",
      resources: [
        {
          title: "The Impact of Digital Learning on Professional Development: A Global Study",
          description: "Comprehensive research analyzing the effectiveness of digital learning platforms in corporate training environments across 25 countries.",
          type: "Research Paper",
          pages: "45 pages",
          published: "December 2024",
          downloadCount: "2,847",
          featured: true
        },
        {
          title: "Business Leadership in the 21st Century: Emerging Trends and Competencies",
          description: "Academic study examining the evolving leadership competencies required for successful business management in the digital age.",
          type: "Research Paper",
          pages: "32 pages",
          published: "November 2024",
          downloadCount: "1,923"
        },
        {
          title: "Psychometric Assessment in Professional Development: Best Practices",
          description: "Research paper exploring the application of psychometric testing in corporate talent development and career advancement programs.",
          type: "Research Paper",
          pages: "28 pages",
          published: "October 2024",
          downloadCount: "1,567"
        }
      ]
    },
    {
      title: "White Papers",
      description: "Expert insights and industry analysis on key business topics",
      icon: BookOpen,
      color: "from-green-500 to-emerald-600",
      resources: [
        {
          title: "Executive Education Transformation: Preparing Leaders for Tomorrow",
          description: "Strategic analysis of how executive education programs are evolving to meet the demands of modern business challenges.",
          type: "White Paper",
          pages: "24 pages",
          published: "January 2025",
          downloadCount: "3,245",
          featured: true
        },
        {
          title: "The ROI of Professional Certification Programs",
          description: "Comprehensive analysis of the return on investment for organizations implementing professional certification programs for their workforce.",
          type: "White Paper",
          pages: "18 pages",
          published: "December 2024",
          downloadCount: "2,156"
        },
        {
          title: "Women in Leadership: Breaking Through the Glass Ceiling",
          description: "Industry insights on advancing women in leadership positions and the business impact of gender-diverse leadership teams.",
          type: "White Paper",
          pages: "22 pages",
          published: "November 2024",
          downloadCount: "2,834"
        }
      ]
    },
    {
      title: "Industry Reports",
      description: "Comprehensive market analysis and industry trend reports",
      icon: BarChart3,
      color: "from-purple-500 to-violet-600",
      resources: [
        {
          title: "Global Business Education Market Report 2025",
          description: "Comprehensive market analysis covering trends, growth drivers, and future projections for the international business education sector.",
          type: "Industry Report",
          pages: "89 pages",
          published: "January 2025",
          downloadCount: "4,523",
          featured: true
        },
        {
          title: "Corporate Learning & Development Trends Report",
          description: "Annual report analyzing emerging trends in corporate training, employee development, and organizational learning strategies.",
          type: "Industry Report",
          pages: "67 pages",
          published: "December 2024",
          downloadCount: "3,678"
        },
        {
          title: "Middle East Professional Development Market Analysis",
          description: "Regional market report focusing on professional development trends, opportunities, and challenges in the Middle East market.",
          type: "Industry Report",
          pages: "54 pages",
          published: "November 2024",
          downloadCount: "2,945"
        }
      ]
    },
    {
      title: "Best Practice Guides",
      description: "Practical guides and implementation frameworks for professional excellence",
      icon: Target,
      color: "from-red-500 to-rose-600",
      resources: [
        {
          title: "Implementing IBDL Qualifications: A Step-by-Step Guide",
          description: "Comprehensive implementation guide for organizations looking to integrate IBDL qualification programs into their professional development framework.",
          type: "Best Practice Guide",
          pages: "36 pages",
          published: "December 2024",
          downloadCount: "2,367"
        },
        {
          title: "Building Effective Learning Pathways for Career Advancement",
          description: "Strategic guide for HR professionals on designing personalized learning journeys that align with career progression and organizational goals.",
          type: "Best Practice Guide",
          pages: "29 pages",
          published: "November 2024",
          downloadCount: "1,892"
        },
        {
          title: "Executive Coaching Excellence: Methodologies and Frameworks",
          description: "Professional guide covering proven methodologies for executive coaching, leadership development, and performance enhancement strategies.",
          type: "Best Practice Guide",
          pages: "41 pages",
          published: "October 2024",
          downloadCount: "2,134"
        }
      ]
    }
  ];

  const featuredTools = [
    {
      title: "IBDL Competency Assessment Tool",
      description: "Interactive assessment platform for evaluating professional competencies across multiple business domains",
      type: "Assessment Tool",
      icon: Brain
    },
    {
      title: "Learning Path Generator",
      description: "AI-powered tool for creating personalized learning journeys based on career goals and current skill levels",
      type: "Planning Tool",
      icon: Target
    },
    {
      title: "ROI Calculator for Training Programs",
      description: "Calculate the return on investment for professional development and training initiatives",
      type: "Business Tool",
      icon: BarChart3
    }
  ];

  return (
    <WebsiteLayout>
      <div className="min-h-screen bg-gray-50">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="text-center">
              <Badge variant="secondary" className="mb-4 text-[#1c1c42] bg-[#ffffff]">
                Knowledge Center
              </Badge>
              <h1 className="text-4xl md:text-5xl font-bold mb-6">
                {(pageData as any)?.title || "Learning Resources"}
              </h1>
              <p className="text-xl text-gray-300 max-w-4xl mx-auto leading-relaxed">
                Access our comprehensive collection of research papers, industry reports, white papers, and best practice guides to enhance your professional knowledge and business expertise
              </p>
              <div className="mt-8 flex flex-col sm:flex-row gap-4 justify-center max-w-md mx-auto">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-5 w-5" />
                  <input
                    type="text"
                    placeholder="Search resources..."
                    className="w-full pl-10 pr-4 py-3 rounded-lg border-0 text-gray-900 placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-[#ED1C24]"
                  />
                </div>
                <Button 
                  size="lg"
                  className="bg-[#ED1C24] hover:bg-[#CC1820] text-white px-8 py-3"
                >
                  Search
                </Button>
              </div>
            </div>
          </div>
        </div>

        {/* Resource Categories */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          {resourceCategories.map((category, categoryIndex) => (
            <div key={categoryIndex} className="mb-16">
              {/* Category Header */}
              <div className="text-center mb-12">
                <div className={`inline-flex items-center justify-center w-16 h-16 rounded-full bg-gradient-to-r ${category.color} text-white mb-4`}>
                  <category.icon className="h-8 w-8" />
                </div>
                <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">{category.title}</h2>
                <p className="text-xl text-gray-600 max-w-3xl mx-auto">{category.description}</p>
              </div>

              {/* Resources Grid */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                {category.resources.map((resource, resourceIndex) => (
                  <Card key={resourceIndex} className="relative overflow-hidden border-0 shadow-lg hover:shadow-xl transition-all duration-300 group">
                    {resource.featured && (
                      <div className="absolute top-4 right-4 z-10">
                        <Badge className="bg-[#ED1C24] text-white">
                          <Star className="w-3 h-3 mr-1" />
                          Featured
                        </Badge>
                      </div>
                    )}
                    
                    <div className={`h-2 bg-gradient-to-r ${category.color}`} />
                    
                    <CardHeader className="pb-4">
                      <div className="flex items-start justify-between mb-2">
                        <Badge variant="outline" className="text-xs">
                          {resource.type}
                        </Badge>
                        <div className="text-xs text-gray-500">{resource.published}</div>
                      </div>
                      <CardTitle className="text-xl font-bold text-[#1A1A40] mb-2 group-hover:text-[#ED1C24] transition-colors">
                        {resource.title}
                      </CardTitle>
                    </CardHeader>

                    <CardContent className="pt-0">
                      <p className="text-gray-600 mb-6 leading-relaxed line-clamp-3">
                        {resource.description}
                      </p>

                      <div className="flex items-center justify-between text-sm text-gray-500 mb-6">
                        <div className="flex items-center">
                          <FileText className="w-4 h-4 mr-1" />
                          {resource.pages}
                        </div>
                        <div className="flex items-center">
                          <Download className="w-4 h-4 mr-1" />
                          {resource.downloadCount} downloads
                        </div>
                      </div>

                      <div className="flex gap-3">
                        <Button className="flex-1 bg-[#1A1A40] hover:bg-[#2A2A50] text-white">
                          <Download className="w-4 h-4 mr-2" />
                          Download PDF
                        </Button>
                        <Button variant="outline" size="sm" className="border-[#1A1A40] text-[#1A1A40] hover:bg-[#1A1A40] hover:text-white">
                          <ArrowRight className="w-4 h-4" />
                        </Button>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>
          ))}
        </div>

        {/* Featured Tools Section */}
        <div className="bg-white py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Professional Development Tools</h2>
              <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                Interactive tools to enhance your learning journey and professional growth
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              {featuredTools.map((tool, index) => (
                <Card key={index} className="text-center border-0 shadow-lg hover:shadow-xl transition-all duration-300 group">
                  <CardContent className="pt-8 pb-6">
                    <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white mb-4 group-hover:scale-110 transition-transform">
                      <tool.icon className="h-8 w-8" />
                    </div>
                    <h3 className="text-xl font-bold text-[#1A1A40] mb-2 group-hover:text-[#ED1C24] transition-colors">
                      {tool.title}
                    </h3>
                    <Badge variant="outline" className="mb-4 text-xs">
                      {tool.type}
                    </Badge>
                    <p className="text-gray-600 mb-6 leading-relaxed">
                      {tool.description}
                    </p>
                    <Button className="w-full bg-[#ED1C24] hover:bg-[#CC1820] text-white">
                      Launch Tool
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </Button>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>

        {/* Newsletter Subscription */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] py-16">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
            <h2 className="text-3xl font-bold text-white mb-4">Stay Updated with New Resources</h2>
            <p className="text-xl text-gray-300 mb-8">
              Subscribe to receive notifications when new research papers, reports, and guides are published
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center max-w-md mx-auto">
              <input
                type="email"
                placeholder="Enter your email"
                className="flex-1 px-4 py-3 rounded-lg border-0 text-gray-900 placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-[#ED1C24]"
              />
              <Button className="bg-[#ED1C24] hover:bg-[#CC1820] text-white px-8 py-3">
                Subscribe
              </Button>
            </div>
          </div>
        </div>

        {/* CMS Integration Notice */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 text-center">
            <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
            <p className="text-blue-700">
              This page content is managed through the CMS and can be updated by administrators.
              {pageData && (
                <span className="block mt-2 text-sm">
                  Page ID: {(pageData as any).id} | Status: {(pageData as any).status}
                </span>
              )}
            </p>
          </div>
        </div>
      </div>
    </WebsiteLayout>
  );
}