import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { 
  Plus, 
  Edit, 
  Trash2, 
  CheckCircle, 
  XCircle, 
  Clock, 
  AlertCircle,
  Eye,
  FileText,
  Building,
  Users,
  GraduationCap,
  Award
} from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { 
  type AccreditationApplication,
  type User
} from "@shared/schema";

export default function QAAccreditation() {
  const [activeTab, setActiveTab] = useState("training_center");
  const [selectedApplication, setSelectedApplication] = useState<AccreditationApplication | null>(null);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch applications
  const { data: applications = [], isLoading } = useQuery({
    queryKey: ["backend/api/accreditation/applications"],
  });

  const { data: users = [] } = useQuery({
    queryKey: ["backend/api/admin/users"],
  });

  // Filter applications by type
  const trainingCenterApps = (applications as AccreditationApplication[]).filter((app: AccreditationApplication) => 
    app.applicationType === "training_center"
  );
  const testingCenterApps = (applications as AccreditationApplication[]).filter((app: AccreditationApplication) => 
    app.applicationType === "testing_center"
  );
  const trainerApps = (applications as AccreditationApplication[]).filter((app: AccreditationApplication) => 
    app.applicationType === "trainer"
  );
  const corporateApps = (applications as AccreditationApplication[]).filter((app: AccreditationApplication) => 
    app.applicationType === "corporate"
  );

  // Update status mutation
  const updateStatusMutation = useMutation({
    mutationFn: ({ 
      id, 
      status, 
      comments, 
      qaCheckpoint1, 
      qaCheckpoint2, 
      qaCheckpoint3 
    }: { 
      id: string; 
      status: string; 
      comments: string;
      qaCheckpoint1?: boolean;
      qaCheckpoint2?: boolean;
      qaCheckpoint3?: boolean;
    }) => 
      apiRequest("backend/api/accreditation/update-status", "POST", { 
        id, 
        status, 
        comments, 
        qaCheckpoint1, 
        qaCheckpoint2, 
        qaCheckpoint3 
      }),
    onSuccess: () => {
      toast({ title: "Application status updated successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/accreditation/applications"] });
      setSelectedApplication(null);
    },
    onError: (error) => {
      toast({ title: "Error updating status", description: error.message, variant: "destructive" });
    },
  });

  const getStatusColor = (status: string) => {
    switch (status) {
      case "approved":
        return "bg-green-100 text-green-800";
      case "rejected":
        return "bg-red-100 text-red-800";
      case "pending":
        return "bg-yellow-100 text-yellow-800";
      case "under_review":
        return "bg-blue-100 text-blue-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "approved":
        return <CheckCircle className="w-4 h-4 text-green-600" />;
      case "rejected":
        return <XCircle className="w-4 h-4 text-red-600" />;
      case "pending":
        return <Clock className="w-4 h-4 text-yellow-600" />;
      case "under_review":
        return <AlertCircle className="w-4 h-4 text-blue-600" />;
      default:
        return <Clock className="w-4 h-4 text-gray-600" />;
    }
  };

  const QAChecklistIndicator = ({ application }: { application: any }) => {
    const checkpoint1 = application.qaCheckpoint1 || false;
    const checkpoint2 = application.qaCheckpoint2 || false;
    const checkpoint3 = application.qaCheckpoint3 || false;

    return (
      <div className="flex items-center gap-1 mt-2">
        <span className="text-xs text-muted-foreground">QA Progress:</span>
        <div className="flex gap-1">
          <div className={`w-3 h-3 rounded-full ${checkpoint1 ? 'bg-green-500' : 'bg-gray-300'}`} title="Document Verification" />
          <div className={`w-3 h-3 rounded-full ${checkpoint2 ? 'bg-green-500' : 'bg-gray-300'}`} title="Facility Assessment" />
          <div className={`w-3 h-3 rounded-full ${checkpoint3 ? 'bg-green-500' : 'bg-gray-300'}`} title="Final Approval Ready" />
        </div>
      </div>
    );
  };

  const ApplicationCard = ({ application }: { application: AccreditationApplication }) => (
    <Card className="hover:shadow-lg transition-shadow cursor-pointer" onClick={() => setSelectedApplication(application)}>
      <CardHeader>
        <div className="flex justify-between items-start">
          <div>
            <CardTitle className="text-lg">{application.organizationName}</CardTitle>
            <CardDescription className="flex items-center gap-2 mt-1">
              {getStatusIcon(application.status)}
              <span className="capitalize">{application.status.replace('_', ' ')}</span>
            </CardDescription>
            <QAChecklistIndicator application={application} />
          </div>
          <Badge className={getStatusColor(application.status)}>
            {application.status.replace('_', ' ')}
          </Badge>
        </div>
      </CardHeader>
      <CardContent>
        <div className="space-y-2">
          <div className="flex justify-between text-sm">
            <span>Contact:</span>
            <span>{application.contactPerson}</span>
          </div>
          <div className="flex justify-between text-sm">
            <span>Email:</span>
            <span>{application.contactEmail}</span>
          </div>
          <div className="flex justify-between text-sm">
            <span>Location:</span>
            <span>{application.city}, {application.country}</span>
          </div>
          <div className="flex justify-between text-sm">
            <span>Submitted:</span>
            <span>{new Date(application.submittedAt).toLocaleDateString()}</span>
          </div>
        </div>
      </CardContent>
    </Card>
  );

  const ReviewDialog = ({ application, onClose }: { application: AccreditationApplication; onClose: () => void }) => {
    const [status, setStatus] = useState(application.status);
    const [comments, setComments] = useState("");
    const [qaCheckpoint1, setQaCheckpoint1] = useState((application as any)?.qaCheckpoint1 || false);
    const [qaCheckpoint2, setQaCheckpoint2] = useState((application as any)?.qaCheckpoint2 || false);
    const [qaCheckpoint3, setQaCheckpoint3] = useState((application as any)?.qaCheckpoint3 || false);

    const handleSubmit = () => {
      updateStatusMutation.mutate({ 
        id: application.id, 
        status, 
        comments,
        qaCheckpoint1,
        qaCheckpoint2,
        qaCheckpoint3
      });
    };

    return (
      <Dialog open={true} onOpenChange={onClose}>
        <DialogContent className="max-w-4xl">
          <DialogHeader>
            <DialogTitle>Review Application - {application.organizationName}</DialogTitle>
          </DialogHeader>
          <div className="space-y-6">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label>Organization Name</Label>
                <p className="text-sm text-muted-foreground">{application.organizationName}</p>
              </div>
              <div>
                <Label>Application Type</Label>
                <p className="text-sm text-muted-foreground capitalize">{application.applicationType.replace('_', ' ')}</p>
              </div>
              <div>
                <Label>Contact Name</Label>
                <p className="text-sm text-muted-foreground">{application.contactName}</p>
              </div>
              <div>
                <Label>Contact Email</Label>
                <p className="text-sm text-muted-foreground">{application.contactEmail}</p>
              </div>
              <div>
                <Label>Phone</Label>
                <p className="text-sm text-muted-foreground">{application.contactPhone}</p>
              </div>
              <div>
                <Label>Location</Label>
                <p className="text-sm text-muted-foreground">{application.location}</p>
              </div>
            </div>

            <div>
              <Label>Description</Label>
              <p className="text-sm text-muted-foreground mt-1">{application.description}</p>
            </div>

            {application.supportingDocuments && (
              <div>
                <Label>Supporting Documents</Label>
                <div className="mt-2 space-y-2">
                  {Object.entries(application.supportingDocuments).map(([key, value]) => (
                    <div key={key} className="flex items-center gap-2">
                      <FileText className="w-4 h-4" />
                      <span className="text-sm">{key}: {value as string}</span>
                    </div>
                  ))}
                </div>
              </div>
            )}

            <div className="space-y-4">
              <div>
                <Label>Update Status</Label>
                <Select value={status} onValueChange={setStatus}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="pending">Pending</SelectItem>
                    <SelectItem value="under_review">Under Review</SelectItem>
                    <SelectItem value="approved">Approved</SelectItem>
                    <SelectItem value="rejected">Rejected</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-3">
                <Label className="text-base font-semibold">QA Review Checklist</Label>
                <div className="grid grid-cols-3 gap-4">
                  <div className="flex items-center space-x-2">
                    <Checkbox 
                      id="checkpoint1" 
                      checked={qaCheckpoint1}
                      onCheckedChange={setQaCheckpoint1}
                    />
                    <Label htmlFor="checkpoint1" className="text-sm">
                      Document Verification
                    </Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Checkbox 
                      id="checkpoint2" 
                      checked={qaCheckpoint2}
                      onCheckedChange={setQaCheckpoint2}
                    />
                    <Label htmlFor="checkpoint2" className="text-sm">
                      Facility Assessment
                    </Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Checkbox 
                      id="checkpoint3" 
                      checked={qaCheckpoint3}
                      onCheckedChange={setQaCheckpoint3}
                    />
                    <Label htmlFor="checkpoint3" className="text-sm">
                      Final Approval Ready
                    </Label>
                  </div>
                </div>
              </div>
              
              <div>
                <Label>Comments</Label>
                <Textarea
                  value={comments}
                  onChange={(e) => setComments(e.target.value)}
                  placeholder="Add comments about the application review..."
                  className="mt-1"
                />
              </div>
            </div>

            <div className="flex justify-end gap-2">
              <Button variant="outline" onClick={onClose}>
                Cancel
              </Button>
              <Button onClick={handleSubmit} disabled={updateStatusMutation.isPending}>
                {updateStatusMutation.isPending ? "Updating..." : "Update Status"}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    );
  };

  return (
    <DashboardLayout userType="qa">
      <div className="space-y-8">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">QA - Accreditation Process</h1>
          <p className="text-muted-foreground">
            Review and approve accreditation applications for training centers, testing centers, trainers, and corporate clients
          </p>
        </div>

        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Training Centers</CardTitle>
              <Building className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{trainingCenterApps.length}</div>
              <p className="text-xs text-muted-foreground">
                {trainingCenterApps.filter((app: any) => app.status === "pending").length} pending
              </p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Testing Centers</CardTitle>
              <Award className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{testingCenterApps.length}</div>
              <p className="text-xs text-muted-foreground">
                {testingCenterApps.filter((app: any) => app.status === "pending").length} pending
              </p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Trainers</CardTitle>
              <GraduationCap className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{trainerApps.length}</div>
              <p className="text-xs text-muted-foreground">
                {trainerApps.filter((app: any) => app.status === "pending").length} pending
              </p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Corporate</CardTitle>
              <Users className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{corporateApps.length}</div>
              <p className="text-xs text-muted-foreground">
                {corporateApps.filter((app: any) => app.status === "pending").length} pending
              </p>
            </CardContent>
          </Card>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="training_center" className="flex items-center gap-2">
              <Building className="w-4 h-4" />
              Training Centers
            </TabsTrigger>
            <TabsTrigger value="testing_center" className="flex items-center gap-2">
              <Award className="w-4 h-4" />
              Testing Centers
            </TabsTrigger>
            <TabsTrigger value="trainer" className="flex items-center gap-2">
              <GraduationCap className="w-4 h-4" />
              Trainers
            </TabsTrigger>
            <TabsTrigger value="corporate" className="flex items-center gap-2">
              <Users className="w-4 h-4" />
              Corporate
            </TabsTrigger>
          </TabsList>

          <TabsContent value="training_center" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Training Center Applications</h2>
              <div className="flex gap-2">
                <Button variant="outline">
                  <FileText className="w-4 h-4 mr-2" />
                  Export List
                </Button>
                <Button variant="outline">
                  <AlertCircle className="w-4 h-4 mr-2" />
                  Pending Only
                </Button>
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {trainingCenterApps.map((application) => (
                <ApplicationCard key={application.id} application={application} />
              ))}
            </div>
          </TabsContent>

          <TabsContent value="testing_center" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Testing Center Applications</h2>
              <div className="flex gap-2">
                <Button variant="outline">
                  <FileText className="w-4 h-4 mr-2" />
                  Export List
                </Button>
                <Button variant="outline">
                  <AlertCircle className="w-4 h-4 mr-2" />
                  Pending Only
                </Button>
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {testingCenterApps.map((application) => (
                <ApplicationCard key={application.id} application={application} />
              ))}
            </div>
          </TabsContent>

          <TabsContent value="trainer" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Trainer Applications</h2>
              <div className="flex gap-2">
                <Button variant="outline">
                  <FileText className="w-4 h-4 mr-2" />
                  Export List
                </Button>
                <Button variant="outline">
                  <AlertCircle className="w-4 h-4 mr-2" />
                  Pending Only
                </Button>
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {trainerApps.map((application) => (
                <ApplicationCard key={application.id} application={application} />
              ))}
            </div>
          </TabsContent>

          <TabsContent value="corporate" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Corporate Applications</h2>
              <div className="flex gap-2">
                <Button variant="outline">
                  <FileText className="w-4 h-4 mr-2" />
                  Export List
                </Button>
                <Button variant="outline">
                  <AlertCircle className="w-4 h-4 mr-2" />
                  Pending Only
                </Button>
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {corporateApps.map((application) => (
                <ApplicationCard key={application.id} application={application} />
              ))}
            </div>
          </TabsContent>
        </Tabs>

        {selectedApplication && (
          <ReviewDialog 
            application={selectedApplication} 
            onClose={() => setSelectedApplication(null)} 
          />
        )}
      </div>
    </DashboardLayout>
  );
}