import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { 
  Calendar, 
  Clock, 
  ArrowRight, 
  Newspaper,
  Users,
  Building2,
  Globe,
  Award,
  BookOpen,
  TrendingUp,
  Briefcase,
  Star
} from "lucide-react";
import WebsiteLayout from "@/components/WebsiteLayout";

export default function News() {
  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/news"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <WebsiteLayout>
        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mt-12">
                {[1, 2, 3, 4, 5, 6].map((i) => (
                  <Skeleton key={i} className="h-64 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
      </WebsiteLayout>
    );
  }

  const newsCategories = [
    {
      title: "Latest News",
      articles: [
        {
          title: "IBDL Learning Group Celebrates 20th Anniversary with Global Recognition",
          excerpt: "IBDL Learning Group marks two decades of excellence in international business education, serving over 222,000 learners across 39 countries with innovative learning solutions.",
          date: "January 15, 2025",
          category: "Company News",
          readTime: "3 min read",
          featured: true,
          image: "backend/api/placeholder/400/250"
        },
        {
          title: "New Partnership with Missouri State University Expands Global Reach",
          excerpt: "Strategic collaboration enhances IBDL's academic credentials and brings internationally recognized business education to more professionals worldwide.",
          date: "December 10, 2024",
          category: "Partnerships",
          readTime: "4 min read",
          image: "backend/api/placeholder/400/250"
        },
        {
          title: "IBDL Master Certification Now Available in Digital Format",
          excerpt: "Executive-level Business Leadership and Strategies certification embraces digital transformation with enhanced interactive content and virtual assessments.",
          date: "November 28, 2024",
          category: "Product Updates",
          readTime: "2 min read",
          image: "backend/api/placeholder/400/250"
        }
      ]
    },
    {
      title: "Industry Insights",
      articles: [
        {
          title: "The Future of Business Education: Digital Learning Trends for 2025",
          excerpt: "Exploring how AI, VR, and personalized learning platforms are revolutionizing professional development and corporate training programs.",
          date: "January 8, 2025",
          category: "Industry Analysis",
          readTime: "6 min read",
          image: "backend/api/placeholder/400/250"
        },
        {
          title: "Executive Leadership in the Post-Digital Era",
          excerpt: "Key insights from IBDL's Executive Leadership Program on navigating business challenges in an increasingly connected world.",
          date: "December 20, 2024",
          category: "Leadership",
          readTime: "5 min read",
          image: "backend/api/placeholder/400/250"
        },
        {
          title: "Women in Leadership: Breaking Barriers in Global Business",
          excerpt: "Success stories and insights from participants in IBDL's Women Leadership Program, showcasing the impact of gender-inclusive leadership development.",
          date: "December 5, 2024",
          category: "Diversity & Inclusion",
          readTime: "4 min read",
          image: "backend/api/placeholder/400/250"
        }
      ]
    },
    {
      title: "Success Stories",
      articles: [
        {
          title: "From IBDL Essential to C-Suite: A Professional Journey",
          excerpt: "Meet Sarah Ahmed, who started with IBDL Essential certification and now leads a multinational corporation, crediting IBDL's comprehensive learning pathway.",
          date: "December 15, 2024",
          category: "Alumni Success",
          readTime: "3 min read",
          image: "backend/api/placeholder/400/250"
        },
        {
          title: "Corporate Transformation Through IBDL Training Solutions",
          excerpt: "How Global Tech Solutions increased employee performance by 40% after implementing IBDL's customized corporate learning programs.",
          date: "November 30, 2024",
          category: "Case Study",
          readTime: "5 min read",
          image: "backend/api/placeholder/400/250"
        },
        {
          title: "Training Center Excellence: Regional Provider Spotlight",
          excerpt: "Celebrating outstanding IBDL training centers across the Middle East and their contribution to professional development in their regions.",
          date: "November 10, 2024",
          category: "Partner Recognition",
          readTime: "4 min read",
          image: "backend/api/placeholder/400/250"
        }
      ]
    }
  ];

  const upcomingEvents = [
    {
      title: "Global Business Leaders Summit 2025",
      date: "March 15-17, 2025",
      location: "Dubai, UAE",
      type: "Conference",
      description: "Join industry leaders for three days of networking, learning, and strategic insights into the future of global business."
    },
    {
      title: "Women Leadership Excellence Workshop",
      date: "February 28, 2025",
      location: "Virtual Event",
      type: "Workshop",
      description: "Interactive workshop focusing on leadership strategies and career advancement for women professionals."
    },
    {
      title: "Digital Transformation in Education Webinar",
      date: "February 10, 2025",
      location: "Online",
      type: "Webinar",
      description: "Exploring the latest trends and technologies shaping the future of professional education and training."
    }
  ];

  return (
    <WebsiteLayout>
      <div className="min-h-screen bg-gray-50">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="text-center">
              <Badge variant="secondary" className="mb-4 bg-[#ffffff] text-[#1d1d43]">
                Media Center
              </Badge>
              <h1 className="text-4xl md:text-5xl font-bold mb-6">
                {(pageData as any)?.title || "News & Updates"}
              </h1>
              <p className="text-xl text-gray-300 max-w-4xl mx-auto leading-relaxed">
                Stay informed with the latest news, industry insights, and success stories from IBDL Learning Group's global community
              </p>
            </div>
          </div>
        </div>

        {/* Featured News */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          {newsCategories.map((category, categoryIndex) => (
            <div key={categoryIndex} className="mb-16">
              <div className="flex items-center justify-between mb-8">
                <h2 className="text-3xl font-bold text-[#1A1A40]">{category.title}</h2>
                <Button variant="outline" className="border-[#1A1A40] text-[#1A1A40] hover:bg-[#1A1A40] hover:text-white">
                  View All
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Button>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                {category.articles.map((article, articleIndex) => (
                  <Card key={articleIndex} className={`overflow-hidden border-0 shadow-lg hover:shadow-xl transition-all duration-300 group ${article.featured ? 'md:col-span-2 lg:col-span-2' : ''}`}>
                    <div className="aspect-w-16 aspect-h-9 bg-gray-200">
                      <div className="w-full h-48 bg-gradient-to-br from-[#1A1A40] to-[#ED1C24] flex items-center justify-center">
                        <Newspaper className="h-12 w-12 text-white" />
                      </div>
                    </div>
                    
                    <CardHeader className="pb-4">
                      <div className="flex items-center justify-between mb-2">
                        <Badge variant="outline" className="text-xs">
                          {article.category}
                        </Badge>
                        {article.featured && (
                          <Badge className="bg-[#ED1C24] text-white text-xs">
                            <Star className="w-3 h-3 mr-1" />
                            Featured
                          </Badge>
                        )}
                      </div>
                      <CardTitle className={`${article.featured ? 'text-2xl' : 'text-xl'} font-bold text-[#1A1A40] mb-2 group-hover:text-[#ED1C24] transition-colors line-clamp-2`}>
                        {article.title}
                      </CardTitle>
                      <div className="flex items-center text-sm text-gray-500 space-x-4">
                        <div className="flex items-center">
                          <Calendar className="w-4 h-4 mr-1" />
                          {article.date}
                        </div>
                        <div className="flex items-center">
                          <Clock className="w-4 h-4 mr-1" />
                          {article.readTime}
                        </div>
                      </div>
                    </CardHeader>

                    <CardContent className="pt-0">
                      <p className="text-gray-600 mb-6 leading-relaxed line-clamp-3">
                        {article.excerpt}
                      </p>
                      <Button className="w-full bg-[#1A1A40] hover:bg-[#2A2A50] text-white">
                        Read Full Article
                        <ArrowRight className="ml-2 h-4 w-4" />
                      </Button>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>
          ))}
        </div>

        {/* Upcoming Events */}
        <div className="bg-white py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Upcoming Events</h2>
              <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                Join us at our upcoming events and connect with industry leaders
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
              {upcomingEvents.map((event, index) => (
                <Card key={index} className="border-0 shadow-lg hover:shadow-xl transition-shadow">
                  <CardHeader className="pb-4">
                    <div className="flex items-center justify-between mb-2">
                      <Badge className="bg-[#ED1C24] text-white">
                        {event.type}
                      </Badge>
                      <div className="text-sm text-gray-500">{event.location}</div>
                    </div>
                    <CardTitle className="text-xl font-bold text-[#1A1A40] mb-2">
                      {event.title}
                    </CardTitle>
                    <div className="flex items-center text-sm text-gray-500">
                      <Calendar className="w-4 h-4 mr-1" />
                      {event.date}
                    </div>
                  </CardHeader>

                  <CardContent className="pt-0">
                    <p className="text-gray-600 mb-6 leading-relaxed">
                      {event.description}
                    </p>
                    <Button className="w-full bg-[#1A1A40] hover:bg-[#2A2A50] text-white">
                      Register Now
                    </Button>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>

        {/* Newsletter Subscription */}
        <div className="bg-gradient-to-r from-[#ED1C24] to-[#CC1820] py-16">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
            <h2 className="text-3xl font-bold text-white mb-4">Stay Updated</h2>
            <p className="text-xl text-red-100 mb-8">
              Subscribe to our newsletter for the latest news, insights, and exclusive content
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center max-w-md mx-auto">
              <input
                type="email"
                placeholder="Enter your email"
                className="flex-1 px-4 py-3 rounded-lg border-0 text-gray-900 placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-white"
              />
              <Button className="bg-white text-[#ED1C24] hover:bg-gray-100 px-8 py-3">
                Subscribe
              </Button>
            </div>
          </div>
        </div>

        {/* CMS Integration Notice */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 text-center">
            <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
            <p className="text-blue-700">
              This page content is managed through the CMS and can be updated by administrators.
              {pageData && (
                <span className="block mt-2 text-sm">
                  Page ID: {(pageData as any).id} | Status: {(pageData as any).status}
                </span>
              )}
            </p>
          </div>
        </div>
      </div>
    </WebsiteLayout>
  );
}