import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { 
  BookOpen, 
  Calendar, 
  Clock, 
  ArrowRight, 
  Download,
  Users,
  Globe,
  TrendingUp,
  Brain,
  Building2,
  Star,
  FileText,
  Quote,
  Book,
  Mail
} from "lucide-react";
import WebsiteLayout from "@/components/WebsiteLayout";

export default function Magazine() {
  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/magazine"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <WebsiteLayout>
        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mt-12">
                {[1, 2, 3, 4, 5, 6].map((i) => (
                  <Skeleton key={i} className="h-64 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
      </WebsiteLayout>
    );
  }

  const currentIssue = {
    title: "The Learning Edit - Volume 5, Issue 1",
    subtitle: "Digital Transformation in Business Education",
    coverStory: "How AI and Virtual Reality are Reshaping Professional Development",
    publishDate: "January 2025",
    issueNumber: "Issue 17",
    coverImage: "backend/api/placeholder/400/600"
  };

  const featuredArticles = [
    {
      title: "The Future of Executive Leadership in a Digital World",
      author: "Dr. Sarah Mitchell, Executive Education Director",
      excerpt: "Exploring how digital transformation is reshaping leadership competencies and executive decision-making processes in the modern business landscape.",
      category: "Leadership",
      readTime: "8 min read",
      featured: true,
      image: "backend/api/placeholder/400/250"
    },
    {
      title: "Women Breaking Barriers: Success Stories from IBDL Alumni",
      author: "Maria Rodriguez, Director of Diversity Programs",
      excerpt: "Inspiring stories of women who leveraged IBDL certifications to advance their careers and make significant impacts in their organizations.",
      category: "Success Stories",
      readTime: "6 min read",
      image: "backend/api/placeholder/400/250"
    },
    {
      title: "The ROI of Professional Development: A Data-Driven Analysis",
      author: "Dr. James Thompson, Research Director",
      excerpt: "Comprehensive analysis of how professional development investments translate into measurable business outcomes and career advancement.",
      category: "Research",
      readTime: "10 min read",
      image: "backend/api/placeholder/400/250"
    }
  ];

  const regularSections = [
    {
      title: "Industry Insights",
      description: "Expert analysis of business trends and market developments",
      articles: [
        {
          title: "Emerging Markets and Global Business Expansion",
          author: "Dr. Ahmed Al-Rashid",
          excerpt: "Strategic insights into expanding business operations in emerging markets across the Middle East and Asia.",
          readTime: "5 min read"
        },
        {
          title: "Sustainable Business Practices in 2025",
          author: "Jennifer Green",
          excerpt: "How companies are integrating sustainability into their core business strategies for long-term success.",
          readTime: "4 min read"
        }
      ]
    },
    {
      title: "Technology Corner",
      description: "Latest developments in educational technology and digital learning",
      articles: [
        {
          title: "Virtual Reality in Business Training: Beyond the Hype",
          author: "Dr. Michael Chen",
          excerpt: "Practical applications of VR technology in corporate training and professional development programs.",
          readTime: "6 min read"
        },
        {
          title: "AI-Powered Learning Analytics: Personalizing Education",
          author: "Lisa Zhang",
          excerpt: "How artificial intelligence is enabling personalized learning experiences and improved educational outcomes.",
          readTime: "7 min read"
        }
      ]
    },
    {
      title: "Global Perspectives",
      description: "International viewpoints on business education and professional development",
      articles: [
        {
          title: "Business Education Trends Across Europe and Asia",
          author: "Prof. Elena Kowalski",
          excerpt: "Comparative analysis of business education approaches and their effectiveness across different cultural contexts.",
          readTime: "8 min read"
        },
        {
          title: "Cross-Cultural Leadership: Lessons from Global Executives",
          author: "David Kumar",
          excerpt: "Insights from international business leaders on managing diverse teams and global operations.",
          readTime: "5 min read"
        }
      ]
    }
  ];

  const pastIssues = [
    {
      issue: "Issue 16",
      title: "Corporate Learning Revolution",
      date: "October 2024",
      highlight: "Special focus on microlearning and just-in-time training solutions"
    },
    {
      issue: "Issue 15", 
      title: "Leadership in Crisis",
      date: "July 2024",
      highlight: "Crisis management strategies and resilient leadership practices"
    },
    {
      issue: "Issue 14",
      title: "The Next Generation Workforce",
      date: "April 2024",
      highlight: "Preparing for Gen Z professionals and their learning preferences"
    }
  ];

  return (
    <WebsiteLayout>
      <div className="min-h-screen bg-gray-50">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="text-center">
              <Badge variant="secondary" className="mb-4 bg-[#ffffff] text-[#1c1c42]">
                Digital Publication
              </Badge>
              <h1 className="text-4xl md:text-5xl font-bold mb-6">
                {(pageData as any)?.title || "The Learning Edit Magazine"}
              </h1>
              <p className="text-xl text-gray-300 max-w-4xl mx-auto leading-relaxed">
                IBDL's premier digital magazine featuring thought leadership, industry insights, and innovative perspectives on business education and professional development
              </p>
            </div>
          </div>
        </div>

        {/* Current Issue Spotlight */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Current Issue</h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Discover the latest insights and trends shaping the future of business education
            </p>
          </div>

          <Card className="overflow-hidden border-0 shadow-2xl">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-0">
              {/* Magazine Cover */}
              <div className="bg-gradient-to-br from-[#1A1A40] to-[#ED1C24] p-12 text-white flex flex-col justify-center">
                <Badge className="self-start mb-4 bg-white text-[#1A1A40]">
                  {currentIssue.issueNumber}
                </Badge>
                <h3 className="text-3xl font-bold mb-2">{currentIssue.title}</h3>
                <h4 className="text-xl text-gray-200 mb-6">{currentIssue.subtitle}</h4>
                <div className="bg-white/10 backdrop-blur-sm rounded-lg p-4 mb-6">
                  <Quote className="h-6 w-6 mb-2" />
                  <p className="text-lg font-medium">{currentIssue.coverStory}</p>
                </div>
                <div className="text-sm text-gray-300 mb-6">{currentIssue.publishDate}</div>
                <div className="flex gap-4">
                  <Button className="bg-white text-[#1A1A40] hover:bg-gray-100">
                    <BookOpen className="mr-2 h-5 w-5" />
                    Read Online
                  </Button>
                  <Button variant="outline" className="border-white text-white bg-transparent hover:bg-white hover:text-[#1A1A40]">
                    <Download className="mr-2 h-5 w-5" />
                    Download PDF
                  </Button>
                </div>
              </div>

              {/* Featured Content */}
              <div className="p-8 bg-white">
                <h4 className="text-2xl font-bold text-[#1A1A40] mb-6">In This Issue</h4>
                <div className="space-y-6">
                  {featuredArticles.map((article, index) => (
                    <div key={index} className="border-l-4 border-[#ED1C24] pl-4">
                      <div className="flex items-center gap-2 mb-1">
                        <Badge variant="outline" className="text-xs">
                          {article.category}
                        </Badge>
                        <span className="text-xs text-gray-500">{article.readTime}</span>
                      </div>
                      <h5 className="font-semibold text-[#1A1A40] mb-1 hover:text-[#ED1C24] cursor-pointer transition-colors">
                        {article.title}
                      </h5>
                      <p className="text-sm text-gray-600 mb-2">{article.excerpt}</p>
                      <p className="text-xs text-gray-500">By {article.author}</p>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          </Card>
        </div>

        {/* Regular Sections */}
        <div className="bg-white py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Regular Sections</h2>
              <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                Explore our recurring content sections covering various aspects of business and professional development
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              {regularSections.map((section, sectionIndex) => (
                <Card key={sectionIndex} className="border-0 shadow-lg hover:shadow-xl transition-shadow">
                  <CardHeader className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
                    <CardTitle className="text-xl font-bold">{section.title}</CardTitle>
                    <p className="text-gray-200 text-sm">{section.description}</p>
                  </CardHeader>
                  <CardContent className="pt-6">
                    <div className="space-y-4">
                      {section.articles.map((article, articleIndex) => (
                        <div key={articleIndex} className="pb-4 border-b border-gray-200 last:border-b-0 last:pb-0">
                          <h6 className="font-semibold text-[#1A1A40] mb-1 hover:text-[#ED1C24] cursor-pointer transition-colors">
                            {article.title}
                          </h6>
                          <p className="text-sm text-gray-600 mb-2">{article.excerpt}</p>
                          <div className="flex items-center justify-between text-xs text-gray-500">
                            <span>By {article.author}</span>
                            <span>{article.readTime}</span>
                          </div>
                        </div>
                      ))}
                    </div>
                    <Button className="w-full mt-6 bg-[#ED1C24] hover:bg-[#CC1820] text-white">
                      View All Articles
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </Button>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>

        {/* Past Issues */}
        <div className="bg-gray-100 py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Past Issues</h2>
              <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                Browse our archive of previous magazine issues and timeless content
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              {pastIssues.map((issue, index) => (
                <Card key={index} className="border-0 shadow-lg hover:shadow-xl transition-shadow">
                  <CardContent className="p-6">
                    <div className="text-center">
                      <Badge className="mb-4 bg-[#1A1A40] text-white">
                        {issue.issue}
                      </Badge>
                      <h3 className="text-xl font-bold text-[#1A1A40] mb-2">{issue.title}</h3>
                      <div className="flex items-center justify-center text-sm text-gray-500 mb-4">
                        <Calendar className="w-4 h-4 mr-1" />
                        {issue.date}
                      </div>
                      <p className="text-gray-600 mb-6 text-sm">{issue.highlight}</p>
                      <Button variant="outline" className="w-full border-[#1A1A40] text-[#1A1A40] hover:bg-[#1A1A40] hover:text-white">
                        <Download className="mr-2 h-4 w-4" />
                        Download
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>

        {/* Subscription Section */}
        <div className="bg-gradient-to-r from-[#ED1C24] to-[#CC1820] py-16">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
            <h2 className="text-3xl font-bold text-white mb-4">Subscribe to The Learning Edit</h2>
            <p className="text-xl text-red-100 mb-8">
              Stay informed with the latest insights in business education and professional development. Delivered monthly to your inbox.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center max-w-md mx-auto">
              <input
                type="email"
                placeholder="Enter your email"
                className="flex-1 px-4 py-3 rounded-lg border-0 text-gray-900 placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-white"
              />
              <Button className="bg-white text-[#ED1C24] hover:bg-gray-100 px-8 py-3">
                Subscribe Free
              </Button>
            </div>
            <p className="text-red-100 text-sm mt-4">Join over 25,000 professionals who trust The Learning Edit for industry insights</p>
          </div>
        </div>

        {/* CMS Integration Notice */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 text-center">
            <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
            <p className="text-blue-700">
              This page content is managed through the CMS and can be updated by administrators.
              {pageData && typeof pageData === 'object' && pageData && 'id' in pageData && (
                <span className="block mt-2 text-sm">
                  Page ID: {String((pageData as any).id)} | Status: {String((pageData as any).status)}
                </span>
              )}
            </p>
          </div>
        </div>
      </div>
    </WebsiteLayout>
  );
}