import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { Home, Eye, EyeOff, Users, Building, User, ArrowRight } from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import ibdlLogo from "@assets/IBDL Icon_1751318336054.png";

interface LoginProps {
  onBack?: () => void;
}

export default function Login({ onBack }: LoginProps) {
  const [, setLocation] = useLocation();
  const [username, setUsername] = useState("");
  const [password, setPassword] = useState("");
  const [showPassword, setShowPassword] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const { toast } = useToast();

  const loginMutation = useMutation({
    mutationFn: async (credentials: { username: string; password: string }) => {
      return await apiRequest("backend/api/auth/login", "POST", credentials);
    },
    onSuccess: (data) => {
      // Check if user is a learner
      if (!['learner', 'applicant'].includes(data.user.userType)) {
        toast({
          title: "Access denied",
          description: "This login portal is for learners and applicants only. Please use the appropriate admin portal.",
          variant: "destructive",
        });
        return;
      }

      toast({
        title: "Login successful",
        description: `Welcome back, ${data.user.firstName}!`,
      });
      
      // Redirect to learner dashboard
      window.location.href = "/";
    },
    onError: (error) => {
      toast({
        title: "Login Failed",
        description: error.message || "Invalid credentials. Please try again.",
        variant: "destructive",
      });
    }
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!username || !password) {
      toast({
        title: "Error",
        description: "Please enter both username and password.",
        variant: "destructive",
      });
      return;
    }
    loginMutation.mutate({ username, password });
  };

  // Demo credentials display - only for learners
  const demoCredentials = [
    { type: "Individual Learner", username: "learner1", password: "password123" },
  ];

  const fillDemoCredentials = (username: string, password: string) => {
    setUsername(username);
    setPassword(password);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-[#1A1A40] to-[#2A2A60]">
      {/* Simplified Header */}
      <div className="bg-white shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <button onClick={onBack} className="mr-4">
                <img src={ibdlLogo} alt="IBDL Logo" className="h-10 w-auto" />
              </button>
            </div>
            <div className="flex items-center space-x-4">
              {onBack && (
                <Button 
                  onClick={onBack}
                  variant="outline" 
                  className="border-[#1A1A40] text-[#1A1A40] hover:bg-[#1A1A40] hover:text-white"
                >
                  <Home className="mr-2 h-4 w-4" />
                  Back to Home
                </Button>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Login Content */}
      <div className="min-h-[calc(100vh-200px)] flex items-center justify-center p-4">
        <div className="w-full max-w-4xl grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* Login Form */}
          <Card className="shadow-2xl border-0">
            <CardHeader className="space-y-6 text-center">
              <div className="flex justify-center">
                <img src={ibdlLogo} alt="IBDL Logo" className="h-12 w-auto" />
              </div>
              <div>
                <CardTitle className="text-2xl font-bold text-[#1A1A40]">
                  Welcome Back
                </CardTitle>
                <p className="text-muted-foreground mt-2">
                  Sign in to your IBDL Learning Platform
                </p>
              </div>
            </CardHeader>
            <CardContent className="space-y-6">
              <form onSubmit={handleSubmit} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="username">Username</Label>
                  <Input
                    id="username"
                    type="text"
                    value={username}
                    onChange={(e) => setUsername(e.target.value)}
                    placeholder="Enter your username"
                    className="border-gray-300 focus:border-[#ED1C24] focus:ring-[#ED1C24]"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="password">Password</Label>
                  <div className="relative">
                    <Input
                      id="password"
                      type={showPassword ? "text" : "password"}
                      value={password}
                      onChange={(e) => setPassword(e.target.value)}
                      placeholder="Enter your password"
                      className="border-gray-300 focus:border-[#ED1C24] focus:ring-[#ED1C24] pr-10"
                    />
                    <Button
                      type="button"
                      variant="ghost"
                      size="sm"
                      className="absolute right-0 top-0 h-full px-3 py-2 hover:bg-transparent"
                      onClick={() => setShowPassword(!showPassword)}
                    >
                      {showPassword ? (
                        <EyeOff className="h-4 w-4 text-gray-400" />
                      ) : (
                        <Eye className="h-4 w-4 text-gray-400" />
                      )}
                    </Button>
                  </div>
                </div>
                <Button
                  type="submit"
                  className="w-full bg-[#ED1C24] hover:bg-red-700 text-white"
                  disabled={loginMutation.isPending}
                >
                  {loginMutation.isPending ? "Signing In..." : "Sign In"}
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Button>
              </form>

              {/* Demo Credentials */}
              <div className="border-t pt-6">
                <h4 className="text-sm font-medium text-gray-700 mb-3">Demo Credentials (Learners Only)</h4>
                <div className="space-y-2">
                  {demoCredentials.map((demo, index) => (
                    <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                      <div className="flex items-center space-x-3">
                        <User className="h-4 w-4 text-gray-500" />
                        <div>
                          <p className="text-sm font-medium text-gray-700">{demo.type}</p>
                          <p className="text-xs text-gray-500">{demo.username} / {demo.password}</p>
                        </div>
                      </div>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => fillDemoCredentials(demo.username, demo.password)}
                        className="border-[#ED1C24] text-[#ED1C24] hover:bg-[#ED1C24] hover:text-white"
                      >
                        Use
                      </Button>
                    </div>
                  ))}
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Login Info */}
          <div className="flex flex-col justify-center space-y-8 text-white">
            <div>
              <h2 className="text-3xl font-bold mb-4">Individual Learner Access</h2>
              <p className="text-xl text-white/80 leading-relaxed">
                Access your personalized learning journey, track progress, and unlock your potential with IBDL's world-class business education.
              </p>
            </div>

            <div className="space-y-6">
              <div className="flex items-start space-x-4">
                <div className="flex-shrink-0 w-8 h-8 bg-[#ED1C24] rounded-full flex items-center justify-center">
                  <User className="h-4 w-4 text-white" />
                </div>
                <div>
                  <h3 className="font-semibold text-lg">Personal Dashboard</h3>
                  <p className="text-white/70">Track your learning progress and achievements</p>
                </div>
              </div>

              <div className="flex items-start space-x-4">
                <div className="flex-shrink-0 w-8 h-8 bg-[#ED1C24] rounded-full flex items-center justify-center">
                  <Building className="h-4 w-4 text-white" />
                </div>
                <div>
                  <h3 className="font-semibold text-lg">Course Access</h3>
                  <p className="text-white/70">Enroll in courses and access learning materials</p>
                </div>
              </div>

              <div className="flex items-start space-x-4">
                <div className="flex-shrink-0 w-8 h-8 bg-[#ED1C24] rounded-full flex items-center justify-center">
                  <Users className="h-4 w-4 text-white" />
                </div>
                <div>
                  <h3 className="font-semibold text-lg">Certifications</h3>
                  <p className="text-white/70">Earn globally recognized business qualifications</p>
                </div>
              </div>
            </div>

            <div className="bg-white/10 rounded-lg p-6">
              <p className="text-sm text-white/80">
                <strong>Welcome to IBDL Learning Platform</strong><br />
                Access your courses, track progress, and earn internationally recognized certifications in business leadership and management.
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}