import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { 
  Building2, User, GraduationCap, Users, CheckCircle, ArrowRight, Home,
  ShoppingBag, Mail, Briefcase, Languages, ChevronDown, ChevronLeft,
  Linkedin, Twitter, Facebook, Instagram, Youtube, MessageCircle
} from "lucide-react";
import ibdlLogo from "@assets/logo_1751230646761.png";
import footerLogo from "@assets/footer-logo_1751304691992.png";
import WebsiteLayout from "@/components/WebsiteLayout";

interface GetStartedProps {
  onRoleSelect: (role: string) => void;
  onBack?: () => void;
}

export default function GetStarted({ onRoleSelect, onBack }: GetStartedProps) {
  const [selectedCard, setSelectedCard] = useState<string | null>(null);
  const [cardStage, setCardStage] = useState<'initial' | 'details' | 'apply' | 'register'>('initial');
  const [centerAccreditationType, setCenterAccreditationType] = useState<string | null>(null);
  const [isRegistering, setIsRegistering] = useState(false);
  const { toast } = useToast();

  // Registration form state
  const [formData, setFormData] = useState({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    title: '',
    company: '',
    password: '',
    confirmPassword: '',
    agreeTerms: false
  });

  const handleInputChange = (field: string, value: string | boolean) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const roles = [
    {
      id: "corporate",
      title: "Corporate",
      subtitle: "Enterprise Learning Solutions",
      description: "Transform your organization with comprehensive learning management solutions. Our corporate platform helps you manage employee training, track progress, and achieve business objectives through structured learning programs tailored to your industry needs.",
      icon: Building2,
      features: [
        "Employee Training Management",
        "Learning Analytics & Reporting", 
        "Custom Learning Paths",
        "Compliance Training",
        "Skills Assessment & Gap Analysis",
        "Multi-tenant Organization Management"
      ],
      requirements: "Corporate email domain verification, organization size validation, and learning objectives assessment required for accreditation.",
      gradient: "from-blue-500 to-blue-700",
      bgGradient: "bg-gradient-to-br from-blue-50 to-blue-100",
      borderColor: "border-blue-200",
      iconBg: "bg-blue-100"
    },
    {
      id: "individual",
      title: "Individual Learner", 
      subtitle: "Personal Development Journey",
      description: "Accelerate your career with world-class business education and internationally recognized certifications. Access self-paced learning modules, practical assessments, and digital credentials that employers value worldwide.",
      icon: User,
      features: [
        "Professional Certifications",
        "Self-Paced Learning Modules",
        "Progress Tracking & Analytics",
        "Blockchain-Verified Certificates", 
        "Career Development Resources",
        "Global Learning Community Access"
      ],
      requirements: "Professional background verification, learning goals assessment, and commitment to complete certification requirements.",
      gradient: "from-green-500 to-green-700",
      bgGradient: "bg-gradient-to-br from-green-50 to-green-100", 
      borderColor: "border-green-200",
      iconBg: "bg-green-100"
    },
    {
      id: "training_center",
      title: "Training/Testing Center",
      subtitle: "Accredited Partner Network",
      description: "Become an IBDL authorized training center and expand your institutional capabilities. Deliver world-class business education through our comprehensive partnership program with full operational support.",
      icon: GraduationCap,
      features: [
        "Training Center Accreditation",
        "Comprehensive Course Library",
        "Instructor Certification Program",
        "Student Management System",
        "Quality Assurance Support",
        "Marketing & Branding Resources"
      ],
      requirements: "Facility assessment, instructor qualification review, technology infrastructure evaluation, and partnership agreement execution.",
      gradient: "from-purple-500 to-purple-700",
      bgGradient: "bg-gradient-to-br from-purple-50 to-purple-100",
      borderColor: "border-purple-200",
      iconBg: "bg-purple-100"
    },
    {
      id: "trainer",
      title: "Trainer",
      subtitle: "Certified IBDL Instructor", 
      description: "Join our elite network of expert trainers and shape the next generation of business leaders. Deliver high-impact education using IBDL's proven methodology and world-class resources.",
      icon: Users,
      features: [
        "Trainer Certification Program",
        "Teaching Resources & Materials",
        "Continuous Professional Development",
        "Global Trainer Network Access",
        "Performance Recognition Program",
        "Revenue Sharing Opportunities"
      ],
      requirements: "Professional experience validation, teaching competency assessment, certification completion, and ongoing performance standards.",
      gradient: "from-orange-500 to-orange-700", 
      bgGradient: "bg-gradient-to-br from-orange-50 to-orange-100",
      borderColor: "border-orange-200",
      iconBg: "bg-orange-100"
    }
  ];

  const handleCardClick = (roleId: string) => {
    if (cardStage === 'initial') {
      setSelectedCard(roleId);
      setCardStage('details');
    }
  };

  const handleNext = () => {
    if (cardStage === 'details') {
      // For training center, require accreditation type selection
      if (selectedCard === 'training_center' && !centerAccreditationType) {
        return; // Don't proceed without selection
      }
      setCardStage('apply');
    }
  };

  const handleBack = () => {
    if (cardStage === 'register') {
      setCardStage('details');
    } else if (cardStage === 'apply') {
      setCardStage('details');
    } else if (cardStage === 'details') {
      setCardStage('initial');
      setSelectedCard(null);
      setCenterAccreditationType(null);
    }
  };

  const handleApply = () => {
    if (selectedCard) {
      if (selectedCard === 'individual') {
        // Individual learners go to registration
        setCardStage('register');
      } else if (selectedCard === 'training_center') {
        // Training centers go to accreditation application form
        const applicationType = centerAccreditationType === 'testing' ? 'testing_center' : 
                               centerAccreditationType === 'training' ? 'training_center' : 
                               'both';
        window.location.href = `/training-center-application?type=${applicationType}`;
      } else {
        // Other roles go to accreditation process
        onRoleSelect(selectedCard);
      }
    }
  };

  const handleRegistration = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Validate form
    if (!formData.firstName || !formData.lastName || !formData.email || !formData.password) {
      toast({
        title: "Missing Information",
        description: "Please fill in all required fields",
        variant: "destructive"
      });
      return;
    }

    if (formData.password !== formData.confirmPassword) {
      toast({
        title: "Password Mismatch",
        description: "Password and confirm password must match",
        variant: "destructive"
      });
      return;
    }

    if (!formData.agreeTerms) {
      toast({
        title: "Terms Required",
        description: "Please agree to the terms and conditions",
        variant: "destructive"
      });
      return;
    }

    setIsRegistering(true);
    
    try {
      // Generate username from email
      const username = formData.email.split('@')[0];
      
      await apiRequest('backend/api/auth/register', 'POST', {
        username,
        email: formData.email,
        password: formData.password,
        firstName: formData.firstName,
        lastName: formData.lastName,
        userType: 'learner'
      });

      toast({
        title: "Registration Successful",
        description: "Your account has been created successfully!",
      });

      // Reset form
      setFormData({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        title: '',
        company: '',
        password: '',
        confirmPassword: '',
        agreeTerms: false
      });

      // Redirect to login
      setTimeout(() => {
        onRoleSelect('learner');
      }, 1500);

    } catch (error: any) {
      toast({
        title: "Registration Failed",
        description: error.message || "Failed to create account. Please try again.",
        variant: "destructive"
      });
    } finally {
      setIsRegistering(false);
    }
  };

  const selectedRole = roles.find(role => role.id === selectedCard);

  return (
    <WebsiteLayout>
      {/* Get Started Content */}
      <div className="min-h-[calc(100vh-200px)] bg-gradient-to-br from-gray-50 to-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          {/* Header */}
          <div className="text-center mb-12">
            <Badge className="mb-4 bg-[#1A1A40]/10 text-[#1A1A40] border-[#1A1A40]/20">
              Getting Started
            </Badge>
            <h1 className="text-4xl md:text-5xl font-bold mb-6 text-[#1A1A40]">
              Who Are You?
            </h1>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Choose your role to get personalized access to IBDL's comprehensive learning ecosystem. 
              Each pathway is designed to meet your specific needs and objectives.
            </p>
          </div>

          {/* Progressive Role Cards - Full IBDL Branded */}
          {cardStage === 'initial' && (
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
              {roles.map((role, index) => {
                const Icon = role.icon;
                const isEven = index % 2 === 0;
                return (
                  <Card 
                    key={role.id}
                    className={`group cursor-pointer transition-all duration-500 hover:shadow-2xl border-0 hover:transform hover:-translate-y-3 hover:scale-105 overflow-hidden ${
                      isEven 
                        ? 'bg-gradient-to-br from-[#1A1A40] to-[#1A1A40]/90 hover:from-[#ED1C24] hover:to-[#ED1C24]/90' 
                        : 'bg-gradient-to-br from-[#ED1C24] to-[#ED1C24]/90 hover:from-[#1A1A40] hover:to-[#1A1A40]/90'
                    }`}
                    onClick={() => handleCardClick(role.id)}
                  >
                    <CardContent className="p-6 text-center h-full flex flex-col justify-center">
                      {/* Icon and Title beside each other */}
                      <div className="flex items-center justify-center mb-4">
                        <div className={`inline-flex p-3 rounded-full mr-3 transition-all duration-300 group-hover:scale-110 ${
                          isEven 
                            ? 'bg-white/20 group-hover:bg-white/30' 
                            : 'bg-white/20 group-hover:bg-white/30'
                        }`}>
                          <Icon className="h-6 w-6 text-white" />
                        </div>
                        <CardTitle className="text-lg font-bold text-white group-hover:text-white transition-colors">
                          {role.title}
                        </CardTitle>
                      </div>
                      {/* Subtle subtitle */}
                      <p className="text-white/90 text-sm mb-4 transition-opacity duration-300">
                        {role.subtitle}
                      </p>
                      {/* White Arrow on Hover */}
                      <div className="opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                        <div className="w-8 h-0.5 bg-white mx-auto mb-2"></div>
                        <ArrowRight className="h-4 w-4 mx-auto text-white" />
                      </div>
                    </CardContent>
                  </Card>
                );
              })}
            </div>
          )}

          {/* Details Stage - Enhanced IBDL Branding */}
          {cardStage === 'details' && selectedRole && (
            <div className="max-w-4xl mx-auto">
              <Card className="border-0 shadow-2xl bg-gradient-to-br from-[#1A1A40] via-[#1A1A40]/95 to-[#1A1A40]/90 text-white overflow-hidden">
                <div className="h-4 bg-gradient-to-r from-[#ED1C24] via-[#ED1C24]/80 to-[#ED1C24]"></div>
                <CardContent className="p-8 bg-gradient-to-br from-[#1A1A40]/90 to-[#1A1A40]">
                  <div className="flex items-center justify-between mb-6">
                    <div className="flex items-center space-x-4">
                      <div className="p-4 rounded-xl bg-[#ED1C24]/20 border border-[#ED1C24]/30">
                        <selectedRole.icon className="h-10 w-10 text-[#ED1C24]" />
                      </div>
                      <div>
                        <h2 className="text-3xl font-bold text-white">{selectedRole.title}</h2>
                        <p className="text-[#ED1C24] font-medium">{selectedRole.subtitle}</p>
                      </div>
                    </div>
                    <Button 
                      onClick={handleBack}
                      variant="ghost"
                      size="sm"
                      className="text-[#1A1A40] bg-white/90 hover:bg-white hover:text-[#1A1A40] border border-white/20"
                    >
                      <ChevronLeft className="h-4 w-4" />
                    </Button>
                  </div>
                  
                  <p className="text-white/90 mb-8 text-lg leading-relaxed">
                    {selectedRole.description}
                  </p>
                  
                  <div className="grid md:grid-cols-2 gap-8 mb-8">
                    <div>
                      <h3 className="text-xl font-semibold text-white mb-4 flex items-center">
                        <CheckCircle className="h-5 w-5 text-[#ED1C24] mr-2" />
                        Key Features
                      </h3>
                      <div className="space-y-3">
                        {selectedRole.features.map((feature, index) => (
                          <div key={index} className="flex items-start bg-white/10 rounded-lg p-3 transition-colors hover:bg-[#ED1C24]/20 border border-white/10">
                            <div className="w-2 h-2 bg-[#ED1C24] rounded-full mt-2 mr-3 flex-shrink-0"></div>
                            <span className="text-white/90 text-sm">{feature}</span>
                          </div>
                        ))}
                      </div>
                    </div>
                    
                    <div>
                      <h3 className="text-xl font-semibold text-white mb-4 flex items-center">
                        <Briefcase className="h-5 w-5 text-[#ED1C24] mr-2" />
                        Requirements
                      </h3>
                      <div className="bg-white/10 rounded-lg p-4 border border-white/10">
                        <p className="text-white/90 text-sm leading-relaxed">
                          {selectedRole.requirements}
                        </p>
                      </div>
                    </div>
                  </div>
                  
                  {/* Training Center Accreditation Type Selection */}
                  {selectedCard === 'training_center' && (
                    <div className="mb-8">
                      <h3 className="text-xl font-semibold text-white mb-4 text-center">Select Accreditation Type</h3>
                      <div className="grid md:grid-cols-3 gap-4 max-w-3xl mx-auto">
                        {[
                          { id: 'testing', title: 'Testing Center Accreditation', description: 'Conduct IBDL examinations and assessments' },
                          { id: 'training', title: 'Training Center Accreditation', description: 'Deliver IBDL training programs' },
                          { id: 'both', title: 'Both Accreditations', description: 'Complete training and testing capabilities' }
                        ].map((type) => (
                          <Card 
                            key={type.id}
                            className={`cursor-pointer transition-all duration-200 border-2 ${
                              centerAccreditationType === type.id 
                                ? 'border-[#ED1C24] bg-[#ED1C24]/20' 
                                : 'border-white/20 hover:border-[#ED1C24]/50 bg-white/5'
                            }`}
                            onClick={() => setCenterAccreditationType(type.id)}
                          >
                            <CardContent className="p-4 text-center">
                              <div className={`w-4 h-4 rounded-full border-2 mx-auto mb-3 ${
                                centerAccreditationType === type.id 
                                  ? 'bg-[#ED1C24] border-[#ED1C24]' 
                                  : 'border-white/50'
                              }`}></div>
                              <h4 className="font-semibold text-white text-sm mb-2">{type.title}</h4>
                              <p className="text-xs text-white/70">{type.description}</p>
                            </CardContent>
                          </Card>
                        ))}
                      </div>
                    </div>
                  )}
                  
                  <div className="text-center">
                    <Button 
                      onClick={handleNext}
                      size="lg"
                      disabled={selectedCard === 'training_center' && !centerAccreditationType}
                      className="bg-gradient-to-r from-[#ED1C24] to-[#ED1C24]/90 hover:from-[#ED1C24]/90 hover:to-[#ED1C24] hover:shadow-xl text-white px-8 py-4 text-lg transform hover:scale-105 transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none border border-[#ED1C24]/20"
                    >
                      {selectedCard === 'individual' ? 'Next: Create Account' : 'Next: Application Process'}
                      <ArrowRight className="ml-2 h-5 w-5" />
                    </Button>
                    {selectedCard === 'training_center' && !centerAccreditationType && (
                      <p className="text-sm text-[#ED1C24] mt-2">Please select an accreditation type to continue</p>
                    )}
                  </div>
                </CardContent>
              </Card>
            </div>
          )}

          {/* Apply Stage - Full IBDL Branding */}
          {cardStage === 'apply' && selectedRole && (
            <div className="max-w-3xl mx-auto">
              <Card className="border-0 shadow-2xl bg-gradient-to-br from-[#ED1C24] via-[#ED1C24]/95 to-[#ED1C24]/90 text-white overflow-hidden">
                <div className="h-4 bg-gradient-to-r from-[#1A1A40] via-[#1A1A40]/80 to-[#1A1A40]"></div>
                <CardContent className="p-8 bg-gradient-to-br from-[#ED1C24]/90 to-[#ED1C24]">
                  {/* Back Button */}
                  <div className="flex justify-end mb-4">
                    <Button 
                      onClick={handleBack}
                      variant="ghost"
                      size="sm"
                      className="text-[#1A1A40] bg-white/90 hover:bg-white hover:text-[#1A1A40] border border-white/20"
                    >
                      <ChevronLeft className="h-4 w-4 mr-1" />
                      Back
                    </Button>
                  </div>
                  
                  <div className="text-center mb-8">
                    <div className="inline-flex p-4 rounded-full bg-[#1A1A40]/20 border border-[#1A1A40]/30 mb-4">
                      <selectedRole.icon className="h-8 w-8 text-[#1A1A40]" />
                    </div>
                    <h2 className="text-3xl font-bold text-white mb-2">Ready to Apply?</h2>
                    <p className="text-white/90">
                      Start your journey as an IBDL {selectedRole.title}
                      {selectedCard === 'training_center' && centerAccreditationType && (
                        <span className="block text-[#1A1A40] font-medium mt-1">
                          {centerAccreditationType === 'testing' && 'Testing Center Accreditation'}
                          {centerAccreditationType === 'training' && 'Training Center Accreditation'}
                          {centerAccreditationType === 'both' && 'Both Training & Testing Accreditations'}
                        </span>
                      )}
                    </p>
                  </div>
                  
                  <div className="bg-white/10 rounded-lg p-6 mb-8 border border-white/20">
                    <h3 className="text-lg font-semibold text-white mb-4">Application Process:</h3>
                    <div className="space-y-3">
                      <div className="flex items-center">
                        <div className="w-8 h-8 bg-[#1A1A40] text-white rounded-full flex items-center justify-center text-sm font-bold mr-3">1</div>
                        <span className="text-white/90">Submit your application form with required documentation</span>
                      </div>
                      <div className="flex items-center">
                        <div className="w-8 h-8 bg-[#1A1A40] text-white rounded-full flex items-center justify-center text-sm font-bold mr-3">2</div>
                        <span className="text-white/90">Undergo accreditation review and assessment</span>
                      </div>
                      <div className="flex items-center">
                        <div className="w-8 h-8 bg-[#1A1A40] text-white rounded-full flex items-center justify-center text-sm font-bold mr-3">3</div>
                        <span className="text-white/90">Receive approval notification and access credentials</span>
                      </div>
                      <div className="flex items-center">
                        <div className="w-8 h-8 bg-[#1A1A40] text-white rounded-full flex items-center justify-center text-sm font-bold mr-3">4</div>
                        <span className="text-white/90">Begin your IBDL journey with full platform access</span>
                      </div>
                    </div>
                  </div>
                  
                  <div className="flex flex-col sm:flex-row gap-4 justify-center">
                    <Button 
                      onClick={handleApply}
                      size="lg"
                      className="bg-gradient-to-r from-[#1A1A40] to-[#1A1A40]/90 hover:from-[#1A1A40]/90 hover:to-[#1A1A40] hover:shadow-xl text-white px-8 py-4 text-lg transform hover:scale-105 transition-all duration-200 border border-[#1A1A40]/20"
                    >
                      {selectedCard === 'individual' ? 'Register Now' : `Apply for ${selectedRole.title} Access`}
                      {selectedCard === 'training_center' && centerAccreditationType && (
                        <>
                          <br />
                          <span className="text-sm font-normal">
                            ({centerAccreditationType === 'testing' && 'Testing Center'}
                            {centerAccreditationType === 'training' && 'Training Center'}
                            {centerAccreditationType === 'both' && 'Both Accreditations'})
                          </span>
                        </>
                      )}
                      <ArrowRight className="ml-2 h-5 w-5" />
                    </Button>
                    <Button 
                      onClick={handleBack}
                      variant="outline"
                      size="lg"
                      className="bg-white/90 border-white text-[#1A1A40] hover:bg-white hover:text-[#1A1A40] px-8 py-4"
                    >
                      <ChevronLeft className="mr-2 h-4 w-4" />
                      Back to Details
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          )}

          {/* Registration Form for Individual Learners */}
          {cardStage === 'register' && selectedCard === 'individual' && (
            <div className="max-w-2xl mx-auto">
              <Card className="border-0 shadow-2xl bg-gradient-to-br from-[#1A1A40] via-[#1A1A40]/95 to-[#1A1A40]/90 text-white overflow-hidden">
                <div className="h-4 bg-gradient-to-r from-[#ED1C24] via-[#ED1C24]/80 to-[#ED1C24]"></div>
                <CardContent className="p-8">
                  {/* Back Button */}
                  <div className="flex justify-end mb-4">
                    <Button 
                      onClick={handleBack}
                      variant="ghost"
                      size="sm"
                      className="text-[#1A1A40] bg-white/90 hover:bg-white hover:text-[#1A1A40] border border-white/20"
                    >
                      <ChevronLeft className="h-4 w-4 mr-1" />
                      Back
                    </Button>
                  </div>
                  
                  <div className="text-center mb-8">
                    <div className="inline-flex p-4 rounded-full bg-[#ED1C24]/20 border border-[#ED1C24]/30 mb-4">
                      <User className="h-8 w-8 text-[#ED1C24]" />
                    </div>
                    <h2 className="text-3xl font-bold text-white mb-2">Join IBDL Learning</h2>
                    <p className="text-white/90">
                      Create your account to start your professional development journey
                    </p>
                  </div>
                  
                  <form className="space-y-6" onSubmit={handleRegistration}>
                    <div className="grid md:grid-cols-2 gap-4">
                      <div>
                        <label className="block text-white text-sm font-medium mb-2">First Name *</label>
                        <input 
                          type="text"
                          required
                          value={formData.firstName}
                          onChange={(e) => handleInputChange('firstName', e.target.value)}
                          className="w-full px-4 py-3 rounded-lg bg-white/10 border border-white/20 text-white placeholder-white/60 focus:border-[#ED1C24] focus:outline-none focus:ring-2 focus:ring-[#ED1C24]/20"
                          placeholder="Enter your first name"
                        />
                      </div>
                      <div>
                        <label className="block text-white text-sm font-medium mb-2">Last Name *</label>
                        <input 
                          type="text"
                          required
                          value={formData.lastName}
                          onChange={(e) => handleInputChange('lastName', e.target.value)}
                          className="w-full px-4 py-3 rounded-lg bg-white/10 border border-white/20 text-white placeholder-white/60 focus:border-[#ED1C24] focus:outline-none focus:ring-2 focus:ring-[#ED1C24]/20"
                          placeholder="Enter your last name"
                        />
                      </div>
                    </div>
                    
                    <div>
                      <label className="block text-white text-sm font-medium mb-2">Email Address *</label>
                      <input 
                        type="email"
                        required
                        value={formData.email}
                        onChange={(e) => handleInputChange('email', e.target.value)}
                        className="w-full px-4 py-3 rounded-lg bg-white/10 border border-white/20 text-white placeholder-white/60 focus:border-[#ED1C24] focus:outline-none focus:ring-2 focus:ring-[#ED1C24]/20"
                        placeholder="Enter your email address"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-white text-sm font-medium mb-2">Phone Number</label>
                      <input 
                        type="tel"
                        value={formData.phone}
                        onChange={(e) => handleInputChange('phone', e.target.value)}
                        className="w-full px-4 py-3 rounded-lg bg-white/10 border border-white/20 text-white placeholder-white/60 focus:border-[#ED1C24] focus:outline-none focus:ring-2 focus:ring-[#ED1C24]/20"
                        placeholder="Enter your phone number"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-white text-sm font-medium mb-2">Professional Title</label>
                      <input 
                        type="text"
                        value={formData.title}
                        onChange={(e) => handleInputChange('title', e.target.value)}
                        className="w-full px-4 py-3 rounded-lg bg-white/10 border border-white/20 text-white placeholder-white/60 focus:border-[#ED1C24] focus:outline-none focus:ring-2 focus:ring-[#ED1C24]/20"
                        placeholder="e.g., Marketing Manager, Business Analyst"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-white text-sm font-medium mb-2">Company/Organization</label>
                      <input 
                        type="text"
                        value={formData.company}
                        onChange={(e) => handleInputChange('company', e.target.value)}
                        className="w-full px-4 py-3 rounded-lg bg-white/10 border border-white/20 text-white placeholder-white/60 focus:border-[#ED1C24] focus:outline-none focus:ring-2 focus:ring-[#ED1C24]/20"
                        placeholder="Enter your organization name"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-white text-sm font-medium mb-2">Password *</label>
                      <input 
                        type="password"
                        required
                        value={formData.password}
                        onChange={(e) => handleInputChange('password', e.target.value)}
                        className="w-full px-4 py-3 rounded-lg bg-white/10 border border-white/20 text-white placeholder-white/60 focus:border-[#ED1C24] focus:outline-none focus:ring-2 focus:ring-[#ED1C24]/20"
                        placeholder="Create a secure password"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-white text-sm font-medium mb-2">Confirm Password *</label>
                      <input 
                        type="password"
                        required
                        value={formData.confirmPassword}
                        onChange={(e) => handleInputChange('confirmPassword', e.target.value)}
                        className="w-full px-4 py-3 rounded-lg bg-white/10 border border-white/20 text-white placeholder-white/60 focus:border-[#ED1C24] focus:outline-none focus:ring-2 focus:ring-[#ED1C24]/20"
                        placeholder="Confirm your password"
                      />
                    </div>
                    
                    <div className="flex items-start">
                      <input 
                        type="checkbox"
                        required
                        checked={formData.agreeTerms}
                        onChange={(e) => handleInputChange('agreeTerms', e.target.checked)}
                        className="mt-1 mr-3 text-[#ED1C24] focus:ring-[#ED1C24]"
                      />
                      <p className="text-white/90 text-sm">
                        I agree to the <span className="text-[#ED1C24] hover:underline cursor-pointer">Terms of Service</span> and <span className="text-[#ED1C24] hover:underline cursor-pointer">Privacy Policy</span>
                      </p>
                    </div>
                    
                    <div className="flex flex-col sm:flex-row gap-4 justify-center mt-8">
                      <Button 
                        type="submit"
                        size="lg"
                        disabled={isRegistering}
                        className="bg-gradient-to-r from-[#ED1C24] to-[#ED1C24]/90 hover:from-[#ED1C24]/90 hover:to-[#ED1C24] hover:shadow-xl text-white px-8 py-4 text-lg transform hover:scale-105 transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none"
                      >
                        {isRegistering ? 'Creating Account...' : 'Create My Account'}
                        <ArrowRight className="ml-2 h-5 w-5" />
                      </Button>
                      <Button 
                        onClick={handleBack}
                        variant="outline"
                        size="lg"
                        disabled={isRegistering}
                        className="border-white/50 text-white hover:bg-white/10 hover:text-white px-8 py-4 hover:border-white disabled:opacity-50"
                      >
                        <ChevronLeft className="mr-2 h-4 w-4" />
                        Back
                      </Button>
                    </div>
                  </form>
                </CardContent>
              </Card>
            </div>
          )}
        </div>
      </div>
    </WebsiteLayout>
  );
}