import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { 
  Plus, 
  Edit, 
  Trash2, 
  Settings, 
  BarChart3, 
  Calendar, 
  Monitor, 
  BookOpen, 
  Clock, 
  CheckCircle, 
  XCircle, 
  AlertCircle,
  Eye
} from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { 
  insertExamConfigurationSchema,
  type ExamConfiguration,
  type AssessmentAttempt,
  type ExaminationSession,
  type QuestionBank
} from "@shared/schema";

export default function Examinations() {
  const [activeTab, setActiveTab] = useState("configurations");
  const [createDialogOpen, setCreateDialogOpen] = useState(false);
  const [editingConfig, setEditingConfig] = useState<any>(null);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch data
  const { data: examConfigurations = [], isLoading: loadingConfigs } = useQuery({
    queryKey: ["backend/api/exam-configurations"],
  });

  const { data: examResults = [], isLoading: loadingResults } = useQuery({
    queryKey: ["backend/api/exam-results"],
  });

  const { data: examSessions = [], isLoading: loadingSessions } = useQuery({
    queryKey: ["backend/api/examination-sessions"],
  });

  const { data: questionBank = [], isLoading: loadingQuestions } = useQuery({
    queryKey: ["backend/api/question-bank"],
  });

  const { data: certificates = [] } = useQuery({
    queryKey: ["backend/api/system-certificates"],
  });

  // Configuration form
  const configForm = useForm({
    resolver: zodResolver(insertExamConfigurationSchema),
    defaultValues: {
      certificateId: "",
      levelId: "",
      examName: "",
      examType: "certification",
      questionCount: 50,
      timeLimit: 120,
      passingScore: 70,
      randomizeQuestions: true,
      allowReview: false,
      showResults: true,
      isProctored: false,
      maxAttempts: 3,
      retakeDelay: 24,
      createdBy: "sys-admin-1",
    },
  });

  // Mutations
  const createConfigMutation = useMutation({
    mutationFn: (data: any) => apiRequest("backend/api/exam-configurations", "POST", data),
    onSuccess: () => {
      toast({ title: "Exam configuration created successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/exam-configurations"] });
      configForm.reset();
      setCreateDialogOpen(false);
    },
    onError: (error) => {
      toast({ title: "Error creating configuration", description: error.message, variant: "destructive" });
    },
  });

  const updateConfigMutation = useMutation({
    mutationFn: ({ id, data }: { id: string; data: any }) => 
      apiRequest(`backend/api/exam-configurations/${id}`, "PATCH", data),
    onSuccess: () => {
      toast({ title: "Exam configuration updated successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/exam-configurations"] });
      setEditingConfig(null);
      setCreateDialogOpen(false);
    },
    onError: (error) => {
      toast({ title: "Error updating configuration", description: error.message, variant: "destructive" });
    },
  });

  const deleteConfigMutation = useMutation({
    mutationFn: (id: string) => apiRequest(`backend/api/exam-configurations/${id}`, "DELETE"),
    onSuccess: () => {
      toast({ title: "Exam configuration deleted successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/exam-configurations"] });
    },
    onError: (error) => {
      toast({ title: "Error deleting configuration", description: error.message, variant: "destructive" });
    },
  });

  const handleCreateConfig = (data: any) => {
    if (editingConfig) {
      updateConfigMutation.mutate({ id: editingConfig.id, data });
    } else {
      createConfigMutation.mutate(data);
    }
  };

  const handleEditConfig = (config: any) => {
    setEditingConfig(config);
    configForm.reset({
      certificateId: config.certificateId,
      examName: config.examName,
      examType: config.examType,
      questionCount: config.questionCount,
      timeLimit: config.timeLimit,
      passingScore: config.passingScore,
      randomizeQuestions: config.randomizeQuestions,
      allowReview: config.allowReview,
      showResults: config.showResults,
      isProctored: config.isProctored,
      maxAttempts: config.maxAttempts,
      retakeDelay: config.retakeDelay,
      createdBy: config.createdBy,
    });
    setCreateDialogOpen(true);
  };

  const handleDeleteConfig = (id: string) => {
    if (confirm("Are you sure you want to delete this exam configuration?")) {
      deleteConfigMutation.mutate(id);
    }
  };

  const updateExamStatusMutation = useMutation({
    mutationFn: ({ id, status }: { id: string; status: string }) => 
      apiRequest(`backend/api/examination-sessions/${id}`, "PATCH", { status }),
    onSuccess: () => {
      toast({ title: "Exam status updated successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/examination-sessions"] });
    },
    onError: (error) => {
      toast({ title: "Error updating exam status", description: error.message, variant: "destructive" });
    },
  });

  const getStatusColor = (status: string) => {
    switch (status) {
      case "active":
        return "bg-green-100 text-green-800";
      case "completed":
        return "bg-blue-100 text-blue-800";
      case "failed":
        return "bg-red-100 text-red-800";
      case "in_progress":
        return "bg-yellow-100 text-yellow-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  return (
    <DashboardLayout userType="admin">
      <div className="space-y-8">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Examinations</h1>
          <p className="text-muted-foreground">
            Manage exam configurations, monitor results, schedule sessions, and oversee question bank
          </p>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
          <TabsList className="grid w-full grid-cols-5">
            <TabsTrigger value="configurations" className="flex items-center gap-2">
              <Settings className="w-4 h-4" />
              Configurations
            </TabsTrigger>
            <TabsTrigger value="results" className="flex items-center gap-2">
              <BarChart3 className="w-4 h-4" />
              Results
            </TabsTrigger>
            <TabsTrigger value="scheduling" className="flex items-center gap-2">
              <Calendar className="w-4 h-4" />
              Scheduling
            </TabsTrigger>
            <TabsTrigger value="monitoring" className="flex items-center gap-2">
              <Monitor className="w-4 h-4" />
              Monitoring
            </TabsTrigger>
            <TabsTrigger value="questions" className="flex items-center gap-2">
              <BookOpen className="w-4 h-4" />
              Questions Bank
            </TabsTrigger>
          </TabsList>

          <TabsContent value="configurations" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Exam Configurations</h2>
              <Dialog open={createDialogOpen} onOpenChange={setCreateDialogOpen}>
                <DialogTrigger asChild>
                  <Button onClick={() => { setEditingConfig(null); configForm.reset(); }}>
                    <Plus className="w-4 h-4 mr-2" />
                    Create Configuration
                  </Button>
                </DialogTrigger>
                <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
                  <DialogHeader>
                    <DialogTitle>
                      {editingConfig ? "Edit Exam Configuration" : "Create Exam Configuration"}
                    </DialogTitle>
                  </DialogHeader>
                  <Form {...configForm}>
                    <form onSubmit={configForm.handleSubmit(handleCreateConfig)} className="space-y-4">
                      <div className="grid grid-cols-2 gap-4">
                        <FormField
                          control={configForm.control}
                          name="examName"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Exam Name</FormLabel>
                              <FormControl>
                                <Input placeholder="Strategic Leadership Certification Exam" {...field} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={configForm.control}
                          name="examType"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Exam Type</FormLabel>
                              <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl>
                                  <SelectTrigger>
                                    <SelectValue placeholder="Select type" />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  <SelectItem value="certification">Certification</SelectItem>
                                  <SelectItem value="placement">Placement Test</SelectItem>
                                  <SelectItem value="practice">Practice Exam</SelectItem>
                                </SelectContent>
                              </Select>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="grid grid-cols-3 gap-4">
                        <FormField
                          control={configForm.control}
                          name="questionCount"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Question Count</FormLabel>
                              <FormControl>
                                <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={configForm.control}
                          name="timeLimit"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Time Limit (Minutes)</FormLabel>
                              <FormControl>
                                <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={configForm.control}
                          name="passingScore"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Passing Score (%)</FormLabel>
                              <FormControl>
                                <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="grid grid-cols-2 gap-4">
                        <FormField
                          control={configForm.control}
                          name="maxAttempts"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Max Attempts</FormLabel>
                              <FormControl>
                                <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={configForm.control}
                          name="retakeDelay"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Retake Delay (Hours)</FormLabel>
                              <FormControl>
                                <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="grid grid-cols-2 gap-4">
                        <FormField
                          control={configForm.control}
                          name="randomizeQuestions"
                          render={({ field }) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                              <div className="space-y-0.5">
                                <FormLabel className="text-base">Randomize Questions</FormLabel>
                                <p className="text-sm text-muted-foreground">
                                  Randomize the order of questions
                                </p>
                              </div>
                              <FormControl>
                                <Switch
                                  checked={field.value}
                                  onCheckedChange={field.onChange}
                                />
                              </FormControl>
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={configForm.control}
                          name="allowReview"
                          render={({ field }) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                              <div className="space-y-0.5">
                                <FormLabel className="text-base">Allow Review</FormLabel>
                                <p className="text-sm text-muted-foreground">
                                  Allow candidates to review answers
                                </p>
                              </div>
                              <FormControl>
                                <Switch
                                  checked={field.value}
                                  onCheckedChange={field.onChange}
                                />
                              </FormControl>
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="grid grid-cols-2 gap-4">
                        <FormField
                          control={configForm.control}
                          name="showResults"
                          render={({ field }) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                              <div className="space-y-0.5">
                                <FormLabel className="text-base">Show Results</FormLabel>
                                <p className="text-sm text-muted-foreground">
                                  Show results immediately after completion
                                </p>
                              </div>
                              <FormControl>
                                <Switch
                                  checked={field.value}
                                  onCheckedChange={field.onChange}
                                />
                              </FormControl>
                            </FormItem>
                          )}
                        />
                        <FormField
                          control={configForm.control}
                          name="isProctored"
                          render={({ field }) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                              <div className="space-y-0.5">
                                <FormLabel className="text-base">Proctored Exam</FormLabel>
                                <p className="text-sm text-muted-foreground">
                                  Enable proctoring for this exam
                                </p>
                              </div>
                              <FormControl>
                                <Switch
                                  checked={field.value}
                                  onCheckedChange={field.onChange}
                                />
                              </FormControl>
                            </FormItem>
                          )}
                        />
                      </div>
                      <div className="flex justify-end space-x-2">
                        <Button type="button" variant="outline" onClick={() => setCreateDialogOpen(false)}>
                          Cancel
                        </Button>
                        <Button type="submit" disabled={createConfigMutation.isPending || updateConfigMutation.isPending}>
                          {createConfigMutation.isPending || updateConfigMutation.isPending 
                            ? "Saving..." 
                            : editingConfig ? "Update Configuration" : "Create Configuration"}
                        </Button>
                      </div>
                    </form>
                  </Form>
                </DialogContent>
              </Dialog>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {examConfigurations.map((config: ExamConfiguration) => (
                <Card key={config.id}>
                  <CardHeader>
                    <div className="flex justify-between items-start">
                      <div>
                        <CardTitle className="text-lg">{config.examName}</CardTitle>
                        <CardDescription>
                          <Badge variant={config.examType === "certification" ? "default" : "secondary"}>
                            {config.examType}
                          </Badge>
                        </CardDescription>
                      </div>
                      <Badge variant={config.isActive ? "default" : "secondary"}>
                        {config.isActive ? "Active" : "Inactive"}
                      </Badge>
                    </div>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>Questions:</span>
                        <span>{config.questionCount}</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span>Time Limit:</span>
                        <span>{config.timeLimit} min</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span>Passing Score:</span>
                        <span>{config.passingScore}%</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span>Max Attempts:</span>
                        <span>{config.maxAttempts}</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span>Proctored:</span>
                        <span>{config.isProctored ? "Yes" : "No"}</span>
                      </div>
                      <div className="flex justify-end gap-2 mt-4">
                        <Button variant="ghost" size="sm" onClick={() => handleEditConfig(config)}>
                          <Edit className="w-4 h-4" />
                        </Button>
                        <Button variant="ghost" size="sm" onClick={() => handleDeleteConfig(config.id)}>
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>

          <TabsContent value="results" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Exam Results</h2>
              <div className="flex gap-2">
                <Button variant="outline">
                  <BarChart3 className="w-4 h-4 mr-2" />
                  Export Results
                </Button>
                <Button variant="outline">
                  <Settings className="w-4 h-4 mr-2" />
                  Filter Results
                </Button>
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Total Exams</CardTitle>
                  <BookOpen className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{examResults.length}</div>
                  <p className="text-xs text-muted-foreground">+12% from last month</p>
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Pass Rate</CardTitle>
                  <CheckCircle className="h-4 w-4 text-green-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">78.5%</div>
                  <p className="text-xs text-muted-foreground">+5.2% from last month</p>
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Average Score</CardTitle>
                  <BarChart3 className="h-4 w-4 text-blue-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">82.3%</div>
                  <p className="text-xs text-muted-foreground">+2.1% from last month</p>
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Completion Rate</CardTitle>
                  <Clock className="h-4 w-4 text-orange-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">94.2%</div>
                  <p className="text-xs text-muted-foreground">+1.8% from last month</p>
                </CardContent>
              </Card>
            </div>

            <Card>
              <CardHeader>
                <CardTitle>Recent Exam Results</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {examResults.slice(0, 10).map((result: AssessmentAttempt) => (
                    <div key={result.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <div>
                          <p className="font-medium">{result.userId}</p>
                          <p className="text-sm text-muted-foreground">{result.assessmentId}</p>
                        </div>
                      </div>
                      <div className="flex items-center gap-4">
                        <div className="text-right">
                          <p className="font-medium">{result.score}%</p>
                          <p className="text-sm text-muted-foreground">{result.completedAt}</p>
                        </div>
                        <Badge className={getStatusColor(result.status)}>
                          {result.status}
                        </Badge>
                        <Button size="sm" variant="outline">
                          <Eye className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="scheduling" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Exam Scheduling</h2>
              <Button>
                <Plus className="w-4 h-4 mr-2" />
                Schedule Exam
              </Button>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Upcoming Exams</CardTitle>
                  <CardDescription>Next 7 days</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="text-3xl font-bold">24</div>
                  <div className="space-y-2 mt-4">
                    <div className="flex justify-between text-sm">
                      <span>Tomorrow:</span>
                      <span>8 exams</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>This week:</span>
                      <span>16 exams</span>
                    </div>
                  </div>
                </CardContent>
              </Card>
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Testing Centers</CardTitle>
                  <CardDescription>Available venues</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="text-3xl font-bold">12</div>
                  <div className="space-y-2 mt-4">
                    <div className="flex justify-between text-sm">
                      <span>Active:</span>
                      <span>10 centers</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Maintenance:</span>
                      <span>2 centers</span>
                    </div>
                  </div>
                </CardContent>
              </Card>
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Capacity</CardTitle>
                  <CardDescription>Current utilization</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="text-3xl font-bold">78%</div>
                  <div className="space-y-2 mt-4">
                    <div className="flex justify-between text-sm">
                      <span>Peak hours:</span>
                      <span>95%</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span>Off-peak:</span>
                      <span>45%</span>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="monitoring" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Exam Monitoring</h2>
              <div className="flex gap-2">
                <Button variant="outline">
                  <AlertCircle className="w-4 h-4 mr-2" />
                  View Alerts
                </Button>
                <Button variant="outline">
                  <Monitor className="w-4 h-4 mr-2" />
                  Live Monitor
                </Button>
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Active Sessions</CardTitle>
                  <Monitor className="h-4 w-4 text-green-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">18</div>
                  <p className="text-xs text-muted-foreground">Currently in progress</p>
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Flagged Events</CardTitle>
                  <AlertCircle className="h-4 w-4 text-red-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">3</div>
                  <p className="text-xs text-muted-foreground">Require attention</p>
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Proctored Exams</CardTitle>
                  <Eye className="h-4 w-4 text-blue-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">12</div>
                  <p className="text-xs text-muted-foreground">Under supervision</p>
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">System Health</CardTitle>
                  <CheckCircle className="h-4 w-4 text-green-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">99.8%</div>
                  <p className="text-xs text-muted-foreground">Uptime</p>
                </CardContent>
              </Card>
            </div>

            <Card>
              <CardHeader>
                <CardTitle>Active Examination Sessions</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {examSessions.filter((session: ExaminationSession) => session.status === "active").map((session: ExaminationSession) => (
                    <div key={session.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <div>
                          <p className="font-medium">{session.userId}</p>
                          <p className="text-sm text-muted-foreground">{session.assessmentId}</p>
                        </div>
                      </div>
                      <div className="flex items-center gap-4">
                        <div className="text-right">
                          <p className="font-medium">{session.status}</p>
                          <p className="text-sm text-muted-foreground">{session.startTime}</p>
                        </div>
                        <div className="flex gap-2">
                          <Button size="sm" variant="outline">
                            <Eye className="w-4 h-4" />
                          </Button>
                          <Button 
                            size="sm" 
                            variant="outline"
                            onClick={() => updateExamStatusMutation.mutate({ id: session.id, status: "paused" })}
                          >
                            Pause
                          </Button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="questions" className="space-y-4">
            <div className="flex justify-between items-center">
              <h2 className="text-xl font-semibold">Questions Bank</h2>
              <div className="flex gap-2">
                <Button variant="outline">
                  <Plus className="w-4 h-4 mr-2" />
                  Import Questions
                </Button>
                <Button>
                  <Plus className="w-4 h-4 mr-2" />
                  Add Question
                </Button>
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Total Questions</CardTitle>
                  <BookOpen className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{questionBank.length}</div>
                  <p className="text-xs text-muted-foreground">+45 this month</p>
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Multiple Choice</CardTitle>
                  <CheckCircle className="h-4 w-4 text-blue-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">
                    {questionBank.filter((q: QuestionBank) => q.questionType === "multiple_choice").length}
                  </div>
                  <p className="text-xs text-muted-foreground">Most common type</p>
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">True/False</CardTitle>
                  <XCircle className="h-4 w-4 text-green-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">
                    {questionBank.filter((q: QuestionBank) => q.questionType === "true_false").length}
                  </div>
                  <p className="text-xs text-muted-foreground">Simple format</p>
                </CardContent>
              </Card>
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Essay Questions</CardTitle>
                  <Edit className="h-4 w-4 text-purple-600" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">
                    {questionBank.filter((q: QuestionBank) => q.questionType === "essay").length}
                  </div>
                  <p className="text-xs text-muted-foreground">Detailed responses</p>
                </CardContent>
              </Card>
            </div>

            <Card>
              <CardHeader>
                <CardTitle>Recent Questions</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {questionBank.slice(0, 10).map((question: QuestionBank) => (
                    <div key={question.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <div>
                          <p className="font-medium">{question.questionText.substring(0, 80)}...</p>
                          <div className="flex items-center gap-2 mt-1">
                            <Badge variant="outline">{question.questionType}</Badge>
                            <Badge variant="secondary">{question.difficultyLevel}</Badge>
                            <Badge variant="outline">{question.topic}</Badge>
                          </div>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <Button size="sm" variant="outline">
                          <Edit className="w-4 h-4" />
                        </Button>
                        <Button size="sm" variant="outline">
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </DashboardLayout>
  );
}