import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { 
  Plus, 
  Play, 
  Book, 
  Monitor, 
  FileText, 
  Upload, 
  Download,
  Edit,
  Trash2,
  Eye,
  Clock,
  Users,
  Award
} from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { 
  insertSystemELearningContentSchema,
  type SystemELearningContent,
  type SystemCertificate,
  type SystemLevel,
  type SystemChapter
} from "@shared/schema";

export default function ELearning() {
  const [selectedCertificate, setSelectedCertificate] = useState<string | null>(null);
  const [selectedLevel, setSelectedLevel] = useState<string | null>(null);
  const [selectedChapter, setSelectedChapter] = useState<string | null>(null);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch data
  const { data: certificates = [] } = useQuery({
    queryKey: ["backend/api/system-certificates"],
  });

  const { data: levels = [] } = useQuery({
    queryKey: ["backend/api/system-levels", selectedCertificate],
    enabled: !!selectedCertificate,
  });

  const { data: chapters = [] } = useQuery({
    queryKey: ["backend/api/system-chapters", selectedLevel],
    enabled: !!selectedLevel,
  });

  const { data: elearningContent = [] } = useQuery({
    queryKey: ["backend/api/elearning-content"],
  });

  // Form
  const contentForm = useForm({
    resolver: zodResolver(insertSystemELearningContentSchema),
    defaultValues: {
      certificateId: "",
      levelId: "",
      chapterId: "",
      titleEn: "",
      titleAr: "",
      titleFr: "",
      contentType: "video",
      content: {},
      duration: 30,
      createdBy: "sys-admin-1",
    },
  });

  // Mutations
  const createContentMutation = useMutation({
    mutationFn: (data: any) => apiRequest("backend/api/elearning-content", "POST", data),
    onSuccess: () => {
      toast({ title: "eLearning content created successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/elearning-content"] });
      contentForm.reset();
    },
    onError: (error) => {
      toast({ title: "Error creating content", description: error.message, variant: "destructive" });
    },
  });

  const getContentTypeIcon = (type: string) => {
    switch (type) {
      case "video":
        return <Play className="w-4 h-4 text-blue-600" />;
      case "interactive":
        return <Monitor className="w-4 h-4 text-green-600" />;
      case "simulation":
        return <FileText className="w-4 h-4 text-purple-600" />;
      case "quiz":
        return <Book className="w-4 h-4 text-orange-600" />;
      default:
        return <FileText className="w-4 h-4 text-gray-600" />;
    }
  };

  const getContentTypeColor = (type: string) => {
    switch (type) {
      case "video":
        return "bg-blue-100 text-blue-800";
      case "interactive":
        return "bg-green-100 text-green-800";
      case "simulation":
        return "bg-purple-100 text-purple-800";
      case "quiz":
        return "bg-orange-100 text-orange-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  return (
    <DashboardLayout userType="admin">
      <div className="space-y-8">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">eLearning Content Management</h1>
          <p className="text-muted-foreground">
            Create and manage interactive learning content for certificates, levels, and chapters
          </p>
        </div>

        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total Content</CardTitle>
              <FileText className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{elearningContent.length}</div>
              <p className="text-xs text-muted-foreground">Across all certificates</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Video Content</CardTitle>
              <Play className="h-4 w-4 text-blue-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {elearningContent.filter((content: SystemELearningContent) => content.contentType === "video").length}
              </div>
              <p className="text-xs text-muted-foreground">Video lessons</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Interactive</CardTitle>
              <Monitor className="h-4 w-4 text-green-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {elearningContent.filter((content: SystemELearningContent) => content.contentType === "interactive").length}
              </div>
              <p className="text-xs text-muted-foreground">Interactive modules</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Simulations</CardTitle>
              <FileText className="h-4 w-4 text-purple-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {elearningContent.filter((content: SystemELearningContent) => content.contentType === "simulation").length}
              </div>
              <p className="text-xs text-muted-foreground">Business simulations</p>
            </CardContent>
          </Card>
        </div>

        <div className="flex justify-between items-center">
          <h2 className="text-xl font-semibold">Content Library</h2>
          <Dialog>
            <DialogTrigger asChild>
              <Button>
                <Plus className="w-4 h-4 mr-2" />
                Add Content
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-4xl">
              <DialogHeader>
                <DialogTitle>Create eLearning Content</DialogTitle>
              </DialogHeader>
              <Form {...contentForm}>
                <form onSubmit={contentForm.handleSubmit((data) => createContentMutation.mutate(data))} className="space-y-4">
                  <div className="grid grid-cols-3 gap-4">
                    <FormField
                      control={contentForm.control}
                      name="certificateId"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Certificate</FormLabel>
                          <Select onValueChange={(value) => {
                            field.onChange(value);
                            setSelectedCertificate(value);
                          }}>
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select certificate" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              {certificates.map((cert: SystemCertificate) => (
                                <SelectItem key={cert.id} value={cert.id}>{cert.nameEn}</SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={contentForm.control}
                      name="levelId"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Level (Optional)</FormLabel>
                          <Select onValueChange={(value) => {
                            field.onChange(value);
                            setSelectedLevel(value);
                          }}>
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select level" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              {levels.map((level: SystemLevel) => (
                                <SelectItem key={level.id} value={level.id}>Level {level.levelNumber} - {level.nameEn}</SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={contentForm.control}
                      name="chapterId"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Chapter (Optional)</FormLabel>
                          <Select onValueChange={field.onChange}>
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select chapter" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              {chapters.map((chapter: SystemChapter) => (
                                <SelectItem key={chapter.id} value={chapter.id}>Chapter {chapter.chapterNumber} - {chapter.titleEn}</SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                  
                  <div className="grid grid-cols-2 gap-4">
                    <FormField
                      control={contentForm.control}
                      name="titleEn"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Title (English)</FormLabel>
                          <FormControl>
                            <Input placeholder="Introduction to Strategic Planning" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={contentForm.control}
                      name="contentType"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Content Type</FormLabel>
                          <Select onValueChange={field.onChange} defaultValue={field.value}>
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select type" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              <SelectItem value="video">Video</SelectItem>
                              <SelectItem value="interactive">Interactive</SelectItem>
                              <SelectItem value="simulation">Simulation</SelectItem>
                              <SelectItem value="quiz">Quiz</SelectItem>
                            </SelectContent>
                          </Select>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <FormField
                      control={contentForm.control}
                      name="titleAr"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Title (Arabic)</FormLabel>
                          <FormControl>
                            <Input placeholder="مقدمة في التخطيط الاستراتيجي" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={contentForm.control}
                      name="titleFr"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Title (French)</FormLabel>
                          <FormControl>
                            <Input placeholder="Introduction à la Planification Stratégique" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>

                  <FormField
                    control={contentForm.control}
                    name="duration"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Duration (Minutes)</FormLabel>
                        <FormControl>
                          <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <div className="space-y-2">
                    <Label>Content Details</Label>
                    <div className="border rounded-lg p-4 space-y-4">
                      <div className="grid grid-cols-2 gap-4">
                        <Input placeholder="Video URL / Interactive Link" />
                        <Input placeholder="Thumbnail URL" />
                      </div>
                      <Textarea placeholder="Content description and learning objectives" />
                      <div className="flex items-center gap-2">
                        <Button type="button" variant="outline" size="sm">
                          <Upload className="w-4 h-4 mr-2" />
                          Upload Content
                        </Button>
                        <Button type="button" variant="outline" size="sm">
                          <FileText className="w-4 h-4 mr-2" />
                          Add Resources
                        </Button>
                      </div>
                    </div>
                  </div>

                  <Button type="submit" disabled={createContentMutation.isPending}>
                    {createContentMutation.isPending ? "Creating..." : "Create Content"}
                  </Button>
                </form>
              </Form>
            </DialogContent>
          </Dialog>
        </div>

        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
          {elearningContent.map((content: SystemELearningContent) => (
            <Card key={content.id} className="hover:shadow-lg transition-shadow">
              <CardHeader>
                <div className="flex justify-between items-start">
                  <div>
                    <CardTitle className="text-lg">{content.titleEn}</CardTitle>
                    <CardDescription className="flex items-center gap-2 mt-1">
                      {getContentTypeIcon(content.contentType)}
                      <span className="capitalize">{content.contentType}</span>
                    </CardDescription>
                  </div>
                  <Badge className={getContentTypeColor(content.contentType)}>
                    {content.contentType}
                  </Badge>
                </div>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  <div className="flex justify-between text-sm">
                    <span>Duration:</span>
                    <span className="flex items-center gap-1">
                      <Clock className="w-3 h-3" />
                      {content.duration} min
                    </span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>Arabic Title:</span>
                    <span>{content.titleAr}</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>French Title:</span>
                    <span>{content.titleFr}</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>Status:</span>
                    <Badge variant={content.isActive ? "default" : "secondary"}>
                      {content.isActive ? "Active" : "Inactive"}
                    </Badge>
                  </div>
                  <div className="flex justify-between pt-2">
                    <Button size="sm" variant="outline">
                      <Eye className="w-4 h-4 mr-1" />
                      Preview
                    </Button>
                    <div className="flex gap-1">
                      <Button size="sm" variant="outline">
                        <Edit className="w-4 h-4" />
                      </Button>
                      <Button size="sm" variant="outline">
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    </DashboardLayout>
  );
}