import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { 
  Plus, 
  Book, 
  Download, 
  Upload, 
  FileText, 
  Edit,
  Trash2,
  Eye,
  Users,
  Award,
  HardDrive
} from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { 
  insertSystemEBookSchema,
  type SystemEBook,
  type SystemCertificate,
  type SystemLevel,
  type SystemChapter
} from "@shared/schema";

export default function EBooks() {
  const [selectedCertificate, setSelectedCertificate] = useState<string | null>(null);
  const [selectedLevel, setSelectedLevel] = useState<string | null>(null);
  const [selectedChapter, setSelectedChapter] = useState<string | null>(null);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch data
  const { data: certificates = [] } = useQuery({
    queryKey: ["backend/api/system-certificates"],
  });

  const { data: levels = [] } = useQuery({
    queryKey: ["backend/api/system-levels", selectedCertificate],
    enabled: !!selectedCertificate,
  });

  const { data: chapters = [] } = useQuery({
    queryKey: ["backend/api/system-chapters", selectedLevel],
    enabled: !!selectedLevel,
  });

  const { data: ebooks = [] } = useQuery({
    queryKey: ["backend/api/ebooks"],
  });

  // Form
  const ebookForm = useForm({
    resolver: zodResolver(insertSystemEBookSchema),
    defaultValues: {
      certificateId: "",
      levelId: "",
      chapterId: "",
      titleEn: "",
      titleAr: "",
      titleFr: "",
      author: "",
      description: "",
      fileUrl: "",
      fileSize: 0,
      createdBy: "sys-admin-1",
    },
  });

  // Mutations
  const createEbookMutation = useMutation({
    mutationFn: (data: any) => apiRequest("backend/api/ebooks", "POST", data),
    onSuccess: () => {
      toast({ title: "eBook created successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/ebooks"] });
      ebookForm.reset();
    },
    onError: (error) => {
      toast({ title: "Error creating eBook", description: error.message, variant: "destructive" });
    },
  });

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return "0 Bytes";
    const k = 1024;
    const sizes = ["Bytes", "KB", "MB", "GB"];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
  };

  return (
    <DashboardLayout userType="admin">
      <div className="space-y-8">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">eBooks Management</h1>
          <p className="text-muted-foreground">
            Manage digital books and resources for certificates, levels, and chapters
          </p>
        </div>

        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total eBooks</CardTitle>
              <Book className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{ebooks.length}</div>
              <p className="text-xs text-muted-foreground">Digital library</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total Downloads</CardTitle>
              <Download className="h-4 w-4 text-green-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {ebooks.reduce((total: number, book: SystemEBook) => total + book.downloadCount, 0)}
              </div>
              <p className="text-xs text-muted-foreground">All time</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Storage Used</CardTitle>
              <HardDrive className="h-4 w-4 text-blue-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {formatFileSize(ebooks.reduce((total: number, book: SystemEBook) => total + book.fileSize, 0))}
              </div>
              <p className="text-xs text-muted-foreground">Total size</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Active eBooks</CardTitle>
              <Award className="h-4 w-4 text-orange-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {ebooks.filter((book: SystemEBook) => book.isActive).length}
              </div>
              <p className="text-xs text-muted-foreground">Available for download</p>
            </CardContent>
          </Card>
        </div>

        <div className="flex justify-between items-center">
          <h2 className="text-xl font-semibold">Digital Library</h2>
          <Dialog>
            <DialogTrigger asChild>
              <Button>
                <Plus className="w-4 h-4 mr-2" />
                Add eBook
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-4xl">
              <DialogHeader>
                <DialogTitle>Add New eBook</DialogTitle>
              </DialogHeader>
              <Form {...ebookForm}>
                <form onSubmit={ebookForm.handleSubmit((data) => createEbookMutation.mutate(data))} className="space-y-4">
                  <div className="grid grid-cols-3 gap-4">
                    <FormField
                      control={ebookForm.control}
                      name="certificateId"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Certificate</FormLabel>
                          <Select onValueChange={(value) => {
                            field.onChange(value);
                            setSelectedCertificate(value);
                          }}>
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select certificate" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              {certificates.map((cert: SystemCertificate) => (
                                <SelectItem key={cert.id} value={cert.id}>{cert.nameEn}</SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={ebookForm.control}
                      name="levelId"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Level (Optional)</FormLabel>
                          <Select onValueChange={(value) => {
                            field.onChange(value);
                            setSelectedLevel(value);
                          }}>
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select level" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              {levels.map((level: SystemLevel) => (
                                <SelectItem key={level.id} value={level.id}>Level {level.levelNumber} - {level.nameEn}</SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={ebookForm.control}
                      name="chapterId"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Chapter (Optional)</FormLabel>
                          <Select onValueChange={field.onChange}>
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select chapter" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              {chapters.map((chapter: SystemChapter) => (
                                <SelectItem key={chapter.id} value={chapter.id}>Chapter {chapter.chapterNumber} - {chapter.titleEn}</SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                  
                  <div className="grid grid-cols-2 gap-4">
                    <FormField
                      control={ebookForm.control}
                      name="titleEn"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Title (English)</FormLabel>
                          <FormControl>
                            <Input placeholder="Strategic Leadership Handbook" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={ebookForm.control}
                      name="author"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Author</FormLabel>
                          <FormControl>
                            <Input placeholder="Dr. Ahmed Hassan" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <FormField
                      control={ebookForm.control}
                      name="titleAr"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Title (Arabic)</FormLabel>
                          <FormControl>
                            <Input placeholder="دليل القيادة الاستراتيجية" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={ebookForm.control}
                      name="titleFr"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Title (French)</FormLabel>
                          <FormControl>
                            <Input placeholder="Manuel de Leadership Stratégique" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>

                  <FormField
                    control={ebookForm.control}
                    name="description"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Description</FormLabel>
                        <FormControl>
                          <Textarea placeholder="Comprehensive guide to strategic leadership principles and practices..." {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <div className="grid grid-cols-2 gap-4">
                    <FormField
                      control={ebookForm.control}
                      name="fileUrl"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>File URL</FormLabel>
                          <FormControl>
                            <Input placeholder="https://storage.ibdl.net/ebooks/strategic-leadership.pdf" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={ebookForm.control}
                      name="fileSize"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>File Size (Bytes)</FormLabel>
                          <FormControl>
                            <Input type="number" {...field} onChange={(e) => field.onChange(Number(e.target.value))} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>

                  <div className="space-y-2">
                    <Label>File Upload</Label>
                    <div className="border rounded-lg p-6 border-dashed">
                      <div className="flex flex-col items-center justify-center space-y-2">
                        <Upload className="w-8 h-8 text-muted-foreground" />
                        <p className="text-sm text-muted-foreground">
                          Drag and drop your eBook file here, or click to browse
                        </p>
                        <p className="text-xs text-muted-foreground">
                          Supported formats: PDF, EPUB, MOBI (Max 50MB)
                        </p>
                        <Button type="button" variant="outline" size="sm">
                          <Upload className="w-4 h-4 mr-2" />
                          Choose File
                        </Button>
                      </div>
                    </div>
                  </div>

                  <Button type="submit" disabled={createEbookMutation.isPending}>
                    {createEbookMutation.isPending ? "Creating..." : "Create eBook"}
                  </Button>
                </form>
              </Form>
            </DialogContent>
          </Dialog>
        </div>

        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
          {ebooks.map((book: SystemEBook) => (
            <Card key={book.id} className="hover:shadow-lg transition-shadow">
              <CardHeader>
                <div className="flex justify-between items-start">
                  <div>
                    <CardTitle className="text-lg">{book.titleEn}</CardTitle>
                    <CardDescription className="flex items-center gap-2 mt-1">
                      <Users className="w-4 h-4" />
                      <span>by {book.author}</span>
                    </CardDescription>
                  </div>
                  <Badge variant={book.isActive ? "default" : "secondary"}>
                    {book.isActive ? "Active" : "Inactive"}
                  </Badge>
                </div>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  <div className="flex justify-between text-sm">
                    <span>Downloads:</span>
                    <span className="flex items-center gap-1">
                      <Download className="w-3 h-3" />
                      {book.downloadCount}
                    </span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>File Size:</span>
                    <span>{formatFileSize(book.fileSize)}</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>Arabic Title:</span>
                    <span>{book.titleAr}</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>French Title:</span>
                    <span>{book.titleFr}</span>
                  </div>
                  <div className="pt-2">
                    <p className="text-sm text-muted-foreground line-clamp-2">
                      {book.description}
                    </p>
                  </div>
                  <div className="flex justify-between pt-2">
                    <Button size="sm" variant="outline">
                      <Download className="w-4 h-4 mr-1" />
                      Download
                    </Button>
                    <div className="flex gap-1">
                      <Button size="sm" variant="outline">
                        <Eye className="w-4 h-4" />
                      </Button>
                      <Button size="sm" variant="outline">
                        <Edit className="w-4 h-4" />
                      </Button>
                      <Button size="sm" variant="outline">
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    </DashboardLayout>
  );
}