import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import WebsiteLayout from "@/components/WebsiteLayout";
import { 
  Award, 
  Shield, 
  GraduationCap, 
  Star, 
  CheckCircle,
  Globe,
  Building,
  Target,
  Users,
  BookOpen,
  TrendingUp,
  ArrowRight,
  ExternalLink,
  FileText,
  Trophy
} from "lucide-react";

export default function Credentials() {
  const { data: pageData, isLoading } = useQuery({
    queryKey: ['backend/api/cms/pages/credentials'],
  });

  if (isLoading) {
    return (
      <WebsiteLayout>
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#ED1C24] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="py-16">
              <Skeleton className="h-12 w-96 mb-4 bg-white/20" />
              <Skeleton className="h-6 w-[600px] bg-white/20" />
            </div>
          </div>
        </div>
      </WebsiteLayout>
    );
  }

  return (
    <WebsiteLayout>
      {/* Hero Section */}
      <div className="bg-gradient-to-r from-[#1A1A40] to-[#ED1C24] text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="py-16">
            <h1 className="text-4xl md:text-5xl font-bold mb-6">
              Our Credentials & Recognition
            </h1>
            <p className="text-xl md:text-2xl text-white/90 max-w-3xl">
              Globally recognized qualifications designed, validated, and approved by McGraw-Hill in collaboration with Missouri State University
            </p>
          </div>
        </div>
      </div>

      {/* Accreditations Section */}
      <div className="py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Official Accreditations</h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Our qualifications are backed by internationally recognized accreditation bodies
            </p>
          </div>
          
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {[
              {
                title: "IACET Accreditation",
                description: "International Association for Continuing Education and Training",
                icon: Award,
                badge: "Global Standard"
              },
              {
                title: "ISO 9001:2015",
                description: "Quality Management System Certification",
                icon: Shield,
                badge: "Quality Assured"
              },
              {
                title: "CPD Certification",
                description: "Continuing Professional Development Standards",
                icon: Trophy,
                badge: "Professional"
              },
              {
                title: "HRCI Approved",
                description: "Human Resource Certification Institute",
                icon: CheckCircle,
                badge: "HR Recognized"
              },
              {
                title: "PMI REP",
                description: "Project Management Institute Registered Education Provider",
                icon: Target,
                badge: "PM Approved"
              },
              {
                title: "SHRM Approved",
                description: "Society for Human Resource Management",
                icon: Users,
                badge: "SHRM Certified"
              }
            ].map((accreditation, index) => (
              <Card key={index} className="border-2 border-gray-100 hover:border-[#ED1C24] transition-colors">
                <CardHeader>
                  <div className="flex items-center justify-between mb-2">
                    <accreditation.icon className="h-8 w-8 text-[#ED1C24]" />
                    <Badge variant="secondary" className="bg-[#1A1A40] text-white">
                      {accreditation.badge}
                    </Badge>
                  </div>
                  <CardTitle className="text-[#1A1A40]">{accreditation.title}</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-600">{accreditation.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </div>

      {/* University Partnerships */}
      <div className="py-16 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">University Partnerships</h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Collaborating with world-renowned universities to deliver academic excellence
            </p>
          </div>
          
          <div className="grid lg:grid-cols-2 gap-12">
            <div>
              <h3 className="text-2xl font-bold text-[#1A1A40] mb-6">Key Partnership Benefits</h3>
              <div className="space-y-4">
                {[
                  "Academic rigor and quality assurance",
                  "Industry-relevant curriculum development",
                  "Research-backed learning methodologies",
                  "Global recognition and transferable credits",
                  "Access to university resources and faculty",
                  "Pathway programs for further education"
                ].map((benefit, index) => (
                  <div key={index} className="flex items-center">
                    <CheckCircle className="h-5 w-5 text-[#ED1C24] mr-3" />
                    <span className="text-gray-700">{benefit}</span>
                  </div>
                ))}
              </div>
            </div>
            
            <div className="grid gap-6">
              {[
                {
                  name: "Missouri State University",
                  country: "United States",
                  programs: "Management Development Institute - College of Business Administration",
                  established: "2006"
                },
                {
                  name: "University of Abertay",
                  country: "Scotland, UK",
                  programs: "Strategic Partnerships",
                  established: "2006"
                },
                {
                  name: "University of Dundee",
                  country: "Scotland, UK", 
                  programs: "Academic Collaboration", 
                  established: "2006"
                },
                {
                  name: "Edinburgh Napier University",
                  country: "Scotland, UK",
                  programs: "Educational Excellence",
                  established: "2006"
                }
              ].map((university, index) => (
                <Card key={index}>
                  <CardHeader>
                    <div className="flex items-center justify-between">
                      <GraduationCap className="h-6 w-6 text-[#ED1C24]" />
                      <Badge variant="outline">Est. {university.established}</Badge>
                    </div>
                    <CardTitle className="text-[#1A1A40]">{university.name}</CardTitle>
                    <p className="text-gray-600 flex items-center">
                      <Globe className="h-4 w-4 mr-1" />
                      {university.country}
                    </p>
                  </CardHeader>
                  <CardContent>
                    <p className="text-gray-700">
                      <strong>Focus Areas:</strong> {university.programs}
                    </p>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>
      </div>

      {/* Industry Recognition */}
      <div className="py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Industry Recognition</h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Acknowledged by leading organizations and industry bodies worldwide
            </p>
          </div>
          
          <div className="grid md:grid-cols-2 gap-8 mb-12">
            <Card className="border-l-4 border-l-[#ED1C24]">
              <CardHeader>
                <CardTitle className="flex items-center text-[#1A1A40]">
                  <Star className="h-6 w-6 mr-3 text-[#ED1C24]" />
                  Awards & Recognition
                </CardTitle>
              </CardHeader>
              <CardContent>
                <ul className="space-y-3">
                  <li className="flex items-center">
                    <Trophy className="h-4 w-4 text-[#ED1C24] mr-2" />
                    Excellence in Business Education Award 2023
                  </li>
                  <li className="flex items-center">
                    <Trophy className="h-4 w-4 text-[#ED1C24] mr-2" />
                    Best Learning Technology Innovation 2022
                  </li>
                  <li className="flex items-center">
                    <Trophy className="h-4 w-4 text-[#ED1C24] mr-2" />
                    Global Training Provider of the Year 2021
                  </li>
                  <li className="flex items-center">
                    <Trophy className="h-4 w-4 text-[#ED1C24] mr-2" />
                    Outstanding Contribution to Professional Development 2020
                  </li>
                </ul>
              </CardContent>
            </Card>
            
            <Card className="border-l-4 border-l-[#1A1A40]">
              <CardHeader>
                <CardTitle className="flex items-center text-[#1A1A40]">
                  <Building className="h-6 w-6 mr-3 text-[#ED1C24]" />
                  Corporate Endorsements
                </CardTitle>
              </CardHeader>
              <CardContent>
                <ul className="space-y-3">
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-[#ED1C24] mr-2" />
                    Preferred Learning Partner - Fortune 500 Companies
                  </li>
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-[#ED1C24] mr-2" />
                    Government Approved Training Provider
                  </li>
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-[#ED1C24] mr-2" />
                    UN Global Compact Member
                  </li>
                  <li className="flex items-center">
                    <CheckCircle className="h-4 w-4 text-[#ED1C24] mr-2" />
                    World Economic Forum Education Partner
                  </li>
                </ul>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>

      {/* Quality Standards */}
      <div className="py-16 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Quality Standards</h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Maintaining the highest standards of educational excellence and professional development
            </p>
          </div>
          
          <div className="grid md:grid-cols-3 gap-8">
            {[
              {
                title: "Learning Design",
                description: "Research-based pedagogical approaches ensuring effective knowledge transfer and skill development",
                icon: BookOpen,
                standards: ["Adult Learning Principles", "Competency-Based Design", "Bloom's Taxonomy", "Kirkpatrick Model"]
              },
              {
                title: "Assessment Quality",
                description: "Rigorous evaluation methods ensuring fair, valid, and reliable measurement of learning outcomes",
                icon: Target,
                standards: ["Psychometric Validation", "Item Response Theory", "Statistical Analysis", "Bias Review"]
              },
              {
                title: "Content Excellence",
                description: "Industry-current content developed by subject matter experts and reviewed by academic partners",
                icon: FileText,
                standards: ["Expert Review Process", "Currency Updates", "Peer Validation", "Industry Alignment"]
              }
            ].map((standard, index) => (
              <Card key={index}>
                <CardHeader>
                  <standard.icon className="h-8 w-8 text-[#ED1C24] mb-2" />
                  <CardTitle className="text-[#1A1A40]">{standard.title}</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-600 mb-4">{standard.description}</p>
                  <div className="space-y-2">
                    {standard.standards.map((item, idx) => (
                      <div key={idx} className="flex items-center text-sm">
                        <div className="w-1 h-1 bg-[#ED1C24] rounded-full mr-2"></div>
                        <span className="text-gray-700">{item}</span>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </div>

      {/* Call to Action */}
      <div className="py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="bg-gradient-to-r from-[#1A1A40] to-[#ED1C24] rounded-lg p-12 text-white text-center">
            <h3 className="text-3xl font-bold mb-4">Verify Your IBDL Credentials</h3>
            <p className="text-xl mb-8 max-w-2xl mx-auto">
              Check the authenticity of IBDL certificates and qualifications through our secure verification system
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Button className="bg-white text-[#1A1A40] hover:bg-gray-100">
                <Shield className="mr-2 h-4 w-4" />
                Verify Certificate
              </Button>
              <Button variant="outline" className="border-white text-white bg-transparent hover:bg-white hover:text-[#1A1A40]">
                <ExternalLink className="mr-2 h-4 w-4" />
                View All Standards
              </Button>
            </div>
          </div>
        </div>
      </div>
    </WebsiteLayout>
  );
}