import { useState } from 'react';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { useToast } from '@/hooks/use-toast';
import { Building, Users, Calendar, CreditCard, FileText, Plus, Edit, Trash2, UserPlus, Upload } from 'lucide-react';
import DashboardLayout from '@/components/layouts/DashboardLayout';
import { apiRequest } from '@/lib/queryClient';
import type { User, Organization, ExamBooking, CorporateCredits } from '@shared/schema';

interface Employee extends User {
  department?: string;
  position?: string;
  enrollmentDate?: string;
  status?: string;
}

interface CorporateStats {
  totalEmployees: number;
  activeEmployees: number;
  totalCredits: number;
  usedCredits: number;
  upcomingExams: number;
  completedExams: number;
  totalBookings: number;
  pendingBookings: number;
}

export default function CorporateAccountManagement() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [selectedTab, setSelectedTab] = useState('overview');
  const [employeeModalOpen, setEmployeeModalOpen] = useState(false);
  const [bulkUploadModalOpen, setBulkUploadModalOpen] = useState(false);
  const [examBookingModalOpen, setExamBookingModalOpen] = useState(false);
  const [selectedEmployee, setSelectedEmployee] = useState<Employee | null>(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [departmentFilter, setDepartmentFilter] = useState('all');

  // Fetch corporate stats
  const { data: corporateStats, isLoading: statsLoading } = useQuery<CorporateStats>({
    queryKey: ['backend/api/corporate/stats'],
  });

  // Fetch employees
  const { data: employees = [], isLoading: employeesLoading } = useQuery<Employee[]>({
    queryKey: ['backend/api/corporate/employees'],
  });

  // Fetch exam bookings
  const { data: examBookings = [], isLoading: bookingsLoading } = useQuery<ExamBooking[]>({
    queryKey: ['backend/api/corporate/exam-bookings'],
  });

  // Fetch corporate credits
  const { data: credits, isLoading: creditsLoading } = useQuery<CorporateCredits>({
    queryKey: ['backend/api/corporate/credits'],
  });

  // Add employee mutation
  const addEmployeeMutation = useMutation({
    mutationFn: async (employeeData: Partial<Employee>) => {
      return await apiRequest('backend/api/corporate/employees', {
        method: 'POST',
        body: employeeData,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/employees'] });
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/stats'] });
      setEmployeeModalOpen(false);
      setSelectedEmployee(null);
      toast({
        title: "Success",
        description: "Employee added successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message || "Failed to add employee",
        variant: "destructive",
      });
    },
  });

  // Update employee mutation
  const updateEmployeeMutation = useMutation({
    mutationFn: async ({ id, data }: { id: string; data: Partial<Employee> }) => {
      return await apiRequest(`backend/api/corporate/employees/${id}`, {
        method: 'PUT',
        body: data,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/employees'] });
      setEmployeeModalOpen(false);
      setSelectedEmployee(null);
      toast({
        title: "Success",
        description: "Employee updated successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message || "Failed to update employee",
        variant: "destructive",
      });
    },
  });

  // Delete employee mutation
  const deleteEmployeeMutation = useMutation({
    mutationFn: async (id: string) => {
      return await apiRequest(`backend/api/corporate/employees/${id}`, {
        method: 'DELETE',
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/employees'] });
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/stats'] });
      toast({
        title: "Success",
        description: "Employee deleted successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message || "Failed to delete employee",
        variant: "destructive",
      });
    },
  });

  // Book exam mutation
  const bookExamMutation = useMutation({
    mutationFn: async (bookingData: any) => {
      return await apiRequest('backend/api/corporate/exam-bookings', {
        method: 'POST',
        body: bookingData,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/exam-bookings'] });
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/stats'] });
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/credits'] });
      setExamBookingModalOpen(false);
      toast({
        title: "Success",
        description: "Exam booked successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message || "Failed to book exam",
        variant: "destructive",
      });
    },
  });

  // Bulk upload employees mutation
  const bulkUploadMutation = useMutation({
    mutationFn: async (file: File) => {
      const formData = new FormData();
      formData.append('file', file);
      return await apiRequest('backend/api/corporate/employees/bulk-upload', {
        method: 'POST',
        body: formData,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/employees'] });
      queryClient.invalidateQueries({ queryKey: ['backend/api/corporate/stats'] });
      setBulkUploadModalOpen(false);
      toast({
        title: "Success",
        description: "Employees uploaded successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message || "Failed to upload employees",
        variant: "destructive",
      });
    },
  });

  // Filter employees
  const filteredEmployees = employees.filter(employee => {
    const matchesSearch = employee.firstName?.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         employee.lastName?.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         employee.email?.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDepartment = departmentFilter === 'all' || employee.department === departmentFilter;
    return matchesSearch && matchesDepartment;
  });

  const handleEmployeeSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const formData = new FormData(e.currentTarget);
    const employeeData = {
      firstName: formData.get('firstName') as string,
      lastName: formData.get('lastName') as string,
      email: formData.get('email') as string,
      department: formData.get('department') as string,
      position: formData.get('position') as string,
      userType: 'learner',
    };

    if (selectedEmployee) {
      updateEmployeeMutation.mutate({ id: selectedEmployee.id, data: employeeData });
    } else {
      addEmployeeMutation.mutate(employeeData);
    }
  };

  const handleExamBookingSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const formData = new FormData(e.currentTarget);
    const bookingData = {
      employeeId: formData.get('employeeId') as string,
      examId: formData.get('examId') as string,
      testingCenterId: formData.get('testingCenterId') as string,
      preferredDate: formData.get('preferredDate') as string,
      preferredTime: formData.get('preferredTime') as string,
      notes: formData.get('notes') as string,
    };

    bookExamMutation.mutate(bookingData);
  };

  const handleBulkUpload = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const formData = new FormData(e.currentTarget);
    const file = formData.get('file') as File;
    if (file) {
      bulkUploadMutation.mutate(file);
    }
  };

  return (
    <DashboardLayout userType="corporate">
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Corporate Account Management</h1>
            <p className="text-gray-600 mt-2">Manage your corporate employees and exam bookings</p>
          </div>
          <div className="flex items-center gap-3">
            <Button
              onClick={() => setExamBookingModalOpen(true)}
              className="bg-blue-600 hover:bg-blue-700"
            >
              <Calendar className="w-4 h-4 mr-2" />
              Book Exam
            </Button>
            <Button
              onClick={() => setBulkUploadModalOpen(true)}
              variant="outline"
            >
              <Upload className="w-4 h-4 mr-2" />
              Bulk Upload
            </Button>
            <Button
              onClick={() => {
                setSelectedEmployee(null);
                setEmployeeModalOpen(true);
              }}
              className="bg-green-600 hover:bg-green-700"
            >
              <UserPlus className="w-4 h-4 mr-2" />
              Add Employee
            </Button>
          </div>
        </div>

        <Tabs value={selectedTab} onValueChange={setSelectedTab}>
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="employees">Employees</TabsTrigger>
            <TabsTrigger value="bookings">Exam Bookings</TabsTrigger>
            <TabsTrigger value="credits">Credits</TabsTrigger>
          </TabsList>

          <TabsContent value="overview" className="space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Total Employees</CardTitle>
                  <Users className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{statsLoading ? '...' : corporateStats?.totalEmployees || 0}</div>
                  <p className="text-xs text-muted-foreground">
                    {statsLoading ? '...' : corporateStats?.activeEmployees || 0} active
                  </p>
                </CardContent>
              </Card>
              
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Available Credits</CardTitle>
                  <CreditCard className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{creditsLoading ? '...' : (credits?.totalCredits || 0) - (credits?.usedCredits || 0)}</div>
                  <p className="text-xs text-muted-foreground">
                    {creditsLoading ? '...' : credits?.usedCredits || 0} used of {creditsLoading ? '...' : credits?.totalCredits || 0}
                  </p>
                </CardContent>
              </Card>
              
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Upcoming Exams</CardTitle>
                  <Calendar className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{statsLoading ? '...' : corporateStats?.upcomingExams || 0}</div>
                  <p className="text-xs text-muted-foreground">
                    {statsLoading ? '...' : corporateStats?.completedExams || 0} completed
                  </p>
                </CardContent>
              </Card>
              
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Total Bookings</CardTitle>
                  <FileText className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{statsLoading ? '...' : corporateStats?.totalBookings || 0}</div>
                  <p className="text-xs text-muted-foreground">
                    {statsLoading ? '...' : corporateStats?.pendingBookings || 0} pending approval
                  </p>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="employees" className="space-y-6">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-4">
                <Input
                  placeholder="Search employees..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-64"
                />
                <Select value={departmentFilter} onValueChange={setDepartmentFilter}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Filter by department" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Departments</SelectItem>
                    <SelectItem value="HR">HR</SelectItem>
                    <SelectItem value="IT">IT</SelectItem>
                    <SelectItem value="Finance">Finance</SelectItem>
                    <SelectItem value="Marketing">Marketing</SelectItem>
                    <SelectItem value="Operations">Operations</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <Card>
              <CardHeader>
                <CardTitle>Employee List</CardTitle>
              </CardHeader>
              <CardContent>
                {employeesLoading ? (
                  <div className="flex items-center justify-center h-32">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                  </div>
                ) : (
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Name</TableHead>
                        <TableHead>Email</TableHead>
                        <TableHead>Department</TableHead>
                        <TableHead>Position</TableHead>
                        <TableHead>Status</TableHead>
                        <TableHead>Actions</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {filteredEmployees.map((employee) => (
                        <TableRow key={employee.id}>
                          <TableCell>{employee.firstName} {employee.lastName}</TableCell>
                          <TableCell>{employee.email}</TableCell>
                          <TableCell>{employee.department || 'N/A'}</TableCell>
                          <TableCell>{employee.position || 'N/A'}</TableCell>
                          <TableCell>
                            <Badge variant={employee.status === 'active' ? 'default' : 'secondary'}>
                              {employee.status || 'Active'}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            <div className="flex items-center gap-2">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => {
                                  setSelectedEmployee(employee);
                                  setEmployeeModalOpen(true);
                                }}
                              >
                                <Edit className="w-4 h-4" />
                              </Button>
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => deleteEmployeeMutation.mutate(employee.id)}
                                disabled={deleteEmployeeMutation.isPending}
                              >
                                <Trash2 className="w-4 h-4" />
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="bookings" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Exam Bookings</CardTitle>
              </CardHeader>
              <CardContent>
                {bookingsLoading ? (
                  <div className="flex items-center justify-center h-32">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                  </div>
                ) : (
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Employee</TableHead>
                        <TableHead>Exam</TableHead>
                        <TableHead>Testing Center</TableHead>
                        <TableHead>Date</TableHead>
                        <TableHead>Status</TableHead>
                        <TableHead>Actions</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {examBookings.map((booking) => (
                        <TableRow key={booking.id}>
                          <TableCell>{booking.employeeName}</TableCell>
                          <TableCell>{booking.examTitle}</TableCell>
                          <TableCell>{booking.testingCenterName}</TableCell>
                          <TableCell>{booking.examDate}</TableCell>
                          <TableCell>
                            <Badge variant={booking.status === 'confirmed' ? 'default' : 'secondary'}>
                              {booking.status}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            <Button variant="outline" size="sm">
                              View Details
                            </Button>
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="credits" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Credit Management</CardTitle>
              </CardHeader>
              <CardContent>
                {creditsLoading ? (
                  <div className="flex items-center justify-center h-32">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                  </div>
                ) : (
                  <div className="space-y-4">
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                      <div className="text-center p-4 bg-blue-50 rounded-lg">
                        <div className="text-2xl font-bold text-blue-600">{credits?.totalCredits || 0}</div>
                        <div className="text-sm text-gray-600">Total Credits</div>
                      </div>
                      <div className="text-center p-4 bg-green-50 rounded-lg">
                        <div className="text-2xl font-bold text-green-600">{(credits?.totalCredits || 0) - (credits?.usedCredits || 0)}</div>
                        <div className="text-sm text-gray-600">Available Credits</div>
                      </div>
                      <div className="text-center p-4 bg-red-50 rounded-lg">
                        <div className="text-2xl font-bold text-red-600">{credits?.usedCredits || 0}</div>
                        <div className="text-sm text-gray-600">Used Credits</div>
                      </div>
                    </div>
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>Credit Usage</span>
                        <span>{credits?.usedCredits || 0} / {credits?.totalCredits || 0}</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div 
                          className="bg-blue-600 h-2 rounded-full" 
                          style={{ width: `${credits ? (credits.usedCredits / credits.totalCredits) * 100 : 0}%` }}
                        ></div>
                      </div>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        {/* Add/Edit Employee Modal */}
        <Dialog open={employeeModalOpen} onOpenChange={setEmployeeModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>{selectedEmployee ? 'Edit Employee' : 'Add Employee'}</DialogTitle>
            </DialogHeader>
            <form onSubmit={handleEmployeeSubmit} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="firstName">First Name</Label>
                  <Input
                    id="firstName"
                    name="firstName"
                    defaultValue={selectedEmployee?.firstName || ''}
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="lastName">Last Name</Label>
                  <Input
                    id="lastName"
                    name="lastName"
                    defaultValue={selectedEmployee?.lastName || ''}
                    required
                  />
                </div>
              </div>
              <div>
                <Label htmlFor="email">Email</Label>
                <Input
                  id="email"
                  name="email"
                  type="email"
                  defaultValue={selectedEmployee?.email || ''}
                  required
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="department">Department</Label>
                  <Select name="department" defaultValue={selectedEmployee?.department || ''}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select department" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="HR">HR</SelectItem>
                      <SelectItem value="IT">IT</SelectItem>
                      <SelectItem value="Finance">Finance</SelectItem>
                      <SelectItem value="Marketing">Marketing</SelectItem>
                      <SelectItem value="Operations">Operations</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label htmlFor="position">Position</Label>
                  <Input
                    id="position"
                    name="position"
                    defaultValue={selectedEmployee?.position || ''}
                  />
                </div>
              </div>
              <div className="flex justify-end gap-3">
                <Button type="button" variant="outline" onClick={() => setEmployeeModalOpen(false)}>
                  Cancel
                </Button>
                <Button 
                  type="submit" 
                  disabled={addEmployeeMutation.isPending || updateEmployeeMutation.isPending}
                >
                  {selectedEmployee ? 'Update' : 'Add'} Employee
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>

        {/* Bulk Upload Modal */}
        <Dialog open={bulkUploadModalOpen} onOpenChange={setBulkUploadModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Bulk Upload Employees</DialogTitle>
            </DialogHeader>
            <form onSubmit={handleBulkUpload} className="space-y-4">
              <div>
                <Label htmlFor="file">CSV File</Label>
                <Input
                  id="file"
                  name="file"
                  type="file"
                  accept=".csv"
                  required
                />
                <p className="text-sm text-gray-600 mt-1">
                  Upload a CSV file with columns: firstName, lastName, email, department, position
                </p>
              </div>
              <div className="flex justify-end gap-3">
                <Button type="button" variant="outline" onClick={() => setBulkUploadModalOpen(false)}>
                  Cancel
                </Button>
                <Button type="submit" disabled={bulkUploadMutation.isPending}>
                  Upload Employees
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>

        {/* Exam Booking Modal */}
        <Dialog open={examBookingModalOpen} onOpenChange={setExamBookingModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Book Exam for Employee</DialogTitle>
            </DialogHeader>
            <form onSubmit={handleExamBookingSubmit} className="space-y-4">
              <div>
                <Label htmlFor="employeeId">Employee</Label>
                <Select name="employeeId" required>
                  <SelectTrigger>
                    <SelectValue placeholder="Select employee" />
                  </SelectTrigger>
                  <SelectContent>
                    {employees.map((employee) => (
                      <SelectItem key={employee.id} value={employee.id}>
                        {employee.firstName} {employee.lastName}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="examId">Exam</Label>
                <Select name="examId" required>
                  <SelectTrigger>
                    <SelectValue placeholder="Select exam" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="leadership-cert">Leadership Certification</SelectItem>
                    <SelectItem value="project-mgmt">Project Management</SelectItem>
                    <SelectItem value="digital-marketing">Digital Marketing</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="testingCenterId">Testing Center</Label>
                <Select name="testingCenterId" required>
                  <SelectTrigger>
                    <SelectValue placeholder="Select testing center" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="center-1">Downtown Testing Center</SelectItem>
                    <SelectItem value="center-2">Business District Center</SelectItem>
                    <SelectItem value="center-3">University Campus Center</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="preferredDate">Preferred Date</Label>
                  <Input
                    id="preferredDate"
                    name="preferredDate"
                    type="date"
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="preferredTime">Preferred Time</Label>
                  <Input
                    id="preferredTime"
                    name="preferredTime"
                    type="time"
                    required
                  />
                </div>
              </div>
              <div>
                <Label htmlFor="notes">Notes</Label>
                <Textarea
                  id="notes"
                  name="notes"
                  placeholder="Any special requirements or notes..."
                />
              </div>
              <div className="flex justify-end gap-3">
                <Button type="button" variant="outline" onClick={() => setExamBookingModalOpen(false)}>
                  Cancel
                </Button>
                <Button type="submit" disabled={bookExamMutation.isPending}>
                  Book Exam
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>
      </div>
    </DashboardLayout>
  );
}