import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { 
  MapPin, Phone, Mail, Clock, Globe, MessageSquare, 
  Send, User, Building, Headphones, Calendar 
} from "lucide-react";

export default function Contact() {
  const contactInfo = [
    {
      icon: MapPin,
      title: "Global Headquarters",
      details: [
        "IBDL Learning Group",
        "International Business Center",
        "Cairo, Egypt"
      ],
      color: "bg-blue-50 text-blue-700"
    },
    {
      icon: Phone,
      title: "Phone Support",
      details: [
        "+20 2 123 4567",
        "+1 800 IBDL-LEARN",
        "Available 24/7"
      ],
      color: "bg-green-50 text-green-700"
    },
    {
      icon: Mail,
      title: "Email Support",
      details: [
        "info@ibdl.net",
        "support@ibdl.net",
        "admissions@ibdl.net"
      ],
      color: "bg-purple-50 text-purple-700"
    },
    {
      icon: Clock,
      title: "Business Hours",
      details: [
        "Monday - Friday: 9AM - 6PM",
        "Saturday: 10AM - 4PM",
        "Sunday: Closed"
      ],
      color: "bg-orange-50 text-orange-700"
    }
  ];

  const departments = [
    {
      name: "General Inquiries",
      email: "info@ibdl.net",
      description: "General questions about IBDL programs and services"
    },
    {
      name: "Admissions",
      email: "admissions@ibdl.net",
      description: "Course enrollment and qualification requirements"
    },
    {
      name: "Technical Support",
      email: "support@ibdl.net",
      description: "Platform issues and technical assistance"
    },
    {
      name: "Partnership",
      email: "partners@ibdl.net",
      description: "Training center and corporate partnerships"
    },
    {
      name: "Accreditation",
      email: "accreditation@ibdl.net",
      description: "Accreditation services and certification"
    },
    {
      name: "Media & Press",
      email: "media@ibdl.net",
      description: "Media inquiries and press releases"
    }
  ];

  const regions = [
    {
      region: "Middle East & North Africa",
      countries: ["Egypt", "Saudi Arabia", "UAE", "Jordan", "Morocco"],
      contact: "mena@ibdl.net"
    },
    {
      region: "Europe",
      countries: ["United Kingdom", "Germany", "France", "Netherlands"],
      contact: "europe@ibdl.net"
    },
    {
      region: "Asia Pacific",
      countries: ["Singapore", "Malaysia", "Indonesia", "Philippines"],
      contact: "apac@ibdl.net"
    },
    {
      region: "Americas",
      countries: ["United States", "Canada", "Brazil", "Mexico"],
      contact: "americas@ibdl.net"
    }
  ];

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="text-center">
            <Badge variant="secondary" className="mb-4 bg-red-100 text-red-700">
              Get In Touch
            </Badge>
            <h1 className="text-4xl md:text-5xl font-bold mb-6">
              Contact IBDL
            </h1>
            <p className="text-xl text-gray-300 max-w-3xl mx-auto">
              We're here to help you on your learning journey. Get in touch with our team 
              for questions about programs, partnerships, or general inquiries.
            </p>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        {/* Contact Information */}
        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8 mb-16">
          {contactInfo.map((info, index) => (
            <Card key={index} className={`${info.color} border-2 hover:shadow-lg transition-shadow`}>
              <CardHeader>
                <div className="flex items-center space-x-3">
                  <div className="w-10 h-10 bg-white rounded-full flex items-center justify-center">
                    <info.icon className="w-5 h-5 text-[#1A1A40]" />
                  </div>
                  <CardTitle className="text-lg font-bold text-[#1A1A40]">{info.title}</CardTitle>
                </div>
              </CardHeader>
              <CardContent>
                <div className="space-y-1">
                  {info.details.map((detail, detailIndex) => (
                    <p key={detailIndex} className="text-sm text-gray-700">{detail}</p>
                  ))}
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Contact Form and Departments */}
        <div className="grid lg:grid-cols-2 gap-12 mb-16">
          {/* Contact Form */}
          <Card>
            <CardHeader>
              <CardTitle className="text-2xl font-bold text-[#1A1A40] flex items-center">
                <MessageSquare className="w-6 h-6 mr-3" />
                Send us a Message
              </CardTitle>
              <p className="text-gray-600">
                Fill out the form below and we'll get back to you within 24 hours
              </p>
            </CardHeader>
            <CardContent>
              <form className="space-y-6">
                <div className="grid md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      First Name *
                    </label>
                    <Input placeholder="Your first name" className="w-full" />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Last Name *
                    </label>
                    <Input placeholder="Your last name" className="w-full" />
                  </div>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Email Address *
                  </label>
                  <Input type="email" placeholder="your.email@example.com" className="w-full" />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Organization
                  </label>
                  <Input placeholder="Your organization or company" className="w-full" />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Inquiry Type
                  </label>
                  <select className="w-full p-2 border border-gray-300 rounded-md">
                    <option>General Inquiry</option>
                    <option>Program Information</option>
                    <option>Partnership Opportunity</option>
                    <option>Technical Support</option>
                    <option>Media/Press</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Message *
                  </label>
                  <Textarea 
                    placeholder="Tell us about your inquiry..." 
                    className="w-full h-32" 
                  />
                </div>
                <Button className="w-full bg-[#ED1C24] hover:bg-[#C41E1E] text-white">
                  <Send className="w-5 h-5 mr-2" />
                  Send Message
                </Button>
              </form>
            </CardContent>
          </Card>

          {/* Departments */}
          <Card>
            <CardHeader>
              <CardTitle className="text-2xl font-bold text-[#1A1A40] flex items-center">
                <Building className="w-6 h-6 mr-3" />
                Contact by Department
              </CardTitle>
              <p className="text-gray-600">
                Reach out to the specific department for faster assistance
              </p>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {departments.map((dept, index) => (
                  <div key={index} className="border-l-4 border-[#ED1C24] pl-4">
                    <h3 className="font-semibold text-[#1A1A40]">{dept.name}</h3>
                    <p className="text-sm text-gray-600 mb-1">{dept.description}</p>
                    <a 
                      href={`mailto:${dept.email}`}
                      className="text-sm text-[#ED1C24] hover:underline"
                    >
                      {dept.email}
                    </a>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Regional Offices */}
        <div className="mb-16">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Regional Offices</h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              Connect with our regional teams for localized support and services
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8">
            {regions.map((region, index) => (
              <Card key={index} className="hover:shadow-lg transition-shadow">
                <CardHeader>
                  <CardTitle className="text-lg font-bold text-[#1A1A40] flex items-center">
                    <Globe className="w-5 h-5 mr-2" />
                    {region.region}
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div>
                      <h4 className="font-semibold text-sm text-gray-700 mb-1">Countries:</h4>
                      <div className="flex flex-wrap gap-1">
                        {region.countries.map((country, countryIndex) => (
                          <Badge key={countryIndex} variant="outline" className="text-xs">
                            {country}
                          </Badge>
                        ))}
                      </div>
                    </div>
                    <div>
                      <h4 className="font-semibold text-sm text-gray-700 mb-1">Contact:</h4>
                      <a 
                        href={`mailto:${region.contact}`}
                        className="text-sm text-[#ED1C24] hover:underline"
                      >
                        {region.contact}
                      </a>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Support Options */}
        <div className="grid md:grid-cols-3 gap-8 mb-16">
          <Card className="bg-blue-50 border-blue-200">
            <CardHeader>
              <CardTitle className="text-xl font-bold text-[#1A1A40] flex items-center">
                <Headphones className="w-6 h-6 mr-3" />
                Live Chat Support
              </CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-gray-600 mb-4">
                Get instant help with our 24/7 live chat support for urgent inquiries
              </p>
              <Button className="w-full bg-[#ED1C24] hover:bg-[#C41E1E] text-white">
                Start Chat
              </Button>
            </CardContent>
          </Card>

          <Card className="bg-green-50 border-green-200">
            <CardHeader>
              <CardTitle className="text-xl font-bold text-[#1A1A40] flex items-center">
                <Calendar className="w-6 h-6 mr-3" />
                Schedule a Call
              </CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-gray-600 mb-4">
                Book a consultation call with our education advisors
              </p>
              <Button className="w-full bg-[#ED1C24] hover:bg-[#C41E1E] text-white">
                Book Call
              </Button>
            </CardContent>
          </Card>

          <Card className="bg-purple-50 border-purple-200">
            <CardHeader>
              <CardTitle className="text-xl font-bold text-[#1A1A40] flex items-center">
                <User className="w-6 h-6 mr-3" />
                Student Portal
              </CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-gray-600 mb-4">
                Access your account and get support through the student portal
              </p>
              <Button className="w-full bg-[#ED1C24] hover:bg-[#C41E1E] text-white">
                Portal Login
              </Button>
            </CardContent>
          </Card>
        </div>

        {/* Map Section */}
        <Card className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <CardContent className="py-16 text-center">
            <h2 className="text-3xl font-bold mb-4">Visit Our Global Headquarters</h2>
            <p className="text-xl text-gray-300 mb-8 max-w-2xl mx-auto">
              Located in Cairo, Egypt, our headquarters serves as the hub for our global operations
            </p>
            <div className="bg-white rounded-lg h-64 flex items-center justify-center text-gray-500">
              <div className="text-center">
                <MapPin className="w-12 h-12 mx-auto mb-4" />
                <p>Interactive Map Coming Soon</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}