import React from "react";
import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Target, TrendingUp, Globe, Users, Zap, BookOpen, Award, Building } from "lucide-react";
import Header from "@/components/Header";
import Footer from "@/components/Footer";

export default function CMSStrategy() {


  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/strategy"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <>
        <Header />

        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mt-12">
                {[1, 2, 3, 4].map((i) => (
                  <Skeleton key={i} className="h-48 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
        <Footer />
      </>
    );
  }

  const strategicGoals = [
    {
      number: "01",
      title: "Catalyzing Exponential Value Generation",
      description: "The organization aims to expand its offerings through open innovation and partnerships, delivering cutting-edge certifications and educational solutions to meet evolving customer needs, thereby driving tangible outcomes in education and professional growth.",
      icon: Zap,
      color: "bg-red-50 text-red-700",
      borderColor: "border-red-200"
    },
    {
      number: "02", 
      title: "Unparalleled Learning-Driven Organization Model",
      description: "The company has established a sustainable learning culture that integrates people learning, leadership development, and machine learning. Through investing in employee growth and development, they foster innovation, collaboration, and continuous improvement, ensuring success within the educational landscape.",
      icon: BookOpen,
      color: "bg-blue-50 text-blue-700",
      borderColor: "border-blue-200"
    },
    {
      number: "03",
      title: "Digital Revolution with Disruptive Solutions",
      description: "The organization is strategically expanding into a digital ecosystem tailored for education. Their goal is to advance digital transformation, empowering learners and institutions to maximize investments. Innovative digital products are designed to enhance learning experiences, outcomes, and long-term success in the digital era.",
      icon: TrendingUp,
      color: "bg-purple-50 text-purple-700",
      borderColor: "border-purple-200"
    },
    {
      number: "04",
      title: "Empowering Universities for Unprecedented Excellence",
      description: "Providing support to universities, the organization aims to enhance their international standing and performance. Customized solutions align with labor market demands, strengthening digital capabilities to contribute to sustained excellence and success in today's digitally-driven world.",
      icon: Building,
      color: "bg-emerald-50 text-emerald-700",
      borderColor: "border-emerald-200"
    }
  ];

  const strategicApproaches = [
    {
      title: "Disruptive Innovation",
      description: "We will challenge traditional learning paradigms and embrace disruptive innovation as the driving force behind our solutions. By pushing the boundaries of what is possible, we will pioneer cutting-edge technologies, methodologies, and approaches that revolutionize how learning is delivered, experienced, and absorbed."
    },
    {
      title: "Agile Development",
      description: "We intend to implement an agile development methodology in order to swiftly alter and improve our solutions. This approach will allow us to quickly adapt to emerging trends, learner feedback, and technological advancements, ensuring that our solutions remain at the forefront of innovation."
    },
    {
      title: "Moonshot Collaborations",
      description: "We will actively seek bold collaborations with visionary partners, such as leading educational institutions, forward-thinking organizations, and groundbreaking technology companies. By pooling our collective expertise, resources, and audacious aspirations, we will tackle the most formidable learning challenges."
    },
    {
      title: "Exponential Scale",
      description: "We will keep to find ways to have an endlessly large influence. We will strive to reach and empower many individuals and organizations worldwide through scalable technology infrastructure, strategic partnerships, and widespread adoption."
    },
    {
      title: "Continuous Learning Loop",
      description: "We will establish a dynamic, iterative approach learning loop that fuels continuous improvement. We will continuously refine our solutions by actively gathering and analyzing data, feedback, and insights from learners, educators, and organizations."
    },
    {
      title: "Bold Leadership",
      description: "We will cultivate a culture of bold leadership that encourages audacious thinking, calculated risk-taking, and relentless pursuit of excellence. Our leaders will inspire and empower our teams to push boundaries, challenge conventions, and embrace failure as a catalyst for growth and innovation."
    }
  ];

  return (
    <>
      <Header />
      
      <div className="min-h-screen bg-gray-50">
        {/* CMS Content Header */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="text-center">
              <Badge variant="secondary" className="mb-4 bg-red-100 text-red-700">
                Strategic Vision
              </Badge>
              <h1 className="text-4xl md:text-5xl font-bold mb-6">
                {(pageData && typeof pageData === 'object' && 'title' in pageData) 
                  ? String((pageData as any).title) 
                  : "IBDL Strategic Framework"}
              </h1>
              <p className="text-xl text-gray-300 max-w-3xl mx-auto">
                {(pageData && typeof pageData === 'object' && 'content' in pageData) 
                  ? String((pageData as any).content) 
                  : "Our strategic roadmap guides us toward becoming the world's leading learning-driven organization."}
              </p>
              <div className="mt-8 text-sm text-gray-400">
                {(pageData && typeof pageData === 'object' && 'updatedAt' in pageData) ? (
                  <span>Last updated: {new Date((pageData as any).updatedAt).toLocaleDateString()}</span>
                ) : null}
              </div>
            </div>
          </div>
        </div>

      {/* Mission Statement */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#ED1C24] text-white rounded-2xl p-8 mb-16">
          <div className="text-center">
            <h2 className="text-3xl font-bold mb-6">Our Mission</h2>
            <p className="text-xl leading-relaxed max-w-4xl mx-auto">
              We strive to create innovative and digitalized learning solutions that empower individuals and organizations to progress and thrive.
            </p>
          </div>
        </div>

        {/* Strategic Approach */}
        <div className="mb-16">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Our Strategic Approach</h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              In order to accomplish our mission, we will implement the following bold and strategic approach
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-16">
            {strategicApproaches.map((approach, index) => (
              <Card key={index} className="border-l-4 border-l-[#ED1C24] hover:shadow-lg transition-shadow duration-300">
                <CardHeader>
                  <CardTitle className="text-lg font-bold text-[#1A1A40]">{index + 1}. {approach.title}</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-600 leading-relaxed text-sm">{approach.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Strategic Goals */}
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">Our Goals</h2>
          <p className="text-lg text-gray-600 max-w-2xl mx-auto">
            Four transformative goals that drive our mission to revolutionize global learning
          </p>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mb-16">
          {strategicGoals.map((goal, index) => {
            const IconComponent = goal.icon;
            return (
              <Card key={index} className={`border-2 ${goal.borderColor} hover:shadow-lg transition-shadow duration-300`}>
                <CardHeader className={`${goal.color} rounded-t-lg`}>
                  <div className="flex items-center space-x-4">
                    <div className="w-12 h-12 bg-white rounded-full flex items-center justify-center">
                      <IconComponent className="w-6 h-6 text-gray-700" />
                    </div>
                    <div>
                      <Badge variant="outline" className="mb-2 border-current">
                        Goal {goal.number}
                      </Badge>
                      <CardTitle className="text-xl font-bold">{goal.title}</CardTitle>
                    </div>
                  </div>
                </CardHeader>
                <CardContent className="p-6">
                  <p className="text-gray-600 leading-relaxed">{goal.description}</p>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* CMS Integration Notice */}
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 text-center">
          <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
          <p className="text-blue-700">
            This page content is managed through the CMS and can be updated by administrators.
            {pageData && typeof pageData === 'object' && 'id' in pageData ? (
              <span className="block mt-2 text-sm">
                Page ID: {String((pageData as any).id)} | Status: {String((pageData as any).status)}
              </span>
            ) : null}
          </p>
        </div>
      </div>
      </div>

      
      <Footer />
    </>
  );
}