import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { 
  Building2, 
  Users, 
  Globe, 
  Award, 
  BookOpen, 
  Target, 
  Lightbulb,
  TrendingUp,
  GraduationCap,
  Briefcase,
  UserCheck,
  Settings,
  Zap,
  CheckCircle,
  Star,
  ArrowRight
} from "lucide-react";
import SharedLayout from "@/components/SharedLayout";

export default function CMSSolutions() {
  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/solutions"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <SharedLayout>
        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mt-12">
                {[1, 2, 3, 4, 5, 6].map((i) => (
                  <Skeleton key={i} className="h-64 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  const enterpriseSolutions = [
    {
      title: "Corporate Learning Solutions",
      tagline: "Empower Your Workforce",
      description: "Comprehensive learning programs designed to enhance organizational capabilities and drive business performance through skill development and professional growth.",
      features: [
        "Customized learning pathways aligned with business objectives",
        "Executive leadership development programs", 
        "Digital transformation training modules",
        "Performance analytics and progress tracking",
        "Multi-language support for global teams"
      ],
      benefits: [
        "Increased employee engagement and retention",
        "Enhanced organizational performance",
        "Improved competitive advantage",
        "Accelerated digital transformation"
      ],
      icon: Building2,
      color: "from-blue-400 to-indigo-500",
      ctaText: "Transform Your Organization"
    },
    {
      title: "Training Center Accreditation",
      tagline: "Become an Authorized Provider",
      description: "Join our global network of accredited training centers and deliver world-class IBDL qualifications to learners in your region with comprehensive support and resources.",
      features: [
        "Comprehensive accreditation process and guidelines",
        "Access to all IBDL curriculum materials",
        "Ongoing trainer certification and development",
        "Marketing and promotional support materials",
        "Quality assurance and assessment tools"
      ],
      benefits: [
        "Revenue generation through certification programs",
        "Enhanced institutional reputation",
        "Access to global learning network",
        "Continuous professional development opportunities"
      ],
      icon: GraduationCap,
      color: "from-green-400 to-emerald-500",
      ctaText: "Start Accreditation Process"
    },
    {
      title: "Executive Education Programs",
      tagline: "Strategic Leadership Development",
      description: "Elite programs designed for senior executives and C-suite leaders focusing on strategic thinking, innovation management, and organizational transformation.",
      features: [
        "Executive-level curriculum design",
        "Case study methodologies from global organizations",
        "Strategic leadership and decision-making modules",
        "Innovation and change management frameworks",
        "Peer networking and collaboration opportunities"
      ],
      benefits: [
        "Enhanced strategic thinking capabilities",
        "Improved organizational leadership",
        "Expanded professional network",
        "Competitive edge in dynamic markets"
      ],
      icon: Briefcase,
      color: "from-purple-400 to-pink-500",
      ctaText: "Explore Executive Programs"
    },
    {
      title: "Professional Assessment Services",
      tagline: "Validate Competencies",
      description: "Comprehensive assessment solutions including psychometric evaluations, skills assessments, and competency validations to support talent development and recruitment.",
      features: [
        "Psychometric assessment batteries",
        "Skills and competency evaluations",
        "Personality and leadership assessments",
        "Custom assessment development",
        "Comprehensive reporting and analytics"
      ],
      benefits: [
        "Data-driven talent decisions",
        "Improved hiring accuracy",
        "Enhanced employee development",
        "Reduced recruitment risks"
      ],
      icon: UserCheck,
      color: "from-red-400 to-orange-500",
      ctaText: "Discover Assessment Solutions"
    },
    {
      title: "Digital Learning Platforms",
      tagline: "Technology-Enabled Learning",
      description: "State-of-the-art digital learning infrastructure providing seamless online education experiences with advanced analytics and engagement features.",
      features: [
        "Cloud-based learning management system",
        "Mobile-responsive design and accessibility",
        "Advanced analytics and reporting dashboard",
        "Interactive content delivery mechanisms",
        "Integration with existing enterprise systems"
      ],
      benefits: [
        "Scalable learning delivery",
        "Enhanced learner engagement",
        "Cost-effective training solutions",
        "Real-time performance insights"
      ],
      icon: Settings,
      color: "from-teal-400 to-cyan-500",
      ctaText: "Explore Digital Solutions"
    },
    {
      title: "Consulting & Advisory Services",
      tagline: "Strategic Guidance",
      description: "Expert consulting services to help organizations design and implement effective learning strategies, organizational development programs, and performance improvement initiatives.",
      features: [
        "Organizational learning strategy development",
        "Competency framework design",
        "Learning needs analysis and gap assessment",
        "Change management and implementation support",
        "ROI measurement and evaluation frameworks"
      ],
      benefits: [
        "Customized learning strategies",
        "Optimized training investments",
        "Accelerated implementation timelines",
        "Measurable performance improvements"
      ],
      icon: Lightbulb,
      color: "from-yellow-400 to-amber-500",
      ctaText: "Schedule Consultation"
    }
  ];

  const industryApplications = [
    {
      title: "Financial Services",
      description: "Specialized programs for banking, insurance, and fintech organizations",
      icon: TrendingUp
    },
    {
      title: "Healthcare",
      description: "Leadership and management training for healthcare professionals",
      icon: Target
    },
    {
      title: "Technology",
      description: "Digital transformation and innovation management programs",
      icon: Zap
    },
    {
      title: "Manufacturing",
      description: "Operations excellence and supply chain management training",
      icon: Settings
    },
    {
      title: "Government",
      description: "Public sector leadership and governance development",
      icon: Building2
    },
    {
      title: "Education",
      description: "Academic leadership and institutional development programs",
      icon: BookOpen
    }
  ];

  return (
    <SharedLayout>
      <div className="min-h-screen bg-gray-50">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="text-center">
              <Badge variant="secondary" className="mb-4 bg-red-100 text-red-700">
                Our Services
              </Badge>
              <h1 className="text-4xl md:text-5xl font-bold mb-6">
                {(pageData as any)?.title || "Enterprise Learning Solutions"}
              </h1>
              <p className="text-xl text-gray-300 max-w-4xl mx-auto leading-relaxed">
                Comprehensive learning solutions designed to empower organizations, develop talent, and drive sustainable business growth through innovative educational programs
              </p>
              <div className="mt-8 text-sm text-gray-400">
                {(pageData as any)?.updatedAt && (
                  <span>Last updated: {new Date((pageData as any).updatedAt).toLocaleDateString()}</span>
                )}
              </div>
            </div>
          </div>
        </div>

        {/* Value Proposition */}
        <div className="bg-gradient-to-r from-[#ED1C24] to-[#FF3344] text-white py-12">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center">
              <h2 className="text-3xl font-bold mb-4">Why Choose IBDL Solutions?</h2>
              <div className="grid grid-cols-1 md:grid-cols-4 gap-8 mt-8">
                <div className="text-center">
                  <Globe className="w-12 h-12 mx-auto mb-3" />
                  <h3 className="font-semibold mb-2">Global Recognition</h3>
                  <p className="text-sm opacity-90">International accreditation from Missouri State University</p>
                </div>
                <div className="text-center">
                  <Users className="w-12 h-12 mx-auto mb-3" />
                  <h3 className="font-semibold mb-2">Expert Faculty</h3>
                  <p className="text-sm opacity-90">Industry leaders and academic experts</p>
                </div>
                <div className="text-center">
                  <Award className="w-12 h-12 mx-auto mb-3" />
                  <h3 className="font-semibold mb-2">Proven Results</h3>
                  <p className="text-sm opacity-90">222,000+ successful certifications worldwide</p>
                </div>
                <div className="text-center">
                  <Target className="w-12 h-12 mx-auto mb-3" />
                  <h3 className="font-semibold mb-2">Custom Solutions</h3>
                  <p className="text-sm opacity-90">Tailored programs for your specific needs</p>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Main Content */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          
          {/* Solutions Grid */}
          <div className="mb-16">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-gray-900 mb-4">Our Enterprise Solutions</h2>
              <p className="text-lg text-gray-600 max-w-2xl mx-auto">
                Comprehensive learning solutions designed to meet diverse organizational needs and drive measurable business outcomes
              </p>
            </div>
            
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
              {enterpriseSolutions.map((solution, index) => {
                const IconComponent = solution.icon;
                return (
                  <Card key={index} className="overflow-hidden hover:shadow-2xl transition-all duration-300 group h-full">
                    {/* Header with Gradient */}
                    <div className={`h-2 bg-gradient-to-r ${solution.color}`}></div>
                    
                    <CardHeader className="bg-white pb-4">
                      <div className="flex items-center justify-between mb-4">
                        <div className={`w-16 h-16 rounded-full bg-gradient-to-r ${solution.color} flex items-center justify-center group-hover:scale-110 transition-transform`}>
                          <IconComponent className="w-8 h-8 text-white" />
                        </div>
                        <Badge className={`bg-gradient-to-r ${solution.color} text-white`}>
                          Enterprise Solution
                        </Badge>
                      </div>
                      <CardTitle className="text-xl font-bold text-[#1A1A40] mb-2">
                        {solution.title}
                      </CardTitle>
                      <p className="text-lg text-[#ED1C24] font-medium mb-3">
                        {solution.tagline}
                      </p>
                      <p className="text-gray-600 leading-relaxed text-sm">
                        {solution.description}
                      </p>
                    </CardHeader>

                    <CardContent className="space-y-4">
                      {/* Features */}
                      <div>
                        <h4 className="font-semibold text-[#1A1A40] mb-3 flex items-center text-sm">
                          <CheckCircle className="w-4 h-4 mr-2 text-green-500" />
                          Key Features:
                        </h4>
                        <ul className="space-y-1">
                          {solution.features.map((feature, idx) => (
                            <li key={idx} className="flex items-start text-xs text-gray-600">
                              <ArrowRight className="w-3 h-3 mr-2 text-[#ED1C24] mt-0.5 flex-shrink-0" />
                              {feature}
                            </li>
                          ))}
                        </ul>
                      </div>

                      {/* Benefits */}
                      <div>
                        <h4 className="font-semibold text-[#1A1A40] mb-3 flex items-center text-sm">
                          <Star className="w-4 h-4 mr-2 text-yellow-500" />
                          Key Benefits:
                        </h4>
                        <ul className="space-y-1">
                          {solution.benefits.map((benefit, idx) => (
                            <li key={idx} className="flex items-start text-xs text-gray-600">
                              <CheckCircle className="w-3 h-3 mr-2 text-green-500 mt-0.5 flex-shrink-0" />
                              {benefit}
                            </li>
                          ))}
                        </ul>
                      </div>

                      {/* CTA */}
                      <div className="pt-4">
                        <Button 
                          className={`w-full bg-gradient-to-r ${solution.color} text-white hover:shadow-lg transition-all duration-300`}
                        >
                          {solution.ctaText}
                        </Button>
                      </div>
                    </CardContent>
                  </Card>
                );
              })}
            </div>
          </div>

          {/* Industry Applications */}
          <div className="mb-16">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-gray-900 mb-4">Industry Applications</h2>
              <p className="text-lg text-gray-600 max-w-2xl mx-auto">
                Our solutions are tailored to meet the unique challenges and requirements of various industries
              </p>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {industryApplications.map((industry, index) => {
                const IconComponent = industry.icon;
                return (
                  <Card key={index} className="text-center hover:shadow-lg transition-shadow duration-300 group">
                    <CardHeader>
                      <div className="w-12 h-12 mx-auto mb-4 bg-gradient-to-r from-[#1A1A40] to-[#ED1C24] rounded-full flex items-center justify-center group-hover:scale-110 transition-transform">
                        <IconComponent className="w-6 h-6 text-white" />
                      </div>
                      <CardTitle className="text-lg font-bold text-[#1A1A40]">{industry.title}</CardTitle>
                    </CardHeader>
                    <CardContent>
                      <p className="text-gray-600 text-sm">{industry.description}</p>
                    </CardContent>
                  </Card>
                );
              })}
            </div>
          </div>

          {/* Call to Action */}
          <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white rounded-2xl p-12 text-center">
            <h2 className="text-3xl font-bold mb-6">Ready to Transform Your Organization?</h2>
            <p className="text-xl mb-8 max-w-3xl mx-auto">
              Partner with IBDL to develop customized learning solutions that drive real business results and empower your workforce for success
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Button 
                size="lg"
                className="bg-[#ED1C24] hover:bg-[#CC1820] text-white px-8 py-3"
              >
                Schedule a Consultation
              </Button>
              <Button 
                size="lg"
                variant="outline" 
                className="border-white text-white hover:bg-white hover:text-[#1A1A40] px-8 py-3"
              >
                Download Solutions Brochure
              </Button>
            </div>
          </div>

          {/* CMS Integration Notice */}
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 text-center mt-16">
            <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
            <p className="text-blue-700">
              This page content is managed through the CMS and can be updated by administrators.
              {pageData && (
                <span className="block mt-2 text-sm">
                  Page ID: {(pageData as any).id} | Status: {(pageData as any).status}
                </span>
              )}
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}