import React from "react";
import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Award, BookOpen, TrendingUp, Users, Star, Target, Download, DollarSign, Globe, CheckCircle, Clock, FileText, GraduationCap } from "lucide-react";
import Header from "@/components/Header";
import Footer from "@/components/Footer";

export default function CMSQualifications() {


  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/qualifications"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <>
        <Header />
        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mt-12">
                {[1, 2].map((i) => (
                  <Skeleton key={i} className="h-96 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
        <Footer />
      </>
    );
  }

  const qualifications = [
    {
      title: "IBDL Essential",
      tagline: "Your gateway to the business world",
      description: "IBDL Essential is your gateway to the business world, giving you an engaging educational experience specially designed for you. Moving from understanding the fundamentals of business, to studying examples of leading organizations globally, to allow you to acquire the knowledge and abilities you need for your next move to boost your career potential and/or start your own business.",
      duration: "90 days access",
      examAttempts: "3 attempts",
      certification: "International certificate from Missouri State University",
      examCost: "$360",
      courseCost: "$800",
      modules: [
        "Entrepreneurship and Starting a Small Business",
        "Management: Functions and Styles", 
        "Leadership and Motivation",
        "Human Resources Management",
        "Marketing: Product and Price",
        "Marketing: Place and Promotion",
        "Understanding Financial Information and Accounting",
        "Business in Global Markets",
        "Demonstrating Ethical Behavior and Social Responsibility",
        "Information Technology in Business"
      ],
      includes: [
        "90 days access to IBDL's E-Learning course",
        "IBDL Essential Textbook",
        "Online practice test",
        "IBDL official examination (3 attempts)",
        "The international certificate from Missouri State University upon passing the exam"
      ],
      testimonials: [
        {
          text: "Enrolling in the IBDL Business Essential Professional Qualification was a pivotal step in advancing my career in finance. The program's emphasis on practical knowledge, strategic thinking, and analytical skills perfectly aligned with the demands of my role as a Financial Analyst. The global perspective provided by the curriculum has been invaluable in navigating complex financial markets. I highly recommend the IBDL qualification to anyone seeking to excel in the dynamic world of business and finance.",
          author: "Alessandro Rossi",
          country: "Italy"
        },
        {
          text: "I am thoroughly impressed with the comprehensive knowledge and practical skills that the IBDL Business Essential Professional Qualification provided me. The program's well-structured modules and expert instructors equipped me with a solid foundation in various business disciplines. This qualification has undoubtedly contributed to my professional growth and effectiveness in a competitive business environment.",
          author: "Sofia Müller",
          country: "Germany"
        }
      ],
      color: "from-green-400 to-emerald-500",
      icon: CheckCircle
    },
    {
      title: "IBDL Professional",
      tagline: "Your powerful tool to develop advanced skills",
      description: "IBDL Professional is your powerful tool to develop the skills and knowledge that will help you to move forward in your professional career. IBDL Professional enables organizations to prepare second-line leaders and managers through developing their management skills, which enable them to lead their organizations efficiently. Specially designed in an interactive and enjoyable way, in which you move from studying the management skills, through understanding the basics of planning and strategic management, providing you with the knowledge and skills you need to grow on the personal and professional levels and invest in your future.",
      duration: "90 days access",
      examAttempts: "3 attempts", 
      certification: "International certificate from Missouri State University",
      examCost: "$380",
      courseCost: "$1000",
      modules: [
        "Managing",
        "Managing International Management",
        "Planning and Strategic Management",
        "Managerial Decision Making",
        "Managerial Control",
        "Teamwork",
        "Communicating",
        "Human Resources Management",
        "Organization Structure",
        "Organizational Agility",
        "Managing the Diverse Workforce",
        "The External Environment and Organizational Culture",
        "Ethics and Corporate Responsibility",
        "Managing Technology and Innovation",
        "Creating and Managing Change"
      ],
      includes: [
        "90 days access to IBDL's E-Learning course",
        "IBDL Professional Textbook",
        "Online practice test",
        "IBDL official examination (3 attempts)",
        "The international certificate from Missouri State University upon passing the exam"
      ],
      testimonials: [
        {
          text: "The IBDL Professional has been an eye-opening journey that expanded my horizons in the business world. The program's diverse modules and insightful instructors helped me grasp essential concepts, which I've successfully applied to real-world situations. This qualification is an investment in your professional growth, and I'm truly grateful for the knowledge gained.",
          author: "Lars Andersen",
          country: "Denmark"
        },
        {
          text: "Embarking on IBDL Professional was a game-changer for my career. The program's well-structured curriculum and interactive learning environment enabled me to develop a profound understanding of various business disciplines. The knowledge I acquired has given me the confidence to take on new challenges and excel in my role. I highly recommend the IBDL Professional to anyone aiming for excellence in the business realm.",
          author: "Isabella Francesco",
          country: "Italy"
        }
      ],
      color: "from-blue-400 to-indigo-500",
      icon: Award
    }
  ];

  return (
    <>
      <Header />
      
      <div className="min-h-screen bg-gray-50">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="text-center">
            <Badge variant="secondary" className="mb-4 bg-[#ffffff] text-[#1d1d43]">
              Professional Development
            </Badge>
            <h1 className="text-4xl md:text-5xl font-bold mb-6">
              {(pageData && typeof pageData === 'object' && 'title' in pageData) 
                ? String((pageData as any).title) 
                : "IBDL Qualifications & Certifications"}
            </h1>
            <p className="text-xl text-gray-300 max-w-4xl mx-auto leading-relaxed">
              Internationally recognized professional qualifications designed to advance careers and drive organizational excellence across diverse industries
            </p>
            <div className="mt-8 text-sm text-gray-400">
              {(pageData && typeof pageData === 'object' && 'updatedAt' in pageData) ? (
                <span>Last updated: {new Date((pageData as any).updatedAt).toLocaleDateString()}</span>
              ) : null}
            </div>
          </div>
        </div>
      </div>

      {/* Certification Programs */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">Our Professional Qualifications</h2>
          <p className="text-lg text-gray-600 max-w-2xl mx-auto">
            Choose the right certification level to advance your professional development and career growth
          </p>
        </div>
        
        {/* International Recognition Banner */}
        <div className="bg-gradient-to-r from-[#ED1C24] to-[#FF3344] text-white py-8 mb-16 rounded-2xl">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="flex items-center justify-center space-x-6">
              <Globe className="w-8 h-8" />
              <div className="text-center">
                <h2 className="text-2xl font-bold mb-2">INTERNATIONAL RECOGNITION</h2>
                <p className="text-lg">Add a real competitive advantage to your career by obtaining IBDL certificates with international accreditation from Missouri State University</p>
              </div>
              <GraduationCap className="w-8 h-8" />
            </div>
          </div>
        </div>

        {/* Qualifications Grid */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 mb-16">
          {qualifications.map((qualification, index) => {
            const IconComponent = qualification.icon;
            return (
              <Card key={index} className="overflow-hidden hover:shadow-2xl transition-all duration-300 group">
                {/* Header with Gradient */}
                <div className={`h-2 bg-gradient-to-r ${qualification.color}`}></div>
                
                <CardHeader className="bg-white pb-4">
                  <div className="flex items-center justify-between mb-4">
                    <div className={`w-16 h-16 rounded-full bg-gradient-to-r ${qualification.color} flex items-center justify-center group-hover:scale-110 transition-transform`}>
                      <IconComponent className="w-8 h-8 text-white" />
                    </div>
                    <Badge className={`bg-gradient-to-r ${qualification.color} text-white`}>
                      Professional Qualification
                    </Badge>
                  </div>
                  <CardTitle className="text-2xl font-bold text-[#1A1A40] mb-2">
                    {qualification.title}
                  </CardTitle>
                  <p className="text-lg text-[#ED1C24] font-medium mb-4">
                    {qualification.tagline}
                  </p>
                  <p className="text-gray-600 leading-relaxed">
                    {qualification.description}
                  </p>
                </CardHeader>

                <CardContent className="space-y-6">
                  {/* Key Features */}
                  <div>
                    <h4 className="font-semibold text-[#1A1A40] mb-3 flex items-center">
                      <CheckCircle className="w-5 h-5 mr-2 text-green-500" />
                      What You'll Receive:
                    </h4>
                    <ul className="space-y-2">
                      {qualification.includes.map((item, idx) => (
                        <li key={idx} className="flex items-start text-sm text-gray-600">
                          <CheckCircle className="w-4 h-4 mr-2 text-green-500 mt-0.5 flex-shrink-0" />
                          {item}
                        </li>
                      ))}
                    </ul>
                  </div>

                  {/* Learning Journey */}
                  <div>
                    <h4 className="font-semibold text-[#1A1A40] mb-3 flex items-center">
                      <BookOpen className="w-5 h-5 mr-2 text-blue-500" />
                      Components of Your Learning Journey:
                    </h4>
                    <div className="max-h-40 overflow-y-auto">
                      <ol className="space-y-1">
                        {qualification.modules.map((module, idx) => (
                          <li key={idx} className="text-sm text-gray-600 flex items-start">
                            <span className="text-[#ED1C24] font-medium mr-2 mt-0.5">{(idx + 1).toString().padStart(2, '0')}.</span>
                            {module}
                          </li>
                        ))}
                      </ol>
                    </div>
                  </div>

                  {/* Pricing */}
                  <div className="bg-gray-50 rounded-lg p-4">
                    <h4 className="font-semibold text-[#1A1A40] mb-3 flex items-center">
                      <DollarSign className="w-5 h-5 mr-2 text-green-500" />
                      Certification and Training Cost:
                    </h4>
                    <div className="grid grid-cols-2 gap-4 text-sm">
                      <div>
                        <span className="text-gray-600">Exam and Certification:</span>
                        <span className="font-bold text-[#ED1C24] ml-2">{qualification.examCost}</span>
                      </div>
                      <div>
                        <span className="text-gray-600">Preparation Course:</span>
                        <span className="font-bold text-[#ED1C24] ml-2">{qualification.courseCost}</span>
                      </div>
                    </div>
                  </div>

                  {/* Download Flyers */}
                  <div className="flex space-x-4">
                    <Button 
                      variant="outline" 
                      size="sm" 
                      className="flex items-center space-x-2 hover:bg-[#ED1C24] hover:text-white transition-colors"
                    >
                      <Download className="w-4 h-4" />
                      <span>Arabic Flyer</span>
                    </Button>
                    <Button 
                      variant="outline" 
                      size="sm"
                      className="flex items-center space-x-2 hover:bg-[#1A1A40] hover:text-white transition-colors"
                    >
                      <Download className="w-4 h-4" />
                      <span>English Flyer</span>
                    </Button>
                  </div>

                  {/* Testimonials */}
                  <div>
                    <h4 className="font-semibold text-[#1A1A40] mb-4 flex items-center">
                      <Star className="w-5 h-5 mr-2 text-yellow-500" />
                      What People Say About {qualification.title}:
                    </h4>
                    <div className="space-y-4">
                      {qualification.testimonials.map((testimonial, idx) => (
                        <div key={idx} className="bg-white border rounded-lg p-4 shadow-sm">
                          <p className="text-sm text-gray-600 italic mb-3 leading-relaxed">
                            "{testimonial.text}"
                          </p>
                          <div className="flex items-center justify-between text-xs">
                            <span className="font-medium text-[#1A1A40]">
                              {testimonial.author}
                            </span>
                            <span className="text-gray-500">
                              {testimonial.country}
                            </span>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* CMS Integration Notice */}
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 text-center">
          <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
          <p className="text-blue-700">
            This page content is managed through the CMS and can be updated by administrators.
            {pageData && typeof pageData === 'object' && 'id' in pageData ? (
              <span className="block mt-2 text-sm">
                Page ID: {String((pageData as any).id)} | Status: {String((pageData as any).status)}
              </span>
            ) : null}
          </p>
        </div>
      </div>
      </div>

      
      <Footer />
    </>
  );
}