import React from "react";
import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { MapPin, Phone, Mail, Globe, Clock } from "lucide-react";
import Header from "@/components/Header";
import Footer from "@/components/Footer";

export default function CMSContact() {

  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/contact"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <>
        <Header />
        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-3 gap-8 mt-12">
                {[1, 2, 3].map((i) => (
                  <Skeleton key={i} className="h-48 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
        <Footer />
      </>
    );
  }

  const globalOffices = [
    {
      region: "United Kingdom",
      title: "Global Headquarters",
      address: "IBDL Learning Group\nLondon, United Kingdom",
      phone: "+44 20 7123 4567",
      email: "info@ibdl.net",
      hours: "Mon-Fri: 9:00 AM - 6:00 PM GMT",
      icon: MapPin,
      color: "bg-red-50 border-red-200"
    },
    {
      region: "United States",
      title: "Americas Office",
      address: "IBDL USA Office\nNew York, NY, USA",
      phone: "+1 212 555 0123",
      email: "usa@ibdl.net",
      hours: "Mon-Fri: 9:00 AM - 6:00 PM EST",
      icon: Globe,
      color: "bg-blue-50 border-blue-200"
    },
    {
      region: "Middle East",
      title: "MENA Regional Office",
      address: "IBDL Regional Office\nDubai, UAE",
      phone: "+971 4 123 4567",
      email: "mena@ibdl.net",
      hours: "Sun-Thu: 9:00 AM - 6:00 PM GST",
      icon: MapPin,
      color: "bg-emerald-50 border-emerald-200"
    }
  ];

  return (
    <>
      <Header />

      <div className="min-h-screen bg-gray-50">
        {/* CMS Content Header */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="text-center">
            <Badge variant="secondary" className="mb-4 bg-red-100 text-red-700">
              Get in Touch
            </Badge>
            <h1 className="text-4xl md:text-5xl font-bold mb-6">
              {(pageData && typeof pageData === 'object' && 'title' in pageData) 
                ? String((pageData as any).title) 
                : "Contact IBDL"}
            </h1>
            <p className="text-xl text-gray-300 max-w-3xl mx-auto">
              {(pageData && typeof pageData === 'object' && 'content' in pageData) 
                ? String((pageData as any).content) 
                : "Connect with IBDL to explore learning opportunities, partnerships, and professional development solutions."}
            </p>
            <div className="mt-8 text-sm text-gray-400">
              {(pageData && typeof pageData === 'object' && 'updatedAt' in pageData) ? (
                <span>Last updated: {new Date((pageData as any).updatedAt).toLocaleDateString()}</span>
              ) : null}
            </div>
          </div>
        </div>
      </div>

      {/* Contact Form and Offices */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
          {/* Contact Form */}
          <div>
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Send us a Message</h2>
            <Card>
              <CardContent className="p-6">
                <form className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">
                        First Name
                      </label>
                      <Input placeholder="John" />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">
                        Last Name
                      </label>
                      <Input placeholder="Doe" />
                    </div>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Email
                    </label>
                    <Input type="email" placeholder="john.doe@company.com" />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Subject
                    </label>
                    <Input placeholder="How can we help you?" />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Message
                    </label>
                    <Textarea 
                      rows={4} 
                      placeholder="Tell us about your inquiry..." 
                    />
                  </div>
                  <Button className="w-full bg-[#ED1C24] hover:bg-red-700">
                    Send Message
                  </Button>
                </form>
              </CardContent>
            </Card>
          </div>

          {/* Global Offices */}
          <div>
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Global Offices</h2>
            <div className="space-y-6">
              {globalOffices.map((office, index) => {
                const IconComponent = office.icon;
                return (
                  <Card key={index} className={`border-2 ${office.color}`}>
                    <CardHeader>
                      <div className="flex items-center space-x-3">
                        <div className="w-10 h-10 bg-white rounded-full flex items-center justify-center">
                          <IconComponent className="w-5 h-5 text-gray-700" />
                        </div>
                        <div>
                          <CardTitle className="text-lg">{office.region}</CardTitle>
                          <p className="text-sm text-gray-600">{office.title}</p>
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent className="space-y-3">
                      <div className="flex items-start space-x-3">
                        <MapPin className="w-4 h-4 text-gray-500 mt-1" />
                        <div className="text-sm text-gray-600 whitespace-pre-line">
                          {office.address}
                        </div>
                      </div>
                      <div className="flex items-center space-x-3">
                        <Phone className="w-4 h-4 text-gray-500" />
                        <span className="text-sm text-gray-600">{office.phone}</span>
                      </div>
                      <div className="flex items-center space-x-3">
                        <Mail className="w-4 h-4 text-gray-500" />
                        <span className="text-sm text-gray-600">{office.email}</span>
                      </div>
                      <div className="flex items-center space-x-3">
                        <Clock className="w-4 h-4 text-gray-500" />
                        <span className="text-sm text-gray-600">{office.hours}</span>
                      </div>
                    </CardContent>
                  </Card>
                );
              })}
            </div>
          </div>
        </div>

        {/* CMS Integration Notice */}
        <div className="mt-12 bg-blue-50 border border-blue-200 rounded-lg p-6 text-center">
          <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
          <p className="text-blue-700">
            This page content is managed through the CMS and can be updated by administrators.
            {pageData && typeof pageData === 'object' && pageData && 'id' in pageData ? (
              <span className="block mt-2 text-sm">
                Page ID: {String((pageData as any).id)} | Status: {String((pageData as any).status)}
              </span>
            ) : null}
          </p>
        </div>
      </div>
      </div>

      <Footer />
    </>
  );
}