import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import SharedLayout from "@/components/SharedLayout";
import { 
  Building2, 
  Target, 
  Users, 
  CheckCircle, 
  ArrowRight,
  FileText,
  Calendar,
  Briefcase,
  GraduationCap,
  Clock,
  MapPin,
  Phone,
  Mail,
  Download,
  Star
} from "lucide-react";

export default function CMSAccreditation() {
  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/accreditation"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading) {
    return (
      <SharedLayout>
        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mt-12">
                {[1, 2, 3, 4].map((i) => (
                  <Skeleton key={i} className="h-64 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
      </SharedLayout>
    );
  }

  const accreditationTypes = [
    {
      title: "Testing Center Accreditation",
      description: "Become an authorized IBDL examination center and deliver secure, proctored assessments to candidates in your region",
      requirements: [
        "Secure testing facility with proper infrastructure",
        "Qualified proctoring staff with certification", 
        "Compliance with IBDL security protocols",
        "Technical requirements for digital examinations",
        "Regular quality assurance audits"
      ],
      benefits: [
        "Generate revenue through examination fees",
        "Serve local professional community",
        "Access to IBDL marketing support",
        "Ongoing technical and operational support"
      ],
      process: [
        "Submit initial application and documentation",
        "Facility assessment and security review",
        "Staff training and certification", 
        "Trial examination period",
        "Full accreditation approval"
      ],
      duration: "4-6 weeks",
      icon: Target,
      color: "from-blue-400 to-indigo-500"
    },
    {
      title: "Training Provider Accreditation", 
      description: "Deliver authorized IBDL training programs and develop the next generation of business leaders in your market",
      requirements: [
        "Qualified instructors with IBDL certification",
        "Appropriate training facilities and equipment",
        "Curriculum delivery standards compliance", 
        "Student assessment and tracking capabilities",
        "Ongoing professional development commitment"
      ],
      benefits: [
        "Authority to deliver IBDL qualifications",
        "Access to comprehensive training materials",
        "Marketing support and brand recognition",
        "Revenue sharing opportunities"
      ],
      process: [
        "Application submission with credentials",
        "Instructor qualification assessment",
        "Facility and resource evaluation",
        "Training delivery trial period",
        "Full provider accreditation"
      ],
      duration: "6-8 weeks",
      icon: GraduationCap,
      color: "from-green-400 to-emerald-500"
    }
  ];

  return (
    <SharedLayout>
      <div className="min-h-screen bg-gray-50">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#ED1C24] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="text-center">
              <Badge variant="secondary" className="mb-4 bg-red-100 text-red-700">
                IBDL Accreditation
              </Badge>
              <h1 className="text-4xl md:text-5xl font-bold mb-6">
                {(pageData as any)?.title || "Partner with IBDL"}
              </h1>
              <p className="text-xl text-gray-300 max-w-4xl mx-auto leading-relaxed">
                Join our global network of authorized training providers and testing centers. Deliver world-class business education and assessments in your region.
              </p>
            </div>
          </div>
        </div>

        {/* Accreditation Types */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="text-center mb-16">
            <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Accreditation Opportunities</h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Choose the accreditation path that aligns with your business goals and capabilities
            </p>
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
            {accreditationTypes.map((type, index) => (
              <Card key={index} className="overflow-hidden border-0 shadow-xl">
                <div className={`h-2 bg-gradient-to-r ${type.color}`}></div>
                
                <CardHeader className="text-center pb-6">
                  <div className={`inline-flex items-center justify-center w-16 h-16 rounded-full bg-gradient-to-r ${type.color} text-white mb-4 mx-auto`}>
                    <type.icon className="h-8 w-8" />
                  </div>
                  <CardTitle className="text-2xl font-bold text-[#1A1A40] mb-2">{type.title}</CardTitle>
                  <p className="text-gray-600 leading-relaxed">{type.description}</p>
                </CardHeader>

                <CardContent className="pt-0">
                  {/* Requirements */}
                  <div className="mb-8">
                    <h4 className="font-semibold text-[#1A1A40] mb-3 flex items-center">
                      <CheckCircle className="h-5 w-5 text-green-500 mr-2" />
                      Requirements
                    </h4>
                    <div className="space-y-2">
                      {type.requirements.map((req, idx) => (
                        <div key={idx} className="flex items-start text-sm text-gray-600">
                          <div className="w-1 h-1 bg-gray-400 rounded-full mt-2 mr-3 flex-shrink-0"></div>
                          {req}
                        </div>
                      ))}
                    </div>
                  </div>

                  {/* Benefits */}
                  <div className="mb-8">
                    <h4 className="font-semibold text-[#1A1A40] mb-3 flex items-center">
                      <Star className="h-5 w-5 text-yellow-500 mr-2" />
                      Benefits
                    </h4>
                    <div className="space-y-2">
                      {type.benefits.map((benefit, idx) => (
                        <div key={idx} className="flex items-start text-sm text-gray-600">
                          <div className="w-1 h-1 bg-gray-400 rounded-full mt-2 mr-3 flex-shrink-0"></div>
                          {benefit}
                        </div>
                      ))}
                    </div>
                  </div>

                  {/* Process Timeline */}
                  <div className="mb-8">
                    <h4 className="font-semibold text-[#1A1A40] mb-3 flex items-center">
                      <Clock className="h-5 w-5 text-blue-500 mr-2" />
                      Process Timeline ({type.duration})
                    </h4>
                    <div className="space-y-3">
                      {type.process.map((step, idx) => (
                        <div key={idx} className="flex items-center text-sm">
                          <div className="w-6 h-6 rounded-full bg-blue-100 text-blue-600 text-xs font-semibold flex items-center justify-center mr-3 flex-shrink-0">
                            {idx + 1}
                          </div>
                          <span className="text-gray-600">{step}</span>
                        </div>
                      ))}
                    </div>
                  </div>

                  <Button className={`w-full bg-gradient-to-r ${type.color} text-white hover:opacity-90 transition-opacity`}>
                    Apply for {type.title}
                    <ArrowRight className="ml-2 h-4 w-4" />
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Support Section */}
        <div className="bg-white py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-[#1A1A40] mb-4">Need Help with Your Application?</h2>
              <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                Our accreditation team is here to guide you through the process
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              {[
                {
                  icon: Phone,
                  title: "Phone Support",
                  description: "Speak directly with our accreditation specialists",
                  contact: "+1-800-IBDL-ACC",
                  action: "Call Now"
                },
                {
                  icon: Mail,
                  title: "Email Support", 
                  description: "Get detailed answers to your accreditation questions",
                  contact: "accreditation@ibdl.net",
                  action: "Send Email"
                },
                {
                  icon: Download,
                  title: "Application Guide",
                  description: "Download our comprehensive accreditation guide",
                  contact: "Complete application requirements and process",
                  action: "Download Guide"
                }
              ].map((support, index) => (
                <Card key={index} className="text-center border-2 border-gray-100 hover:border-[#ED1C24] transition-colors">
                  <CardContent className="pt-8 pb-6">
                    <div className="inline-flex items-center justify-center w-12 h-12 rounded-full bg-[#ED1C24] text-white mb-4">
                      <support.icon className="h-6 w-6" />
                    </div>
                    <h3 className="text-lg font-semibold text-[#1A1A40] mb-2">{support.title}</h3>
                    <p className="text-gray-600 text-sm mb-4">{support.description}</p>
                    <p className="text-[#ED1C24] font-medium text-sm mb-4">{support.contact}</p>
                    <Button size="sm" className="bg-[#1A1A40] hover:bg-[#2A2A50] text-white">
                      {support.action}
                    </Button>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>

        {/* CMS Integration Notice */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 text-center">
            <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
            <p className="text-blue-700">
              This page content is managed through the CMS and can be updated by administrators.
              {pageData && (
                <span className="block mt-2 text-sm">
                  Page ID: {(pageData as any).id} | Status: {(pageData as any).status}
                </span>
              )}
            </p>
          </div>
        </div>
      </div>
    </SharedLayout>
  );
}