import { useQuery } from "@tanstack/react-query";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { 
  Building2, 
  Users, 
  Globe, 
  Award, 
  BookOpen, 
  Target, 
  Handshake, 
  TrendingUp,
  GraduationCap,
  Lightbulb,
  Linkedin,
  Twitter,
  MessageSquare
} from "lucide-react";
import SharedLayout from "@/components/SharedLayout";
import { useState } from 'react';

// Function to render sections based on template type
function renderSection(
  section: any, 
  index: number, 
  expandedTrustees: Set<string>,
  toggleTrusteeExpansion: (key: string) => void
) {
  const { templateType, title, content, data, image } = section;

  switch (templateType) {
    case 'board_trustees':
      const trustees = (() => {
        try {
          return data?.trustee_profiles ? JSON.parse(data.trustee_profiles) : [];
        } catch {
          return [];
        }
      })();

      return (
        <div key={index} className="mb-16">
          <div className="text-center mb-12">
            <h2 className="text-4xl font-bold text-[#ED1C24] dark:text-white mb-6 hover:text-[#ED1C24] transition-colors duration-300">
              {title}
            </h2>
            {content && (
              <p className="text-lg text-gray-700 dark:text-gray-300 leading-relaxed max-w-4xl mx-auto">
                {content}
              </p>
            )}
          </div>
          
          {/* Trustees Cards Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {trustees.map((trustee: any, trusteeIndex: number) => (
              <div 
                key={trusteeIndex}
                className="bg-white dark:bg-slate-800 rounded-xl shadow-md hover:shadow-lg transition-all duration-500 p-6 group hover:scale-[1.02] border border-gray-100 dark:border-slate-700 h-full"
              >
                <div className="text-center h-full flex flex-col">
                  <div className="relative mx-auto w-24 h-24 mb-4">
                    {trustee.image ? (
                      <img 
                        src={`backend/api/uploads/${trustee.image}`} 
                        alt={trustee.name}
                        className="w-full h-full rounded-full object-cover shadow-md group-hover:scale-110 transition-transform duration-300"
                      />
                    ) : (
                      <div className="w-full h-full rounded-full bg-gray-200 flex items-center justify-center">
                        <Users className="w-8 h-8 text-gray-400" />
                      </div>
                    )}
                    <div className="absolute inset-0 rounded-full bg-gradient-to-br from-[#ED1C24]/20 to-[#1A1A40]/20 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                  </div>
                  
                  <h3 className="text-lg font-bold text-[#1A1A40] dark:text-white mb-2">
                    {trustee.name}
                  </h3>
                  
                  <p className="text-sm font-semibold text-[#ED1C24] mb-3">
                    {trustee.title}
                  </p>
                  
                  <div className="flex-grow">
                    {(() => {
                      const description = trustee.description || '';
                      const trusteeKey = `${index}-${trusteeIndex}`;
                      const isExpanded = expandedTrustees.has(trusteeKey);
                      const shouldTruncate = description.length > 120;
                      
                      return (
                        <>
                          <p className="text-xs text-gray-600 dark:text-gray-400 leading-relaxed mb-4">
                            {shouldTruncate && !isExpanded 
                              ? `${description.substring(0, 120)}...` 
                              : description
                            }
                          </p>
                          {shouldTruncate && (
                            <button
                              onClick={() => toggleTrusteeExpansion(trusteeKey)}
                              className="text-xs text-[#ED1C24] font-semibold hover:text-[#ED1C24]/80 transition-colors duration-200 mb-4"
                            >
                              {isExpanded ? 'Read Less' : 'Read More'}
                            </button>
                          )}
                        </>
                      );
                    })()}
                  </div>
                  
                  {/* Social Media Icons */}
                  {(trustee.linkedin || trustee.twitter) && (
                    <div className="flex justify-center space-x-2 pt-2 border-t border-gray-100 dark:border-slate-600 mt-auto">
                      {trustee.linkedin && (
                        <a 
                          href={trustee.linkedin} 
                          target="_blank"
                          rel="noopener noreferrer"
                          className="p-1.5 rounded-full bg-[#0077B5] text-white hover:bg-[#005885] transition-colors duration-200"
                          title="LinkedIn"
                        >
                          <Linkedin className="h-3 w-3" />
                        </a>
                      )}
                      {trustee.twitter && (
                        <a 
                          href={trustee.twitter} 
                          target="_blank"
                          rel="noopener noreferrer"
                          className="p-1.5 rounded-full bg-[#1DA1F2] text-white hover:bg-[#0d8bd9] transition-colors duration-200"
                          title="Twitter"
                        >
                          <Twitter className="h-3 w-3" />
                        </a>
                      )}
                    </div>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>
      );

    case 'ceo_message':
      // Helper function to generate initials from CEO name
      const getInitials = (name: string) => {
        if (!name) return 'CEO';
        return name.split(' ').map(n => n[0]).join('').toUpperCase().slice(0, 2);
      };

      // Parse dynamic content items
      const contentItems = (() => {
        try {
          return data?.content_items ? JSON.parse(data.content_items) : [];
        } catch {
          return [];
        }
      })();

      return (
        <div key={index} className="mb-16">
          <div className="relative overflow-hidden rounded-2xl shadow-md">
            <div className="absolute inset-0 bg-gradient-to-r from-[#1A1A40] via-slate-800 to-[#ED1C24]"></div>
            <div className="absolute inset-0 bg-black/10"></div>
            
            <div className="relative py-16 px-12 text-white">
              <div className="flex items-center justify-center mb-8">
                <div className="p-4 bg-white/10 rounded-xl backdrop-blur-sm">
                  <MessageSquare className="h-8 w-8 text-white" />
                </div>
                <h2 className="text-4xl font-bold ml-6">{title}</h2>
              </div>
              
              <div className="max-w-4xl mx-auto">
                {data?.quote && (
                  <blockquote className="text-2xl font-semibold text-center mb-8 italic">
                    "{data.quote}"
                  </blockquote>
                )}
                
                {/* Dynamic Content Items */}
                {contentItems.length > 0 ? (
                  <div className="space-y-6 text-lg leading-relaxed">
                    {contentItems.map((item: any, itemIndex: number) => (
                      <p key={itemIndex}>{item.content}</p>
                    ))}
                  </div>
                ) : (
                  /* Fallback to regular content if no content items */
                  content && (
                    <div className="space-y-6 text-lg leading-relaxed">
                      <p>{content}</p>
                    </div>
                  )
                )}
                
                {/* CEO Profile Section */}
                <div className="mt-12 pt-8 border-t border-white/20">
                  <div className="flex items-center justify-center space-x-6">
                    <div className="relative w-20 h-20 flex-shrink-0">
                      {data?.ceo_image ? (
                        <img 
                          src={`backend/api/uploads/${data.ceo_image}`} 
                          alt={data?.ceo_name || 'CEO'}
                          className="w-full h-full rounded-full object-cover shadow-md border-2 border-white/20"
                        />
                      ) : (
                        /* Avatar with initials fallback */
                        <div className="w-full h-full rounded-full bg-white/20 backdrop-blur-sm border-2 border-white/20 flex items-center justify-center">
                          <span className="text-2xl font-bold text-white">
                            {getInitials(data?.ceo_name || '')}
                          </span>
                        </div>
                      )}
                    </div>
                    <div className="text-center">
                      {data?.ceo_name && (
                        <p className="text-2xl font-bold">{data.ceo_name}</p>
                      )}
                      {data?.ceo_title && (
                        <p className="text-lg text-white/80">{data.ceo_title}</p>
                      )}
                      <p className="text-base text-white/60">IBDL Learning Group</p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      );

    case 'centered_card':
      return (
        <div key={index} className="mb-16">
          <div className="bg-white rounded-2xl p-8 shadow-sm">
            <div className="max-w-4xl mx-auto text-center">
              <h2 className="text-3xl font-bold text-[#1A1A40] mb-6">{title}</h2>
              {content && (
                <p className="text-lg text-gray-700 leading-relaxed">{content}</p>
              )}
            </div>
          </div>
        </div>
      );

    default:
      // Fallback for unknown template types
      return (
        <div key={index} className="mb-16">
          <div className="bg-white rounded-2xl p-8 shadow-sm">
            <div className="max-w-4xl mx-auto">
              <h2 className="text-3xl font-bold text-[#1A1A40] mb-6">{title}</h2>
              {content && (
                <p className="text-lg text-gray-700 leading-relaxed">{content}</p>
              )}
              {image && (
                <img src={`backend/api/uploads/${image}`} alt={title} className="mt-6 rounded-lg w-full max-w-2xl mx-auto" />
              )}
            </div>
          </div>
        </div>
      );
  }
}

export default function CMSAbout() {
  // State for expandable trustee cards
  const [expandedTrustees, setExpandedTrustees] = useState<Set<string>>(new Set());

  const toggleTrusteeExpansion = (trusteeKey: string) => {
    setExpandedTrustees(prev => {
      const newSet = new Set(prev);
      if (newSet.has(trusteeKey)) {
        newSet.delete(trusteeKey);
      } else {
        newSet.add(trusteeKey);
      }
      return newSet;
    });
  };

  // Fetch CMS content for this page
  const { data: pageData, isLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/about"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  // Fetch CMS sections for this page
  const { data: sectionsData, isLoading: sectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages/about/sections"],
    staleTime: 5 * 60 * 1000, // 5 minutes
  });

  if (isLoading || sectionsLoading) {
    return (
      <SharedLayout>
        <div className="min-h-screen bg-gray-50">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="space-y-6">
              <Skeleton className="h-12 w-3/4 mx-auto" />
              <Skeleton className="h-6 w-1/2 mx-auto" />
              <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mt-12">
                {[1, 2, 3, 4].map((i) => (
                  <Skeleton key={i} className="h-48 w-full" />
                ))}
              </div>
            </div>
          </div>
        </div>
      </SharedLayout>
    );
  }

  const keyHighlights = [
    {
      icon: Building2,
      title: "Established 2006",
      description: "From humble beginnings to global recognition with 22 dynamic learning solutions",
      color: "bg-blue-50 text-blue-700"
    },
    {
      icon: Handshake,
      title: "Strategic Partnerships",
      description: "Partnerships with Missouri State University, Abertay University, University of Dundee, and Edinburgh Napier University",
      color: "bg-green-50 text-green-700"
    },
    {
      icon: Award,
      title: "Industry Leaders",
      description: "Collaborations with McGraw-Hill and Missouri State University's ILTC",
      color: "bg-purple-50 text-purple-700"
    },
    {
      icon: Globe,
      title: "Global Impact",
      description: "Internationally recognized certificates and personalized solutions worldwide",
      color: "bg-red-50 text-red-700"
    }
  ];

  const coreValues = [
    {
      icon: Lightbulb,
      title: "Innovation",
      description: "We challenge traditional learning paradigms and embrace disruptive innovation as our driving force",
      color: "from-yellow-400 to-orange-500"
    },
    {
      icon: Target,
      title: "Excellence",
      description: "We maintain the highest standards in training, certification, and delivery of learning solutions",
      color: "from-blue-400 to-purple-500"
    },
    {
      icon: Users,
      title: "Empowerment",
      description: "We inspire individuals and organizations to push boundaries and realize their full potential",
      color: "from-green-400 to-teal-500"
    },
    {
      icon: GraduationCap,
      title: "Lifelong Learning",
      description: "We foster a culture of continuous learning, growth, and professional development",
      color: "from-red-400 to-pink-500"
    }
  ];

  return (
    <SharedLayout>
      <div className="min-h-screen bg-gray-50">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-[#1A1A40] to-[#2A2A50] text-white">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
            <div className="text-center">
              <Badge variant="secondary" className="mb-4 bg-red-100 text-red-700">
                Who We Are
              </Badge>
              <h1 className="text-4xl md:text-5xl font-bold mb-6">
                {(pageData as any)?.title || "About IBDL Learning Group"}
              </h1>
              <p className="text-xl text-gray-300 max-w-4xl mx-auto leading-relaxed">
                The driving force behind extraordinary success, motivating professionals to embark on a transformative journey of continuous learning
              </p>
              <div className="mt-8 text-sm text-gray-400">
                {(pageData as any)?.updatedAt && (
                  <span>Last updated: {new Date((pageData as any).updatedAt).toLocaleDateString()}</span>
                )}
              </div>
            </div>
          </div>
        </div>

        {/* Main Content */}
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          
          {/* Dynamic CMS Sections */}
          {sectionsData && Array.isArray(sectionsData) && sectionsData.length > 0 ? (
            sectionsData.map((section: any, index: number) => renderSection(section, index, expandedTrustees, toggleTrusteeExpansion))
          ) : (
            <>
              {/* Fallback content when no sections */}
              <div className="bg-white rounded-2xl p-8 mb-16 shadow-sm">
                <div className="max-w-4xl mx-auto text-center">
                  <h2 className="text-3xl font-bold text-[#1A1A40] mb-6">Our Story</h2>
                  <p className="text-lg text-gray-700 leading-relaxed mb-6">
                    IBDL Learning Group, the driving force behind extraordinary success, motivates professionals to embark on a transformative journey of continuous learning. International business education has been completely transformed by IBDL Learning Group, the unwavering owner of the IBDL brand certificates worldwide.
                  </p>
                  <p className="text-lg text-gray-700 leading-relaxed">
                    IBDL Learning Group encourages people to achieve extraordinary levels with its cutting-edge learning solutions and firm dedication to quality.
                  </p>
                </div>
              </div>
            </>
          )}


          {/* CMS Integration Notice */}
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 text-center">
            <h3 className="text-lg font-semibold text-blue-900 mb-2">Content Management</h3>
            <p className="text-blue-700">
              This page content is managed through the CMS and can be updated by administrators.
              {pageData && typeof pageData === 'object' && pageData && 'id' in pageData ? (
                <span className="block mt-2 text-sm">
                  Page ID: {String((pageData as any).id)} | Status: {String((pageData as any).status)}
                </span>
              ) : null}
            </p>
          </div>
        </div>
      </div>
    </SharedLayout>
  );
}