import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { 
  Plus, 
  Edit, 
  Trash2, 
  Eye, 
  Globe, 
  FileText, 
  Settings, 
  BarChart3,
  LogOut,
  Image,
  Type,
  Layout,
  X,
  ExternalLink,
  Users,
  Award,
  MessageSquare,
  Target,
  Shield,
  BookOpen,
  TrendingUp,
  Heart,
  GripVertical,
  DollarSign,
  Lightbulb,
  Handshake,
  GraduationCap,
  Cog,
  Telescope,
  Briefcase,
  Cpu,
  Rocket,
  Building2,
  FileCheck
} from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { useAuth } from "@/hooks/useAuth";
import ibdlLogo from "@assets/logo_1751230646761.png";
import { TemplateSelectionPanel } from "@/components/cms/TemplateSelectionPanel";
import {
  DndContext,
  closestCenter,
  KeyboardSensor,
  PointerSensor,
  useSensor,
  useSensors,
  DragEndEvent,
} from '@dnd-kit/core';
import {
  arrayMove,
  SortableContext,
  sortableKeyboardCoordinates,
  verticalListSortingStrategy,
} from '@dnd-kit/sortable';
import {
  useSortable,
} from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';

// Template definitions
export const SECTION_TEMPLATES = {
  TEXT_LEFT_IMAGE_RIGHT: 'text_left_image_right',
  IMAGE_LEFT_TEXT_RIGHT: 'image_left_text_right',
  TEXT_CARD_LEFT_IMAGE_RIGHT: 'text_left_image_right_with_card',
  IMAGE_LEFT_TEXT_CARD_RIGHT: 'image_left_text_card_right',
  MISSION_VISION: 'centered_card',
  GRADIENT_CENTERED: 'gradient_centered',
  GOVERNANCE_PRINCIPLES: 'governance_principles',
  BOARD_TRUSTEES: 'board_trustees',
  CEO_MESSAGE: 'ceo_message',
  IMPACT_STATISTICS: 'impact_statistics',
  WHY_NUMBERS_MATTER: 'why_numbers_matter',
  NARRATIVE_SUMMARY: 'narrative_summary'
} as const;

export const TEMPLATE_CONFIG = {
  [SECTION_TEMPLATES.TEXT_LEFT_IMAGE_RIGHT]: {
    name: 'Text Left, Image Right',
    description: 'Text content on left side with supporting image on right',
    icon: Layout,
    preview: 'Like "A Legacy of Learning Excellence" - text flows left, image anchors right',
    fields: ['title', 'card_content', 'image']
  },
  [SECTION_TEMPLATES.IMAGE_LEFT_TEXT_RIGHT]: {
    name: 'Image Left, Text Right',
    description: 'Eye-catching image on left with descriptive text on right',
    icon: Image,
    preview: 'Like "Our Vision – Inspiring the Future" - image leads, text follows',
    fields: ['title', 'card_content', 'image']
  },
  [SECTION_TEMPLATES.TEXT_CARD_LEFT_IMAGE_RIGHT]: {
    name: 'Text + Card Left, Image Right',
    description: 'Main text with expandable card details, image on right',
    icon: FileText,
    preview: 'Like "Our Global Footprint" - text with stats card, supporting image',
    fields: ['title', 'content', 'card_title', 'card_subtitle', 'card_items', 'image']
  },
  [SECTION_TEMPLATES.IMAGE_LEFT_TEXT_CARD_RIGHT]: {
    name: 'Image Left, Text + Card Right',
    description: 'Leading image with text and detailed card information',
    icon: BookOpen,
    preview: 'Reverse of Global Footprint - image leads, text and details follow',
    fields: ['title', 'content', 'card_title', 'card_subtitle', 'card_items', 'image']
  },
  [SECTION_TEMPLATES.MISSION_VISION]: {
    name: 'Mission & Vision Card',
    description: 'Side-by-side mission and vision statements in elegant cards',
    icon: Target,
    preview: 'Like "Our Mission & Vision" - dual cards with customizable labels',
    fields: ['title', 'mission_label', 'mission', 'vision_label', 'vision']
  },
  [SECTION_TEMPLATES.GRADIENT_CENTERED]: {
    name: 'Gradient Background Card',
    description: 'Centered content with beautiful gradient background',
    icon: Heart,
    preview: 'Like "Closing Statement" - elegant centered text with gradient backdrop',
    fields: ['title', 'content', 'gradient']
  },
  [SECTION_TEMPLATES.GOVERNANCE_PRINCIPLES]: {
    name: 'Governance Principles',
    description: 'Structured display of organizational principles with main icon',
    icon: Shield,
    preview: 'Like Leadership & Governance - principles grid with central icon',
    fields: ['title', 'content', 'principle_items', 'main_icon', 'numbered_cards', 'governance_structure']
  },
  [SECTION_TEMPLATES.BOARD_TRUSTEES]: {
    name: 'Board of Trustees',
    description: 'Professional profiles showcase for leadership team',
    icon: Users,
    preview: 'Like Leadership & Governance - trustee profiles with photos and bios',
    fields: ['title', 'content', 'trustee_profiles']
  },
  [SECTION_TEMPLATES.CEO_MESSAGE]: {
    name: 'CEO Message',
    description: 'Personal message from leadership with photo and signature',
    icon: MessageSquare,
    preview: 'Like Leadership & Governance - inspirational quote with CEO profile',
    fields: ['title', 'quote', 'content_items', 'ceo_name', 'ceo_title', 'ceo_image']
  },
  [SECTION_TEMPLATES.IMPACT_STATISTICS]: {
    name: 'Impact Statistics',
    description: 'Impressive metrics and milestones with visual emphasis',
    icon: TrendingUp,
    preview: 'Like "Our Impact" milestones - key numbers with compelling descriptions',
    fields: ['title', 'content', 'table_title', 'table_icon', 'metrics_data']
  },
  [SECTION_TEMPLATES.WHY_NUMBERS_MATTER]: {
    name: 'Why Numbers Matter',
    description: 'Explanation section with supporting points and reasoning',
    icon: BarChart3,
    preview: 'Like "Our Impact" explanation - why these achievements matter',
    fields: ['title', 'matter_items']
  },
  [SECTION_TEMPLATES.NARRATIVE_SUMMARY]: {
    name: 'Narrative Summary',
    description: 'Storytelling conclusion with call-to-action elements',
    icon: Award,
    preview: 'Like "Our Impact" conclusion - comprehensive narrative with next steps',
    fields: ['title', 'content', 'cta_text', 'cta_link']
  }
} as const;

// Helper function to serialize section data for form based on template type
function serializeSectionDataForForm(section: any) {
  const sectionData = section.data || {};
  const templateType = section.templateType;
  
  // Base form data with common fields
  const formData: any = {
    // Top-level fields
    title: section.title || '',
    content: section.content || '',
    subtitle: section.subtitle || sectionData.subtitle || '',
    image_url: section.image_url || section.image || '',
    video_url: section.video_url || '',
    
    // String fields
    card_title: sectionData.card_title || '',
    card_subtitle: sectionData.card_subtitle || '',
    table_title: sectionData.table_title || '',
    table_icon: sectionData.table_icon || 'BarChart3',
    cta_text: sectionData.cta_text || '',
    cta_link: sectionData.cta_link || '',
    mission: sectionData.mission || '',
    vision: sectionData.vision || '',
    mission_label: sectionData.mission_label || 'Our Mission',
    vision_label: sectionData.vision_label || 'Our Vision',
    gradient: sectionData.gradient || '',
    main_icon: sectionData.main_icon || '',
    governance_structure: sectionData.governance_structure || '',
    quote: sectionData.quote || '',
    ceo_name: sectionData.ceo_name || '',
    ceo_title: sectionData.ceo_title || '',
    ceo_image: sectionData.ceo_image || '',
    gallery_layout: sectionData.gallery_layout || 'grid',
    
    // Arrays as comma-separated strings
    card_items: Array.isArray(sectionData.card_items) ? sectionData.card_items.join(', ') : (sectionData.card_items || ''),
    
    // Arrays/Objects as JSON strings for JSON editors
    metrics_data: typeof sectionData.metrics_data === 'object' ? JSON.stringify(sectionData.metrics_data) : (sectionData.metrics_data || ''),
    principle_items: Array.isArray(sectionData.principle_items) ? JSON.stringify(sectionData.principle_items) : (sectionData.principle_items || ''),
    numbered_cards: Array.isArray(sectionData.numbered_cards) ? JSON.stringify(sectionData.numbered_cards) : (sectionData.numbered_cards || ''),
    trustee_profiles: typeof sectionData.trustee_profiles === 'object' ? JSON.stringify(sectionData.trustee_profiles) : (sectionData.trustee_profiles || ''),
    matter_items: Array.isArray(sectionData.matter_items) ? JSON.stringify(sectionData.matter_items) : (sectionData.matter_items || ''),
  };
  
  // Handle card_content - ContentWithCardEditor expects JSON string with specific structure
  if (sectionData.content_items || sectionData.bullet_points || sectionData.card_position !== undefined || sectionData.final_content) {
    // Build card_content JSON from individual fields in data JSONB
    const cardContentData: any = {};
    if (sectionData.content_items) cardContentData.content_items = sectionData.content_items;
    if (sectionData.card_position !== undefined) cardContentData.card_position = sectionData.card_position;
    if (sectionData.bullet_points) cardContentData.bullet_points = sectionData.bullet_points;
    if (sectionData.bullet_points_position !== undefined) cardContentData.bullet_points_position = sectionData.bullet_points_position;
    if (sectionData.card_content) cardContentData.card_content = sectionData.card_content;
    if (sectionData.final_content) cardContentData.final_content = sectionData.final_content;
    formData.card_content = JSON.stringify(cardContentData);
  } else if (sectionData.card_content && typeof sectionData.card_content === 'string') {
    // If only card_content exists as a string, wrap it in the expected structure
    formData.card_content = JSON.stringify({ card_content: sectionData.card_content });
  } else {
    formData.card_content = '';
  }
  
  // Template-specific handling for content_items
  if (templateType === 'ceo_message') {
    // CEO Message uses ContentParagraphsEditor which expects double-newline-delimited string
    const contentItems = sectionData.content_items;
    if (Array.isArray(contentItems)) {
      formData.content_items = contentItems.map((item: any) => typeof item === 'string' ? item : item.content).join('\n\n');
    } else {
      formData.content_items = contentItems || '';
    }
  } else {
    // Other templates expect JSON string
    formData.content_items = Array.isArray(sectionData.content_items) ? JSON.stringify(sectionData.content_items) : (sectionData.content_items || '');
  }
  
  return formData;
}

// Template type field mapping - defines which fields each template type should show
const TEMPLATE_FIELD_MAPPING = {
  // 1. Text Left, Image Right - like "A Legacy of Learning Excellence" section
  text_left_image_right: {
    fields: ['title', 'card_content', 'image'],
    name: 'Text Left, Image Right (like "A Legacy of Learning Excellence")'
  },
  
  // 2. Image Left, Text Right - like "Our Vision – Inspiring the Future" section  
  image_left_text_right: {
    fields: ['title', 'card_content', 'image'], 
    name: 'Image Left, Text Right (like "Our Vision – Inspiring the Future")'
  },
  
  // 3. Text + Card Left, Image Right - like "Our Global Footprint" section
  text_left_image_right_with_card: {
    fields: ['title', 'content', 'image', 'card_title', 'card_subtitle', 'card_items'],
    name: 'Text + Card Left, Image Right (like "Our Global Footprint")'
  },
  
  // 4. Image Left, Text + Card Right - reverse of #3
  image_left_text_card_right: {
    fields: ['title', 'content', 'image', 'card_title', 'card_subtitle', 'card_items'],
    name: 'Image Left, Text + Card Right (reverse of Global Footprint)'
  },
  
  // 5. Horizontal Mission & Vision Card - like "Our Mission & Vision" section
  centered_card: {
    fields: ['title', 'mission', 'vision', 'mission_label', 'vision_label', 'text_color', 'background_color'],
    name: 'Mission & Vision Card (like "Our Mission & Vision")'
  },
  
  // 6. Gradient Background Card - like "Closing Statement" section
  gradient_centered: {
    fields: ['title', 'content', 'gradient', 'text_color'],
    name: 'Gradient Background Card (like "Closing Statement")'
  },
  
  // 7. Governance Principles Design - icon + title + content + principle cards  
  governance_principles: {
    fields: ['title', 'content', 'principle_items', 'main_icon', 'numbered_cards', 'governance_structure'],
    name: 'Governance Principles (like Leadership & Governance page)'
  },
  
  // 8. Board of Trustees Design - grid of profile cards
  board_trustees: {
    fields: ['title', 'content', 'trustee_profiles'],
    name: 'Board of Trustees (like Leadership & Governance page)'
  },
  
  // 9. CEO Message Design - gradient with quote and profile
  ceo_message: {
    fields: ['title', 'quote', 'content_items', 'ceo_name', 'ceo_title', 'ceo_image', 'gradient', 'text_color'],
    name: 'CEO Message (like Leadership & Governance page)'
  },
  
  // 10. Impact Statistics Design - title + metrics list  
  impact_statistics: {
    fields: ['title', 'content', 'table_title', 'table_icon', 'metrics_data'],
    name: 'Impact Statistics (like "Our Impact" milestones section)'
  },
  
  // 11. Why Numbers Matter Design - icon grid with explanations
  why_numbers_matter: {
    fields: ['title', 'subtitle', 'main_icon', 'matter_items'],
    name: 'Why Numbers Matter (like "Our Impact" page explanation)'
  },
  
  // 12. Narrative Summary Design - full gradient with centered text
  narrative_summary: {
    fields: ['title', 'subtitle', 'content_items', 'main_icon'],
    name: 'Narrative Summary (like "Our Impact" conclusion)'
  }
} as const;

// Form schema for section editor - includes all possible fields
const createSectionSchema = z.object({
  // Basic fields
  title: z.string().optional(),
  content: z.string().optional(),
  subtitle: z.string().optional(),
  
  // CTA fields
  cta_text: z.string().optional(),
  cta_link: z.string().optional(),
  
  // Card fields
  card_title: z.string().optional(),
  card_subtitle: z.string().optional(),
  card_items: z.string().optional(), // comma-separated list
  card_content: z.string().optional(), // Optional reddish blue card content
  card_position: z.string().optional(), // Position of card after which content paragraph
  numbered_cards: z.string().optional(), // JSON array of numbered cards for Core Learning Modules style
  
  // Mission & Vision fields
  mission: z.string().optional(),
  vision: z.string().optional(),
  mission_label: z.string().optional(),
  vision_label: z.string().optional(),
  
  // Color fields
  text_color: z.string().optional(),
  background_color: z.string().optional(),
  gradient: z.string().optional(),
  
  // New fields for advanced templates
  quote: z.string().optional(), // CEO quote
  content_items: z.string().optional(), // JSON string for dynamic content paragraphs
  ceo_name: z.string().optional(),
  ceo_title: z.string().optional(),
  ceo_image: z.union([z.instanceof(FileList), z.string()]).optional(),
  main_icon: z.string().optional(), // icon name
  principle_items: z.string().optional(), // JSON string for governance principles array
  governance_structure: z.string().optional(), // governance structure text box
  trustee_profiles: z.string().optional(), // JSON string for trustee data
  table_title: z.string().optional(), // Table title for impact statistics
  table_icon: z.string().optional(), // Icon name for impact statistics table
  metrics_data: z.string().optional(), // JSON string for metrics
  matter_items: z.string().optional(), // JSON string for why numbers matter
  
  // Image fields - accept both FileList (new uploads) and string (existing filenames)
  image: z.union([z.instanceof(FileList), z.string()]).optional(),
  background_image: z.union([z.instanceof(FileList), z.string()]).optional(),
  images: z.union([z.instanceof(FileList), z.string()]).optional(),
  gallery_layout: z.string().optional(),
});

type CreateSectionData = z.infer<typeof createSectionSchema>;

// Dynamic card items management component
function CardItemsEditor({ value, onChange }: { value?: string; onChange: (value: string) => void }) {
  const items = value ? value.split(',').map(item => item.trim()).filter(item => item) : [];
  const [newItem, setNewItem] = useState('');
  
  const addItem = () => {
    if (newItem.trim()) {
      const updatedItems = [...items, newItem.trim()];
      onChange(updatedItems.join(', '));
      setNewItem('');
    }
  };
  
  const removeItem = (index: number) => {
    const updatedItems = items.filter((_, i) => i !== index);
    onChange(updatedItems.join(', '));
  };
  
  const updateItem = (index: number, newValue: string) => {
    const updatedItems = [...items];
    updatedItems[index] = newValue;
    onChange(updatedItems.join(', '));
  };
  
  return (
    <div className="space-y-2">
      {/* Existing items */}
      {items.map((item, index) => (
        <div key={index} className="flex items-center gap-2 p-3 border rounded-lg bg-gray-50">
          <Input
            value={item}
            onChange={(e) => updateItem(index, e.target.value)}
            className="flex-1"
            data-testid={`card-item-${index}`}
          />
          <Button
            type="button"
            variant="outline"
            size="sm"
            onClick={() => removeItem(index)}
            data-testid={`remove-card-item-${index}`}
          >
            <X className="h-4 w-4" />
          </Button>
        </div>
      ))}
      
      {/* Add new item */}
      <div className="flex items-center gap-2 p-3 border-2 border-dashed border-gray-300 rounded-lg">
        <Input
          value={newItem}
          onChange={(e) => setNewItem(e.target.value)}
          placeholder="Add a new partnership (e.g., Harvard University)"
          className="flex-1"
          onKeyDown={(e) => {
            if (e.key === 'Enter') {
              e.preventDefault();
              addItem();
            }
          }}
          data-testid="input-new-card-item"
        />
        <Button
          type="button"
          onClick={addItem}
          disabled={!newItem.trim()}
          data-testid="button-add-card-item"
        >
          <Plus className="h-4 w-4 mr-1" />
          Add
        </Button>
      </div>
      
      {items.length === 0 && (
        <p className="text-sm text-gray-500 italic">No partnerships added yet. Add your first one above.</p>
      )}
    </div>
  );
}

// Content paragraphs management component for CEO Message template
function ContentParagraphsEditor({ value, onChange }: { value?: string; onChange: (value: string) => void }) {
  // Parse paragraphs from double-line-break separated string
  const paragraphs = value ? value.split('\n\n').map(p => p.trim()).filter(p => p) : [];
  const [newParagraph, setNewParagraph] = useState('');
  
  const addParagraph = () => {
    if (newParagraph.trim()) {
      const updatedParagraphs = [...paragraphs, newParagraph.trim()];
      onChange(updatedParagraphs.join('\n\n'));
      setNewParagraph('');
    }
  };
  
  const removeParagraph = (index: number) => {
    const updatedParagraphs = paragraphs.filter((_, i) => i !== index);
    onChange(updatedParagraphs.join('\n\n'));
  };
  
  const updateParagraph = (index: number, newValue: string) => {
    const updatedParagraphs = [...paragraphs];
    updatedParagraphs[index] = newValue;
    onChange(updatedParagraphs.join('\n\n'));
  };
  
  return (
    <div className="space-y-2">
      {/* Existing paragraphs */}
      {paragraphs.map((paragraph, index) => (
        <div key={index} className="flex items-start gap-2 p-3 border rounded-lg bg-gray-50">
          <Textarea
            value={paragraph}
            onChange={(e) => updateParagraph(index, e.target.value)}
            className="flex-1 min-h-[80px]"
            placeholder="Enter paragraph content"
            data-testid={`content-paragraph-${index}`}
          />
          <Button
            type="button"
            variant="outline"
            size="sm"
            onClick={() => removeParagraph(index)}
            data-testid={`remove-paragraph-${index}`}
          >
            <X className="h-4 w-4" />
          </Button>
        </div>
      ))}
      
      {/* Add new paragraph */}
      <div className="flex items-start gap-2 p-3 border-2 border-dashed border-gray-300 rounded-lg">
        <Textarea
          value={newParagraph}
          onChange={(e) => setNewParagraph(e.target.value)}
          placeholder="Add a new content paragraph..."
          className="flex-1 min-h-[80px]"
          onKeyDown={(e) => {
            if (e.key === 'Enter' && e.ctrlKey) {
              e.preventDefault();
              addParagraph();
            }
          }}
          data-testid="input-new-paragraph"
        />
        <Button
          type="button"
          onClick={addParagraph}
          disabled={!newParagraph.trim()}
          data-testid="button-add-paragraph"
        >
          <Plus className="h-4 w-4 mr-1" />
          Add
        </Button>
      </div>
      
      {paragraphs.length === 0 && (
        <p className="text-sm text-gray-500 italic">No content paragraphs added yet. Add your first one above.</p>
      )}
      
      <p className="text-xs text-gray-500">
        Tip: Press Ctrl+Enter in the text area to quickly add a paragraph
      </p>
    </div>
  );
}

// Metrics data management component for Impact Statistics template
function MetricsDataEditor({ value, onChange }: { value?: string; onChange: (value: string) => void }) {
  // Parse existing metrics from JSON string
  const metrics = (() => {
    try {
      return value ? JSON.parse(value) : [];
    } catch {
      return [];
    }
  })();
  
  const [newMetric, setNewMetric] = useState({ metric: '', description: '', category: '' });
  
  const addMetric = () => {
    if (newMetric.metric.trim() && newMetric.description.trim()) {
      const updatedMetrics = [...metrics, { ...newMetric }];
      onChange(JSON.stringify(updatedMetrics));
      setNewMetric({ metric: '', description: '', category: '' });
    }
  };
  
  const removeMetric = (index: number) => {
    const updatedMetrics = metrics.filter((_: any, i: number) => i !== index);
    onChange(JSON.stringify(updatedMetrics));
  };
  
  const updateMetric = (index: number, field: string, newValue: string) => {
    const updatedMetrics = [...metrics];
    updatedMetrics[index] = { ...updatedMetrics[index], [field]: newValue };
    onChange(JSON.stringify(updatedMetrics));
  };
  
  return (
    <div className="space-y-4">
      {/* Existing metrics */}
      {metrics.map((metric: any, index: number) => (
        <div key={index} className="p-4 border rounded-lg bg-gray-50 space-y-3">
          <div className="flex items-center justify-between">
            <h4 className="font-medium text-sm">Metric {index + 1}</h4>
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={() => removeMetric(index)}
              data-testid={`remove-metric-${index}`}
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Category (optional)</label>
              <Input
                value={metric.category || ''}
                onChange={(e) => updateMetric(index, 'category', e.target.value)}
                placeholder="e.g., Participants, Service Providers"
                className="text-sm"
                data-testid={`metric-category-${index}`}
              />
            </div>
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Metric/Value *</label>
              <Input
                value={metric.metric}
                onChange={(e) => updateMetric(index, 'metric', e.target.value)}
                placeholder="e.g., 256,000+"
                className="text-sm"
                data-testid={`metric-value-${index}`}
              />
            </div>
          </div>
          
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Description *</label>
            <Textarea
              value={metric.description}
              onChange={(e) => updateMetric(index, 'description', e.target.value)}
              placeholder="e.g., Programs participants benefited from this e-learning experience."
              rows={2}
              className="text-sm resize-none"
              data-testid={`metric-description-${index}`}
            />
          </div>
        </div>
      ))}
      
      {/* Add new metric */}
      <div className="p-4 border-2 border-dashed border-gray-300 rounded-lg space-y-3">
        <h4 className="font-medium text-sm text-gray-600">Add New Metric</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Category (optional)</label>
            <Input
              value={newMetric.category}
              onChange={(e) => setNewMetric(prev => ({ ...prev, category: e.target.value }))}
              placeholder="e.g., Participants, Service Providers"
              className="text-sm"
              data-testid="input-new-metric-category"
            />
          </div>
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Metric/Value *</label>
            <Input
              value={newMetric.metric}
              onChange={(e) => setNewMetric(prev => ({ ...prev, metric: e.target.value }))}
              placeholder="e.g., 256,000+"
              className="text-sm"
              data-testid="input-new-metric-value"
            />
          </div>
        </div>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Description *</label>
          <Textarea
            value={newMetric.description}
            onChange={(e) => setNewMetric(prev => ({ ...prev, description: e.target.value }))}
            placeholder="e.g., Programs participants benefited from this e-learning experience."
            rows={2}
            className="text-sm resize-none"
            data-testid="input-new-metric-description"
          />
        </div>
        
        <Button
          type="button"
          onClick={addMetric}
          disabled={!newMetric.metric.trim() || !newMetric.description.trim()}
          className="w-full"
          data-testid="button-add-metric"
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Metric
        </Button>
      </div>
      
      {metrics.length === 0 && (
        <p className="text-sm text-gray-500 italic">No metrics added yet. Add your first metric above.</p>
      )}
      
      <p className="text-xs text-gray-500">
        Add impact statistics like participant counts, service providers, achievements, etc.
      </p>
    </div>
  );
}

// Combined items management component for Why Numbers Matter template (supports both matter and pricing items)
function MatterItemsEditor({ value, onChange }: { value?: string; onChange: (value: string) => void }) {
  // Parse existing items from JSON string
  const items = (() => {
    try {
      return value ? JSON.parse(value) : [];
    } catch {
      return [];
    }
  })();
  
  const [newMatterItem, setNewMatterItem] = useState({ type: 'matter', icon: '', title: '', description: '' });
  const [newPricingItem, setNewPricingItem] = useState({ type: 'pricing', icon: '', title: '', price: '', footer: '' });
  const [newTestimonialItem, setNewTestimonialItem] = useState({ type: 'testimonial', quote: '', name: '' });
  
  // Available icon options for items
  const iconOptions = [
    'Scale', 'Users', 'Building', 'Target', 'Award', 'Shield', 'Heart', 'Globe', 'Lock', 'Star', 
    'CheckCircle', 'Compass', 'Eye', 'Handshake', 'Lightbulb', 'TrendingUp',
    'BookOpen', 'Zap', 'Settings', 'Crown', 'Rocket', 'DollarSign'
  ];
  
  const addMatterItem = () => {
    if (newMatterItem.icon.trim() && newMatterItem.title.trim() && newMatterItem.description.trim()) {
      const updatedItems = [...items, { ...newMatterItem }];
      onChange(JSON.stringify(updatedItems));
      setNewMatterItem({ type: 'matter', icon: '', title: '', description: '' });
    }
  };
  
  const addPricingItem = () => {
    if (newPricingItem.icon.trim() && newPricingItem.title.trim() && newPricingItem.price.trim()) {
      const updatedItems = [...items, { ...newPricingItem }];
      onChange(JSON.stringify(updatedItems));
      setNewPricingItem({ type: 'pricing', icon: '', title: '', price: '', footer: '' });
    }
  };
  
  const addTestimonialItem = () => {
    if (newTestimonialItem.quote.trim() && newTestimonialItem.name.trim()) {
      const updatedItems = [...items, { ...newTestimonialItem }];
      onChange(JSON.stringify(updatedItems));
      setNewTestimonialItem({ type: 'testimonial', quote: '', name: '' });
    }
  };
  
  const removeItem = (index: number) => {
    const updatedItems = items.filter((_: any, i: number) => i !== index);
    onChange(JSON.stringify(updatedItems));
  };
  
  const updateItem = (index: number, field: string, newValue: string) => {
    const updatedItems = [...items];
    updatedItems[index] = { ...updatedItems[index], [field]: newValue };
    onChange(JSON.stringify(updatedItems));
  };
  
  // Separate matter, pricing, and testimonial items
  // Only include items with valid data (not empty objects or undefined values)
  const matterItems = items.filter((item: any) => 
    (item.type === 'matter' || item.type === undefined) && 
    item.icon && item.title && item.description
  );
  const pricingItems = items.filter((item: any) => 
    item.type === 'pricing' && 
    item.icon && item.title && item.price
  );
  const testimonialItems = items.filter((item: any) => 
    item.type === 'testimonial' && 
    item.quote && item.name
  );

  return (
    <div className="space-y-6">
      
      {/* Add new matter item */}
      <div className="p-4 border-2 border-dashed border-gray-300 rounded-lg space-y-3">
        <h4 className="font-medium text-sm text-gray-600">Add New Matter Item</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Icon</label>
            <Select value={newMatterItem.icon} onValueChange={(value) => setNewMatterItem(prev => ({ ...prev, icon: value }))}>
              <SelectTrigger data-testid="select-new-matter-item-icon">
                <SelectValue placeholder="Select icon" />
              </SelectTrigger>
              <SelectContent>
                {iconOptions.map((iconName) => (
                  <SelectItem key={iconName} value={iconName}>
                    {iconName}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Title</label>
            <Input
              value={newMatterItem.title}
              onChange={(e) => setNewMatterItem(prev => ({ ...prev, title: e.target.value }))}
              placeholder="e.g., Scale with Quality"
              className="text-sm"
              data-testid="input-new-matter-item-title"
            />
          </div>
        </div>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Description</label>
          <Textarea
            value={newMatterItem.description}
            onChange={(e) => setNewMatterItem(prev => ({ ...prev, description: e.target.value }))}
            placeholder="e.g., Serving over a quarter-million learners across more than 34 nations demonstrates that IBDL scales effectively..."
            rows={3}
            className="text-sm"
            data-testid="input-new-matter-item-description"
          />
        </div>
        
        <Button
          type="button"
          onClick={addMatterItem}
          disabled={!newMatterItem.icon.trim() || !newMatterItem.title.trim() || !newMatterItem.description.trim()}
          className="w-full"
          data-testid="button-add-matter-item"
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Matter Item
        </Button>
      </div>

      {/* Existing matter items */}
      {matterItems.map((item: any, index: number) => {
        const originalIndex = items.findIndex((originalItem: any) => originalItem === item);
        return (
          <div key={originalIndex} className="p-4 border rounded-lg bg-gray-50 space-y-3">
            <div className="flex items-center justify-between">
              <h4 className="font-medium text-sm">
                Matter Item {index + 1}
                <span className="ml-2 text-xs bg-blue-100 text-blue-800 px-2 py-1 rounded">
                  Matter
                </span>
              </h4>
              <Button
                type="button"
                variant="outline"
                size="sm"
                onClick={() => removeItem(originalIndex)}
                data-testid={`remove-matter-item-${index}`}
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
              <div>
                <label className="text-xs font-medium text-gray-700 mb-1 block">Icon</label>
                <Select value={item.icon} onValueChange={(value) => updateItem(originalIndex, 'icon', value)}>
                  <SelectTrigger data-testid={`matter-item-icon-${index}`}>
                    <SelectValue placeholder="Select icon" />
                  </SelectTrigger>
                  <SelectContent>
                    {iconOptions.map((iconName) => (
                      <SelectItem key={iconName} value={iconName}>
                        {iconName}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <label className="text-xs font-medium text-gray-700 mb-1 block">Title</label>
                <Input
                  value={item.title}
                  onChange={(e) => updateItem(originalIndex, 'title', e.target.value)}
                  placeholder="e.g., Scale with Quality"
                  className="text-sm"
                  data-testid={`matter-item-title-${index}`}
                />
              </div>
            </div>
            
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Description</label>
              <Textarea
                value={item.description}
                onChange={(e) => updateItem(originalIndex, 'description', e.target.value)}
                placeholder="e.g., Serving over a quarter-million learners across more than 34 nations demonstrates that IBDL scales effectively..."
                rows={3}
                className="text-sm"
                data-testid={`matter-item-description-${index}`}
              />
            </div>
          </div>
        );
      })}

      {/* Add new pricing item */}
      <div className="p-4 border-2 border-dashed border-gray-300 rounded-lg space-y-3">
        <h4 className="font-medium text-sm text-gray-600">Add New Pricing Item</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Icon</label>
            <Select value={newPricingItem.icon} onValueChange={(value) => setNewPricingItem(prev => ({ ...prev, icon: value }))}>
              <SelectTrigger data-testid="select-new-pricing-item-icon">
                <SelectValue placeholder="Select icon" />
              </SelectTrigger>
              <SelectContent>
                {iconOptions.map((iconName) => (
                  <SelectItem key={iconName} value={iconName}>
                    {iconName}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Title</label>
            <Input
              value={newPricingItem.title}
              onChange={(e) => setNewPricingItem(prev => ({ ...prev, title: e.target.value }))}
              placeholder="e.g., Basic Plan"
              className="text-sm"
              data-testid="input-new-pricing-item-title"
            />
          </div>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Price</label>
            <Input
              value={newPricingItem.price}
              onChange={(e) => setNewPricingItem(prev => ({ ...prev, price: e.target.value }))}
              placeholder="e.g., £299"
              className="text-sm"
              data-testid="input-new-pricing-item-price"
            />
          </div>
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Footer</label>
            <Input
              value={newPricingItem.footer}
              onChange={(e) => setNewPricingItem(prev => ({ ...prev, footer: e.target.value }))}
              placeholder="e.g., per qualification"
              className="text-sm"
              data-testid="input-new-pricing-item-footer"
            />
          </div>
        </div>
        
        <Button
          type="button"
          onClick={addPricingItem}
          disabled={!newPricingItem.icon.trim() || !newPricingItem.title.trim() || !newPricingItem.price.trim()}
          className="w-full"
          data-testid="button-add-pricing-item"
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Pricing Item
        </Button>
      </div>

      {/* Existing pricing items */}
      {pricingItems.map((item: any, index: number) => {
        const originalIndex = items.findIndex((originalItem: any) => originalItem === item);
        return (
          <div key={originalIndex} className="p-4 border rounded-lg bg-gray-50 space-y-3">
            <div className="flex items-center justify-between">
              <h4 className="font-medium text-sm">
                Pricing Item {index + 1}
                <span className="ml-2 text-xs bg-blue-100 text-blue-800 px-2 py-1 rounded">
                  Pricing
                </span>
              </h4>
              <Button
                type="button"
                variant="outline"
                size="sm"
                onClick={() => removeItem(originalIndex)}
                data-testid={`remove-pricing-item-${index}`}
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
              <div>
                <label className="text-xs font-medium text-gray-700 mb-1 block">Icon</label>
                <Select value={item.icon} onValueChange={(value) => updateItem(originalIndex, 'icon', value)}>
                  <SelectTrigger data-testid={`pricing-item-icon-${index}`}>
                    <SelectValue placeholder="Select icon" />
                  </SelectTrigger>
                  <SelectContent>
                    {iconOptions.map((iconName) => (
                      <SelectItem key={iconName} value={iconName}>
                        {iconName}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <label className="text-xs font-medium text-gray-700 mb-1 block">Title</label>
                <Input
                  value={item.title}
                  onChange={(e) => updateItem(originalIndex, 'title', e.target.value)}
                  placeholder="e.g., Basic Plan"
                  className="text-sm"
                  data-testid={`pricing-item-title-${index}`}
                />
              </div>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
              <div>
                <label className="text-xs font-medium text-gray-700 mb-1 block">Price</label>
                <Input
                  value={item.price}
                  onChange={(e) => updateItem(originalIndex, 'price', e.target.value)}
                  placeholder="e.g., £299"
                  className="text-sm"
                  data-testid={`pricing-item-price-${index}`}
                />
              </div>
              <div>
                <label className="text-xs font-medium text-gray-700 mb-1 block">Footer</label>
                <Input
                  value={item.footer}
                  onChange={(e) => updateItem(originalIndex, 'footer', e.target.value)}
                  placeholder="e.g., per qualification"
                  className="text-sm"
                  data-testid={`pricing-item-footer-${index}`}
                />
              </div>
            </div>
          </div>
        );
      })}

      {/* Add new testimonial item */}
      <div className="p-4 border-2 border-dashed border-gray-300 rounded-lg space-y-3">
        <h4 className="font-medium text-sm text-gray-600">Add New Testimonial</h4>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Quote</label>
          <Textarea
            value={newTestimonialItem.quote}
            onChange={(e) => setNewTestimonialItem(prev => ({ ...prev, quote: e.target.value }))}
            placeholder="e.g., Enrolling in the IBDL Business Essential Professional Qualification was a pivotal step in advancing my career..."
            rows={4}
            className="text-sm"
            data-testid="input-new-testimonial-quote"
          />
        </div>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Name</label>
          <Input
            value={newTestimonialItem.name}
            onChange={(e) => setNewTestimonialItem(prev => ({ ...prev, name: e.target.value }))}
            placeholder="e.g., Alessandro Rossi, Italy"
            className="text-sm"
            data-testid="input-new-testimonial-name"
          />
        </div>
        
        <Button
          type="button"
          onClick={addTestimonialItem}
          disabled={!newTestimonialItem.quote.trim() || !newTestimonialItem.name.trim()}
          className="w-full"
          data-testid="button-add-testimonial"
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Testimonial
        </Button>
      </div>

      {/* Existing testimonial items */}
      {testimonialItems.map((item: any, index: number) => {
        const originalIndex = items.findIndex((originalItem: any) => originalItem === item);
        return (
          <div key={originalIndex} className="p-4 border rounded-lg bg-gray-50 space-y-3">
            <div className="flex items-center justify-between">
              <h4 className="font-medium text-sm">
                Testimonial {index + 1}
                <span className="ml-2 text-xs bg-green-100 text-green-800 px-2 py-1 rounded">
                  Testimonial
                </span>
              </h4>
              <Button
                type="button"
                variant="outline"
                size="sm"
                onClick={() => removeItem(originalIndex)}
                data-testid={`remove-testimonial-${index}`}
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
            
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Quote</label>
              <Textarea
                value={item.quote}
                onChange={(e) => updateItem(originalIndex, 'quote', e.target.value)}
                placeholder="e.g., Enrolling in the IBDL Business Essential Professional Qualification was a pivotal step in advancing my career..."
                rows={4}
                className="text-sm"
                data-testid={`testimonial-quote-${index}`}
              />
            </div>
            
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Name</label>
              <Input
                value={item.name}
                onChange={(e) => updateItem(originalIndex, 'name', e.target.value)}
                placeholder="e.g., Alessandro Rossi, Italy"
                className="text-sm"
                data-testid={`testimonial-name-${index}`}
              />
            </div>
          </div>
        );
      })}
      
      {items.length === 0 && (
        <p className="text-sm text-gray-500 italic">No items added yet. Add your first one above.</p>
      )}
    </div>
  );
}


// Narrative content items management component for Narrative Summary template
function NarrativeContentEditor({ value, onChange }: { value?: string; onChange: (value: string) => void }) {
  // Parse existing content items from JSON string
  const contentItems = (() => {
    try {
      return value ? JSON.parse(value) : [];
    } catch {
      return [];
    }
  })();
  
  const [newContentItem, setNewContentItem] = useState('');
  
  const addContentItem = () => {
    if (newContentItem.trim()) {
      const updatedContentItems = [...contentItems, newContentItem.trim()];
      onChange(JSON.stringify(updatedContentItems));
      setNewContentItem('');
    }
  };
  
  const removeContentItem = (index: number) => {
    const updatedContentItems = contentItems.filter((_: any, i: number) => i !== index);
    onChange(JSON.stringify(updatedContentItems));
  };
  
  const updateContentItem = (index: number, newValue: string) => {
    const updatedContentItems = [...contentItems];
    updatedContentItems[index] = newValue;
    onChange(JSON.stringify(updatedContentItems));
  };
  
  return (
    <div className="space-y-4">
      {/* Existing content items */}
      {contentItems.map((item: string, index: number) => (
        <div key={index} className="p-4 border rounded-lg bg-gray-50 space-y-3">
          <div className="flex items-center justify-between">
            <h4 className="font-medium text-sm">Content Paragraph {index + 1}</h4>
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={() => removeContentItem(index)}
              data-testid={`remove-content-item-${index}`}
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
          
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Content</label>
            <Textarea
              value={item}
              onChange={(e) => updateContentItem(index, e.target.value)}
              placeholder="Enter content paragraph..."
              rows={4}
              className="text-sm"
              data-testid={`content-item-${index}`}
            />
          </div>
        </div>
      ))}
      
      {/* Add new content item */}
      <div className="p-4 border-2 border-dashed border-gray-300 rounded-lg space-y-3">
        <h4 className="font-medium text-sm text-gray-600">Add New Content Paragraph</h4>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Content</label>
          <Textarea
            value={newContentItem}
            onChange={(e) => setNewContentItem(e.target.value)}
            placeholder="Enter content paragraph..."
            rows={4}
            className="text-sm"
            data-testid="input-new-content-item"
          />
        </div>
        
        <Button
          type="button"
          onClick={addContentItem}
          disabled={!newContentItem.trim()}
          className="w-full"
          data-testid="button-add-content-item"
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Content Paragraph
        </Button>
      </div>
      
      {contentItems.length === 0 && (
        <p className="text-sm text-gray-500 italic">No content paragraphs added yet. Add your first one above.</p>
      )}
    </div>
  );
}

// Content with optional positioned card editor for Text Left/Right Image templates
function ContentWithCardEditor({ value, onChange }: { value?: string; onChange: (value: string) => void }) {
  // Parse existing content items and all optional features from JSON string
  const data = (() => {
    try {
      return value ? JSON.parse(value) : { 
        content_items: [], 
        card_content: '', 
        card_position: 0, 
        bullet_points: [], 
        bullet_points_position: 0,
        final_content: ''
      };
    } catch {
      return { 
        content_items: [], 
        card_content: '', 
        card_position: 0, 
        bullet_points: [], 
        bullet_points_position: 0,
        final_content: ''
      };
    }
  })();
  
  const [contentItems, setContentItems] = useState<string[]>(data.content_items || []);
  const [cardContent, setCardContent] = useState<string>(data.card_content || '');
  const [cardPosition, setCardPosition] = useState<number>(data.card_position || 0);
  const [bulletPoints, setBulletPoints] = useState<{title: string; description: string}[]>(data.bullet_points || []);
  const [bulletPointsPosition, setBulletPointsPosition] = useState<number>(data.bullet_points_position || 0);
  const [finalContent, setFinalContent] = useState<string>(data.final_content || '');
  const [newContentItem, setNewContentItem] = useState('');
  const [newBulletPoint, setNewBulletPoint] = useState({ title: '', description: '' });

  // Update the parent form whenever local state changes
  const updateData = (
    newContentItems: string[], 
    newCardContent: string, 
    newCardPosition: number, 
    newBulletPoints: {title: string; description: string}[], 
    newBulletPointsPosition: number,
    newFinalContent: string
  ) => {
    const updatedData = {
      content_items: newContentItems,
      card_content: newCardContent,
      card_position: newCardPosition,
      bullet_points: newBulletPoints,
      bullet_points_position: newBulletPointsPosition,
      final_content: newFinalContent
    };
    onChange(JSON.stringify(updatedData));
  };

  const addContentItem = () => {
    if (newContentItem.trim()) {
      const updatedContentItems = [...contentItems, newContentItem.trim()];
      setContentItems(updatedContentItems);
      updateData(updatedContentItems, cardContent, cardPosition, bulletPoints, bulletPointsPosition, finalContent);
      setNewContentItem('');
    }
  };
  
  const removeContentItem = (index: number) => {
    const updatedContentItems = contentItems.filter((_, i) => i !== index);
    setContentItems(updatedContentItems);
    // Adjust positions if they're after the removed item
    const newCardPosition = cardPosition > index ? Math.max(0, cardPosition - 1) : cardPosition;
    const newBulletPointsPosition = bulletPointsPosition > index ? Math.max(0, bulletPointsPosition - 1) : bulletPointsPosition;
    setCardPosition(newCardPosition);
    setBulletPointsPosition(newBulletPointsPosition);
    updateData(updatedContentItems, cardContent, newCardPosition, bulletPoints, newBulletPointsPosition, finalContent);
  };
  
  const updateContentItem = (index: number, newValue: string) => {
    const updatedContentItems = [...contentItems];
    updatedContentItems[index] = newValue;
    setContentItems(updatedContentItems);
    updateData(updatedContentItems, cardContent, cardPosition, bulletPoints, bulletPointsPosition, finalContent);
  };

  const updateCardContent = (newCardContent: string) => {
    setCardContent(newCardContent);
    updateData(contentItems, newCardContent, cardPosition, bulletPoints, bulletPointsPosition, finalContent);
  };

  const updateCardPosition = (newCardPosition: number) => {
    setCardPosition(newCardPosition);
    updateData(contentItems, cardContent, newCardPosition, bulletPoints, bulletPointsPosition, finalContent);
  };

  // Bullet points management
  const addBulletPoint = () => {
    if (newBulletPoint.title.trim()) {
      const updatedBulletPoints = [...bulletPoints, { title: newBulletPoint.title.trim(), description: newBulletPoint.description.trim() }];
      setBulletPoints(updatedBulletPoints);
      updateData(contentItems, cardContent, cardPosition, updatedBulletPoints, bulletPointsPosition, finalContent);
      setNewBulletPoint({ title: '', description: '' });
    }
  };

  const removeBulletPoint = (index: number) => {
    const updatedBulletPoints = bulletPoints.filter((_, i) => i !== index);
    setBulletPoints(updatedBulletPoints);
    updateData(contentItems, cardContent, cardPosition, updatedBulletPoints, bulletPointsPosition, finalContent);
  };

  const updateBulletPoint = (index: number, field: 'title' | 'description', newValue: string) => {
    const updatedBulletPoints = [...bulletPoints];
    updatedBulletPoints[index][field] = newValue;
    setBulletPoints(updatedBulletPoints);
    updateData(contentItems, cardContent, cardPosition, updatedBulletPoints, bulletPointsPosition, finalContent);
  };

  const updateBulletPointsPosition = (newBulletPointsPosition: number) => {
    setBulletPointsPosition(newBulletPointsPosition);
    updateData(contentItems, cardContent, cardPosition, bulletPoints, newBulletPointsPosition, finalContent);
  };

  const updateFinalContent = (newFinalContent: string) => {
    setFinalContent(newFinalContent);
    updateData(contentItems, cardContent, cardPosition, bulletPoints, bulletPointsPosition, newFinalContent);
  };
  
  return (
    <div className="space-y-4">
      {/* Existing content items */}
      {contentItems.map((item: string, index: number) => (
        <div key={index} className="p-4 border rounded-lg bg-gray-50 space-y-3">
          <div className="flex items-center justify-between">
            <h4 className="font-medium text-sm">Content Paragraph {index + 1}</h4>
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={() => removeContentItem(index)}
              data-testid={`remove-content-item-${index}`}
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
          
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Content</label>
            <Textarea
              value={item}
              onChange={(e) => updateContentItem(index, e.target.value)}
              placeholder="Enter content paragraph..."
              rows={4}
              className="text-sm"
              data-testid={`content-item-${index}`}
            />
          </div>
        </div>
      ))}
      
      {/* Add new content item */}
      <div className="p-4 border-2 border-dashed border-gray-300 rounded-lg space-y-3">
        <h4 className="font-medium text-sm text-gray-600">Add New Content Paragraph</h4>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Content</label>
          <Textarea
            value={newContentItem}
            onChange={(e) => setNewContentItem(e.target.value)}
            placeholder="Enter content paragraph..."
            rows={4}
            className="text-sm"
            data-testid="input-new-content-item"
          />
        </div>
        
        <Button
          type="button"
          onClick={addContentItem}
          disabled={!newContentItem.trim()}
          className="w-full"
          data-testid="button-add-content-item"
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Content Paragraph
        </Button>
      </div>

      {/* Card Configuration */}
      <div className="p-4 border-2 border-dashed border-blue-300 rounded-lg space-y-3 bg-blue-50">
        <h4 className="font-medium text-sm text-blue-700">Optional Reddish Blue Card</h4>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Card Content (Quote/Message)</label>
          <Textarea
            value={cardContent}
            onChange={(e) => updateCardContent(e.target.value)}
            placeholder='e.g., "Rigorous knowledge. Global standards. Transformative leadership."'
            rows={3}
            className="text-sm"
            data-testid="input-card-content"
          />
        </div>

        {cardContent.trim() && (
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">
              Card Position (After which content paragraph?)
            </label>
            <Select 
              value={cardPosition.toString()} 
              onValueChange={(value) => updateCardPosition(parseInt(value))}
            >
              <SelectTrigger data-testid="select-card-position">
                <SelectValue placeholder="Select position" />
              </SelectTrigger>
              <SelectContent>
                {contentItems.map((_, index) => (
                  <SelectItem key={index} value={index.toString()}>
                    After Content Paragraph {index + 1}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        )}
      </div>

      {/* Bullet Points Configuration */}
      <div className="p-4 border-2 border-dashed border-green-300 rounded-lg space-y-3 bg-green-50">
        <h4 className="font-medium text-sm text-green-700">Optional Bullet Points</h4>
        
        {/* Existing bullet points */}
        {bulletPoints.map((bullet, index) => (
          <div key={index} className="p-3 border rounded-lg bg-white space-y-2">
            <div className="flex items-center justify-between">
              <h5 className="font-medium text-xs">Bullet Point {index + 1}</h5>
              <Button
                type="button"
                variant="outline"
                size="sm"
                onClick={() => removeBulletPoint(index)}
                data-testid={`remove-bullet-point-${index}`}
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
            
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Point Title</label>
              <Input
                value={bullet.title}
                onChange={(e) => updateBulletPoint(index, 'title', e.target.value)}
                placeholder="e.g., Cross-cultural and organizational behavior"
                className="text-sm"
                data-testid={`bullet-point-title-${index}`}
              />
            </div>
            
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Description (Optional)</label>
              <Textarea
                value={bullet.description}
                onChange={(e) => updateBulletPoint(index, 'description', e.target.value)}
                placeholder="Optional description for this point..."
                rows={2}
                className="text-sm"
                data-testid={`bullet-point-description-${index}`}
              />
            </div>
          </div>
        ))}
        
        {/* Add new bullet point */}
        <div className="p-3 border-2 border-dashed border-gray-300 rounded-lg space-y-2">
          <h5 className="font-medium text-xs text-gray-600">Add New Bullet Point</h5>
          
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Point Title</label>
            <Input
              value={newBulletPoint.title}
              onChange={(e) => setNewBulletPoint({...newBulletPoint, title: e.target.value})}
              placeholder="e.g., International strategic management"
              className="text-sm"
              data-testid="input-new-bullet-title"
            />
          </div>
          
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Description (Optional)</label>
            <Textarea
              value={newBulletPoint.description}
              onChange={(e) => setNewBulletPoint({...newBulletPoint, description: e.target.value})}
              placeholder="Optional description for this point..."
              rows={2}
              className="text-sm"
              data-testid="input-new-bullet-description"
            />
          </div>
          
          <Button
            type="button"
            onClick={addBulletPoint}
            disabled={!newBulletPoint.title.trim()}
            className="w-full"
            data-testid="button-add-bullet-point"
          >
            <Plus className="h-4 w-4 mr-2" />
            Add Bullet Point
          </Button>
        </div>

        {bulletPoints.length > 0 && (
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">
              Bullet Points Position (After which content paragraph?)
            </label>
            <Select 
              value={bulletPointsPosition.toString()} 
              onValueChange={(value) => updateBulletPointsPosition(parseInt(value))}
            >
              <SelectTrigger data-testid="select-bullet-points-position">
                <SelectValue placeholder="Select position" />
              </SelectTrigger>
              <SelectContent>
                {contentItems.map((_, index) => (
                  <SelectItem key={index} value={index.toString()}>
                    After Content Paragraph {index + 1}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        )}
      </div>

      {/* Final Content Configuration */}
      <div className="p-4 border-2 border-dashed border-purple-300 rounded-lg space-y-3 bg-purple-50">
        <h4 className="font-medium text-sm text-purple-700">Optional Final Content</h4>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Final Descriptive Text</label>
          <Textarea
            value={finalContent}
            onChange={(e) => updateFinalContent(e.target.value)}
            placeholder="e.g., This edition emphasizes comprehensiveness, customization, measurable outcomes, and alignment with international leadership standards."
            rows={3}
            className="text-sm"
            data-testid="input-final-content"
          />
        </div>
      </div>
      
      {contentItems.length === 0 && (
        <p className="text-sm text-gray-500 italic">No content paragraphs added yet. Add your first one above.</p>
      )}
    </div>
  );
}

// Numbered cards management component for Governance Principles template (Core Learning Modules style)
function NumberedCardsEditor({ value, onChange }: { value?: string; onChange: (value: string) => void }) {
  // Parse existing numbered cards from JSON string
  const cards = (() => {
    try {
      return value ? JSON.parse(value) : [];
    } catch {
      return [];
    }
  })();
  
  const [numberedCards, setNumberedCards] = useState<{title: string; content: string}[]>(cards);
  const [newCard, setNewCard] = useState({ title: '', content: '' });

  // Update the parent form whenever local state changes
  const updateCards = (newCards: {title: string; content: string}[]) => {
    setNumberedCards(newCards);
    onChange(JSON.stringify(newCards));
  };

  const addCard = () => {
    if (newCard.title.trim()) {
      const updatedCards = [...numberedCards, { title: newCard.title.trim(), content: newCard.content.trim() }];
      updateCards(updatedCards);
      setNewCard({ title: '', content: '' });
    }
  };
  
  const removeCard = (index: number) => {
    const updatedCards = numberedCards.filter((_, i) => i !== index);
    updateCards(updatedCards);
  };
  
  const updateCard = (index: number, field: 'title' | 'content', newValue: string) => {
    const updatedCards = [...numberedCards];
    updatedCards[index][field] = newValue;
    updateCards(updatedCards);
  };
  
  return (
    <div className="space-y-4">
      {/* Existing numbered cards */}
      {numberedCards.map((card, index) => (
        <div key={index} className="p-4 border rounded-lg bg-gray-50 space-y-3">
          <div className="flex items-center justify-between">
            <h4 className="font-medium text-sm flex items-center">
              <div className="w-6 h-6 bg-[#ED1C24] text-white rounded-full flex items-center justify-center text-xs font-bold mr-2">
                {index + 1}
              </div>
              Card {index + 1}
            </h4>
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={() => removeCard(index)}
              data-testid={`remove-numbered-card-${index}`}
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
          
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Card Title</label>
            <Input
              value={card.title}
              onChange={(e) => updateCard(index, 'title', e.target.value)}
              placeholder="e.g., Entrepreneurship & Starting a Small Business"
              className="text-sm"
              data-testid={`numbered-card-title-${index}`}
            />
          </div>
          
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Card Content</label>
            <Textarea
              value={card.content}
              onChange={(e) => updateCard(index, 'content', e.target.value)}
              placeholder="Optional content for this card..."
              rows={3}
              className="text-sm"
              data-testid={`numbered-card-content-${index}`}
            />
          </div>
        </div>
      ))}
      
      {/* Add new numbered card */}
      <div className="p-4 border-2 border-dashed border-gray-300 rounded-lg space-y-3">
        <h4 className="font-medium text-sm text-gray-600 flex items-center">
          <div className="w-6 h-6 bg-gray-400 text-white rounded-full flex items-center justify-center text-xs font-bold mr-2">
            {numberedCards.length + 1}
          </div>
          Add New Numbered Card
        </h4>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Card Title</label>
          <Input
            value={newCard.title}
            onChange={(e) => setNewCard({...newCard, title: e.target.value})}
            placeholder="e.g., Management: Functions & Styles"
            className="text-sm"
            data-testid="input-new-numbered-card-title"
          />
        </div>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Card Content (Optional)</label>
          <Textarea
            value={newCard.content}
            onChange={(e) => setNewCard({...newCard, content: e.target.value})}
            placeholder="Optional content for this card..."
            rows={3}
            className="text-sm"
            data-testid="input-new-numbered-card-content"
          />
        </div>
        
        <Button
          type="button"
          onClick={addCard}
          disabled={!newCard.title.trim()}
          className="w-full"
          data-testid="button-add-numbered-card"
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Numbered Card
        </Button>
      </div>
      
      {numberedCards.length === 0 && (
        <p className="text-sm text-gray-500 italic">No numbered cards added yet. Add your first one above.</p>
      )}
    </div>
  );
}

// Principle items management component
function PrincipleItemsEditor({ value, onChange }: { value?: string; onChange: (value: string) => void }) {
  // Parse existing principle items from JSON string
  const principles = (() => {
    try {
      return value ? JSON.parse(value) : [];
    } catch {
      return [];
    }
  })();
  
  const [newPrinciple, setNewPrinciple] = useState({ icon: '', title: '', description: '' });
  
  // Available icon options
  const iconOptions = [
    'Shield', 'Award', 'Users', 'Heart', 'Target', 'Globe', 'Lock', 'Star', 
    'CheckCircle', 'Compass', 'Eye', 'Handshake', 'Lightbulb', 'TrendingUp',
    'Building', 'BookOpen', 'Zap', 'Settings', 'Crown', 'Rocket'
  ];
  
  const addPrinciple = () => {
    if (newPrinciple.icon.trim() && newPrinciple.title.trim() && newPrinciple.description.trim()) {
      const updatedPrinciples = [...principles, { ...newPrinciple }];
      onChange(JSON.stringify(updatedPrinciples));
      setNewPrinciple({ icon: '', title: '', description: '' });
    }
  };
  
  const removePrinciple = (index: number) => {
    const updatedPrinciples = principles.filter((_: any, i: number) => i !== index);
    onChange(JSON.stringify(updatedPrinciples));
  };
  
  const updatePrinciple = (index: number, field: string, newValue: string) => {
    const updatedPrinciples = [...principles];
    updatedPrinciples[index] = { ...updatedPrinciples[index], [field]: newValue };
    onChange(JSON.stringify(updatedPrinciples));
  };
  
  return (
    <div className="space-y-4">
      {/* Existing principles */}
      {principles.map((principle: any, index: number) => (
        <div key={index} className="p-4 border rounded-lg bg-gray-50 space-y-3">
          <div className="flex items-center justify-between">
            <h4 className="font-medium text-sm">Principle {index + 1}</h4>
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={() => removePrinciple(index)}
              data-testid={`remove-principle-${index}`}
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Icon</label>
              <Select value={principle.icon} onValueChange={(value) => updatePrinciple(index, 'icon', value)}>
                <SelectTrigger className="text-sm" data-testid={`principle-icon-${index}`}>
                  <SelectValue placeholder="Select an icon" />
                </SelectTrigger>
                <SelectContent>
                  {iconOptions.map((iconName) => (
                    <SelectItem key={iconName} value={iconName}>
                      {iconName}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Title</label>
              <Input
                value={principle.title}
                onChange={(e) => updatePrinciple(index, 'title', e.target.value)}
                placeholder="e.g., Integrity & Ethical Conduct"
                className="text-sm"
                data-testid={`principle-title-${index}`}
              />
            </div>
          </div>
          
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Description</label>
            <Textarea
              value={principle.description}
              onChange={(e) => updatePrinciple(index, 'description', e.target.value)}
              placeholder="e.g., We are committed to the highest standards of ethical behavior, transparency, and accountability..."
              rows={3}
              className="text-sm"
              data-testid={`principle-description-${index}`}
            />
          </div>
        </div>
      ))}
      
      {/* Add new principle */}
      <div className="p-4 border-2 border-dashed border-gray-300 rounded-lg space-y-3">
        <h4 className="font-medium text-sm text-gray-700">Add New Principle</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Icon</label>
            <Select value={newPrinciple.icon} onValueChange={(value) => setNewPrinciple({ ...newPrinciple, icon: value })}>
              <SelectTrigger className="text-sm" data-testid="input-new-principle-icon">
                <SelectValue placeholder="Select an icon" />
              </SelectTrigger>
              <SelectContent>
                {iconOptions.map((iconName) => (
                  <SelectItem key={iconName} value={iconName}>
                    {iconName}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Title</label>
            <Input
              value={newPrinciple.title}
              onChange={(e) => setNewPrinciple({ ...newPrinciple, title: e.target.value })}
              placeholder="Integrity & Ethical Conduct"
              className="text-sm"
              data-testid="input-new-principle-title"
            />
          </div>
        </div>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Description</label>
          <Textarea
            value={newPrinciple.description}
            onChange={(e) => setNewPrinciple({ ...newPrinciple, description: e.target.value })}
            placeholder="We are committed to the highest standards of ethical behavior, transparency, and accountability, ensuring that our actions reflect the best educational and business practices and stakeholder interests."
            rows={3}
            className="text-sm"
            data-testid="input-new-principle-description"
          />
        </div>
        
        <Button
          type="button"
          onClick={addPrinciple}
          disabled={!newPrinciple.icon.trim() || !newPrinciple.title.trim() || !newPrinciple.description.trim()}
          className="w-full"
          data-testid="button-add-principle"
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Principle
        </Button>
      </div>
      
      {principles.length === 0 && (
        <p className="text-sm text-gray-500 italic">No principles added yet. Add your first principle above.</p>
      )}
    </div>
  );
}

// Content items management component for dynamic CEO message content
function ContentItemsEditor({ value, onChange }: { value?: string; onChange: (value: string) => void }) {
  // Parse existing content items from JSON string
  const contentItems = (() => {
    try {
      return value ? JSON.parse(value) : [];
    } catch {
      return [];
    }
  })();
  
  const [newContent, setNewContent] = useState('');
  
  const addContent = () => {
    if (newContent.trim()) {
      const updatedItems = [...contentItems, { content: newContent.trim() }];
      onChange(JSON.stringify(updatedItems));
      setNewContent('');
    }
  };
  
  const removeContent = (index: number) => {
    const updatedItems = contentItems.filter((_: any, i: number) => i !== index);
    onChange(JSON.stringify(updatedItems));
  };
  
  const updateContent = (index: number, newValue: string) => {
    const updatedItems = [...contentItems];
    updatedItems[index] = { content: newValue };
    onChange(JSON.stringify(updatedItems));
  };
  
  return (
    <div className="space-y-4">
      {/* Existing content items */}
      {contentItems.map((item: any, index: number) => (
        <div key={index} className="flex gap-2 items-start">
          <Textarea
            value={item.content || ''}
            onChange={(e) => updateContent(index, e.target.value)}
            placeholder="Enter content paragraph..."
            rows={3}
            className="flex-1"
            data-testid={`input-content-item-${index}`}
          />
          <Button
            type="button"
            variant="destructive"
            size="icon"
            onClick={() => removeContent(index)}
            data-testid={`button-remove-content-${index}`}
          >
            <X className="h-4 w-4" />
          </Button>
        </div>
      ))}
      
      {/* Add new content item */}
      <div className="flex gap-2 items-start">
        <Textarea
          value={newContent}
          onChange={(e) => setNewContent(e.target.value)}
          placeholder="Enter new content paragraph..."
          rows={3}
          className="flex-1"
          data-testid="input-new-content"
        />
        <Button
          type="button"
          onClick={addContent}
          data-testid="button-add-content"
        >
          <Plus className="h-4 w-4 mr-1" />
          Add
        </Button>
      </div>
      
      <p className="text-sm text-gray-500">
        Add multiple content paragraphs for the CEO message. Each paragraph will be displayed separately.
      </p>
    </div>
  );
}

// Trustee profiles management component
function TrusteeProfilesEditor({ 
  value, 
  onChange, 
  onFilesChange 
}: { 
  value?: string; 
  onChange: (value: string) => void;
  onFilesChange?: (files: Record<string, File>) => void;
}) {
  // Parse existing trustee profiles from JSON string
  const [trustees, setTrustees] = useState(() => {
    try {
      return value ? JSON.parse(value) : [];
    } catch {
      return [];
    }
  });
  
  const [newTrustee, setNewTrustee] = useState({ 
    name: '', 
    title: '', 
    description: '', 
    image: '',
    linkedin: '',
    twitter: ''
  });
  
  // Store files separately to avoid JSON serialization issues
  const [trusteeFiles, setTrusteeFiles] = useState<Record<string, File>>({});
  
  // Sync local state when value prop changes
  useEffect(() => {
    try {
      const parsedTrustees = value ? JSON.parse(value) : [];
      setTrustees(parsedTrustees);
    } catch {
      setTrustees([]);
    }
  }, [value]);

  // Notify parent of file changes
  useEffect(() => {
    if (onFilesChange) {
      onFilesChange(trusteeFiles);
    }
  }, [trusteeFiles, onFilesChange]);

  // Check if new trustee has pending file upload
  const hasNewTrusteePendingUpload = () => {
    return !!trusteeFiles['new_trustee'];
  };
  
  // Generate file key for trustees
  const getTrusteeFileKey = (index: number) => `trustee_${index}`;
  
  const addTrustee = () => {
    if (newTrustee.name.trim() && newTrustee.title.trim() && newTrustee.description.trim()) {
      let trusteeToAdd = { ...newTrustee };
      
      // Handle file if selected
      if (trusteeFiles['new_trustee']) {
        const fileKey = getTrusteeFileKey(trustees.length);
        setTrusteeFiles(prev => {
          const updated = { ...prev, [fileKey]: prev['new_trustee'] };
          delete updated['new_trustee'];
          return updated;
        });
        trusteeToAdd.image = fileKey;
      }
      
      const updatedTrustees = [...trustees, trusteeToAdd];
      setTrustees(updatedTrustees);
      onChange(JSON.stringify(updatedTrustees));
      setNewTrustee({ name: '', title: '', description: '', image: '', linkedin: '', twitter: '' });
    }
  };
  
  const removeTrustee = (index: number) => {
    const updatedTrustees = trustees.filter((_: any, i: number) => i !== index);
    setTrustees(updatedTrustees);
    onChange(JSON.stringify(updatedTrustees));
  };
  
  const updateTrustee = (index: number, field: string, newValue: string | FileList | null) => {
    if (field === 'image' && newValue && (newValue as FileList).length > 0) {
      // Store file separately and use placeholder in JSON
      const file = (newValue as FileList)[0];
      const fileKey = getTrusteeFileKey(index);
      setTrusteeFiles(prev => ({ ...prev, [fileKey]: file }));
      
      const updatedTrustees = [...trustees];
      updatedTrustees[index] = { ...updatedTrustees[index], [field]: fileKey };
      setTrustees(updatedTrustees);
      onChange(JSON.stringify(updatedTrustees));
    } else {
      const updatedTrustees = [...trustees];
      updatedTrustees[index] = { ...updatedTrustees[index], [field]: newValue };
      setTrustees(updatedTrustees);
      onChange(JSON.stringify(updatedTrustees));
    }
  };
  
  return (
    <div className="space-y-4">
      {/* Existing trustees */}
      {trustees.map((trustee: any, index: number) => (
        <div key={index} className="p-4 border rounded-lg bg-gray-50 space-y-3">
          <div className="flex items-center justify-between">
            <h4 className="font-medium text-sm">Trustee {index + 1}</h4>
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={() => removeTrustee(index)}
              data-testid={`remove-trustee-${index}`}
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Name</label>
              <Input
                value={trustee.name}
                onChange={(e) => updateTrustee(index, 'name', e.target.value)}
                placeholder="Dr. Ahmed M. Darwish"
                className="text-sm"
                data-testid={`trustee-name-${index}`}
              />
            </div>
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Title</label>
              <Input
                value={trustee.title}
                onChange={(e) => updateTrustee(index, 'title', e.target.value)}
                placeholder="Chairman, Board of Trustees"
                className="text-sm"
                data-testid={`trustee-title-${index}`}
              />
            </div>
          </div>
          
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Description</label>
            <Textarea
              value={trustee.description}
              onChange={(e) => updateTrustee(index, 'description', e.target.value)}
              placeholder="Computer Engineering Ph.D. (UC Davis, 1988), Egypt's National Engineering Award (1999), Singapore's Middle East Eminent Persons Award (2007)"
              rows={3}
              className="text-sm"
              data-testid={`trustee-description-${index}`}
            />
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Image Upload</label>
              <Input
                type="file"
                accept="image/*"
                onChange={(e) => updateTrustee(index, 'image', e.target.files)}
                className="text-sm"
                data-testid={`trustee-image-${index}`}
              />
              {trusteeFiles[getTrusteeFileKey(index)] && (
                <p className="text-xs text-blue-500 mt-1">File selected: {trusteeFiles[getTrusteeFileKey(index)]?.name}</p>
              )}
              {trustee.image && typeof trustee.image === 'string' && !trusteeFiles[getTrusteeFileKey(index)] && (
                <p className="text-xs text-gray-500 mt-1">Current: {trustee.image}</p>
              )}
            </div>
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">LinkedIn URL</label>
              <Input
                value={trustee.linkedin || ''}
                onChange={(e) => updateTrustee(index, 'linkedin', e.target.value)}
                placeholder="https://linkedin.com/in/username"
                className="text-sm"
                data-testid={`trustee-linkedin-${index}`}
              />
            </div>
            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Twitter URL</label>
              <Input
                value={trustee.twitter || ''}
                onChange={(e) => updateTrustee(index, 'twitter', e.target.value)}
                placeholder="https://twitter.com/username"
                className="text-sm"
                data-testid={`trustee-twitter-${index}`}
              />
            </div>
          </div>
        </div>
      ))}
      
      {/* Add new trustee */}
      <div className="p-4 border-2 border-dashed border-gray-300 rounded-lg space-y-3">
        <h4 className="font-medium text-sm text-gray-700">Add New Trustee</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Name</label>
            <Input
              value={newTrustee.name}
              onChange={(e) => setNewTrustee({ ...newTrustee, name: e.target.value })}
              placeholder="Dr. Ahmed M. Darwish"
              className="text-sm"
              data-testid="input-new-trustee-name"
            />
          </div>
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Title</label>
            <Input
              value={newTrustee.title}
              onChange={(e) => setNewTrustee({ ...newTrustee, title: e.target.value })}
              placeholder="Chairman, Board of Trustees"
              className="text-sm"
              data-testid="input-new-trustee-title"
            />
          </div>
        </div>
        
        <div>
          <label className="text-xs font-medium text-gray-700 mb-1 block">Description</label>
          <Textarea
            value={newTrustee.description}
            onChange={(e) => setNewTrustee({ ...newTrustee, description: e.target.value })}
            placeholder="Computer Engineering Ph.D. (UC Davis, 1988), Egypt's National Engineering Award (1999), Singapore's Middle East Eminent Persons Award (2007)"
            rows={3}
            className="text-sm"
            data-testid="input-new-trustee-description"
          />
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Image Upload</label>
            <Input
              type="file"
              accept="image/*"
              onChange={(e) => {
                if (e.target.files && e.target.files.length > 0) {
                  setTrusteeFiles(prev => ({ ...prev, 'new_trustee': e.target.files![0] }));
                } else {
                  setTrusteeFiles(prev => {
                    const updated = { ...prev };
                    delete updated['new_trustee'];
                    return updated;
                  });
                }
              }}
              className="text-sm"
              data-testid="input-new-trustee-image"
            />
            {trusteeFiles['new_trustee'] && (
              <p className="text-xs text-blue-500 mt-1">File selected: {trusteeFiles['new_trustee']?.name}</p>
            )}
          </div>
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">LinkedIn URL</label>
            <Input
              value={newTrustee.linkedin}
              onChange={(e) => setNewTrustee({ ...newTrustee, linkedin: e.target.value })}
              placeholder="https://linkedin.com/in/username"
              className="text-sm"
              data-testid="input-new-trustee-linkedin"
            />
          </div>
          <div>
            <label className="text-xs font-medium text-gray-700 mb-1 block">Twitter URL</label>
            <Input
              value={newTrustee.twitter}
              onChange={(e) => setNewTrustee({ ...newTrustee, twitter: e.target.value })}
              placeholder="https://twitter.com/username"
              className="text-sm"
              data-testid="input-new-trustee-twitter"
            />
          </div>
        </div>
        
        <Button
          type="button"
          onClick={addTrustee}
          disabled={!newTrustee.name.trim() || !newTrustee.title.trim() || !newTrustee.description.trim()}
          className="w-full"
          data-testid="button-add-trustee"
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Trustee
        </Button>
      </div>
      
      {trustees.length === 0 && (
        <p className="text-sm text-gray-500 italic">No trustees added yet. Add your first trustee above.</p>
      )}
    </div>
  );
}

// Sortable section item component for drag and drop
function SortableItem({ 
  section, 
  onEdit, 
  onDelete, 
  isUpdatePending, 
  isDeletePending 
}: { 
  section: any; 
  onEdit: () => void; 
  onDelete: () => void; 
  isUpdatePending: boolean; 
  isDeletePending: boolean; 
}) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: section.id });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
  };

  // Get template name if title is empty
  const displayText = section.title || 
    (section.templateType && TEMPLATE_CONFIG[section.templateType as keyof typeof TEMPLATE_CONFIG]?.name) ||
    'Untitled Section';

  return (
    <div
      ref={setNodeRef}
      style={style}
      className={`flex items-center justify-between p-4 border border-gray-200 rounded-lg bg-white ${
        isDragging ? 'shadow-lg opacity-50' : ''
      }`}
      data-testid={`section-item-${section.id}`}
    >
      <div className="flex items-center gap-3">
        <div
          {...attributes}
          {...listeners}
          className="cursor-grab active:cursor-grabbing text-gray-400 hover:text-gray-600"
          data-testid={`drag-handle-${section.id}`}
        >
          <GripVertical className="h-5 w-5" />
        </div>
        <span className={`font-medium ${!section.title ? 'text-gray-500 italic' : ''}`}>
          {displayText}
        </span>
      </div>
      <div className="flex space-x-2">
        <Button 
          variant="outline" 
          size="sm" 
          data-testid={`button-edit-${section.id}`}
          onClick={onEdit}
          disabled={isUpdatePending}
        >
          {isUpdatePending ? 'Loading...' : 'Edit'}
        </Button>
        <Button 
          variant="outline" 
          size="sm" 
          data-testid={`button-delete-${section.id}`}
          onClick={onDelete}
          disabled={isDeletePending}
        >
          {isDeletePending ? 'Deleting...' : 'Delete'}
        </Button>
      </div>
    </div>
  );
}

export default function CMS() {
  const [isOurStoryDialogOpen, setIsOurStoryDialogOpen] = useState(false);
  const [isLeadershipGovernanceDialogOpen, setIsLeadershipGovernanceDialogOpen] = useState(false);
  const [isOurImpactDialogOpen, setIsOurImpactDialogOpen] = useState(false);
  const [isKnowledgeFrameworkDialogOpen, setIsKnowledgeFrameworkDialogOpen] = useState(false);
  const [isPartnersInExcellenceDialogOpen, setIsPartnersInExcellenceDialogOpen] = useState(false);
  const [isIBDLEssentialDialogOpen, setIsIBDLEssentialDialogOpen] = useState(false);
  const [isIBDLProfessionalDialogOpen, setIsIBDLProfessionalDialogOpen] = useState(false);
  const [isIBDLMasterDialogOpen, setIsIBDLMasterDialogOpen] = useState(false);
  const [isCertifiedEducatorDialogOpen, setIsCertifiedEducatorDialogOpen] = useState(false);
  const [isCertifiedIndEngineerDialogOpen, setIsCertifiedIndEngineerDialogOpen] = useState(false);
  const [isCNPOPractitionerDialogOpen, setIsCNPOPractitionerDialogOpen] = useState(false);
  const [isCNPOManagementDialogOpen, setIsCNPOManagementDialogOpen] = useState(false);
  const [isForesightAssociateDialogOpen, setIsForesightAssociateDialogOpen] = useState(false);
  const [isForesightPractitionerDialogOpen, setIsForesightPractitionerDialogOpen] = useState(false);
  const [isForesightStrategistDialogOpen, setIsForesightStrategistDialogOpen] = useState(false);
  const [isWinVsWarDialogOpen, setIsWinVsWarDialogOpen] = useState(false);
  const [isMasterBoardGameDialogOpen, setIsMasterBoardGameDialogOpen] = useState(false);
  const [isSpartaDialogOpen, setIsSpartaDialogOpen] = useState(false);
  const [isMogulCEODialogOpen, setIsMogulCEODialogOpen] = useState(false);
  const [isMavenMarketingDialogOpen, setIsMavenMarketingDialogOpen] = useState(false);
  const [isMicromaticDialogOpen, setIsMicromaticDialogOpen] = useState(false);
  const [isPitchDeckDialogOpen, setIsPitchDeckDialogOpen] = useState(false);
  const [isSynergyStackDialogOpen, setIsSynergyStackDialogOpen] = useState(false);
  const [isCertDiscEntDialogOpen, setIsCertDiscEntDialogOpen] = useState(false);
  const [isCertIncubMgmtDialogOpen, setIsCertIncubMgmtDialogOpen] = useState(false);
  const [isAngelPlaybookDialogOpen, setIsAngelPlaybookDialogOpen] = useState(false);
  const [isCapitalRaiseDialogOpen, setIsCapitalRaiseDialogOpen] = useState(false);
  const [isDataRoomDialogOpen, setIsDataRoomDialogOpen] = useState(false);
  const [isPQPDialogOpen, setIsPQPDialogOpen] = useState(false);
  const [isCPATDialogOpen, setIsCPATDialogOpen] = useState(false);
  const [isExecutiveEducationDialogOpen, setIsExecutiveEducationDialogOpen] = useState(false);
  const [isTrainingProviderDialogOpen, setIsTrainingProviderDialogOpen] = useState(false);
  const [isExamProviderDialogOpen, setIsExamProviderDialogOpen] = useState(false);
  const [isCorpGovDialogOpen, setIsCorpGovDialogOpen] = useState(false);
  const [isProfTrainerDialogOpen, setIsProfTrainerDialogOpen] = useState(false);
  const [isProfCourseDialogOpen, setIsProfCourseDialogOpen] = useState(false);
  const [isPartnerWithDialogOpen, setIsPartnerWithDialogOpen] = useState(false);
  const [isJoinCommDialogOpen, setIsJoinCommDialogOpen] = useState(false);
  const [isTemplateSelectionOpen, setIsTemplateSelectionOpen] = useState(false);
  const [selectedTemplate, setSelectedTemplate] = useState<string | null>(null);
  const [isSectionEditorOpen, setIsSectionEditorOpen] = useState(false);
  const [currentPageId, setCurrentPageId] = useState<string>("gRsEOE7ovOW3_2_cHOqaO"); // Default to Our Story
  const [editingSection, setEditingSection] = useState<any>(null);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [sectionToDelete, setSectionToDelete] = useState<{ id: string; title: string } | null>(null);
  const { toast } = useToast();
  const { user } = useAuth();

  // Drag and drop sensors
  const sensors = useSensors(
    useSensor(PointerSensor),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    })
  );

  // State for trustee files (separate from form to avoid serialization issues)
  const [trusteeFiles, setTrusteeFiles] = useState<Record<string, File>>({});

  // Empty section values to ensure clean form resets
  const EMPTY_SECTION_VALUES = {
    title: "",
    content: "",
    subtitle: "",
    cta_text: "",
    cta_link: "",
    gallery_layout: "grid" as const,
    image_url: "",
    video_url: "",
    stats: [],
    achievements: [],
    certifications: [],
    partnerships: [],
    testimonials: [],
    team_members: [],
    case_studies: [],
    leadership_profiles: [],
    governance_structure: "",
    compliance_info: "",
    mission_statement: "",
    vision_statement: "",
    core_values: [],
    strategic_goals: [],
    impact_metrics: [],
    success_stories: [],
    community_programs: [],
    social_responsibility: [],
    environmental_initiatives: [],
    diversity_inclusion: [],
    quality_assurance: [],
    accreditation_info: [],
    research_publications: [],
    innovation_projects: [],
    technology_stack: [],
    methodology: "",
    frameworks: [],
    competency_models: [],
    assessment_criteria: [],
    learning_outcomes: [],
    content_paragraphs: [],
  };

  // Form for section editor
  const sectionForm = useForm<CreateSectionData>({
    resolver: zodResolver(createSectionSchema),
    defaultValues: EMPTY_SECTION_VALUES,
  });

  // Logout function
  const handleLogout = async () => {
    try {
      const response = await fetch('backend/api/logout', { 
        method: 'POST',
        credentials: 'include'
      });
      
      if (response.ok) {
        toast({
          title: "Logged out successfully",
          description: "You have been logged out.",
        });
      }
      
      // Always redirect regardless of response
      window.location.href = "/admin";
    } catch (error) {
      console.error("Logout error:", error);
      // Force redirect even on error
      window.location.href = "/admin";
    }
  };

  // Query for Our Story page sections
  const { data: ourStorySections = [], isLoading: sectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "gRsEOE7ovOW3_2_cHOqaO", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/gRsEOE7ovOW3_2_cHOqaO/sections", "GET"),
  });

  // Query for Leadership & Governance page sections
  const { data: leadershipGovernanceSections = [], isLoading: leadershipSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "LGov_abc123xyz456789", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/LGov_abc123xyz456789/sections", "GET"),
  });

  // Query for Our Impact page sections  
  const { data: ourImpactSections = [], isLoading: impactSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "Impact_def456ghi789012", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/Impact_def456ghi789012/sections", "GET"),
  });

  // Query for Knowledge Framework page sections
  const { data: knowledgeFrameworkSections = [], isLoading: knowledgeFrameworkSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "KnowFramework_xyz789abc123", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/KnowFramework_xyz789abc123/sections", "GET"),
  });

  // Query for Partners in Excellence page sections
  const { data: partnersInExcellenceSections = [], isLoading: partnersInExcellenceSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "PartnersExcel_qwe456rty789", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/PartnersExcel_qwe456rty789/sections", "GET"),
  });

  // Query for IBDL Essential page sections
  const { data: ibdlEssentialSections = [], isLoading: ibdlEssentialSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "IBDLEssential_abc123xyz789", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/IBDLEssential_abc123xyz789/sections", "GET"),
  });

  // Query for IBDL Professional page sections
  const { data: ibdlProfessionalSections = [], isLoading: ibdlProfessionalSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "IBDLProfessional_xyz456abc789", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/IBDLProfessional_xyz456abc789/sections", "GET"),
  });

  // Query for IBDL Master page sections
  const { data: ibdlMasterSections = [], isLoading: ibdlMasterSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "IBDLMaster_def789ghi012", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/IBDLMaster_def789ghi012/sections", "GET"),
  });

  // Query for Certified Educator page sections
  const { data: certifiedEducatorSections = [], isLoading: certifiedEducatorSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "CertEducator_xyz123abc456", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/CertEducator_xyz123abc456/sections", "GET"),
  });

  // Query for Certified Industrial Engineer page sections
  const { data: certifiedIndEngineerSections = [], isLoading: certifiedIndEngineerSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "CertIndEng_abc456def789", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/CertIndEng_abc456def789/sections", "GET"),
  });

  // Query for CNPO Practitioner page sections
  const { data: cnpoPractitionerSections = [], isLoading: cnpoPractitionerSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "CNPOPract_def789ghi012", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/CNPOPract_def789ghi012/sections", "GET"),
  });

  // Query for CNPO Management Professional page sections
  const { data: cnpoManagementSections = [], isLoading: cnpoManagementSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "CNPOMgmt_ghi012jkl345", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/CNPOMgmt_ghi012jkl345/sections", "GET"),
  });

  // Query for Foresight & Innovation Associate page sections
  const { data: foresightAssociateSections = [], isLoading: foresightAssociateSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "ForesightAssoc_jkl345mno678", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/ForesightAssoc_jkl345mno678/sections", "GET"),
  });

  // Query for Foresight & Innovation Practitioner page sections
  const { data: foresightPractitionerSections = [], isLoading: foresightPractitionerSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "ForesightPract_mno678pqr901", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/ForesightPract_mno678pqr901/sections", "GET"),
  });

  // Query for Foresight & Innovation Strategist page sections
  const { data: foresightStrategistSections = [], isLoading: foresightStrategistSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "ForesightStrat_pqr901stu234", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/ForesightStrat_pqr901stu234/sections", "GET"),
  });

  // Query for Win vs. War page sections
  const { data: winVsWarSections = [], isLoading: winVsWarSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "WinVsWar_stu234vwx567", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/WinVsWar_stu234vwx567/sections", "GET"),
  });

  // Query for Master Board Game page sections
  const { data: masterBoardGameSections = [], isLoading: masterBoardGameSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "MasterBoard_vwx567yz890", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/MasterBoard_vwx567yz890/sections", "GET"),
  });

  // Query for Sparta page sections
  const { data: spartaSections = [], isLoading: spartaSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "Sparta_yz890abc123", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/Sparta_yz890abc123/sections", "GET"),
  });

  // Query for Mogul CEO page sections
  const { data: mogulCEOSections = [], isLoading: mogulCEOSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "MogulCEO_abc123def456", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/MogulCEO_abc123def456/sections", "GET"),
  });

  // Query for Maven Marketing page sections
  const { data: mavenMarketingSections = [], isLoading: mavenMarketingSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "MavenMkt_def456ghi789", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/MavenMkt_def456ghi789/sections", "GET"),
  });

  // Query for Micromatic page sections
  const { data: micromaticSections = [], isLoading: micromaticSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "Micromatic_ghi789jkl012", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/Micromatic_ghi789jkl012/sections", "GET"),
  });

  // Query for Pitch Deck page sections
  const { data: pitchDeckSections = [], isLoading: pitchDeckSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "PitchDeck_jkl012mno345", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/PitchDeck_jkl012mno345/sections", "GET"),
  });

  // Query for Synergy Stack page sections
  const { data: synergyStackSections = [], isLoading: synergyStackSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "SynergyStack_mno345pqr678", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/SynergyStack_mno345pqr678/sections", "GET"),
  });

  // Query for Certified Disciplined Entrepreneur page sections
  const { data: certDiscEntSections = [], isLoading: certDiscEntSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "CertDiscEnt_pqr678stu901", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/CertDiscEnt_pqr678stu901/sections", "GET"),
  });

  // Query for Certified Incubation Management page sections
  const { data: certIncubMgmtSections = [], isLoading: certIncubMgmtSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "CertIncubMgmt_stu901vwx234", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/CertIncubMgmt_stu901vwx234/sections", "GET"),
  });

  // Query for Angel Playbook Bootcamp page sections
  const { data: angelPlaybookSections = [], isLoading: angelPlaybookSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "AngelPlaybook_vwx234yz567", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/AngelPlaybook_vwx234yz567/sections", "GET"),
  });

  // Query for Capital Raise Bootcamp page sections
  const { data: capitalRaiseSections = [], isLoading: capitalRaiseSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "CapitalRaise_yz567abc890", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/CapitalRaise_yz567abc890/sections", "GET"),
  });

  // Query for Data Room Preparation Bootcamp page sections
  const { data: dataRoomSections = [], isLoading: dataRoomSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "DataRoom_abc890def123", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/DataRoom_abc890def123/sections", "GET"),
  });

  // Query for PQP page sections
  const { data: pqpSections = [], isLoading: pqpSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "PQP_def123ghi456", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/PQP_def123ghi456/sections", "GET"),
  });

  // Query for CPAT page sections
  const { data: cpatSections = [], isLoading: cpatSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "CPAT_ghi456jkl789", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/CPAT_ghi456jkl789/sections", "GET"),
  });

  // Query for Executive Education page sections
  const { data: executiveEducationSections = [], isLoading: executiveEducationSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "ExecEd_jkl789mno012", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/ExecEd_jkl789mno012/sections", "GET"),
  });

  // Query for Training Provider page sections
  const { data: trainingProviderSections = [], isLoading: trainingProviderSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "TrainingProv_mno012pqr345", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/TrainingProv_mno012pqr345/sections", "GET"),
  });

  // Query for Exam Provider page sections
  const { data: examProviderSections = [], isLoading: examProviderSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "ExamProv_pqr345stu678", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/ExamProv_pqr345stu678/sections", "GET"),
  });

  // Query for Corporate/Government page sections
  const { data: corpGovSections = [], isLoading: corpGovSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "CorpGov_stu678vwx901", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/CorpGov_stu678vwx901/sections", "GET"),
  });

  // Query for Professional IBDL Trainer page sections
  const { data: profTrainerSections = [], isLoading: profTrainerSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "ProfTrainer_vwx901yz234abc", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/ProfTrainer_vwx901yz234abc/sections", "GET"),
  });

  // Query for Professional Course page sections
  const { data: profCourseSections = [], isLoading: profCourseSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "ProfCourse_yz234abc567def", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/ProfCourse_yz234abc567def/sections", "GET"),
  });

  // Query for Partner With Us page sections
  const { data: partnerWithSections = [], isLoading: partnerWithSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "PartnerWith_abc567def890ghi", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/PartnerWith_abc567def890ghi/sections", "GET"),
  });

  // Query for Join Our Global Community page sections
  const { data: joinCommSections = [], isLoading: joinCommSectionsLoading } = useQuery({
    queryKey: ["backend/api/cms/pages", "JoinComm_def890ghi123jkl", "sections"],
    queryFn: () => apiRequest("backend/api/cms/pages/JoinComm_def890ghi123jkl/sections", "GET"),
  });

  // QueryClient for cache invalidation
  const queryClient = useQueryClient();

  // Create section mutation
  const createSectionMutation = useMutation({
    mutationFn: async (data: CreateSectionData) => {
      // Dynamic data transformation based on template type
      const templateFields = selectedTemplate && TEMPLATE_FIELD_MAPPING[selectedTemplate as keyof typeof TEMPLATE_FIELD_MAPPING];
      const dataObject: any = {};
      
      if (templateFields) {
        templateFields.fields.forEach((fieldName: string) => {
          switch (fieldName) {
            case 'subtitle':
              if (data.subtitle) dataObject.subtitle = data.subtitle;
              break;
            case 'cta_text':
              if (data.cta_text) dataObject.cta_text = data.cta_text;
              break;
            case 'cta_link':
              if (data.cta_link) dataObject.cta_link = data.cta_link;
              break;
            case 'card_title':
              if (data.card_title) dataObject.card_title = data.card_title;
              break;
            case 'card_subtitle':
              if (data.card_subtitle) dataObject.card_subtitle = data.card_subtitle;
              break;
            case 'card_items':
              if (data.card_items) {
                // Convert comma-separated string to array
                dataObject.card_items = data.card_items.split(',').map(item => item.trim()).filter(item => item);
              }
              break;
            case 'mission':
              if (data.mission) dataObject.mission = data.mission;
              break;
            case 'vision':
              if (data.vision) dataObject.vision = data.vision;
              break;
            case 'text_color':
              if (data.text_color) dataObject.textColor = data.text_color; // Use database field name
              break;
            case 'background_color':
              if (data.background_color) dataObject.backgroundColor = data.background_color; // Use database field name
              break;
            case 'gradient':
              if (data.gradient) dataObject.gradient = data.gradient;
              break;
            case 'gallery_layout':
              if (data.gallery_layout) dataObject.gallery_layout = data.gallery_layout;
              break;
            case 'quote':
              if (data.quote) dataObject.quote = data.quote;
              break;
            case 'content_items':
              if (data.content_items) {
                try {
                  // Try to parse as JSON first (new format for Narrative Summary)
                  dataObject.content_items = JSON.parse(data.content_items);
                } catch {
                  // Fall back to splitting by double line breaks for backward compatibility
                  const paragraphs = data.content_items.split('\n\n').map(p => p.trim()).filter(p => p);
                  dataObject.content_items = paragraphs.map(content => ({ content }));
                }
              }
              break;
            case 'card_content':
              if (data.card_content) {
                try {
                  // Parse the combined content with card, bullet points, and final content data
                  const cardData = JSON.parse(data.card_content);
                  if (cardData.content_items) dataObject.content_items = cardData.content_items;
                  if (cardData.card_content) dataObject.card_content = cardData.card_content;
                  if (cardData.card_position !== undefined) dataObject.card_position = cardData.card_position;
                  if (cardData.bullet_points) dataObject.bullet_points = cardData.bullet_points;
                  if (cardData.bullet_points_position !== undefined) dataObject.bullet_points_position = cardData.bullet_points_position;
                  if (cardData.final_content) dataObject.final_content = cardData.final_content;
                } catch {
                  // Handle as simple string if not JSON
                  dataObject.card_content = data.card_content;
                }
              }
              break;
            case 'card_position':
              // This is handled within card_content processing
              break;
            case 'ceo_name':
              if (data.ceo_name) dataObject.ceo_name = data.ceo_name;
              break;
            case 'ceo_title':
              if (data.ceo_title) dataObject.ceo_title = data.ceo_title;
              break;
            case 'main_icon':
              if (data.main_icon) dataObject.main_icon = data.main_icon;
              break;
            case 'principle_items':
              if (data.principle_items) {
                try {
                  // Try to parse as JSON first (new format)
                  dataObject.principle_items = JSON.parse(data.principle_items);
                } catch {
                  // Fall back to comma-separated format for backward compatibility
                  dataObject.principle_items = data.principle_items.split(',').map(item => item.trim()).filter(item => item);
                }
              }
              break;
            case 'governance_structure':
              if (data.governance_structure) dataObject.governance_structure = data.governance_structure;
              break;
            case 'numbered_cards':
              if (data.numbered_cards) {
                try {
                  // Parse numbered cards as JSON array
                  dataObject.numbered_cards = JSON.parse(data.numbered_cards);
                } catch {
                  // Handle as empty array if parsing fails
                  dataObject.numbered_cards = [];
                }
              }
              break;
            case 'trustee_profiles':
              if (data.trustee_profiles) dataObject.trustee_profiles = data.trustee_profiles;
              break;
            case 'table_title':
              if (data.table_title) dataObject.table_title = data.table_title;
              break;
            case 'table_icon':
              if (data.table_icon) dataObject.table_icon = data.table_icon;
              break;
            case 'metrics_data':
              if (data.metrics_data) dataObject.metrics_data = data.metrics_data;
              break;
            case 'matter_items':
              if (data.matter_items) {
                try {
                  // Parse matter items as JSON array
                  dataObject.matter_items = JSON.parse(data.matter_items);
                } catch {
                  // Handle as empty array if parsing fails
                  dataObject.matter_items = [];
                }
              }
              break;
            case 'image':
              if (data.image) dataObject.image = 'uploaded'; // File placeholder
              break;
            case 'background_image':
              if (data.background_image) dataObject.background_image = 'uploaded'; // File placeholder
              break;
            case 'images':
              if (data.images) dataObject.images = 'uploaded'; // File placeholder
              break;
            case 'ceo_image':
              if (data.ceo_image) dataObject.ceo_image = 'uploaded'; // File placeholder
              break;
            default:
              break;
          }
        });
      }

      // Determine which sections array to use based on current page
      const getCurrentPageSections = () => {
        switch (currentPageId) {
          case "gRsEOE7ovOW3_2_cHOqaO": return ourStorySections;
          case "LGov_abc123xyz456789": return leadershipGovernanceSections;
          case "Impact_def456ghi789012": return ourImpactSections;
          case "KnowFramework_xyz789abc123": return knowledgeFrameworkSections;
          case "PartnersExcel_qwe456rty789": return partnersInExcellenceSections;
          case "IBDLEssential_abc123xyz789": return ibdlEssentialSections;
          case "IBDLProfessional_xyz456abc789": return ibdlProfessionalSections;
          case "IBDLMaster_def789ghi012": return ibdlMasterSections;
          case "CertEducator_xyz123abc456": return certifiedEducatorSections;
          case "CertIndEng_abc456def789": return certifiedIndEngineerSections;
          case "CNPOPract_def789ghi012": return cnpoPractitionerSections;
          case "CNPOMgmt_ghi012jkl345": return cnpoManagementSections;
          case "ForesightAssoc_jkl345mno678": return foresightAssociateSections;
          case "ForesightPract_mno678pqr901": return foresightPractitionerSections;
          case "ForesightStrat_pqr901stu234": return foresightStrategistSections;
          case "WinVsWar_stu234vwx567": return winVsWarSections;
          case "MasterBoard_vwx567yz890": return masterBoardGameSections;
          case "Sparta_yz890abc123": return spartaSections;
          case "MogulCEO_abc123def456": return mogulCEOSections;
          case "MavenMkt_def456ghi789": return mavenMarketingSections;
          case "Micromatic_ghi789jkl012": return micromaticSections;
          case "PitchDeck_jkl012mno345": return pitchDeckSections;
          case "SynergyStack_mno345pqr678": return synergyStackSections;
          case "CertDiscEnt_pqr678stu901": return certDiscEntSections;
          case "CertIncubMgmt_stu901vwx234": return certIncubMgmtSections;
          case "AngelPlaybook_vwx234yz567": return angelPlaybookSections;
          case "CapitalRaise_yz567abc890": return capitalRaiseSections;
          case "DataRoom_abc890def123": return dataRoomSections;
          case "PQP_def123ghi456": return pqpSections;
          case "CPAT_ghi456jkl789": return cpatSections;
          case "ExecEd_jkl789mno012": return executiveEducationSections;
          case "TrainingProv_mno012pqr345": return trainingProviderSections;
          case "ExamProv_pqr345stu678": return examProviderSections;
          case "CorpGov_stu678vwx901": return corpGovSections;
          case "ProfTrainer_vwx901yz234abc": return profTrainerSections;
          case "ProfCourse_yz234abc567def": return profCourseSections;
          case "PartnerWith_abc567def890ghi": return partnerWithSections;
          case "JoinComm_def890ghi123jkl": return joinCommSections;
          default: return [];
        }
      };

      const transformedData = {
        title: data.title || '',
        templateType: selectedTemplate,
        sectionTemplateId: selectedTemplate, // Fix database constraint
        content: data.content || '',
        data: dataObject,
        sortOrder: getCurrentPageSections().length,
        isActive: true,
      };
      
      // Check if we have any file uploads (FileList objects, not strings)
      const hasTrusteeFiles = Object.keys(trusteeFiles).length > 0;
      const hasFiles = (data.image instanceof FileList && data.image.length > 0) || 
                      (data.background_image instanceof FileList && data.background_image.length > 0) ||
                      (data.images instanceof FileList && data.images.length > 0) ||
                      (data.ceo_image instanceof FileList && data.ceo_image.length > 0) ||
                      hasTrusteeFiles;
      
      let response;
      
      if (hasFiles) {
        // Use FormData for file uploads
        const formData = new FormData();
        
        // Add image files (only if they're actual FileList objects)
        if (data.image instanceof FileList && data.image.length > 0) {
          formData.append('image', data.image[0]);
        }
        if (data.background_image instanceof FileList && data.background_image.length > 0) {
          formData.append('background_image', data.background_image[0]);
        }
        if (data.images instanceof FileList && data.images.length > 0) {
          Array.from(data.images).forEach((file) => {
            formData.append('images', file);
          });
        }
        if (data.ceo_image instanceof FileList && data.ceo_image.length > 0) {
          formData.append('ceo_image', data.ceo_image[0]);
        }
        
        // Add trustee images
        if (hasTrusteeFiles) {
          Object.entries(trusteeFiles).forEach(([key, file]) => {
            formData.append(`trustee_image_${key}`, file as File);
          });
        }
        
        // Add other data as JSON string
        formData.append('data', JSON.stringify(transformedData));
        
        response = await fetch(`backend/api/cms/pages/${currentPageId}/sections`, {
          method: 'POST',
          credentials: 'include',
          body: formData,
        });
      } else {
        // Use JSON for text-only sections
        response = await fetch(`backend/api/cms/pages/${currentPageId}/sections`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          credentials: 'include',
          body: JSON.stringify(transformedData),
        });
      }
      
      if (!response.ok) {
        throw new Error('Failed to create section');
      }
      
      return response.json();
    },
    onSuccess: () => {
      // Invalidate and refetch sections
      queryClient.invalidateQueries({
        queryKey: ["backend/api/cms/pages", currentPageId, "sections"]
      });
      toast({
        title: "Section Created",
        description: "Your section has been created successfully.",
      });
      setIsSectionEditorOpen(false);
      setSelectedTemplate(null);
      sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
      setTrusteeFiles({});
    },
    onError: (error) => {
      console.error('Create section error:', error);
      toast({
        title: "Error",
        description: "Failed to create section. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Update section mutation  
  const updateSectionMutation = useMutation({
    mutationFn: async ({ id, data }: { id: string; data: Partial<CreateSectionData> }) => {
      // Dynamic data transformation based on template type (same as create)
      const currentTemplate = editingSection?.templateType || selectedTemplate;
      const templateFields = currentTemplate && TEMPLATE_FIELD_MAPPING[currentTemplate as keyof typeof TEMPLATE_FIELD_MAPPING];
      const dataObject: any = {};
      
      if (templateFields) {
        templateFields.fields.forEach((fieldName: string) => {
          switch (fieldName) {
            case 'subtitle':
              if (data.subtitle !== undefined) dataObject.subtitle = data.subtitle;
              break;
            case 'cta_text':
              if (data.cta_text !== undefined && data.cta_text.trim() !== '') dataObject.cta_text = data.cta_text;
              break;
            case 'cta_link':
              if (data.cta_link !== undefined && data.cta_link.trim() !== '') dataObject.cta_link = data.cta_link;
              break;
            case 'card_title':
              if (data.card_title !== undefined && data.card_title.trim() !== '') dataObject.card_title = data.card_title;
              break;
            case 'card_subtitle':
              if (data.card_subtitle !== undefined && data.card_subtitle.trim() !== '') dataObject.card_subtitle = data.card_subtitle;
              break;
            case 'card_items':
              if (data.card_items !== undefined && data.card_items.trim() !== '') {
                // Convert comma-separated string to array - only if not empty
                dataObject.card_items = data.card_items.split(',').map(item => item.trim()).filter(item => item);
              }
              break;
            case 'mission':
              if (data.mission !== undefined && data.mission.trim() !== '') dataObject.mission = data.mission;
              break;
            case 'vision':
              if (data.vision !== undefined && data.vision.trim() !== '') dataObject.vision = data.vision;
              break;
            case 'mission_label':
              if (data.mission_label !== undefined && data.mission_label.trim() !== '') dataObject.mission_label = data.mission_label;
              else if (data.mission && data.mission.trim() !== '') dataObject.mission_label = 'Our Mission'; // Set default if mission exists but label is missing
              break;
            case 'vision_label':
              if (data.vision_label !== undefined && data.vision_label.trim() !== '') dataObject.vision_label = data.vision_label;
              else if (data.vision && data.vision.trim() !== '') dataObject.vision_label = 'Our Vision'; // Set default if vision exists but label is missing
              break;
            case 'text_color':
              if (data.text_color !== undefined && data.text_color.trim() !== '') dataObject.textColor = data.text_color; // Use database field name
              break;
            case 'background_color':
              if (data.background_color !== undefined && data.background_color.trim() !== '') dataObject.backgroundColor = data.background_color; // Use database field name
              break;
            case 'gradient':
              if (data.gradient !== undefined && data.gradient.trim() !== '') dataObject.gradient = data.gradient;
              break;
            case 'gallery_layout':
              if (data.gallery_layout !== undefined && data.gallery_layout.trim() !== '') dataObject.gallery_layout = data.gallery_layout;
              break;
            case 'quote':
              if (data.quote !== undefined && data.quote.trim() !== '') dataObject.quote = data.quote;
              break;
            case 'content_items':
              if (data.content_items !== undefined && data.content_items.trim() !== '') {
                try {
                  // Try to parse as JSON first (new format for Narrative Summary)
                  dataObject.content_items = JSON.parse(data.content_items);
                } catch {
                  // Fall back to splitting by double line breaks for backward compatibility
                  const paragraphs = data.content_items.split('\n\n').map(p => p.trim()).filter(p => p);
                  dataObject.content_items = paragraphs.map(content => ({ content }));
                }
              }
              break;
            case 'card_content':
              if (data.card_content !== undefined && data.card_content.trim() !== '') {
                try {
                  // Parse the combined content with card, bullet points, and final content data
                  const cardData = JSON.parse(data.card_content);
                  if (cardData.content_items) dataObject.content_items = cardData.content_items;
                  if (cardData.card_content) dataObject.card_content = cardData.card_content;
                  if (cardData.card_position !== undefined) dataObject.card_position = cardData.card_position;
                  if (cardData.bullet_points) dataObject.bullet_points = cardData.bullet_points;
                  if (cardData.bullet_points_position !== undefined) dataObject.bullet_points_position = cardData.bullet_points_position;
                  if (cardData.final_content) dataObject.final_content = cardData.final_content;
                } catch {
                  // Handle as simple string if not JSON
                  dataObject.card_content = data.card_content;
                }
              }
              break;
            case 'card_position':
              // This is handled within card_content processing
              break;
            case 'ceo_name':
              if (data.ceo_name !== undefined && data.ceo_name.trim() !== '') dataObject.ceo_name = data.ceo_name;
              break;
            case 'ceo_title':
              if (data.ceo_title !== undefined && data.ceo_title.trim() !== '') dataObject.ceo_title = data.ceo_title;
              break;
            case 'main_icon':
              if (data.main_icon !== undefined && data.main_icon.trim() !== '') dataObject.main_icon = data.main_icon;
              break;
            case 'principle_items':
              if (data.principle_items !== undefined && data.principle_items.trim() !== '') {
                try {
                  // Try to parse as JSON first (new format)
                  dataObject.principle_items = JSON.parse(data.principle_items);
                } catch {
                  // Fall back to comma-separated format for backward compatibility
                  dataObject.principle_items = data.principle_items.split(',').map(item => item.trim()).filter(item => item);
                }
              }
              break;
            case 'governance_structure':
              if (data.governance_structure !== undefined && data.governance_structure.trim() !== '') dataObject.governance_structure = data.governance_structure;
              break;
            case 'numbered_cards':
              if (data.numbered_cards !== undefined && data.numbered_cards.trim() !== '') {
                try {
                  // Parse numbered cards as JSON array
                  dataObject.numbered_cards = JSON.parse(data.numbered_cards);
                } catch {
                  // Handle as empty array if parsing fails
                  dataObject.numbered_cards = [];
                }
              }
              break;
            case 'trustee_profiles':
              if (data.trustee_profiles !== undefined && data.trustee_profiles.trim() !== '') dataObject.trustee_profiles = data.trustee_profiles;
              break;
            case 'table_title':
              if (data.table_title !== undefined && data.table_title.trim() !== '') dataObject.table_title = data.table_title;
              break;
            case 'table_icon':
              if (data.table_icon !== undefined && data.table_icon.trim() !== '') dataObject.table_icon = data.table_icon;
              break;
            case 'metrics_data':
              if (data.metrics_data !== undefined && data.metrics_data.trim() !== '') dataObject.metrics_data = data.metrics_data;
              break;
            case 'matter_items':
              if (data.matter_items !== undefined && data.matter_items.trim() !== '') {
                try {
                  // Parse matter items as JSON array
                  dataObject.matter_items = JSON.parse(data.matter_items);
                } catch {
                  // Handle as empty array if parsing fails
                  dataObject.matter_items = [];
                }
              }
              break;
            default:
              break;
          }
        });
      }

      const transformedData: any = {
        templateType: currentTemplate,
        sectionTemplateId: currentTemplate, // Fix database constraint
        data: dataObject,
        isActive: true,
      };
      
      // Only include title/content if they are actually provided and not empty
      if (data.title !== undefined && data.title.trim() !== '') {
        transformedData.title = data.title;
      }
      if (data.content !== undefined && data.content.trim() !== '') {
        transformedData.content = data.content;
      }
      
      // Check if we have any file uploads (FileList objects, not strings)
      const hasFiles = data.image instanceof FileList && data.image.length > 0;
      
      let response;
      
      if (hasFiles) {
        // Use FormData for file uploads
        const formData = new FormData();
        
        // Add the image file (only if it's an actual FileList object)
        if (data.image instanceof FileList && data.image.length > 0) {
          formData.append('image', data.image[0]);
        }
        
        // Add other data as JSON string
        formData.append('data', JSON.stringify(transformedData));
        
        response = await fetch(`backend/api/cms/sections/${id}`, {
          method: 'PUT',
          credentials: 'include',
          body: formData,
        });
      } else {
        // Use JSON for text-only updates
        response = await fetch(`backend/api/cms/sections/${id}`, {
          method: 'PUT',
          headers: { 'Content-Type': 'application/json' },
          credentials: 'include',
          body: JSON.stringify(transformedData),
        });
      }
      
      if (!response.ok) {
        throw new Error('Failed to update section');
      }
      
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ["backend/api/cms/pages", currentPageId, "sections"]
      });
      toast({
        title: "Section Updated",
        description: "Your section has been updated successfully.",
      });
      // Close editor and reset state like create does
      setIsSectionEditorOpen(false);
      setSelectedTemplate(null);
      setEditingSection(null);
      sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
      setTrusteeFiles({});
    },
    onError: () => {
      toast({
        title: "Error", 
        description: "Failed to update section. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Delete section mutation
  const deleteSectionMutation = useMutation({
    mutationFn: async (id: string) => {
      const response = await fetch(`backend/api/cms/sections/${id}`, {
        method: 'DELETE',
        credentials: 'include',
      });
      
      if (!response.ok) {
        throw new Error('Failed to delete section');
      }
      
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ["backend/api/cms/pages", currentPageId, "sections"]
      });
      toast({
        title: "Section Deleted",
        description: "The section has been deleted successfully.",
      });
      setDeleteConfirmOpen(false);
      setSectionToDelete(null);
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to delete section. Please try again.", 
        variant: "destructive",
      });
      setDeleteConfirmOpen(false);
      setSectionToDelete(null);
    },
  });

  // Reorder sections mutation
  const reorderSectionsMutation = useMutation({
    mutationFn: async (reorderedSections: any[]) => {
      const response = await fetch(`backend/api/cms/pages/${currentPageId}/sections/reorder`, {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({ sections: reorderedSections }),
      });
      
      if (!response.ok) {
        throw new Error('Failed to reorder sections');
      }
      
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ["backend/api/cms/pages", currentPageId, "sections"]
      });
      toast({
        title: "Sections Reordered",
        description: "Section order has been updated successfully.",
      });
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to reorder sections. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Handle drag end for reordering
  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;

    if (active.id !== over?.id) {
      const oldIndex = ourStorySections.findIndex((section: any) => section.id === active.id);
      const newIndex = ourStorySections.findIndex((section: any) => section.id === over?.id);

      const reorderedSections = arrayMove(ourStorySections, oldIndex, newIndex);
      
      // Update sort order and send to backend
      const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
        id: section.id,
        sortOrder: index
      }));
      
      reorderSectionsMutation.mutate(sectionsWithNewOrder);
    }
  };

  // Handle delete confirmation
  const handleConfirmDelete = () => {
    if (sectionToDelete) {
      deleteSectionMutation.mutate(sectionToDelete.id);
    }
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center py-6">
            <div className="flex items-center">
              <img src={ibdlLogo} alt="IBDL" className="h-8 w-auto mr-4" />
              <div>
                <h1 className="text-2xl font-bold text-gray-900">Content Management System</h1>
                <p className="text-gray-600">Manage your website content and settings</p>
              </div>
            </div>
            <div className="flex items-center space-x-4">
              <Button variant="outline" size="sm">
                <Globe className="h-4 w-4 mr-2" />
                View Website
              </Button>
              <span className="text-gray-600">Welcome, {(user as any)?.firstName || 'Website Admin'}</span>
              <Button variant="outline" onClick={handleLogout}>
                <LogOut className="h-4 w-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </div>

      {/* Content Management */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="flex justify-between items-center mb-8">
          <div>
            <h2 className="text-xl font-semibold text-gray-900">Content Management</h2>
            <p className="text-gray-600">Manage your website content, pages, and settings</p>
          </div>
          <div className="flex items-center space-x-4">
            <Button variant="outline" size="sm">
              <BarChart3 className="h-4 w-4 mr-2" />
              Analytics
            </Button>
            <Button variant="outline" size="sm">
              <Settings className="h-4 w-4 mr-2" />
              Settings
            </Button>
          </div>
        </div>

        {/* Page Management Cards */}
        {/* Grouped Content Management Cards */}
        <div className="space-y-12">
          {/* Who We Are Section */}
          <div className="space-y-6">
            {/* Section Header */}
            <div className="flex items-center space-x-4">
              <div className="w-12 h-12 bg-gradient-to-br from-[#ED1C24] to-[#ED1C24]/80 rounded-xl flex items-center justify-center shadow-md">
                <Users className="h-6 w-6 text-white" />
              </div>
              <div>
                <h2 className="text-2xl font-bold text-[#1A1A40] dark:text-white">Who We Are</h2>
                <p className="text-gray-600 dark:text-gray-400">Manage pages about our organization, leadership, and impact</p>
              </div>
            </div>
            
            {/* Section Cards */}
            <div className="ml-8 border-l-2 border-gray-200 dark:border-gray-700 pl-8">
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {/* Our Story Card */}
                <div className="max-w-md">
                  <Card 
                    className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-red-500 h-32"
                    onClick={() => {
                      setCurrentPageId("gRsEOE7ovOW3_2_cHOqaO");
                      setIsOurStoryDialogOpen(true);
                    }}
                    data-testid="card-our-story"
                  >
                    <CardHeader>
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-3">
                          <div className="w-10 h-10 bg-red-50 rounded-lg flex items-center justify-center">
                            <FileText className="h-5 w-5 text-red-600" />
                          </div>
                          <div>
                            <CardTitle className="text-lg">Our Story</CardTitle>
                            <CardDescription>{sectionsLoading ? 'Loading...' : `${ourStorySections.length} sections`}</CardDescription>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          <a href="/who-we-are/our-story" target="_blank" rel="noopener noreferrer">
                            <Button variant="ghost" size="sm" data-testid="button-preview" className="flex items-center gap-2">
                              <ExternalLink className="h-4 w-4" />
                              Preview
                            </Button>
                          </a>
                        </div>
                      </div>
                    </CardHeader>
                  </Card>
                </div>

                {/* Leadership & Governance Card */}
                <div className="max-w-md">
                  <Card 
                    className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-blue-500 h-32"
                    onClick={() => {
                      setCurrentPageId("LGov_abc123xyz456789");
                      setIsLeadershipGovernanceDialogOpen(true);
                    }}
                    data-testid="card-leadership-governance"
                  >
                    <CardHeader>
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-3">
                          <div className="w-10 h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                            <Shield className="h-5 w-5 text-blue-600" />
                          </div>
                          <div>
                            <CardTitle className="text-lg">Leadership & Governance</CardTitle>
                            <CardDescription>{leadershipSectionsLoading ? 'Loading...' : `${leadershipGovernanceSections.length} sections`}</CardDescription>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          <a href="/who-we-are/leadership-governance" target="_blank" rel="noopener noreferrer">
                            <Button variant="ghost" size="sm" data-testid="button-preview-leadership" className="flex items-center gap-2">
                              <ExternalLink className="h-4 w-4" />
                              Preview
                            </Button>
                          </a>
                        </div>
                      </div>
                    </CardHeader>
                  </Card>
                </div>

                {/* Our Impact Card */}
                <div className="max-w-md">
                  <Card 
                    className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-green-500 h-32"
                    onClick={() => {
                      setCurrentPageId("Impact_def456ghi789012");
                      setIsOurImpactDialogOpen(true);
                    }}
                    data-testid="card-our-impact"
                  >
                    <CardHeader>
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-3">
                          <div className="w-10 h-10 bg-green-50 rounded-lg flex items-center justify-center">
                            <TrendingUp className="h-5 w-5 text-green-600" />
                          </div>
                          <div>
                            <CardTitle className="text-lg">Our Impact</CardTitle>
                            <CardDescription>{impactSectionsLoading ? 'Loading...' : `${ourImpactSections.length} sections`}</CardDescription>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          <a href="/who-we-are/our-impact" target="_blank" rel="noopener noreferrer">
                            <Button variant="ghost" size="sm" data-testid="button-preview-impact" className="flex items-center gap-2">
                              <ExternalLink className="h-4 w-4" />
                              Preview
                            </Button>
                          </a>
                        </div>
                      </div>
                    </CardHeader>
                  </Card>
                </div>

                {/* Knowledge Framework Card */}
                <div className="max-w-md">
                  <Card 
                    className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-purple-500 h-32"
                    onClick={() => {
                      setCurrentPageId("KnowFramework_xyz789abc123");
                      setIsKnowledgeFrameworkDialogOpen(true);
                    }}
                    data-testid="card-knowledge-framework"
                  >
                    <CardHeader>
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-3">
                          <div className="w-10 h-10 bg-purple-50 rounded-lg flex items-center justify-center">
                            <Lightbulb className="h-5 w-5 text-purple-600" />
                          </div>
                          <div>
                            <CardTitle className="text-lg">Knowledge Framework</CardTitle>
                            <CardDescription>{knowledgeFrameworkSectionsLoading ? 'Loading...' : `${knowledgeFrameworkSections.length} sections`}</CardDescription>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          <a href="/who-we-are/knowledge-framework" target="_blank" rel="noopener noreferrer">
                            <Button variant="ghost" size="sm" data-testid="button-preview-knowledge-framework" className="flex items-center gap-2">
                              <ExternalLink className="h-4 w-4" />
                              Preview
                            </Button>
                          </a>
                        </div>
                      </div>
                    </CardHeader>
                  </Card>
                </div>

                {/* Partners in Excellence Card */}
                <div className="max-w-md">
                  <Card 
                    className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-amber-500 h-32"
                    onClick={() => {
                      setCurrentPageId("PartnersExcel_qwe456rty789");
                      setIsPartnersInExcellenceDialogOpen(true);
                    }}
                    data-testid="card-partners-in-excellence"
                  >
                    <CardHeader>
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-3">
                          <div className="w-10 h-10 bg-amber-50 rounded-lg flex items-center justify-center">
                            <Handshake className="h-5 w-5 text-amber-600" />
                          </div>
                          <div>
                            <CardTitle className="text-lg">Partners in Excellence</CardTitle>
                            <CardDescription>{partnersInExcellenceSectionsLoading ? 'Loading...' : `${partnersInExcellenceSections.length} sections`}</CardDescription>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          <a href="/who-we-are/partners-in-excellence" target="_blank" rel="noopener noreferrer">
                            <Button variant="ghost" size="sm" data-testid="button-preview-partners-in-excellence" className="flex items-center gap-2">
                              <ExternalLink className="h-4 w-4" />
                              Preview
                            </Button>
                          </a>
                        </div>
                      </div>
                    </CardHeader>
                  </Card>
                </div>
              </div>
            </div>
          </div>

          {/* Our Products Section */}
          <div className="space-y-6">
            {/* Section Header */}
            <div className="flex items-center space-x-4">
              <div className="w-12 h-12 bg-gradient-to-br from-[#1A1A40] to-[#1A1A40]/80 rounded-xl flex items-center justify-center shadow-md">
                <Award className="h-6 w-6 text-white" />
              </div>
              <div>
                <h2 className="text-2xl font-bold text-[#1A1A40] dark:text-white">Our Products</h2>
                <p className="text-gray-600 dark:text-gray-400">Manage qualifications, certifications, and training programs</p>
              </div>
            </div>
            
            {/* Qualifications Subsection */}
            <div className="ml-8 border-l-2 border-gray-200 dark:border-gray-700 pl-8 space-y-6">
              <div>
                <h3 className="text-lg font-semibold text-gray-700 dark:text-gray-300 mb-4">Qualifications</h3>
                
                {/* Core Qualifications */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Core Qualifications</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* IBDL Essential Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-yellow-500 h-32"
                        onClick={() => {
                          setCurrentPageId("IBDLEssential_abc123xyz789");
                          setIsIBDLEssentialDialogOpen(true);
                        }}
                        data-testid="card-ibdl-essential"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-yellow-50 rounded-lg flex items-center justify-center">
                                <Award className="h-5 w-5 text-yellow-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">IBDL Essential</CardTitle>
                                <CardDescription>{ibdlEssentialSectionsLoading ? 'Loading...' : `${ibdlEssentialSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/core-qualifications/ibdl-essential" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-ibdl-essential" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* IBDL Professional Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-blue-500 h-32"
                        onClick={() => {
                          setCurrentPageId("IBDLProfessional_xyz456abc789");
                          setIsIBDLProfessionalDialogOpen(true);
                        }}
                        data-testid="card-ibdl-professional"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                                <Award className="h-5 w-5 text-blue-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">IBDL Professional</CardTitle>
                                <CardDescription>{ibdlProfessionalSectionsLoading ? 'Loading...' : `${ibdlProfessionalSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/core-qualifications/ibdl-professional" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-ibdl-professional" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* IBDL Master Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-purple-500 h-32"
                        onClick={() => {
                          setCurrentPageId("IBDLMaster_def789ghi012");
                          setIsIBDLMasterDialogOpen(true);
                        }}
                        data-testid="card-ibdl-master"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-purple-50 rounded-lg flex items-center justify-center">
                                <Award className="h-5 w-5 text-purple-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">IBDL Master</CardTitle>
                                <CardDescription>{ibdlMasterSectionsLoading ? 'Loading...' : `${ibdlMasterSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/core-qualifications/ibdl-master" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-ibdl-master" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>
                
                {/* Specialized Qualifications */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6 mt-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Specialized Qualifications</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* Certified Educator Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-green-500 h-32"
                        onClick={() => {
                          setCurrentPageId("CertEducator_xyz123abc456");
                          setIsCertifiedEducatorDialogOpen(true);
                        }}
                        data-testid="card-certified-educator"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-green-50 rounded-lg flex items-center justify-center">
                                <GraduationCap className="h-5 w-5 text-green-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Certified Educator</CardTitle>
                                <CardDescription>{certifiedEducatorSectionsLoading ? 'Loading...' : `${certifiedEducatorSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/specialized-qualifications/certified-educator" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-certified-educator" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Certified Industrial Engineer Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-teal-500 h-32"
                        onClick={() => {
                          setCurrentPageId("CertIndEng_abc456def789");
                          setIsCertifiedIndEngineerDialogOpen(true);
                        }}
                        data-testid="card-certified-industrial-engineer"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-teal-50 rounded-lg flex items-center justify-center">
                                <Cog className="h-5 w-5 text-teal-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Certified Industrial Engineer</CardTitle>
                                <CardDescription>{certifiedIndEngineerSectionsLoading ? 'Loading...' : `${certifiedIndEngineerSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/specialized-qualifications/certified-industrial-engineer" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-certified-industrial-engineer" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* CNPO Practitioner Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-indigo-500 h-32"
                        onClick={() => {
                          setCurrentPageId("CNPOPract_def789ghi012");
                          setIsCNPOPractitionerDialogOpen(true);
                        }}
                        data-testid="card-cnpo-practitioner"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-indigo-50 rounded-lg flex items-center justify-center">
                                <Shield className="h-5 w-5 text-indigo-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">CNPO Practitioner</CardTitle>
                                <CardDescription>{cnpoPractitionerSectionsLoading ? 'Loading...' : `${cnpoPractitionerSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/specialized-qualifications/cnpo-practitioner" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-cnpo-practitioner" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* CNPO Management Professional Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-rose-500 h-32"
                        onClick={() => {
                          setCurrentPageId("CNPOMgmt_ghi012jkl345");
                          setIsCNPOManagementDialogOpen(true);
                        }}
                        data-testid="card-cnpo-management"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-rose-50 rounded-lg flex items-center justify-center">
                                <Target className="h-5 w-5 text-rose-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">CNPO Management Professional</CardTitle>
                                <CardDescription>{cnpoManagementSectionsLoading ? 'Loading...' : `${cnpoManagementSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/specialized-qualifications/cnpo-management-professional" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-cnpo-management" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>
                
                {/* Foresight Qualifications */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6 mt-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Foresight Qualifications</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* Foresight & Innovation Associate Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-cyan-500 h-32"
                        onClick={() => {
                          setCurrentPageId("ForesightAssoc_jkl345mno678");
                          setIsForesightAssociateDialogOpen(true);
                        }}
                        data-testid="card-foresight-associate"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-cyan-50 rounded-lg flex items-center justify-center">
                                <Lightbulb className="h-5 w-5 text-cyan-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Foresight & Innovation Associate</CardTitle>
                                <CardDescription>{foresightAssociateSectionsLoading ? 'Loading...' : `${foresightAssociateSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/foresight-qualifications/foresight-associate" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-foresight-associate" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Foresight & Innovation Practitioner Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-sky-500 h-32"
                        onClick={() => {
                          setCurrentPageId("ForesightPract_mno678pqr901");
                          setIsForesightPractitionerDialogOpen(true);
                        }}
                        data-testid="card-foresight-practitioner"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-sky-50 rounded-lg flex items-center justify-center">
                                <TrendingUp className="h-5 w-5 text-sky-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Foresight & Innovation Practitioner</CardTitle>
                                <CardDescription>{foresightPractitionerSectionsLoading ? 'Loading...' : `${foresightPractitionerSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/foresight-qualifications/foresight-practitioner" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-foresight-practitioner" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Foresight & Innovation Strategist Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-blue-500 h-32"
                        onClick={() => {
                          setCurrentPageId("ForesightStrat_pqr901stu234");
                          setIsForesightStrategistDialogOpen(true);
                        }}
                        data-testid="card-foresight-strategist"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                                <Telescope className="h-5 w-5 text-blue-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Foresight & Innovation Strategist</CardTitle>
                                <CardDescription>{foresightStrategistSectionsLoading ? 'Loading...' : `${foresightStrategistSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/our-products/qualifications/foresight-qualifications/foresight-strategist" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-foresight-strategist" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            
            {/* Learning Games Subsection */}
            <div className="border-l-4 border-[#1A1A40]/20 pl-6 ml-8 space-y-6">
              <div>
                <h3 className="text-xl font-semibold text-[#1A1A40] dark:text-white mb-2">Learning Games</h3>
                <p className="text-sm text-gray-600 dark:text-gray-400">Interactive board games and simulations for learning</p>
              </div>
              
              <div>
                {/* Core Games */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Core Games</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* Win vs. War Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-orange-500 h-32"
                        onClick={() => {
                          setCurrentPageId("WinVsWar_stu234vwx567");
                          setIsWinVsWarDialogOpen(true);
                        }}
                        data-testid="card-win-vs-war"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-orange-50 rounded-lg flex items-center justify-center">
                                <Target className="h-5 w-5 text-orange-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Win vs. War</CardTitle>
                                <CardDescription>{winVsWarSectionsLoading ? 'Loading...' : `${winVsWarSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/learning-games/win-vs-war" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-win-vs-war" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Master Board Game Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-amber-500 h-32"
                        onClick={() => {
                          setCurrentPageId("MasterBoard_vwx567yz890");
                          setIsMasterBoardGameDialogOpen(true);
                        }}
                        data-testid="card-master-board-game"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-amber-50 rounded-lg flex items-center justify-center">
                                <Award className="h-5 w-5 text-amber-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Master Board Game</CardTitle>
                                <CardDescription>{masterBoardGameSectionsLoading ? 'Loading...' : `${masterBoardGameSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/learning-games/master-board-game" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-master-board-game" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Sparta Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-red-500 h-32"
                        onClick={() => {
                          setCurrentPageId("Sparta_yz890abc123");
                          setIsSpartaDialogOpen(true);
                        }}
                        data-testid="card-sparta"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-red-50 rounded-lg flex items-center justify-center">
                                <Shield className="h-5 w-5 text-red-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Sparta</CardTitle>
                                <CardDescription>{spartaSectionsLoading ? 'Loading...' : `${spartaSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/learning-games/sparta" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-sparta" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>

                {/* Online Games */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6 mt-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Online Games</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* Mogul CEO Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-purple-500 h-32"
                        onClick={() => {
                          setCurrentPageId("MogulCEO_abc123def456");
                          setIsMogulCEODialogOpen(true);
                        }}
                        data-testid="card-mogul-ceo"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-purple-50 rounded-lg flex items-center justify-center">
                                <Briefcase className="h-5 w-5 text-purple-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Mogul CEO</CardTitle>
                                <CardDescription>{mogulCEOSectionsLoading ? 'Loading...' : `${mogulCEOSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/learning-games/mogul-ceo" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-mogul-ceo" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Maven Marketing Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-indigo-500 h-32"
                        onClick={() => {
                          setCurrentPageId("MavenMkt_def456ghi789");
                          setIsMavenMarketingDialogOpen(true);
                        }}
                        data-testid="card-maven-marketing"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-indigo-50 rounded-lg flex items-center justify-center">
                                <TrendingUp className="h-5 w-5 text-indigo-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Maven Marketing</CardTitle>
                                <CardDescription>{mavenMarketingSectionsLoading ? 'Loading...' : `${mavenMarketingSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/learning-games/maven-marketing" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-maven-marketing" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Micromatic Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-teal-500 h-32"
                        onClick={() => {
                          setCurrentPageId("Micromatic_ghi789jkl012");
                          setIsMicromaticDialogOpen(true);
                        }}
                        data-testid="card-micromatic"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-teal-50 rounded-lg flex items-center justify-center">
                                <Cpu className="h-5 w-5 text-teal-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Micromatic</CardTitle>
                                <CardDescription>{micromaticSectionsLoading ? 'Loading...' : `${micromaticSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/learning-games/micromatic" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-micromatic" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>

                {/* Specialized Games */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6 mt-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Specialized Games</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* Pitch Deck Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-rose-500 h-32"
                        onClick={() => {
                          setCurrentPageId("PitchDeck_jkl012mno345");
                          setIsPitchDeckDialogOpen(true);
                        }}
                        data-testid="card-pitch-deck"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-rose-50 rounded-lg flex items-center justify-center">
                                <FileText className="h-5 w-5 text-rose-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Pitch Deck</CardTitle>
                                <CardDescription>{pitchDeckSectionsLoading ? 'Loading...' : `${pitchDeckSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/learning-games/pitch-deck" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-pitch-deck" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Synergy Stack Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-cyan-500 h-32"
                        onClick={() => {
                          setCurrentPageId("SynergyStack_mno345pqr678");
                          setIsSynergyStackDialogOpen(true);
                        }}
                        data-testid="card-synergy-stack"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-cyan-50 rounded-lg flex items-center justify-center">
                                <Layout className="h-5 w-5 text-cyan-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Synergy Stack</CardTitle>
                                <CardDescription>{synergyStackSectionsLoading ? 'Loading...' : `${synergyStackSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/learning-games/synergy-stack" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-synergy-stack" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            
            {/* IBDL Venture Lab Subsection */}
            <div className="border-l-4 border-[#1A1A40]/20 pl-6 ml-8 space-y-6">
              <div>
                <h3 className="text-xl font-semibold text-[#1A1A40] dark:text-white mb-2">IBDL Venture Lab</h3>
                <p className="text-sm text-gray-600 dark:text-gray-400">Entrepreneurship and Innovation Qualifications</p>
              </div>
              
              <div>
                {/* Core Qualifications */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Core Qualifications</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* Certified Disciplined Entrepreneur Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-amber-500 h-32"
                        onClick={() => {
                          setCurrentPageId("CertDiscEnt_pqr678stu901");
                          setIsCertDiscEntDialogOpen(true);
                        }}
                        data-testid="card-cert-disc-ent"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-amber-50 rounded-lg flex items-center justify-center">
                                <Lightbulb className="h-5 w-5 text-amber-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Certified Disciplined Entrepreneur</CardTitle>
                                <CardDescription>{certDiscEntSectionsLoading ? 'Loading...' : `${certDiscEntSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/venture-lab/certified-disciplined-entrepreneur" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-cert-disc-ent" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Certified Incubation Management Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-orange-500 h-32"
                        onClick={() => {
                          setCurrentPageId("CertIncubMgmt_stu901vwx234");
                          setIsCertIncubMgmtDialogOpen(true);
                        }}
                        data-testid="card-cert-incub-mgmt"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-orange-50 rounded-lg flex items-center justify-center">
                                <Building2 className="h-5 w-5 text-orange-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Certified Incubation Management</CardTitle>
                                <CardDescription>{certIncubMgmtSectionsLoading ? 'Loading...' : `${certIncubMgmtSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/venture-lab/certified-incubation-management" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-cert-incub-mgmt" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>
              </div>
              
              <div>
                {/* Specialized Executive Programs */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Specialized Executive Programs</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* Angel Playbook Bootcamp Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-blue-500 h-32"
                        onClick={() => {
                          setCurrentPageId("AngelPlaybook_vwx234yz567");
                          setIsAngelPlaybookDialogOpen(true);
                        }}
                        data-testid="card-angel-playbook"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                                <Target className="h-5 w-5 text-blue-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Angel Playbook Bootcamp</CardTitle>
                                <CardDescription>{angelPlaybookSectionsLoading ? 'Loading...' : `${angelPlaybookSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/venture-hub/angel-playbook-bootcamp" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-angel-playbook" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Capital Raise Bootcamp Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-green-500 h-32"
                        onClick={() => {
                          setCurrentPageId("CapitalRaise_yz567abc890");
                          setIsCapitalRaiseDialogOpen(true);
                        }}
                        data-testid="card-capital-raise"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-green-50 rounded-lg flex items-center justify-center">
                                <DollarSign className="h-5 w-5 text-green-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Capital Raise Bootcamp</CardTitle>
                                <CardDescription>{capitalRaiseSectionsLoading ? 'Loading...' : `${capitalRaiseSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/venture-hub/capital-raise-bootcamp" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-capital-raise" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Data-Room Preparation Bootcamp Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-indigo-500 h-32"
                        onClick={() => {
                          setCurrentPageId("DataRoom_abc890def123");
                          setIsDataRoomDialogOpen(true);
                        }}
                        data-testid="card-data-room"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-indigo-50 rounded-lg flex items-center justify-center">
                                <FileText className="h-5 w-5 text-indigo-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Data-Room Preparation Bootcamp</CardTitle>
                                <CardDescription>{dataRoomSectionsLoading ? 'Loading...' : `${dataRoomSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/venture-hub/data-room-preparation-bootcamp" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-data-room" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            
            {/* Assessment Services Subsection */}
            <div className="border-l-4 border-[#1A1A40]/20 pl-6 ml-8 space-y-6">
              <div>
                <h3 className="text-xl font-semibold text-[#1A1A40] dark:text-white mb-2">Assessment Services</h3>
                <p className="text-sm text-gray-600 dark:text-gray-400">Professional psychometric and competency assessments</p>
              </div>
              
              <div>
                {/* Core Assessment */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Core Assessment</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* PQP Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-teal-500 h-32"
                        onClick={() => {
                          setCurrentPageId("PQP_def123ghi456");
                          setIsPQPDialogOpen(true);
                        }}
                        data-testid="card-pqp"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-teal-50 rounded-lg flex items-center justify-center">
                                <BarChart3 className="h-5 w-5 text-teal-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">PQP</CardTitle>
                                <CardDescription>{pqpSectionsLoading ? 'Loading...' : `${pqpSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/assessment-services/pqp" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-pqp" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* CPAT Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-emerald-500 h-32"
                        onClick={() => {
                          setCurrentPageId("CPAT_ghi456jkl789");
                          setIsCPATDialogOpen(true);
                        }}
                        data-testid="card-cpat"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-emerald-50 rounded-lg flex items-center justify-center">
                                <Users className="h-5 w-5 text-emerald-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">CPAT</CardTitle>
                                <CardDescription>{cpatSectionsLoading ? 'Loading...' : `${cpatSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/products/assessment-services/cpat" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-cpat" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Executive Education Subsection */}
            <div className="border-l-4 border-[#1A1A40]/20 pl-6 ml-8 space-y-6">
              <div>
                <h3 className="text-xl font-semibold text-[#1A1A40] dark:text-white mb-2">Executive Education</h3>
                <p className="text-sm text-gray-600 dark:text-gray-400">Advanced programs for executive leadership development</p>
              </div>
              
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {/* Executive Education Card */}
                <div className="max-w-md">
                  <Card 
                    className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-purple-500 h-32"
                    onClick={() => {
                      setCurrentPageId("ExecEd_jkl789mno012");
                      setIsExecutiveEducationDialogOpen(true);
                    }}
                    data-testid="card-executive-education"
                  >
                    <CardHeader>
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-3">
                          <div className="w-10 h-10 bg-purple-50 rounded-lg flex items-center justify-center">
                            <GraduationCap className="h-5 w-5 text-purple-600" />
                          </div>
                          <div>
                            <CardTitle className="text-lg">Executive Education</CardTitle>
                            <CardDescription>{executiveEducationSectionsLoading ? 'Loading...' : `${executiveEducationSections.length} sections`}</CardDescription>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          <a href="/products/executive-education" target="_blank" rel="noopener noreferrer">
                            <Button variant="ghost" size="sm" data-testid="button-preview-executive-education" className="flex items-center gap-2">
                              <ExternalLink className="h-4 w-4" />
                              Preview
                            </Button>
                          </a>
                        </div>
                      </div>
                    </CardHeader>
                  </Card>
                </div>
              </div>
            </div>
          </div>

          {/* Our Services Section */}
          <div className="space-y-6">
            {/* Section Header */}
            <div className="flex items-center space-x-4">
              <div className="w-12 h-12 bg-gradient-to-br from-[#1A1A40] to-[#1A1A40]/80 rounded-xl flex items-center justify-center shadow-md">
                <Briefcase className="h-6 w-6 text-white" />
            </div>
            <div>
              <h2 className="text-2xl font-bold text-[#1A1A40] dark:text-white">Our Services</h2>
              <p className="text-gray-600 dark:text-gray-400">Service offerings and support programs</p>
            </div>
          </div>

          <div className="space-y-8">
            {/* Accreditation Services Subsection */}
            <div className="border-l-4 border-[#1A1A40]/20 pl-6 ml-8 space-y-6">
              <div>
                <h3 className="text-xl font-semibold text-[#1A1A40] dark:text-white mb-2">Accreditation Services</h3>
                <p className="text-sm text-gray-600 dark:text-gray-400">Professional accreditation and certification programs</p>
              </div>
              
              <div>
                {/* Provider Types */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Provider Types</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* Training Provider Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-blue-500 h-32"
                        onClick={() => {
                          setCurrentPageId("TrainingProv_mno012pqr345");
                          setIsTrainingProviderDialogOpen(true);
                        }}
                        data-testid="card-training-provider"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                                <BookOpen className="h-5 w-5 text-blue-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Training Provider</CardTitle>
                                <CardDescription>{trainingProviderSectionsLoading ? 'Loading...' : `${trainingProviderSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/services/accreditation-services/training-provider" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-training-provider" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Exam Provider Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-cyan-500 h-32"
                        onClick={() => {
                          setCurrentPageId("ExamProv_pqr345stu678");
                          setIsExamProviderDialogOpen(true);
                        }}
                        data-testid="card-exam-provider"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-cyan-50 rounded-lg flex items-center justify-center">
                                <FileCheck className="h-5 w-5 text-cyan-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Exam Provider</CardTitle>
                                <CardDescription>{examProviderSectionsLoading ? 'Loading...' : `${examProviderSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/services/accreditation-services/exam-provider" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-exam-provider" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Corporate/Government Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-slate-500 h-32"
                        onClick={() => {
                          setCurrentPageId("CorpGov_stu678vwx901");
                          setIsCorpGovDialogOpen(true);
                        }}
                        data-testid="card-corp-gov"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-slate-50 rounded-lg flex items-center justify-center">
                                <Building2 className="h-5 w-5 text-slate-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Corporate/Government</CardTitle>
                                <CardDescription>{corpGovSectionsLoading ? 'Loading...' : `${corpGovSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/services/accreditation-services/corporate-government" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-corp-gov" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>
              </div>

              <div>
                {/* Professional Services */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Professional Services</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {/* Professional IBDL Trainer Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-indigo-500 h-32"
                        onClick={() => {
                          setCurrentPageId("ProfTrainer_vwx901yz234abc");
                          setIsProfTrainerDialogOpen(true);
                        }}
                        data-testid="card-prof-trainer"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-indigo-50 rounded-lg flex items-center justify-center">
                                <GraduationCap className="h-5 w-5 text-indigo-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Professional IBDL Trainer</CardTitle>
                                <CardDescription>{profTrainerSectionsLoading ? 'Loading...' : `${profTrainerSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/services/accreditation-services/professional-trainer" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-prof-trainer" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>

                    {/* Professional Course Card */}
                    <div className="max-w-md">
                      <Card 
                        className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-violet-500 h-32"
                        onClick={() => {
                          setCurrentPageId("ProfCourse_yz234abc567def");
                          setIsProfCourseDialogOpen(true);
                        }}
                        data-testid="card-prof-course"
                      >
                        <CardHeader>
                          <div className="flex items-center justify-between">
                            <div className="flex items-center space-x-3">
                              <div className="w-10 h-10 bg-violet-50 rounded-lg flex items-center justify-center">
                                <BookOpen className="h-5 w-5 text-violet-600" />
                              </div>
                              <div>
                                <CardTitle className="text-lg">Professional Course</CardTitle>
                                <CardDescription>{profCourseSectionsLoading ? 'Loading...' : `${profCourseSections.length} sections`}</CardDescription>
                              </div>
                            </div>
                            <div className="flex items-center space-x-2">
                              <a href="/services/accreditation-services/professional-course" target="_blank" rel="noopener noreferrer">
                                <Button variant="ghost" size="sm" data-testid="button-preview-prof-course" className="flex items-center gap-2">
                                  <ExternalLink className="h-4 w-4" />
                                  Preview
                                </Button>
                              </a>
                            </div>
                          </div>
                        </CardHeader>
                      </Card>
                    </div>
                  </div>
                </div>
              </div>

            {/* Be Part of Our Story Subsection */}
            <div className="border-l-4 border-[#1A1A40]/20 pl-6 ml-8 space-y-6">
              <div>
                <h3 className="text-xl font-semibold text-[#1A1A40] dark:text-white mb-2">Be Part of Our Story</h3>
                <p className="text-sm text-gray-600 dark:text-gray-400">Partnership opportunities and community engagement</p>
              </div>
              
              <div>
                {/* Partnership Sub-subsection */}
                <div className="ml-6 border-l-2 border-gray-200 dark:border-gray-700 pl-6">
                  <h4 className="text-md font-medium text-gray-600 dark:text-gray-400 mb-4">Partnership</h4>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        {/* Partner With Us Card */}
                        <div className="max-w-md">
                          <Card 
                            className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-emerald-500 h-32"
                            onClick={() => {
                              setCurrentPageId("PartnerWith_abc567def890ghi");
                              setIsPartnerWithDialogOpen(true);
                            }}
                            data-testid="card-partner-with"
                          >
                            <CardHeader>
                              <div className="flex items-center justify-between">
                                <div className="flex items-center space-x-3">
                                  <div className="w-10 h-10 bg-emerald-50 rounded-lg flex items-center justify-center">
                                    <Handshake className="h-5 w-5 text-emerald-600" />
                                  </div>
                                  <div>
                                    <CardTitle className="text-lg">Partner With Us</CardTitle>
                                    <CardDescription>{partnerWithSectionsLoading ? 'Loading...' : `${partnerWithSections.length} sections`}</CardDescription>
                                  </div>
                                </div>
                                <div className="flex items-center space-x-2">
                                  <a href="/services/be-part-of-our-story/partner-with-us" target="_blank" rel="noopener noreferrer">
                                    <Button variant="ghost" size="sm" data-testid="button-preview-partner-with" className="flex items-center gap-2">
                                      <ExternalLink className="h-4 w-4" />
                                      Preview
                                    </Button>
                                  </a>
                                </div>
                              </div>
                            </CardHeader>
                          </Card>
                        </div>

                        {/* Join Our Global Community Card */}
                        <div className="max-w-md">
                          <Card 
                            className="cursor-pointer hover:shadow-lg transition-shadow duration-200 border-l-4 border-l-sky-500 h-32"
                            onClick={() => {
                              setCurrentPageId("JoinComm_def890ghi123jkl");
                              setIsJoinCommDialogOpen(true);
                            }}
                            data-testid="card-join-comm"
                          >
                            <CardHeader>
                              <div className="flex items-center justify-between">
                                <div className="flex items-center space-x-3">
                                  <div className="w-10 h-10 bg-sky-50 rounded-lg flex items-center justify-center">
                                    <Users className="h-5 w-5 text-sky-600" />
                                  </div>
                                  <div>
                                    <CardTitle className="text-lg">Join Our Global Community</CardTitle>
                                    <CardDescription>{joinCommSectionsLoading ? 'Loading...' : `${joinCommSections.length} sections`}</CardDescription>
                                  </div>
                                </div>
                                <div className="flex items-center space-x-2">
                                  <a href="/services/be-part-of-our-story/join-global-community" target="_blank" rel="noopener noreferrer">
                                    <Button variant="ghost" size="sm" data-testid="button-preview-join-comm" className="flex items-center gap-2">
                                      <ExternalLink className="h-4 w-4" />
                                      Preview
                                    </Button>
                                  </a>
                                </div>
                              </div>
                            </CardHeader>
                          </Card>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
            </div>
          </div>
        </div>
        </div>

        {/* Our Story Sections Dialog */}
        <Dialog open={isOurStoryDialogOpen} onOpenChange={setIsOurStoryDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Our Story Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Our Story page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  // Clear all form state when creating new section
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              {/* Template Selection Panel */}
              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    // Close Our Story dialog and open section editor
                    setCurrentPageId("gRsEOE7ovOW3_2_cHOqaO");
                    setIsOurStoryDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    // Ensure completely clean form for new section
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template"
                closeButtonTestId="button-close-template-selection"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {sectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={handleDragEnd}
                    >
                      <SortableContext 
                        items={ourStorySections.map((section: any) => section.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {ourStorySections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={updateSectionMutation.isPending}
                              isDeletePending={deleteSectionMutation.isPending}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Leadership & Governance Sections Dialog */}
        <Dialog open={isLeadershipGovernanceDialogOpen} onOpenChange={setIsLeadershipGovernanceDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Leadership & Governance Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Leadership & Governance page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  // Clear all form state when creating new section
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-leadership"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              {/* Template Selection Panel for Leadership & Governance */}
              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    // Close Leadership & Governance dialog and open section editor
                    setCurrentPageId("LGov_abc123xyz456789");
                    setIsLeadershipGovernanceDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    // Ensure completely clean form for new section
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-leadership"
                closeButtonTestId="button-cancel-template"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {leadershipSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = leadershipGovernanceSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = leadershipGovernanceSections.findIndex((section: any) => section.id === over.id);
                          
                          const newSections = arrayMove(leadershipGovernanceSections, oldIndex, newIndex);
                          
                          // Update the database with new order
                          const reorderData = newSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index + 1
                          }));
                          
                          reorderSectionsMutation.mutate(reorderData);
                        }
                      }}
                    >
                      <SortableContext 
                        items={leadershipGovernanceSections.map((section: any) => section.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {leadershipGovernanceSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsLeadershipGovernanceDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  )}
                  
                  {!leadershipSectionsLoading && leadershipGovernanceSections.length === 0 && (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Our Impact Sections Dialog */}
        <Dialog open={isOurImpactDialogOpen} onOpenChange={setIsOurImpactDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Our Impact Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Our Impact page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  // Clear all form state when creating new section
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-impact"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              {/* Template Selection Panel for Our Impact */}
              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    // Close Our Impact dialog and open section editor
                    setCurrentPageId("Impact_def456ghi789012");
                    setIsOurImpactDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    // Ensure completely clean form for new section
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-impact"
                closeButtonTestId="button-cancel-template-impact"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {impactSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = ourImpactSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = ourImpactSections.findIndex((section: any) => section.id === over.id);
                          
                          const newSections = arrayMove(ourImpactSections, oldIndex, newIndex);
                          
                          // Update the database with new order
                          const reorderData = newSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index + 1
                          }));
                          
                          reorderSectionsMutation.mutate(reorderData);
                        }
                      }}
                    >
                      <SortableContext 
                        items={ourImpactSections.map((section: any) => section.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {ourImpactSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsOurImpactDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Knowledge Framework Sections Dialog */}
        <Dialog open={isKnowledgeFrameworkDialogOpen} onOpenChange={setIsKnowledgeFrameworkDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Knowledge Framework Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Knowledge Framework page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  // Clear all form state when creating new section
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-knowledge-framework"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              {/* Template Selection Panel for Knowledge Framework */}
              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    // Close Knowledge Framework dialog and open section editor
                    setCurrentPageId("KnowFramework_xyz789abc123");
                    setIsKnowledgeFrameworkDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    // Ensure completely clean form for new section
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template"
                closeButtonTestId="button-close-template-selection"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {knowledgeFrameworkSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = knowledgeFrameworkSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = knowledgeFrameworkSections.findIndex((section: any) => section.id === over.id);
                          
                          const newSections = arrayMove(knowledgeFrameworkSections, oldIndex, newIndex);
                          
                          // Update the database with new order
                          const reorderData = newSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(reorderData);
                        }
                      }}
                    >
                      <SortableContext 
                        items={knowledgeFrameworkSections.map((section: any) => section.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {knowledgeFrameworkSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsKnowledgeFrameworkDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  )}
                  
                  {!knowledgeFrameworkSectionsLoading && knowledgeFrameworkSections.length === 0 && (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Partners in Excellence Sections Dialog */}
        <Dialog open={isPartnersInExcellenceDialogOpen} onOpenChange={setIsPartnersInExcellenceDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Partners in Excellence Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Partners in Excellence page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  // Clear all form state when creating new section
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-partners-in-excellence"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              {/* Template Selection Panel for Partners in Excellence */}
              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    // Close Partners in Excellence dialog and open section editor
                    setCurrentPageId("PartnersExcel_qwe456rty789");
                    setIsPartnersInExcellenceDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    // Ensure completely clean form for new section
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-partners-in-excellence"
                closeButtonTestId="button-cancel-template-partners-in-excellence"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {partnersInExcellenceSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : partnersInExcellenceSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={handleDragEnd}
                    >
                      <SortableContext 
                        items={partnersInExcellenceSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {partnersInExcellenceSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                
                                // Handle array fields with proper structure
                                if (section.stats) formData.stats = section.stats;
                                if (section.achievements) formData.achievements = section.achievements;
                                if (section.certifications) formData.certifications = section.certifications;
                                if (section.partnerships) formData.partnerships = section.partnerships;
                                if (section.testimonials) formData.testimonials = section.testimonials;
                                if (section.team_members) formData.team_members = section.team_members;
                                if (section.case_studies) formData.case_studies = section.case_studies;
                                if (section.leadership_profiles) formData.leadership_profiles = section.leadership_profiles;
                                if (section.content_paragraphs) formData.content_paragraphs = section.content_paragraphs;
                                if (section.core_values) formData.core_values = section.core_values;
                                if (section.strategic_goals) formData.strategic_goals = section.strategic_goals;
                                if (section.impact_metrics) formData.impact_metrics = section.impact_metrics;
                                if (section.success_stories) formData.success_stories = section.success_stories;
                                if (section.community_programs) formData.community_programs = section.community_programs;
                                if (section.social_responsibility) formData.social_responsibility = section.social_responsibility;
                                if (section.environmental_initiatives) formData.environmental_initiatives = section.environmental_initiatives;
                                if (section.diversity_inclusion) formData.diversity_inclusion = section.diversity_inclusion;
                                if (section.quality_assurance) formData.quality_assurance = section.quality_assurance;
                                if (section.accreditation_info) formData.accreditation_info = section.accreditation_info;
                                if (section.research_publications) formData.research_publications = section.research_publications;
                                if (section.innovation_projects) formData.innovation_projects = section.innovation_projects;
                                if (section.technology_stack) formData.technology_stack = section.technology_stack;
                                if (section.frameworks) formData.frameworks = section.frameworks;
                                if (section.competency_models) formData.competency_models = section.competency_models;
                                if (section.assessment_criteria) formData.assessment_criteria = section.assessment_criteria;
                                if (section.learning_outcomes) formData.learning_outcomes = section.learning_outcomes;
                                
                                // Handle string fields
                                if (section.mission_statement) formData.mission_statement = section.mission_statement;
                                if (section.vision_statement) formData.vision_statement = section.vision_statement;
                                if (section.governance_structure) formData.governance_structure = section.governance_structure;
                                if (section.compliance_info) formData.compliance_info = section.compliance_info;
                                if (section.methodology) formData.methodology = section.methodology;
                                
                                // Pre-load trustee files from existing section
                                if (section.leadership_profiles && Array.isArray(section.leadership_profiles)) {
                                  const files: Record<string, File> = {};
                                  section.leadership_profiles.forEach((profile: any, index: number) => {
                                    if (profile.image_url) {
                                      // Note: We can't reconstruct File objects from URLs, 
                                      // so we just track which ones exist
                                    }
                                  });
                                  setTrusteeFiles(files);
                                }
                                
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsPartnersInExcellenceDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* IBDL Essential Sections Dialog */}
        <Dialog open={isIBDLEssentialDialogOpen} onOpenChange={setIsIBDLEssentialDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>IBDL Essential Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the IBDL Essential page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  // Clear all form state when creating new section
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-ibdl-essential"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              {/* Template Selection Panel for IBDL Essential */}
              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    // Close IBDL Essential dialog and open section editor
                    setCurrentPageId("IBDLEssential_abc123xyz789");
                    setIsIBDLEssentialDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    // Ensure completely clean form for new section
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-ibdl-essential"
                closeButtonTestId="button-cancel-template-ibdl-essential"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {ibdlEssentialSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : ibdlEssentialSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = ibdlEssentialSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = ibdlEssentialSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(ibdlEssentialSections, oldIndex, newIndex);
                          
                          // Update sort order and send to backend
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={ibdlEssentialSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {ibdlEssentialSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsIBDLEssentialDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* IBDL Professional Sections Dialog */}
        <Dialog open={isIBDLProfessionalDialogOpen} onOpenChange={setIsIBDLProfessionalDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>IBDL Professional Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the IBDL Professional page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-ibdl-professional"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("IBDLProfessional_xyz456abc789");
                    setIsIBDLProfessionalDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-ibdl-professional"
                closeButtonTestId="button-cancel-template-ibdl-professional"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {ibdlProfessionalSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : ibdlProfessionalSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = ibdlProfessionalSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = ibdlProfessionalSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(ibdlProfessionalSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={ibdlProfessionalSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {ibdlProfessionalSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsIBDLProfessionalDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* IBDL Master Sections Dialog */}
        <Dialog open={isIBDLMasterDialogOpen} onOpenChange={setIsIBDLMasterDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>IBDL Master Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the IBDL Master page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-ibdl-master"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("IBDLMaster_def789ghi012");
                    setIsIBDLMasterDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-ibdl-master"
                closeButtonTestId="button-cancel-template-ibdl-master"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {ibdlMasterSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : ibdlMasterSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = ibdlMasterSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = ibdlMasterSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(ibdlMasterSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={ibdlMasterSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {ibdlMasterSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsIBDLMasterDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Certified Educator Sections Dialog */}
        <Dialog open={isCertifiedEducatorDialogOpen} onOpenChange={setIsCertifiedEducatorDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Certified Educator Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Certified Educator page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-certified-educator"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("CertEducator_xyz123abc456");
                    setIsCertifiedEducatorDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-certified-educator"
                closeButtonTestId="button-cancel-template-certified-educator"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {certifiedEducatorSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : certifiedEducatorSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = certifiedEducatorSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = certifiedEducatorSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(certifiedEducatorSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={certifiedEducatorSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {certifiedEducatorSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsCertifiedEducatorDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Certified Industrial Engineer Sections Dialog */}
        <Dialog open={isCertifiedIndEngineerDialogOpen} onOpenChange={setIsCertifiedIndEngineerDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Certified Industrial Engineer Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Certified Industrial Engineer page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-certified-industrial-engineer"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("CertIndEng_abc456def789");
                    setIsCertifiedIndEngineerDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-certified-industrial-engineer"
                closeButtonTestId="button-cancel-template-certified-industrial-engineer"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {certifiedIndEngineerSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : certifiedIndEngineerSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = certifiedIndEngineerSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = certifiedIndEngineerSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(certifiedIndEngineerSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={certifiedIndEngineerSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {certifiedIndEngineerSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsCertifiedIndEngineerDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* CNPO Practitioner Sections Dialog */}
        <Dialog open={isCNPOPractitionerDialogOpen} onOpenChange={setIsCNPOPractitionerDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>CNPO Practitioner Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the CNPO Practitioner page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-cnpo-practitioner"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("CNPOPract_def789ghi012");
                    setIsCNPOPractitionerDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-cnpo-practitioner"
                closeButtonTestId="button-cancel-template-cnpo-practitioner"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {cnpoPractitionerSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : cnpoPractitionerSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = cnpoPractitionerSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = cnpoPractitionerSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(cnpoPractitionerSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={cnpoPractitionerSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {cnpoPractitionerSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsCNPOPractitionerDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* CNPO Management Professional Sections Dialog */}
        <Dialog open={isCNPOManagementDialogOpen} onOpenChange={setIsCNPOManagementDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>CNPO Management Professional Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the CNPO Management Professional page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-cnpo-management"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("CNPOMgmt_ghi012jkl345");
                    setIsCNPOManagementDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-cnpo-management"
                closeButtonTestId="button-cancel-template-cnpo-management"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {cnpoManagementSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : cnpoManagementSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = cnpoManagementSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = cnpoManagementSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(cnpoManagementSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={cnpoManagementSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {cnpoManagementSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsCNPOManagementDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Foresight & Innovation Associate Sections Dialog */}
        <Dialog open={isForesightAssociateDialogOpen} onOpenChange={setIsForesightAssociateDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Foresight & Innovation Associate Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Foresight & Innovation Associate page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-foresight-associate"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("ForesightAssoc_jkl345mno678");
                    setIsForesightAssociateDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-foresight-associate"
                closeButtonTestId="button-cancel-template-foresight-associate"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {foresightAssociateSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : foresightAssociateSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = foresightAssociateSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = foresightAssociateSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(foresightAssociateSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={foresightAssociateSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {foresightAssociateSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsForesightAssociateDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Foresight & Innovation Practitioner Sections Dialog */}
        <Dialog open={isForesightPractitionerDialogOpen} onOpenChange={setIsForesightPractitionerDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Foresight & Innovation Practitioner Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Foresight & Innovation Practitioner page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-foresight-practitioner"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("ForesightPract_mno678pqr901");
                    setIsForesightPractitionerDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-foresight-practitioner"
                closeButtonTestId="button-cancel-template-foresight-practitioner"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {foresightPractitionerSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : foresightPractitionerSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = foresightPractitionerSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = foresightPractitionerSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(foresightPractitionerSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={foresightPractitionerSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {foresightPractitionerSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsForesightPractitionerDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Foresight & Innovation Strategist Sections Dialog */}
        <Dialog open={isForesightStrategistDialogOpen} onOpenChange={setIsForesightStrategistDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Foresight & Innovation Strategist Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Foresight & Innovation Strategist page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-foresight-strategist"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("ForesightStrat_pqr901stu234");
                    setIsForesightStrategistDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-foresight-strategist"
                closeButtonTestId="button-cancel-template-foresight-strategist"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {foresightStrategistSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : foresightStrategistSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = foresightStrategistSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = foresightStrategistSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(foresightStrategistSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={foresightStrategistSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {foresightStrategistSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsForesightStrategistDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Win vs. War Sections Dialog */}
        <Dialog open={isWinVsWarDialogOpen} onOpenChange={setIsWinVsWarDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Win vs. War Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Win vs. War game page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-win-vs-war"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("WinVsWar_stu234vwx567");
                    setIsWinVsWarDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-win-vs-war"
                closeButtonTestId="button-cancel-template-win-vs-war"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {winVsWarSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : winVsWarSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = winVsWarSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = winVsWarSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(winVsWarSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={winVsWarSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {winVsWarSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsWinVsWarDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Master Board Game Sections Dialog */}
        <Dialog open={isMasterBoardGameDialogOpen} onOpenChange={setIsMasterBoardGameDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Master Board Game Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Master Board Game page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-master-board-game"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("MasterBoard_vwx567yz890");
                    setIsMasterBoardGameDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-master-board-game"
                closeButtonTestId="button-cancel-template-master-board-game"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {masterBoardGameSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : masterBoardGameSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = masterBoardGameSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = masterBoardGameSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(masterBoardGameSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={masterBoardGameSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {masterBoardGameSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsMasterBoardGameDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Sparta Sections Dialog */}
        <Dialog open={isSpartaDialogOpen} onOpenChange={setIsSpartaDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Sparta Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Sparta game page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-sparta"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("Sparta_yz890abc123");
                    setIsSpartaDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-sparta"
                closeButtonTestId="button-cancel-template-sparta"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {spartaSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : spartaSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = spartaSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = spartaSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(spartaSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={spartaSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {spartaSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsSpartaDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Mogul CEO Sections Dialog */}
        <Dialog open={isMogulCEODialogOpen} onOpenChange={setIsMogulCEODialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Mogul CEO Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Mogul CEO game page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-mogul-ceo"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("MogulCEO_abc123def456");
                    setIsMogulCEODialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-mogul-ceo"
                closeButtonTestId="button-cancel-template-mogul-ceo"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {mogulCEOSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : mogulCEOSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = mogulCEOSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = mogulCEOSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(mogulCEOSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={mogulCEOSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {mogulCEOSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsMogulCEODialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Maven Marketing Sections Dialog */}
        <Dialog open={isMavenMarketingDialogOpen} onOpenChange={setIsMavenMarketingDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Maven Marketing Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Maven Marketing game page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-maven-marketing"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("MavenMkt_def456ghi789");
                    setIsMavenMarketingDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-maven-marketing"
                closeButtonTestId="button-cancel-template-maven-marketing"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {mavenMarketingSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : mavenMarketingSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = mavenMarketingSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = mavenMarketingSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(mavenMarketingSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={mavenMarketingSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {mavenMarketingSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsMavenMarketingDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Micromatic Sections Dialog */}
        <Dialog open={isMicromaticDialogOpen} onOpenChange={setIsMicromaticDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Micromatic Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Micromatic game page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-micromatic"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("Micromatic_ghi789jkl012");
                    setIsMicromaticDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-micromatic"
                closeButtonTestId="button-cancel-template-micromatic"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {micromaticSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : micromaticSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = micromaticSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = micromaticSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(micromaticSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={micromaticSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {micromaticSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsMicromaticDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Pitch Deck Sections Dialog */}
        <Dialog open={isPitchDeckDialogOpen} onOpenChange={setIsPitchDeckDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Pitch Deck Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Pitch Deck game page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-pitch-deck"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("PitchDeck_jkl012mno345");
                    setIsPitchDeckDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-pitch-deck"
                closeButtonTestId="button-cancel-template-pitch-deck"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {pitchDeckSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : pitchDeckSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = pitchDeckSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = pitchDeckSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(pitchDeckSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={pitchDeckSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {pitchDeckSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsPitchDeckDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Synergy Stack Sections Dialog */}
        <Dialog open={isSynergyStackDialogOpen} onOpenChange={setIsSynergyStackDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Synergy Stack Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Synergy Stack game page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-synergy-stack"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("SynergyStack_mno345pqr678");
                    setIsSynergyStackDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-synergy-stack"
                closeButtonTestId="button-cancel-template-synergy-stack"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {synergyStackSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : synergyStackSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = synergyStackSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = synergyStackSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(synergyStackSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={synergyStackSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {synergyStackSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsSynergyStackDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Certified Disciplined Entrepreneur Sections Dialog */}
        <Dialog open={isCertDiscEntDialogOpen} onOpenChange={setIsCertDiscEntDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Certified Disciplined Entrepreneur Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Certified Disciplined Entrepreneur page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-cert-disc-ent"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("CertDiscEnt_pqr678stu901");
                    setIsCertDiscEntDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-cert-disc-ent"
                closeButtonTestId="button-cancel-template-cert-disc-ent"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {certDiscEntSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : certDiscEntSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = certDiscEntSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = certDiscEntSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(certDiscEntSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={certDiscEntSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {certDiscEntSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsCertDiscEntDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Certified Incubation Management Sections Dialog */}
        <Dialog open={isCertIncubMgmtDialogOpen} onOpenChange={setIsCertIncubMgmtDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Certified Incubation Management Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Certified Incubation Management page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-cert-incub-mgmt"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("CertIncubMgmt_stu901vwx234");
                    setIsCertIncubMgmtDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-cert-incub-mgmt"
                closeButtonTestId="button-cancel-template-cert-incub-mgmt"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {certIncubMgmtSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : certIncubMgmtSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = certIncubMgmtSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = certIncubMgmtSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(certIncubMgmtSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={certIncubMgmtSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {certIncubMgmtSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsCertIncubMgmtDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Angel Playbook Bootcamp Sections Dialog */}
        <Dialog open={isAngelPlaybookDialogOpen} onOpenChange={setIsAngelPlaybookDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Angel Playbook Bootcamp Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Angel Playbook Bootcamp page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-angel-playbook"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("AngelPlaybook_vwx234yz567");
                    setIsAngelPlaybookDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-angel-playbook"
                closeButtonTestId="button-cancel-template-angel-playbook"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {angelPlaybookSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : angelPlaybookSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = angelPlaybookSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = angelPlaybookSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(angelPlaybookSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={angelPlaybookSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {angelPlaybookSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsAngelPlaybookDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Capital Raise Bootcamp Sections Dialog */}
        <Dialog open={isCapitalRaiseDialogOpen} onOpenChange={setIsCapitalRaiseDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Capital Raise Bootcamp Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Capital Raise Bootcamp page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-capital-raise"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("CapitalRaise_yz567abc890");
                    setIsCapitalRaiseDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-capital-raise"
                closeButtonTestId="button-cancel-template-capital-raise"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {capitalRaiseSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : capitalRaiseSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = capitalRaiseSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = capitalRaiseSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(capitalRaiseSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={capitalRaiseSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {capitalRaiseSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsCapitalRaiseDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Data-Room Preparation Bootcamp Sections Dialog */}
        <Dialog open={isDataRoomDialogOpen} onOpenChange={setIsDataRoomDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Data-Room Preparation Bootcamp Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Data-Room Preparation Bootcamp page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-data-room"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("DataRoom_abc890def123");
                    setIsDataRoomDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-data-room"
                closeButtonTestId="button-cancel-template-data-room"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {dataRoomSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : dataRoomSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = dataRoomSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = dataRoomSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(dataRoomSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={dataRoomSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {dataRoomSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsDataRoomDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* PQP Sections Dialog */}
        <Dialog open={isPQPDialogOpen} onOpenChange={setIsPQPDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>PQP Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the PQP page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-pqp"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("PQP_def123ghi456");
                    setIsPQPDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-pqp"
                closeButtonTestId="button-cancel-template-pqp"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {pqpSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : pqpSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = pqpSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = pqpSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(pqpSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={pqpSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {pqpSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsPQPDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* CPAT Sections Dialog */}
        <Dialog open={isCPATDialogOpen} onOpenChange={setIsCPATDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>CPAT Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the CPAT page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-cpat"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("CPAT_ghi456jkl789");
                    setIsCPATDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-cpat"
                closeButtonTestId="button-cancel-template-cpat"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {cpatSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : cpatSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = cpatSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = cpatSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(cpatSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={cpatSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {cpatSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsCPATDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Executive Education Sections Dialog */}
        <Dialog open={isExecutiveEducationDialogOpen} onOpenChange={setIsExecutiveEducationDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Executive Education Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Executive Education page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-executive-education"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("ExecEd_jkl789mno012");
                    setIsExecutiveEducationDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-executive-education"
                closeButtonTestId="button-cancel-template-executive-education"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {executiveEducationSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : executiveEducationSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = executiveEducationSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = executiveEducationSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(executiveEducationSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={executiveEducationSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {executiveEducationSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsExecutiveEducationDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Training Provider Sections Dialog */}
        <Dialog open={isTrainingProviderDialogOpen} onOpenChange={setIsTrainingProviderDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Training Provider Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Training Provider page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-training-provider"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("TrainingProv_mno012pqr345");
                    setIsTrainingProviderDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-training-provider"
                closeButtonTestId="button-cancel-template-training-provider"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {trainingProviderSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : trainingProviderSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = trainingProviderSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = trainingProviderSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(trainingProviderSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={trainingProviderSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {trainingProviderSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsTrainingProviderDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Exam Provider Sections Dialog */}
        <Dialog open={isExamProviderDialogOpen} onOpenChange={setIsExamProviderDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Exam Provider Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Exam Provider page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-exam-provider"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("ExamProv_pqr345stu678");
                    setIsExamProviderDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-exam-provider"
                closeButtonTestId="button-cancel-template-exam-provider"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {examProviderSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : examProviderSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = examProviderSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = examProviderSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(examProviderSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={examProviderSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {examProviderSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsExamProviderDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Corporate/Government Sections Dialog */}
        <Dialog open={isCorpGovDialogOpen} onOpenChange={setIsCorpGovDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Corporate/Government Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Corporate/Government page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-corp-gov"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("CorpGov_stu678vwx901");
                    setIsCorpGovDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-corp-gov"
                closeButtonTestId="button-cancel-template-corp-gov"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {corpGovSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : corpGovSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = corpGovSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = corpGovSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(corpGovSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={corpGovSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {corpGovSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsCorpGovDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Professional IBDL Trainer Sections Dialog */}
        <Dialog open={isProfTrainerDialogOpen} onOpenChange={setIsProfTrainerDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Professional IBDL Trainer Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Professional IBDL Trainer page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-prof-trainer"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("ProfTrainer_vwx901yz234abc");
                    setIsProfTrainerDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-prof-trainer"
                closeButtonTestId="button-cancel-template-prof-trainer"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {profTrainerSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : profTrainerSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = profTrainerSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = profTrainerSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(profTrainerSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={profTrainerSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {profTrainerSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsProfTrainerDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Professional Course Sections Dialog */}
        <Dialog open={isProfCourseDialogOpen} onOpenChange={setIsProfCourseDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Professional Course Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Professional Course page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-prof-course"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("ProfCourse_yz234abc567def");
                    setIsProfCourseDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-prof-course"
                closeButtonTestId="button-cancel-template-prof-course"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {profCourseSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : profCourseSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = profCourseSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = profCourseSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(profCourseSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={profCourseSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {profCourseSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsProfCourseDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Partner With Us Sections Dialog */}
        <Dialog open={isPartnerWithDialogOpen} onOpenChange={setIsPartnerWithDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Partner With Us Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Partner With Us page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-partner-with"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("PartnerWith_abc567def890ghi");
                    setIsPartnerWithDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-partner-with"
                closeButtonTestId="button-cancel-template-partner-with"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {partnerWithSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : partnerWithSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = partnerWithSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = partnerWithSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(partnerWithSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={partnerWithSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {partnerWithSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsPartnerWithDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Join Our Global Community Sections Dialog */}
        <Dialog open={isJoinCommDialogOpen} onOpenChange={setIsJoinCommDialogOpen}>
          <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Join Our Global Community Sections</DialogTitle>
              <DialogDescription>
                Manage content sections for the Join Our Global Community page. Create, edit, and organize your content using different templates.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <Button 
                className="w-full bg-[#ED1C24] hover:bg-red-600 text-white"
                onClick={() => {
                  sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                  setEditingSection(null);
                  setSelectedTemplate(null);
                  setTrusteeFiles({});
                  setIsTemplateSelectionOpen(true);
                }}
                data-testid="button-create-section-join-comm"
              >
                <Plus className="mr-2 h-4 w-4" />
                Create a new Section
              </Button>

              <TemplateSelectionPanel
                isOpen={isTemplateSelectionOpen}
                selectedTemplate={selectedTemplate}
                templateConfig={TEMPLATE_CONFIG}
                onSelectTemplate={setSelectedTemplate}
                onConfirm={() => {
                  if (selectedTemplate) {
                    toast({
                      title: "Template Selected",
                      description: `Creating new ${TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name}...`,
                    });
                    setCurrentPageId("JoinComm_def890ghi123jkl");
                    setIsJoinCommDialogOpen(false);
                    setIsTemplateSelectionOpen(false);
                    setIsSectionEditorOpen(true);
                    setEditingSection(null);
                    sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                    setTrusteeFiles({});
                  }
                }}
                onCancel={() => {
                  setIsTemplateSelectionOpen(false);
                  setSelectedTemplate(null);
                }}
                confirmButtonTestId="button-confirm-template-join-comm"
                closeButtonTestId="button-cancel-template-join-comm"
              />
              
              <div>
                <h3 className="font-semibold text-gray-900 mb-4">Current content sections</h3>
                <div className="space-y-3">
                  {joinCommSectionsLoading ? (
                    <div className="text-center py-8">Loading sections...</div>
                  ) : joinCommSections.length > 0 ? (
                    <DndContext 
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={(event: DragEndEvent) => {
                        const { active, over } = event;
                        
                        if (over && active.id !== over.id) {
                          const oldIndex = joinCommSections.findIndex((section: any) => section.id === active.id);
                          const newIndex = joinCommSections.findIndex((section: any) => section.id === over.id);
                          
                          const reorderedSections = arrayMove(joinCommSections, oldIndex, newIndex);
                          
                          const sectionsWithNewOrder = reorderedSections.map((section: any, index: number) => ({
                            id: section.id,
                            sortOrder: index
                          }));
                          
                          reorderSectionsMutation.mutate(sectionsWithNewOrder);
                        }
                      }}
                    >
                      <SortableContext 
                        items={joinCommSections.map((s: any) => s.id)}
                        strategy={verticalListSortingStrategy}
                      >
                        <div className="space-y-3">
                          {joinCommSections.map((section: any) => (
                            <SortableItem
                              key={section.id}
                              section={section}
                              onEdit={() => {
                                setEditingSection(section);
                                setSelectedTemplate(section.templateType);
                                const formData = serializeSectionDataForForm(section);
                                sectionForm.reset(formData);
                                setIsSectionEditorOpen(true);
                                setIsJoinCommDialogOpen(false);
                              }}
                              onDelete={() => {
                                setSectionToDelete({ id: section.id, title: section.title });
                                setDeleteConfirmOpen(true);
                              }}
                              isUpdatePending={false}
                              isDeletePending={false}
                            />
                          ))}
                        </div>
                      </SortableContext>
                    </DndContext>
                  ) : (
                    <div className="text-center py-8 text-gray-500">
                      No sections created yet. Click "Create a new Section" to get started.
                    </div>
                  )}
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Section Editor Modal */}
        <Dialog open={isSectionEditorOpen} onOpenChange={setIsSectionEditorOpen}>
          <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>
                {editingSection ? 'Edit Section' : `Create ${selectedTemplate ? TEMPLATE_CONFIG[selectedTemplate as keyof typeof TEMPLATE_CONFIG].name : 'Section'}`}
              </DialogTitle>
              <DialogDescription>
                {editingSection ? 'Edit the content and settings for this section.' : 'Configure the content and settings for your new section.'}
              </DialogDescription>
            </DialogHeader>
            
            <Form {...sectionForm}>
              <form onSubmit={sectionForm.handleSubmit(
                (data) => {
                  if (editingSection) {
                    // Update existing section
                    updateSectionMutation.mutate({
                      id: editingSection.id,
                      data: data
                    });
                  } else {
                    // Create new section - add runtime guard
                    if (!selectedTemplate) {
                      toast({
                        title: "Error",
                        description: "Please select a template first.",
                        variant: "destructive",
                      });
                      return;
                    }
                    createSectionMutation.mutate(data);
                  }
                },
                (errors) => {
                  // Form validation failed - show errors to user
                  console.error('Form validation errors:', errors);
                  const errorFields = Object.keys(errors);
                  toast({
                    title: "Validation Error",
                    description: `Please check the following fields: ${errorFields.join(', ')}`,
                    variant: "destructive",
                  });
                }
              )} className="space-y-6">
                
                {/* Dynamic fields based on template type */}
                {(() => {
                  // Determine current template type
                  const currentTemplateType = editingSection?.templateType || selectedTemplate;
                  const templateFields = currentTemplateType && TEMPLATE_FIELD_MAPPING[currentTemplateType as keyof typeof TEMPLATE_FIELD_MAPPING];
                  
                  if (!templateFields) {
                    return (
                      <div className="text-center py-8 text-gray-500">
                        {editingSection ? 'Unknown template type' : 'Please select a template first'}
                      </div>
                    );
                  }

                  return templateFields.fields.map((fieldName: string) => {
                    switch (fieldName) {
                      case 'title':
                        return (
                          <FormField
                            key="title"
                            control={sectionForm.control}
                            name="title"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Section Title</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="Enter section title" data-testid="input-section-title" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'content':
                        return (
                          <FormField
                            key="content"
                            control={sectionForm.control}
                            name="content"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Content</FormLabel>
                                <FormControl>
                                  <Textarea 
                                    {...field} 
                                    placeholder="Enter your content here..." 
                                    rows={6}
                                    data-testid="input-content"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'content_items':
                        return (
                          <FormField
                            key="content_items"
                            control={sectionForm.control}
                            name="content_items"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Content Paragraphs</FormLabel>
                                <FormControl>
                                  {currentTemplateType === 'ceo_message' ? (
                                    <ContentParagraphsEditor
                                      value={field.value || ''}
                                      onChange={field.onChange}
                                    />
                                  ) : (
                                    <NarrativeContentEditor
                                      value={field.value || ''}
                                      onChange={field.onChange}
                                    />
                                  )}
                                </FormControl>
                                <p className="text-sm text-gray-500">Add and edit content paragraphs individually</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'image':
                        return (
                          <FormField
                            key="image"
                            control={sectionForm.control}
                            name="image"
                            render={({ field: { onChange, value, ...field } }) => (
                              <FormItem>
                                <FormLabel>Section Image</FormLabel>
                                <FormControl>
                                  <Input 
                                    {...field}
                                    type="file" 
                                    accept="image/*"
                                    onChange={(e) => {
                                      const files = e.target.files;
                                      if (files) {
                                        onChange(files);
                                      }
                                    }}
                                    data-testid="input-image"
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Upload an image for this section</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'subtitle':
                        return (
                          <FormField
                            key="subtitle"
                            control={sectionForm.control}
                            name="subtitle"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Subtitle</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="Enter subtitle" data-testid="input-subtitle" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'card_content':
                        return (
                          <FormField
                            key="card_content"
                            control={sectionForm.control}
                            name="card_content"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Content Paragraphs</FormLabel>
                                <FormControl>
                                  <ContentWithCardEditor
                                    value={field.value || ''}
                                    onChange={field.onChange}
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Add multiple content paragraphs and optionally position a reddish blue card</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'card_position':
                        // This field is handled within ContentWithCardEditor
                        return null;

                      case 'cta_text':
                        return (
                          <FormField
                            key="cta_text"
                            control={sectionForm.control}
                            name="cta_text"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Call-to-Action Text</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="e.g., Learn More" data-testid="input-cta-text" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'cta_link':
                        return (
                          <FormField
                            key="cta_link"
                            control={sectionForm.control}
                            name="cta_link"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Call-to-Action Link</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="https://..." data-testid="input-cta-link" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'card_title':
                        return (
                          <FormField
                            key="card_title"
                            control={sectionForm.control}
                            name="card_title"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Card Title</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="Enter card title" data-testid="input-card-title" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'card_subtitle':
                        return (
                          <FormField
                            key="card_subtitle"
                            control={sectionForm.control}
                            name="card_subtitle"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Card Subtitle</FormLabel>
                                <FormControl>
                                  <Textarea {...field} placeholder="Enter card subtitle" rows={3} data-testid="input-card-subtitle" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'card_items':
                        return (
                          <FormField
                            key="card_items"
                            control={sectionForm.control}
                            name="card_items"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Card Items</FormLabel>
                                <FormControl>
                                  <CardItemsEditor value={field.value} onChange={field.onChange} />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'mission':
                        return (
                          <FormField
                            key="mission"
                            control={sectionForm.control}
                            name="mission"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Mission Statement</FormLabel>
                                <FormControl>
                                  <Textarea {...field} placeholder="Enter your mission statement" rows={4} data-testid="input-mission" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'vision':
                        return (
                          <FormField
                            key="vision"
                            control={sectionForm.control}
                            name="vision"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Vision Statement</FormLabel>
                                <FormControl>
                                  <Textarea {...field} placeholder="Enter your vision statement" rows={4} data-testid="input-vision" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'mission_label':
                        return (
                          <FormField
                            key="mission_label"
                            control={sectionForm.control}
                            name="mission_label"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Mission Label</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="Our Mission:" data-testid="input-mission-label" />
                                </FormControl>
                                <p className="text-sm text-gray-500">Customize the mission section label (e.g., "Our Mission:", "Mission:")</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'vision_label':
                        return (
                          <FormField
                            key="vision_label"
                            control={sectionForm.control}
                            name="vision_label"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Vision Label</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="Our Vision:" data-testid="input-vision-label" />
                                </FormControl>
                                <p className="text-sm text-gray-500">Customize the vision section label (e.g., "Our Vision:", "Vision:")</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'text_color':
                        return (
                          <FormField
                            key="text_color"
                            control={sectionForm.control}
                            name="text_color"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Text Color</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="#1A1A40" data-testid="input-text-color" />
                                </FormControl>
                                <p className="text-sm text-gray-500">Enter a hex color code (e.g., #1A1A40)</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'background_color':
                        return (
                          <FormField
                            key="background_color"
                            control={sectionForm.control}
                            name="background_color"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Background Color</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="#ffffff" data-testid="input-background-color" />
                                </FormControl>
                                <p className="text-sm text-gray-500">Enter a hex color code (e.g., #ffffff)</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'gradient':
                        return (
                          <FormField
                            key="gradient"
                            control={sectionForm.control}
                            name="gradient"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Gradient</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="linear-gradient(to right, #ED1C24, #1A1A40)" data-testid="input-gradient" />
                                </FormControl>
                                <p className="text-sm text-gray-500">Enter a CSS gradient (e.g., linear-gradient(to right, #ED1C24, #1A1A40))</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'background_image':
                        return (
                          <FormField
                            key="background_image"
                            control={sectionForm.control}
                            name="background_image"
                            render={({ field: { onChange, value, ...field } }) => (
                              <FormItem>
                                <FormLabel>Background Image</FormLabel>
                                <FormControl>
                                  <Input 
                                    {...field}
                                    type="file" 
                                    accept="image/*"
                                    onChange={(e) => {
                                      const files = e.target.files;
                                      if (files) {
                                        onChange(files);
                                      }
                                    }}
                                    data-testid="input-background-image"
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Upload a background image</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'images':
                        return (
                          <FormField
                            key="images"
                            control={sectionForm.control}
                            name="images"
                            render={({ field: { onChange, value, ...field } }) => (
                              <FormItem>
                                <FormLabel>Gallery Images</FormLabel>
                                <FormControl>
                                  <Input 
                                    {...field}
                                    type="file" 
                                    accept="image/*"
                                    multiple
                                    onChange={(e) => {
                                      const files = e.target.files;
                                      if (files) {
                                        onChange(files);
                                      }
                                    }}
                                    data-testid="input-images"
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Upload multiple images for the gallery</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'gallery_layout':
                        return (
                          <FormField
                            key="gallery_layout"
                            control={sectionForm.control}
                            name="gallery_layout"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Gallery Layout</FormLabel>
                                <FormControl>
                                  <select {...field} className="w-full p-2 border rounded-md" data-testid="input-gallery-layout">
                                    <option value="grid">Grid Layout</option>
                                    <option value="masonry">Masonry Layout</option>
                                    <option value="carousel">Carousel Layout</option>
                                  </select>
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'quote':
                        return (
                          <FormField
                            key="quote"
                            control={sectionForm.control}
                            name="quote"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Quote</FormLabel>
                                <FormControl>
                                  <Textarea {...field} placeholder="Enter inspirational quote" rows={3} data-testid="input-quote" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'ceo_name':
                        return (
                          <FormField
                            key="ceo_name"
                            control={sectionForm.control}
                            name="ceo_name"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>CEO Name</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="Enter CEO name" data-testid="input-ceo-name" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'ceo_title':
                        return (
                          <FormField
                            key="ceo_title"
                            control={sectionForm.control}
                            name="ceo_title"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>CEO Title</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="Chief Executive Officer" data-testid="input-ceo-title" />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'ceo_image':
                        return (
                          <FormField
                            key="ceo_image"
                            control={sectionForm.control}
                            name="ceo_image"
                            render={({ field: { onChange, value, ...field } }) => (
                              <FormItem>
                                <FormLabel>CEO Image</FormLabel>
                                <FormControl>
                                  <Input 
                                    {...field}
                                    type="file" 
                                    accept="image/*"
                                    onChange={(e) => {
                                      const files = e.target.files;
                                      if (files) {
                                        onChange(files);
                                      }
                                    }}
                                    data-testid="input-ceo-image"
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Upload CEO profile image</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'main_icon':
                        return (
                          <FormField
                            key="main_icon"
                            control={sectionForm.control}
                            name="main_icon"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Main Icon</FormLabel>
                                <FormControl>
                                  <Select value={field.value || ''} onValueChange={field.onChange}>
                                    <SelectTrigger data-testid="select-main-icon">
                                      <SelectValue placeholder="Select an icon" />
                                    </SelectTrigger>
                                    <SelectContent>
                                      {['Scale', 'Users', 'Building', 'Target', 'Award', 'Shield', 'Heart', 'Globe', 'Lock', 'Star', 'CheckCircle', 'Compass', 'Eye', 'Handshake', 'Lightbulb', 'TrendingUp', 'BookOpen', 'Zap', 'Settings', 'Crown', 'Rocket', 'DollarSign'].map((iconName) => (
                                        <SelectItem key={iconName} value={iconName}>
                                          {iconName}
                                        </SelectItem>
                                      ))}
                                    </SelectContent>
                                  </Select>
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'numbered_cards':
                        return (
                          <FormField
                            key="numbered_cards"
                            control={sectionForm.control}
                            name="numbered_cards"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Numbered Cards (Optional)</FormLabel>
                                <FormControl>
                                  <NumberedCardsEditor
                                    value={field.value || ''}
                                    onChange={field.onChange}
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Add numbered cards like Core Learning Modules section</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'principle_items':
                        return (
                          <FormField
                            key="principle_items"
                            control={sectionForm.control}
                            name="principle_items"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Principle Items</FormLabel>
                                <FormControl>
                                  <PrincipleItemsEditor 
                                    value={field.value} 
                                    onChange={field.onChange}
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Create governance principle cards with icons, titles, and detailed descriptions</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'governance_structure':
                        return (
                          <FormField
                            key="governance_structure"
                            control={sectionForm.control}
                            name="governance_structure"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Governance Structure Text</FormLabel>
                                <FormControl>
                                  <Textarea 
                                    {...field} 
                                    placeholder="Our governance structure ensures a clear separation of responsibilities between oversight (Board of Trustees), execution (Executive Leadership), and operational management—reflecting a balanced, internationally recognized model."
                                    rows={4}
                                    data-testid="input-governance-structure" 
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Text for the governance structure box that appears at the bottom of the principles section</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'trustee_profiles':
                        return (
                          <FormField
                            key="trustee_profiles"
                            control={sectionForm.control}
                            name="trustee_profiles"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Trustee Profiles</FormLabel>
                                <FormControl>
                                  <TrusteeProfilesEditor 
                                    value={field.value} 
                                    onChange={field.onChange}
                                    onFilesChange={setTrusteeFiles}
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Create trustee profile cards with photos, names, titles, descriptions, and social links</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'table_title':
                        return (
                          <FormField
                            key="table_title"
                            control={sectionForm.control}
                            name="table_title"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Table Title</FormLabel>
                                <FormControl>
                                  <Input {...field} placeholder="e.g., Impact by the Numbers" data-testid="input-table-title" />
                                </FormControl>
                                <p className="text-sm text-gray-500">Title that appears above the statistics table</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'table_icon':
                        return (
                          <FormField
                            key="table_icon"
                            control={sectionForm.control}
                            name="table_icon"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Table Icon</FormLabel>
                                <FormControl>
                                  <Select onValueChange={field.onChange} value={field.value || ''}>
                                    <SelectTrigger data-testid="select-table-icon">
                                      <SelectValue placeholder="Select an icon for the table" />
                                    </SelectTrigger>
                                    <SelectContent>
                                      <SelectItem value="BarChart3">📊 Bar Chart</SelectItem>
                                      <SelectItem value="TrendingUp">📈 Trending Up</SelectItem>
                                      <SelectItem value="Target">🎯 Target</SelectItem>
                                      <SelectItem value="Award">🏆 Award</SelectItem>
                                      <SelectItem value="Users">👥 Users</SelectItem>
                                      <SelectItem value="Globe">🌍 Globe</SelectItem>
                                      <SelectItem value="Building">🏢 Building</SelectItem>
                                      <SelectItem value="Star">⭐ Star</SelectItem>
                                      <SelectItem value="CheckCircle">✅ Check Circle</SelectItem>
                                    </SelectContent>
                                  </Select>
                                </FormControl>
                                <p className="text-sm text-gray-500">Icon that appears next to the table title</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'metrics_data':
                        return (
                          <FormField
                            key="metrics_data"
                            control={sectionForm.control}
                            name="metrics_data"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Impact Metrics</FormLabel>
                                <FormControl>
                                  <MetricsDataEditor
                                    value={field.value || ''}
                                    onChange={field.onChange}
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Add and edit impact statistics individually</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      case 'matter_items':
                        return (
                          <FormField
                            key="matter_items"
                            control={sectionForm.control}
                            name="matter_items"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel>Why Numbers Matter</FormLabel>
                                <FormControl>
                                  <MatterItemsEditor
                                    value={field.value || ''}
                                    onChange={field.onChange}
                                  />
                                </FormControl>
                                <p className="text-sm text-gray-500">Add and edit reasons why these statistics matter</p>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        );

                      default:
                        return null;
                    }
                  });
                })()}

                {/* Form Actions */}
                <div className="flex justify-end space-x-3 pt-4">
                  <Button 
                    type="button" 
                    variant="outline"
                    onClick={() => {
                      setIsSectionEditorOpen(false);
                      setSelectedTemplate(null);
                      setEditingSection(null);
                      sectionForm.reset(EMPTY_SECTION_VALUES, { keepDefaultValues: false });
                      setTrusteeFiles({});
                    }}
                    data-testid="button-cancel-section-editor"
                  >
                    Cancel
                  </Button>
                  <Button 
                    type="submit"
                    className="bg-[#ED1C24] hover:bg-red-600 text-white"
                    disabled={createSectionMutation.isPending || updateSectionMutation.isPending}
                    data-testid="button-save-section"
                  >
                    {createSectionMutation.isPending || updateSectionMutation.isPending ? 'Saving...' : (editingSection ? 'Update Section' : 'Create Section')}
                  </Button>
                </div>
              </form>
            </Form>
          </DialogContent>
        </Dialog>

        {/* Delete Confirmation Modal */}
        <Dialog open={deleteConfirmOpen} onOpenChange={setDeleteConfirmOpen}>
          <DialogContent className="sm:max-w-[425px]">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-3">
                <div className="w-12 h-12 rounded-full bg-red-100 dark:bg-red-900/20 flex items-center justify-center">
                  <Trash2 className="h-6 w-6 text-red-600 dark:text-red-400" />
                </div>
                <span>Delete Section</span>
              </DialogTitle>
              <DialogDescription className="pt-2">
                This action cannot be undone. This will permanently delete the section from your page.
              </DialogDescription>
            </DialogHeader>
            
            <div className="py-4">
              <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-4 border">
                <p className="text-sm text-gray-600 dark:text-gray-400 mb-1">Section to delete:</p>
                <p className="font-medium text-gray-900 dark:text-gray-100">
                  {sectionToDelete?.title || 'Untitled Section'}
                </p>
              </div>
            </div>

            <div className="flex justify-end space-x-3">
              <Button
                variant="outline"
                onClick={() => {
                  setDeleteConfirmOpen(false);
                  setSectionToDelete(null);
                }}
                disabled={deleteSectionMutation.isPending}
                data-testid="button-cancel-delete"
              >
                Cancel
              </Button>
              <Button
                variant="destructive"
                onClick={handleConfirmDelete}
                disabled={deleteSectionMutation.isPending}
                data-testid="button-confirm-delete"
                className="bg-red-600 hover:bg-red-700 text-white"
              >
                {deleteSectionMutation.isPending ? (
                  <>
                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                    Deleting...
                  </>
                ) : (
                  <>
                    <Trash2 className="h-4 w-4 mr-2" />
                    Delete Section
                  </>
                )}
              </Button>
            </div>
          </DialogContent>
        </Dialog>
      </div>
    </div>
  );
}