import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { 
  Plus, 
  Edit, 
  Trash2, 
  BarChart3, 
  FileText, 
  Brain, 
  Award,
  Users,
  Calendar,
  CheckCircle,
  AlertCircle
} from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { z } from "zod";

const assessmentFormSchema = z.object({
  title: z.string().min(1, "Title is required"),
  description: z.string().min(1, "Description is required"),
  type: z.enum(["certification", "psychometric", "skills", "personality"]),
  certificateId: z.string().optional(),
  duration: z.number().min(1, "Duration must be at least 1 minute"),
  questionCount: z.number().min(1, "Must have at least 1 question"),
  passingScore: z.number().min(0).max(100).optional(),
  isProctored: z.boolean().default(false),
  allowRetakes: z.boolean().default(true),
  maxAttempts: z.number().min(1).max(10),
  tags: z.array(z.string()).default([]),
  dimensions: z.array(z.string()).optional(),
  settings: z.object({
    timeLimit: z.number().min(1),
    randomizeQuestions: z.boolean().default(false),
    allowReview: z.boolean().default(true),
    showResults: z.boolean().default(true),
    isAdaptive: z.boolean().default(false)
  }).optional(),
});

type AssessmentFormData = z.infer<typeof assessmentFormSchema>;

export default function Assessments() {
  const [activeTab, setActiveTab] = useState("creation");
  const [createDialogOpen, setCreateDialogOpen] = useState(false);
  const [editingAssessment, setEditingAssessment] = useState<any>(null);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch data
  const { data: assessments = [], isLoading: loadingAssessments } = useQuery({
    queryKey: ["backend/api/assessments"],
  });

  const { data: assessmentAttempts = [], isLoading: loadingAttempts } = useQuery({
    queryKey: ["backend/api/assessment-attempts"],
  });

  const { data: certificates = [] } = useQuery({
    queryKey: ["backend/api/system-certificates"],
  });

  // Assessment form
  const assessmentForm = useForm<AssessmentFormData>({
    resolver: zodResolver(assessmentFormSchema),
    defaultValues: {
      title: "",
      description: "",
      type: "certification",
      certificateId: "",
      duration: 60,
      questionCount: 50,
      passingScore: 70,
      isProctored: false,
      allowRetakes: true,
      maxAttempts: 3,
      tags: [],
      dimensions: [],
      settings: {
        timeLimit: 60,
        randomizeQuestions: false,
        allowReview: true,
        showResults: true,
        isAdaptive: false
      }
    },
  });

  const watchedType = assessmentForm.watch("type");

  // Mutations
  const createAssessmentMutation = useMutation({
    mutationFn: (data: AssessmentFormData) => apiRequest("backend/api/assessments", "POST", data),
    onSuccess: () => {
      toast({ title: "Assessment created successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/assessments"] });
      assessmentForm.reset();
      setCreateDialogOpen(false);
    },
    onError: (error) => {
      toast({ title: "Error creating assessment", description: error.message, variant: "destructive" });
    },
  });

  const updateAssessmentMutation = useMutation({
    mutationFn: ({ id, data }: { id: string; data: Partial<AssessmentFormData> }) => 
      apiRequest(`backend/api/assessments/${id}`, "PATCH", data),
    onSuccess: () => {
      toast({ title: "Assessment updated successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/assessments"] });
      setEditingAssessment(null);
    },
    onError: (error) => {
      toast({ title: "Error updating assessment", description: error.message, variant: "destructive" });
    },
  });

  const deleteAssessmentMutation = useMutation({
    mutationFn: (id: string) => apiRequest(`backend/api/assessments/${id}`, "DELETE"),
    onSuccess: () => {
      toast({ title: "Assessment deleted successfully" });
      queryClient.invalidateQueries({ queryKey: ["backend/api/assessments"] });
    },
    onError: (error) => {
      toast({ title: "Error deleting assessment", description: error.message, variant: "destructive" });
    },
  });

  const handleCreateAssessment = (data: AssessmentFormData) => {
    createAssessmentMutation.mutate(data);
  };

  const handleEditAssessment = (assessment: any) => {
    setEditingAssessment(assessment);
    assessmentForm.reset({
      title: assessment.title,
      description: assessment.description,
      type: assessment.type,
      certificateId: assessment.certificateId || "",
      duration: assessment.duration,
      questionCount: assessment.questionCount,
      passingScore: assessment.passingScore,
      isProctored: assessment.isProctored,
      allowRetakes: assessment.allowRetakes,
      maxAttempts: assessment.maxAttempts,
      tags: assessment.tags || [],
      dimensions: assessment.dimensions || [],
      settings: assessment.settings || {
        timeLimit: assessment.duration,
        randomizeQuestions: false,
        allowReview: true,
        showResults: true,
        isAdaptive: false
      }
    });
    setCreateDialogOpen(true);
  };

  const handleDeleteAssessment = (id: string) => {
    if (confirm("Are you sure you want to delete this assessment?")) {
      deleteAssessmentMutation.mutate(id);
    }
  };

  const formatTypeLabel = (type: string) => {
    switch (type) {
      case "certification": return "Certification Exam";
      case "psychometric": return "Psychometric Assessment";
      case "skills": return "Skills Evaluation";
      case "personality": return "Personality Test";
      default: return type;
    }
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case "certification": return <Award className="w-4 h-4" />;
      case "psychometric": return <Brain className="w-4 h-4" />;
      case "skills": return <CheckCircle className="w-4 h-4" />;
      case "personality": return <Users className="w-4 h-4" />;
      default: return <FileText className="w-4 h-4" />;
    }
  };

  const getTypeColor = (type: string) => {
    switch (type) {
      case "certification": return "bg-blue-100 text-blue-800";
      case "psychometric": return "bg-purple-100 text-purple-800";
      case "skills": return "bg-green-100 text-green-800";
      case "personality": return "bg-orange-100 text-orange-800";
      default: return "bg-gray-100 text-gray-800";
    }
  };

  return (
    <DashboardLayout userType="admin">
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">Assessments Management</h1>
            <p className="text-muted-foreground">Create and manage assessments for certification and evaluation</p>
          </div>
          <Dialog open={createDialogOpen} onOpenChange={setCreateDialogOpen}>
            <DialogTrigger asChild>
              <Button onClick={() => { setEditingAssessment(null); assessmentForm.reset(); }}>
                <Plus className="w-4 h-4 mr-2" />
                Create Assessment
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
              <DialogHeader>
                <DialogTitle>
                  {editingAssessment ? "Edit Assessment" : "Create New Assessment"}
                </DialogTitle>
              </DialogHeader>
              <Form {...assessmentForm}>
                <form onSubmit={assessmentForm.handleSubmit(handleCreateAssessment)} className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <FormField
                      control={assessmentForm.control}
                      name="title"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Assessment Title</FormLabel>
                          <FormControl>
                            <Input placeholder="Enter assessment title" {...field} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={assessmentForm.control}
                      name="type"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Assessment Type</FormLabel>
                          <Select onValueChange={field.onChange} defaultValue={field.value}>
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select type" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              <SelectItem value="certification">Certification Exam</SelectItem>
                              <SelectItem value="psychometric">Psychometric Assessment</SelectItem>
                              <SelectItem value="skills">Skills Evaluation</SelectItem>
                              <SelectItem value="personality">Personality Test</SelectItem>
                            </SelectContent>
                          </Select>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>

                  <FormField
                    control={assessmentForm.control}
                    name="description"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Description</FormLabel>
                        <FormControl>
                          <Textarea placeholder="Enter assessment description" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <FormField
                      control={assessmentForm.control}
                      name="duration"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Duration (minutes)</FormLabel>
                          <FormControl>
                            <Input type="number" {...field} onChange={(e) => field.onChange(parseInt(e.target.value))} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={assessmentForm.control}
                      name="questionCount"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Number of Questions</FormLabel>
                          <FormControl>
                            <Input type="number" {...field} onChange={(e) => field.onChange(parseInt(e.target.value))} />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                    {watchedType !== "personality" && (
                      <FormField
                        control={assessmentForm.control}
                        name="passingScore"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Passing Score (%)</FormLabel>
                            <FormControl>
                              <Input type="number" {...field} onChange={(e) => field.onChange(parseInt(e.target.value))} />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    )}
                  </div>

                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <FormField
                      control={assessmentForm.control}
                      name="isProctored"
                      render={({ field }) => (
                        <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3">
                          <div className="space-y-0.5">
                            <FormLabel>Proctored Assessment</FormLabel>
                            <div className="text-sm text-muted-foreground">
                              Requires supervision during assessment
                            </div>
                          </div>
                          <FormControl>
                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                    <FormField
                      control={assessmentForm.control}
                      name="allowRetakes"
                      render={({ field }) => (
                        <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3">
                          <div className="space-y-0.5">
                            <FormLabel>Allow Retakes</FormLabel>
                            <div className="text-sm text-muted-foreground">
                              Allow multiple attempts
                            </div>
                          </div>
                          <FormControl>
                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </div>

                  <FormField
                    control={assessmentForm.control}
                    name="maxAttempts"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Maximum Attempts</FormLabel>
                        <FormControl>
                          <Input type="number" min="1" max="10" {...field} onChange={(e) => field.onChange(parseInt(e.target.value))} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  {(watchedType === "psychometric" || watchedType === "personality") && (
                    <FormField
                      control={assessmentForm.control}
                      name="dimensions"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Assessment Dimensions</FormLabel>
                          <FormControl>
                            <Textarea 
                              placeholder="Enter dimensions separated by commas (e.g., Leadership, Communication, Problem Solving)"
                              value={field.value?.join(", ") || ""}
                              onChange={(e) => field.onChange(e.target.value.split(",").map(d => d.trim()).filter(d => d))}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  )}

                  <div className="flex justify-end space-x-2">
                    <Button type="button" variant="outline" onClick={() => setCreateDialogOpen(false)}>
                      Cancel
                    </Button>
                    <Button type="submit" disabled={createAssessmentMutation.isPending}>
                      {createAssessmentMutation.isPending ? "Creating..." : editingAssessment ? "Update Assessment" : "Create Assessment"}
                    </Button>
                  </div>
                </form>
              </Form>
            </DialogContent>
          </Dialog>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="creation">Assessment Creation</TabsTrigger>
            <TabsTrigger value="reports">Reports & Results</TabsTrigger>
          </TabsList>

          <TabsContent value="creation" className="space-y-4">
            <div className="grid gap-4">
              {loadingAssessments ? (
                <div className="grid gap-4">
                  {[...Array(3)].map((_, i) => (
                    <Card key={i} className="animate-pulse">
                      <CardContent className="p-6">
                        <div className="h-4 bg-gray-200 rounded w-3/4 mb-2"></div>
                        <div className="h-3 bg-gray-200 rounded w-1/2"></div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              ) : assessments.length === 0 ? (
                <Card>
                  <CardContent className="text-center py-8">
                    <FileText className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                    <h3 className="text-lg font-semibold mb-2">No Assessments Created</h3>
                    <p className="text-muted-foreground mb-4">Create your first assessment to get started</p>
                    <Button onClick={() => setCreateDialogOpen(true)}>
                      <Plus className="w-4 h-4 mr-2" />
                      Create Assessment
                    </Button>
                  </CardContent>
                </Card>
              ) : (
                assessments.map((assessment: any) => (
                  <Card key={assessment.id}>
                    <CardHeader>
                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-3">
                          {getTypeIcon(assessment.type)}
                          <div>
                            <CardTitle className="text-lg">{assessment.title}</CardTitle>
                            <CardDescription>{assessment.description}</CardDescription>
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          <Badge className={getTypeColor(assessment.type)}>
                            {formatTypeLabel(assessment.type)}
                          </Badge>
                          <Button variant="ghost" size="sm" onClick={() => handleEditAssessment(assessment)}>
                            <Edit className="w-4 h-4" />
                          </Button>
                          <Button variant="ghost" size="sm" onClick={() => handleDeleteAssessment(assessment.id)}>
                            <Trash2 className="w-4 h-4" />
                          </Button>
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent>
                      <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                        <div>
                          <span className="text-muted-foreground">Duration:</span>
                          <div className="font-medium">{assessment.duration} minutes</div>
                        </div>
                        <div>
                          <span className="text-muted-foreground">Questions:</span>
                          <div className="font-medium">{assessment.questionCount}</div>
                        </div>
                        <div>
                          <span className="text-muted-foreground">Passing Score:</span>
                          <div className="font-medium">{assessment.passingScore || "N/A"}%</div>
                        </div>
                        <div>
                          <span className="text-muted-foreground">Attempts:</span>
                          <div className="font-medium">{assessment.attemptsCount || 0}</div>
                        </div>
                      </div>
                      {assessment.isProctored && (
                        <div className="mt-3">
                          <Badge variant="secondary">
                            <AlertCircle className="w-3 h-3 mr-1" />
                            Proctored
                          </Badge>
                        </div>
                      )}
                    </CardContent>
                  </Card>
                ))
              )}
            </div>
          </TabsContent>

          <TabsContent value="reports" className="space-y-4">
            <div className="grid gap-4">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <BarChart3 className="w-5 h-5" />
                    Assessment Performance Overview
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <div className="text-center">
                      <div className="text-2xl font-bold text-blue-600">{assessments.length}</div>
                      <div className="text-sm text-muted-foreground">Total Assessments</div>
                    </div>
                    <div className="text-center">
                      <div className="text-2xl font-bold text-green-600">{assessmentAttempts.length}</div>
                      <div className="text-sm text-muted-foreground">Total Attempts</div>
                    </div>
                    <div className="text-center">
                      <div className="text-2xl font-bold text-purple-600">
                        {assessmentAttempts.filter((attempt: any) => attempt.status === "completed").length}
                      </div>
                      <div className="text-sm text-muted-foreground">Completed</div>
                    </div>
                    <div className="text-center">
                      <div className="text-2xl font-bold text-orange-600">
                        {Math.round((assessmentAttempts.filter((attempt: any) => attempt.score >= 70).length / assessmentAttempts.length) * 100) || 0}%
                      </div>
                      <div className="text-sm text-muted-foreground">Pass Rate</div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Recent Assessment Results</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {assessmentAttempts.slice(0, 10).map((attempt: any) => (
                      <div key={attempt.id} className="flex items-center justify-between p-3 rounded-lg border">
                        <div className="flex items-center gap-3">
                          <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                          <div>
                            <div className="font-medium">Assessment #{attempt.assessmentId}</div>
                            <div className="text-sm text-muted-foreground">User: {attempt.userId}</div>
                          </div>
                        </div>
                        <div className="text-right">
                          <div className="font-medium">{attempt.score}%</div>
                          <div className="text-sm text-muted-foreground">
                            {attempt.status === "completed" ? "Completed" : "In Progress"}
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>
        </Tabs>
      </div>
    </DashboardLayout>
  );
}