import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { CheckCircle, Clock, AlertCircle, FileText, Users, Building } from "lucide-react";
import { WebsiteLayout } from "@/components/WebsiteLayout";
import { useAuth } from "@/hooks/useAuth";

interface ApplicationStatus {
  id: string;
  organizationName: string;
  applicationType: string;
  status: string;
  qaCheckpoint1: boolean;
  qaCheckpoint2: boolean;
  qaCheckpoint3: boolean;
  submittedAt: string;
  reviewComments?: string;
}

export default function ApplicantDashboard() {
  const { user } = useAuth();
  
  const { data: application, isLoading } = useQuery<ApplicationStatus>({
    queryKey: ["backend/api/accreditation/my-application"],
    enabled: !!user,
  });

  const getStatusInfo = (status: string) => {
    switch (status) {
      case "pending":
        return { 
          label: "Application Submitted", 
          color: "bg-blue-100 text-blue-800", 
          icon: Clock,
          description: "Your application has been received and is in the queue for review."
        };
      case "under_review":
        return { 
          label: "Under Review", 
          color: "bg-yellow-100 text-yellow-800", 
          icon: FileText,
          description: "Our QA team is currently reviewing your application."
        };
      case "pending_approval_1":
        return { 
          label: "QA Review - Level 1", 
          color: "bg-orange-100 text-orange-800", 
          icon: Users,
          description: "Initial quality assurance review in progress."
        };
      case "pending_approval_2":
        return { 
          label: "QA Review - Level 2", 
          color: "bg-orange-100 text-orange-800", 
          icon: Users,
          description: "Secondary quality assurance review in progress."
        };
      case "pending_approval_3":
        return { 
          label: "Final Review", 
          color: "bg-purple-100 text-purple-800", 
          icon: Building,
          description: "Final approval stage - almost complete!"
        };
      case "approved":
        return { 
          label: "Approved", 
          color: "bg-green-100 text-green-800", 
          icon: CheckCircle,
          description: "Congratulations! Your application has been approved."
        };
      case "rejected":
        return { 
          label: "Requires Revision", 
          color: "bg-red-100 text-red-800", 
          icon: AlertCircle,
          description: "Please review the feedback and resubmit your application."
        };
      default:
        return { 
          label: "Unknown Status", 
          color: "bg-gray-100 text-gray-800", 
          icon: Clock,
          description: "Status information is being updated."
        };
    }
  };

  const calculateProgress = (qaCheckpoint1: boolean, qaCheckpoint2: boolean, qaCheckpoint3: boolean, status: string) => {
    if (status === "approved") return 100;
    if (status === "rejected") return 0;
    
    let progress = 20; // Base progress for submission
    if (qaCheckpoint1) progress += 25;
    if (qaCheckpoint2) progress += 25;
    if (qaCheckpoint3) progress += 30;
    
    return Math.min(progress, 100);
  };

  const getQAStepStatus = (checkpoint: boolean, stepNumber: number, currentStatus: string) => {
    if (checkpoint) return "completed";
    if (currentStatus === "rejected") return "failed";
    
    // Determine if this step is currently active
    const isActive = (stepNumber === 1 && currentStatus.includes("approval_1")) ||
                    (stepNumber === 2 && currentStatus.includes("approval_2")) ||
                    (stepNumber === 3 && currentStatus.includes("approval_3"));
    
    return isActive ? "active" : "pending";
  };

  if (isLoading) {
    return (
      <WebsiteLayout>
        <div className="min-h-screen bg-gray-50 py-12">
          <div className="max-w-4xl mx-auto px-4">
            <div className="animate-pulse">
              <div className="h-8 bg-gray-200 rounded w-1/3 mb-6"></div>
              <div className="space-y-4">
                <div className="h-32 bg-gray-200 rounded"></div>
                <div className="h-48 bg-gray-200 rounded"></div>
              </div>
            </div>
          </div>
        </div>
      </WebsiteLayout>
    );
  }

  if (!application) {
    return (
      <WebsiteLayout>
        <div className="min-h-screen bg-gray-50 py-12">
          <div className="max-w-4xl mx-auto px-4">
            <div className="text-center mb-8">
              <h1 className="text-3xl font-bold text-gray-900 mb-2">Accreditation Application Status</h1>
              <p className="text-gray-600">Track your application progress</p>
            </div>
            <Card>
              <CardContent className="pt-6 text-center">
                <FileText className="h-16 w-16 text-gray-400 mx-auto mb-4" />
                <h2 className="text-xl font-semibold text-gray-900 mb-2">No Application Found</h2>
                <p className="text-gray-600">
                  You haven't submitted an application yet, or it may not be linked to your account.
                </p>
              </CardContent>
            </Card>
          </div>
        </div>
      </WebsiteLayout>
    );
  }

  const statusInfo = getStatusInfo(application.status);
  const progress = calculateProgress(application.qaCheckpoint1, application.qaCheckpoint2, application.qaCheckpoint3, application.status);
  const StatusIcon = statusInfo.icon;

  return (
    <WebsiteLayout>
      <div className="min-h-screen bg-gray-50 py-12">
        <div className="max-w-4xl mx-auto px-4">
          <div className="mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Application Dashboard</h1>
            <p className="text-gray-600">Track the progress of your accreditation application</p>
          </div>

          {/* Application Overview */}
          <Card className="mb-8">
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle className="text-xl">{application.organizationName}</CardTitle>
                  <p className="text-gray-600 capitalize">{application.applicationType.replace('_', ' ')} Application</p>
                </div>
                <Badge className={statusInfo.color}>
                  <StatusIcon className="h-4 w-4 mr-1" />
                  {statusInfo.label}
                </Badge>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <div className="flex justify-between text-sm mb-2">
                    <span>Overall Progress</span>
                    <span>{progress}%</span>
                  </div>
                  <Progress value={progress} className="h-3" />
                </div>
                <p className="text-gray-600">{statusInfo.description}</p>
                <div className="text-sm text-gray-500">
                  Submitted: {new Date(application.submittedAt).toLocaleDateString()}
                </div>
              </div>
            </CardContent>
          </Card>

          {/* QA Progress Tracker */}
          <Card className="mb-8">
            <CardHeader>
              <CardTitle>QA Review Progress</CardTitle>
              <p className="text-gray-600">Track the quality assurance review of your application</p>
            </CardHeader>
            <CardContent>
              <div className="space-y-6">
                {[
                  { step: 1, title: "Document Verification", description: "Verification of submitted documents and basic information" },
                  { step: 2, title: "Facility Assessment", description: "Evaluation of training facilities and infrastructure" },
                  { step: 3, title: "Final Approval Ready", description: "Final review and approval preparation" }
                ].map((qa) => {
                  const isCheckpoint1 = qa.step === 1;
                  const isCheckpoint2 = qa.step === 2;
                  const isCheckpoint3 = qa.step === 3;
                  
                  const checkpointValue = isCheckpoint1 ? application.qaCheckpoint1 : 
                                        isCheckpoint2 ? application.qaCheckpoint2 : application.qaCheckpoint3;
                  const stepStatus = getQAStepStatus(checkpointValue, qa.step, application.status);
                  
                  return (
                    <div key={qa.step} className="flex items-start space-x-4">
                      <div className={`flex-shrink-0 w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium
                        ${stepStatus === "completed" ? "bg-green-100 text-green-700" :
                          stepStatus === "active" ? "bg-blue-100 text-blue-700" :
                          stepStatus === "failed" ? "bg-red-100 text-red-700" :
                          "bg-gray-100 text-gray-500"}`}>
                        {stepStatus === "completed" ? (
                          <CheckCircle className="h-5 w-5" />
                        ) : stepStatus === "failed" ? (
                          <AlertCircle className="h-5 w-5" />
                        ) : stepStatus === "active" ? (
                          <Clock className="h-5 w-5" />
                        ) : (
                          qa.step
                        )}
                      </div>
                      <div className="flex-1">
                        <div className="flex items-center space-x-2">
                          <h3 className="font-medium text-gray-900">{qa.title}</h3>
                          {stepStatus === "completed" && (
                            <Badge className="bg-green-100 text-green-800 text-xs">Completed</Badge>
                          )}
                          {stepStatus === "active" && (
                            <Badge className="bg-blue-100 text-blue-800 text-xs">In Progress</Badge>
                          )}
                          {stepStatus === "failed" && (
                            <Badge className="bg-red-100 text-red-800 text-xs">Needs Attention</Badge>
                          )}
                        </div>
                        <p className="text-sm text-gray-600 mt-1">{qa.description}</p>
                      </div>
                    </div>
                  );
                })}
              </div>
            </CardContent>
          </Card>

          {/* Review Comments */}
          {application.reviewComments && (
            <Card>
              <CardHeader>
                <CardTitle>Review Comments</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="bg-gray-50 p-4 rounded-lg">
                  <p className="text-gray-700">{application.reviewComments}</p>
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </WebsiteLayout>
  );
}