import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { Globe, Settings, ArrowRight, Eye, EyeOff, Home } from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import ibdlLogo from "@assets/IBDL Icon_1751318336054.png";

export default function AdminLogin() {
  const [, setLocation] = useLocation();
  const [username, setUsername] = useState("");
  const [password, setPassword] = useState("");
  const [showPassword, setShowPassword] = useState(false);
  const { toast } = useToast();

  const loginMutation = useMutation({
    mutationFn: async (credentials: { username: string; password: string }) => {
      return await apiRequest("backend/api/auth/login", "POST", credentials);
    },
    onSuccess: (data) => {
      // Check if user is website_admin
      if (data.user.userType !== 'website_admin') {
        toast({
          title: "Access denied",
          description: "This portal is for website administrators only",
          variant: "destructive",
        });
        return;
      }

      toast({
        title: "Login successful",
        description: `Welcome back, ${data.user.firstName}!`,
      });

      // Redirect to CMS admin dashboard
      window.location.href = "/cms";
    },
    onError: (error) => {
      toast({
        title: "Login failed",
        description: error.message || "Invalid credentials. Please try again.",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!username || !password) {
      toast({
        title: "Error",
        description: "Please enter both username and password.",
        variant: "destructive",
      });
      return;
    }
    loginMutation.mutate({ username, password });
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-[#1A1A40] via-[#2A2A50] to-[#4A90E2] flex items-center justify-center p-4">
      <div className="w-full max-w-md">
        <Card className="shadow-2xl border-0">
          <CardHeader className="space-y-1 text-center">
            <div className="flex items-center justify-center w-16 h-16 bg-[#4A90E2] rounded-full mx-auto mb-4">
              <Globe className="h-8 w-8 text-white" />
            </div>
            <CardTitle className="text-2xl font-bold text-[#1A1A40]">
              Website Admin Access
            </CardTitle>
            <p className="text-gray-600">
              Content management and website administration portal
            </p>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="username" className="text-[#1A1A40] font-medium">
                  Username
                </Label>
                <Input
                  id="username"
                  type="text"
                  placeholder="Enter your admin username"
                  value={username}
                  onChange={(e) => setUsername(e.target.value)}
                  className="border-gray-300 focus:border-[#4A90E2] focus:ring-[#4A90E2]"
                  disabled={loginMutation.isPending}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="password" className="text-[#1A1A40] font-medium">
                  Password
                </Label>
                <div className="relative">
                  <Input
                    id="password"
                    type={showPassword ? "text" : "password"}
                    placeholder="Enter your admin password"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    className="border-gray-300 focus:border-[#4A90E2] focus:ring-[#4A90E2] pr-10"
                    disabled={loginMutation.isPending}
                  />
                  <Button
                    type="button"
                    variant="ghost"
                    size="sm"
                    className="absolute right-0 top-0 h-full px-3 py-2 hover:bg-transparent"
                    onClick={() => setShowPassword(!showPassword)}
                  >
                    {showPassword ? (
                      <EyeOff className="h-4 w-4 text-gray-400" />
                    ) : (
                      <Eye className="h-4 w-4 text-gray-400" />
                    )}
                  </Button>
                </div>
              </div>
              <Button
                type="submit"
                className="w-full bg-[#4A90E2] hover:bg-blue-700 text-white"
                disabled={loginMutation.isPending}
              >
                {loginMutation.isPending ? "Signing In..." : "Sign In"}
                <ArrowRight className="ml-2 h-4 w-4" />
              </Button>
            </form>

            {/* Demo credentials for website admin */}
            <div className="mt-6 pt-6 border-t border-gray-200">
              <p className="text-sm text-gray-600 mb-3 text-center">Demo Account:</p>
              <div className="bg-gray-50 p-3 rounded-lg">
                <div className="flex items-center justify-between text-sm">
                  <span className="font-medium text-gray-700">Website Admin:</span>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => {
                      setUsername("webadmin");
                      setPassword("webadmin123");
                    }}
                    className="text-xs px-2 py-1 h-auto"
                  >
                    Use Demo
                  </Button>
                </div>
                <div className="text-xs text-gray-500 mt-1">
                  Username: webadmin | Password: webadmin123
                </div>
              </div>
            </div>

            {/* Navigation back to main site */}
            <div className="mt-6 pt-4 border-t border-gray-200 text-center">
              <Button
                variant="ghost"
                onClick={() => setLocation("/")}
                className="text-gray-600 hover:text-[#4A90E2]"
              >
                <Home className="mr-2 h-4 w-4" />
                Back to Main Site
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}