import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import { 
  CheckCircle, 
  XCircle, 
  Clock, 
  FileText, 
  User, 
  Building2, 
  Mail, 
  Phone, 
  MapPin, 
  Calendar,
  AlertCircle,
  Shield,
  Eye,
  Download,
  Users,
  Factory,
  Award,
  TrendingUp
} from "lucide-react";
import { apiRequest } from "@/lib/queryClient";

interface AccreditationApplication {
  id: string;
  userId: string;
  applicationType: string;
  organizationName: string;
  contactPerson: string;
  contactEmail: string;
  contactPhone: string;
  address: string;
  city: string;
  country: string;
  businessLicense: string;
  taxRegistration: string;
  organizationType: string;
  establishedYear: number;
  numberOfEmployees: number;
  facilitiesDescription: string;
  technicalInfrastructure: any;
  qualifiedPersonnel: any;
  certificationPrograms: any;
  operatingHours: any;
  capacityInformation: any;
  qualityAssurance: any;
  supportingDocuments: any;
  facilityPhotos: any;
  agreementToTerms: boolean;
  declarationOfAccuracy: boolean;
  status: string;
  approvalLevel: number;
  approvedBy?: any[];
  reviewerId?: string;
  reviewComments?: string;
  rejectionReason?: string;
  submittedAt: string;
  updatedAt: string;
}

function AccreditationManagement() {
  const [selectedApplication, setSelectedApplication] = useState<AccreditationApplication | null>(null);
  const [reviewModalOpen, setReviewModalOpen] = useState(false);
  const [detailsModalOpen, setDetailsModalOpen] = useState(false);
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [typeFilter, setTypeFilter] = useState<string>("all");
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // FR-AM-003: Query applications with filters
  const { data: applications, isLoading } = useQuery({
    queryKey: ["backend/api/accreditation/applications", statusFilter, typeFilter],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (statusFilter !== "all") params.append("status", statusFilter);
      if (typeFilter !== "all") params.append("type", typeFilter);
      
      const response = await apiRequest(`backend/api/accreditation/applications?${params.toString()}`, "GET");
      return response;
    },
    retry: false,
  });

  // FR-AM-005: Approve application mutation
  const approveApplicationMutation = useMutation({
    mutationFn: async (data: { id: string; level: number; comments?: string }) => {
      return apiRequest("backend/api/accreditation/approve", "POST", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["backend/api/accreditation/applications"] });
      setReviewModalOpen(false);
      setSelectedApplication(null);
      toast({
        title: "Success",
        description: "Application approved successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to approve application",
        variant: "destructive",
      });
    },
  });

  // FR-AM-006: Reject application mutation
  const rejectApplicationMutation = useMutation({
    mutationFn: async (data: { id: string; reason: string }) => {
      return apiRequest("backend/api/accreditation/reject", "POST", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["backend/api/accreditation/applications"] });
      setReviewModalOpen(false);
      setSelectedApplication(null);
      toast({
        title: "Success",
        description: "Application rejected successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to reject application",
        variant: "destructive",
      });
    },
  });

  const getStatusBadge = (status: string) => {
    const statusConfig = {
      'pending_approval_1': { icon: Clock, color: 'bg-yellow-100 text-yellow-800', label: 'Pending L1 Review' },
      'pending_approval_2': { icon: AlertCircle, color: 'bg-blue-100 text-blue-800', label: 'Pending L2 Review' },
      'pending_approval_3': { icon: Shield, color: 'bg-purple-100 text-purple-800', label: 'Pending L3 Review' },
      'approved': { icon: CheckCircle, color: 'bg-green-100 text-green-800', label: 'Approved' },
      'rejected': { icon: XCircle, color: 'bg-red-100 text-red-800', label: 'Rejected' },
    };
    
    const config = statusConfig[status as keyof typeof statusConfig] || statusConfig['pending_approval_1'];
    const Icon = config.icon;
    
    return (
      <Badge className={`${config.color} flex items-center gap-1`}>
        <Icon className="w-3 h-3" />
        {config.label}
      </Badge>
    );
  };

  const getTypeIcon = (type: string) => {
    const typeConfig = {
      'training_center': { icon: Building2, color: 'text-blue-600' },
      'testing_center': { icon: FileText, color: 'text-green-600' },
      'both': { icon: Award, color: 'text-purple-600' },
    };
    
    const config = typeConfig[type as keyof typeof typeConfig] || typeConfig['training_center'];
    const Icon = config.icon;
    
    return <Icon className={`w-5 h-5 ${config.color}`} />;
  };

  const handleApprove = (level: number, comments: string) => {
    if (!selectedApplication) return;
    
    approveApplicationMutation.mutate({
      id: selectedApplication.id,
      level,
      comments,
    });
  };

  const handleReject = (reason: string) => {
    if (!selectedApplication) return;
    
    rejectApplicationMutation.mutate({
      id: selectedApplication.id,
      reason,
    });
  };

  const canApprove = (application: AccreditationApplication) => {
    return ['pending_approval_1', 'pending_approval_2', 'pending_approval_3'].includes(application.status);
  };

  const canReject = (application: AccreditationApplication) => {
    return !['approved', 'rejected'].includes(application.status);
  };

  const getApprovalLevel = (application: AccreditationApplication) => {
    switch (application.status) {
      case 'pending_approval_1': return 1;
      case 'pending_approval_2': return 2;
      case 'pending_approval_3': return 3;
      default: return 0;
    }
  };

  if (isLoading) {
    return (
      <DashboardLayout>
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-red-600"></div>
        </div>
      </DashboardLayout>
    );
  }

  return (
    <DashboardLayout>
      <div className="space-y-6">
        {/* Header with Stats */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600">Total Applications</p>
                  <p className="text-2xl font-bold">{applications?.length || 0}</p>
                </div>
                <FileText className="w-8 h-8 text-blue-500" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600">Pending Review</p>
                  <p className="text-2xl font-bold">
                    {applications?.filter(app => app.status.includes('pending')).length || 0}
                  </p>
                </div>
                <Clock className="w-8 h-8 text-yellow-500" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600">Approved</p>
                  <p className="text-2xl font-bold">
                    {applications?.filter(app => app.status === 'approved').length || 0}
                  </p>
                </div>
                <CheckCircle className="w-8 h-8 text-green-500" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600">Rejected</p>
                  <p className="text-2xl font-bold">
                    {applications?.filter(app => app.status === 'rejected').length || 0}
                  </p>
                </div>
                <XCircle className="w-8 h-8 text-red-500" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Filters and Controls */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <div className="flex items-center gap-2">
              <Label htmlFor="status-filter">Status:</Label>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger className="w-48">
                  <SelectValue placeholder="Filter by status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="pending_approval_1">Pending L1 Review</SelectItem>
                  <SelectItem value="pending_approval_2">Pending L2 Review</SelectItem>
                  <SelectItem value="pending_approval_3">Pending L3 Review</SelectItem>
                  <SelectItem value="approved">Approved</SelectItem>
                  <SelectItem value="rejected">Rejected</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="flex items-center gap-2">
              <Label htmlFor="type-filter">Type:</Label>
              <Select value={typeFilter} onValueChange={setTypeFilter}>
                <SelectTrigger className="w-48">
                  <SelectValue placeholder="Filter by type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="training_center">Training Center</SelectItem>
                  <SelectItem value="testing_center">Testing Center</SelectItem>
                  <SelectItem value="both">Both</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </div>

        {/* Applications List */}
        <div className="space-y-4">
          {applications?.map((application) => (
            <Card key={application.id} className="hover:shadow-lg transition-shadow">
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-4">
                    {getTypeIcon(application.applicationType)}
                    <div>
                      <h3 className="font-semibold text-lg">{application.organizationName}</h3>
                      <div className="flex items-center gap-4 text-sm text-gray-600">
                        <div className="flex items-center gap-1">
                          <User className="w-4 h-4" />
                          {application.contactPerson}
                        </div>
                        <div className="flex items-center gap-1">
                          <Mail className="w-4 h-4" />
                          {application.contactEmail}
                        </div>
                        <div className="flex items-center gap-1">
                          <MapPin className="w-4 h-4" />
                          {application.city}, {application.country}
                        </div>
                      </div>
                      <div className="flex items-center gap-2 mt-2">
                        <Badge variant="outline" className="text-xs">
                          {application.applicationType.replace('_', ' ').toUpperCase()}
                        </Badge>
                        <Badge variant="outline" className="text-xs">
                          {application.organizationType.replace('_', ' ').toUpperCase()}
                        </Badge>
                      </div>
                    </div>
                  </div>
                  <div className="flex items-center gap-3">
                    {getStatusBadge(application.status)}
                    <div className="flex items-center gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => {
                          setSelectedApplication(application);
                          setDetailsModalOpen(true);
                        }}
                      >
                        <Eye className="w-4 h-4 mr-1" />
                        View Details
                      </Button>
                      
                      {canApprove(application) && (
                        <Button
                          variant="default"
                          size="sm"
                          className="bg-green-600 hover:bg-green-700"
                          onClick={() => {
                            setSelectedApplication(application);
                            setReviewModalOpen(true);
                          }}
                        >
                          <CheckCircle className="w-4 h-4 mr-1" />
                          Review
                        </Button>
                      )}
                      
                      {canReject(application) && (
                        <Button
                          variant="outline"
                          size="sm"
                          className="text-red-600 hover:bg-red-50"
                          onClick={() => {
                            setSelectedApplication(application);
                            setReviewModalOpen(true);
                          }}
                        >
                          <XCircle className="w-4 h-4 mr-1" />
                          Reject
                        </Button>
                      )}
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {applications?.length === 0 && (
          <Card>
            <CardContent className="p-12 text-center">
              <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-gray-600 mb-2">No Applications Found</h3>
              <p className="text-gray-500">There are no accreditation applications to display.</p>
            </CardContent>
          </Card>
        )}

        {/* Application Details Modal */}
        <Dialog open={detailsModalOpen} onOpenChange={setDetailsModalOpen}>
          <DialogContent className="max-w-4xl max-h-[80vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2">
                <Building2 className="w-5 h-5" />
                Application Details: {selectedApplication?.organizationName}
              </DialogTitle>
            </DialogHeader>
            
            {selectedApplication && (
              <div className="space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="space-y-4">
                    <h4 className="font-semibold text-lg">Organization Information</h4>
                    <div className="space-y-2">
                      <div className="flex items-center gap-2">
                        <Building2 className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Name:</span>
                        <span>{selectedApplication.organizationName}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <User className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Contact Person:</span>
                        <span>{selectedApplication.contactPerson}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <Mail className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Email:</span>
                        <span>{selectedApplication.contactEmail}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <Phone className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Phone:</span>
                        <span>{selectedApplication.contactPhone}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <MapPin className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Address:</span>
                        <span>{selectedApplication.address}, {selectedApplication.city}, {selectedApplication.country}</span>
                      </div>
                    </div>
                  </div>

                  <div className="space-y-4">
                    <h4 className="font-semibold text-lg">Application Details</h4>
                    <div className="space-y-2">
                      <div className="flex items-center gap-2">
                        <Award className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Type:</span>
                        <span>{selectedApplication.applicationType.replace('_', ' ').toUpperCase()}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <Factory className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Organization Type:</span>
                        <span>{selectedApplication.organizationType.replace('_', ' ').toUpperCase()}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <Calendar className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Established:</span>
                        <span>{selectedApplication.establishedYear}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <Users className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Employees:</span>
                        <span>{selectedApplication.numberOfEmployees}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <Clock className="w-4 h-4 text-gray-500" />
                        <span className="font-medium">Status:</span>
                        {getStatusBadge(selectedApplication.status)}
                      </div>
                    </div>
                  </div>
                </div>

                <div className="space-y-4">
                  <h4 className="font-semibold text-lg">Facilities Description</h4>
                  <p className="text-gray-700 bg-gray-50 p-4 rounded-lg">
                    {selectedApplication.facilitiesDescription}
                  </p>
                </div>

                <div className="space-y-4">
                  <h4 className="font-semibold text-lg">Legal Documents</h4>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div className="bg-gray-50 p-4 rounded-lg">
                      <div className="flex items-center gap-2 mb-2">
                        <FileText className="w-4 h-4 text-blue-500" />
                        <span className="font-medium">Business License</span>
                      </div>
                      <p className="text-sm text-gray-600">{selectedApplication.businessLicense}</p>
                    </div>
                    <div className="bg-gray-50 p-4 rounded-lg">
                      <div className="flex items-center gap-2 mb-2">
                        <FileText className="w-4 h-4 text-green-500" />
                        <span className="font-medium">Tax Registration</span>
                      </div>
                      <p className="text-sm text-gray-600">{selectedApplication.taxRegistration}</p>
                    </div>
                  </div>
                </div>

                {selectedApplication.approvedBy && selectedApplication.approvedBy.length > 0 && (
                  <div className="space-y-4">
                    <h4 className="font-semibold text-lg">Approval History</h4>
                    <div className="space-y-2">
                      {selectedApplication.approvedBy.map((approval: any, index: number) => (
                        <div key={index} className="bg-green-50 p-3 rounded-lg">
                          <div className="flex items-center justify-between">
                            <span className="font-medium">Level {approval.level} Approval</span>
                            <span className="text-sm text-gray-600">
                              {new Date(approval.timestamp).toLocaleDateString()}
                            </span>
                          </div>
                          <p className="text-sm text-gray-700 mt-1">{approval.comments}</p>
                        </div>
                      ))}
                    </div>
                  </div>
                )}

                {selectedApplication.rejectionReason && (
                  <div className="space-y-4">
                    <h4 className="font-semibold text-lg text-red-600">Rejection Reason</h4>
                    <div className="bg-red-50 p-4 rounded-lg">
                      <p className="text-gray-700">{selectedApplication.rejectionReason}</p>
                    </div>
                  </div>
                )}
              </div>
            )}
          </DialogContent>
        </Dialog>

        {/* Review Modal */}
        <Dialog open={reviewModalOpen} onOpenChange={setReviewModalOpen}>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2">
                <Shield className="w-5 h-5" />
                Review Application: {selectedApplication?.organizationName}
              </DialogTitle>
            </DialogHeader>
            
            {selectedApplication && (
              <ReviewForm
                application={selectedApplication}
                onApprove={handleApprove}
                onReject={handleReject}
                onCancel={() => setReviewModalOpen(false)}
                isLoading={approveApplicationMutation.isPending || rejectApplicationMutation.isPending}
              />
            )}
          </DialogContent>
        </Dialog>
      </div>
    </DashboardLayout>
  );
}

// Review Form Component
function ReviewForm({ 
  application, 
  onApprove, 
  onReject, 
  onCancel, 
  isLoading 
}: {
  application: AccreditationApplication;
  onApprove: (level: number, comments: string) => void;
  onReject: (reason: string) => void;
  onCancel: () => void;
  isLoading: boolean;
}) {
  const [action, setAction] = useState<'approve' | 'reject' | null>(null);
  const [comments, setComments] = useState('');
  
  const approvalLevel = getApprovalLevel(application);
  
  const handleSubmit = () => {
    if (action === 'approve') {
      onApprove(approvalLevel, comments);
    } else if (action === 'reject') {
      onReject(comments);
    }
  };

  const getApprovalLevel = (app: AccreditationApplication) => {
    switch (app.status) {
      case 'pending_approval_1': return 1;
      case 'pending_approval_2': return 2;
      case 'pending_approval_3': return 3;
      default: return 0;
    }
  };

  return (
    <div className="space-y-6">
      <div className="bg-blue-50 p-4 rounded-lg">
        <h4 className="font-semibold text-blue-800 mb-2">Application Summary</h4>
        <div className="grid grid-cols-2 gap-4 text-sm">
          <div>
            <span className="font-medium">Organization:</span> {application.organizationName}
          </div>
          <div>
            <span className="font-medium">Type:</span> {application.applicationType.replace('_', ' ').toUpperCase()}
          </div>
          <div>
            <span className="font-medium">Current Status:</span> {application.status.replace('_', ' ').toUpperCase()}
          </div>
          <div>
            <span className="font-medium">Review Level:</span> Level {approvalLevel}
          </div>
        </div>
      </div>

      <div className="space-y-4">
        <div className="flex items-center gap-4">
          <Button
            variant={action === 'approve' ? 'default' : 'outline'}
            onClick={() => setAction('approve')}
            className={action === 'approve' ? 'bg-green-600 hover:bg-green-700' : ''}
          >
            <CheckCircle className="w-4 h-4 mr-2" />
            Approve (Level {approvalLevel})
          </Button>
          
          <Button
            variant={action === 'reject' ? 'default' : 'outline'}
            onClick={() => setAction('reject')}
            className={action === 'reject' ? 'bg-red-600 hover:bg-red-700' : ''}
          >
            <XCircle className="w-4 h-4 mr-2" />
            Reject Application
          </Button>
        </div>

        <div className="space-y-2">
          <Label htmlFor="comments">
            {action === 'approve' ? 'Approval Comments' : 'Rejection Reason'}
          </Label>
          <Textarea
            id="comments"
            value={comments}
            onChange={(e) => setComments(e.target.value)}
            placeholder={
              action === 'approve' 
                ? 'Enter approval comments and any conditions...'
                : 'Enter detailed reason for rejection...'
            }
            required
            className="min-h-[100px]"
          />
        </div>

        <div className="flex items-center justify-end gap-3">
          <Button variant="outline" onClick={onCancel}>
            Cancel
          </Button>
          
          <Button
            onClick={handleSubmit}
            disabled={!action || !comments.trim() || isLoading}
            className={
              action === 'approve' 
                ? 'bg-green-600 hover:bg-green-700' 
                : 'bg-red-600 hover:bg-red-700'
            }
          >
            {isLoading ? (
              <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2" />
            ) : action === 'approve' ? (
              <CheckCircle className="w-4 h-4 mr-2" />
            ) : (
              <XCircle className="w-4 h-4 mr-2" />
            )}
            {action === 'approve' ? 'Approve Application' : 'Reject Application'}
          </Button>
        </div>
      </div>
    </div>
  );
}

export default AccreditationManagement;
