import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { useToast } from "@/hooks/use-toast";
import AccreditationApplicationForm from "@/components/AccreditationApplicationForm";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Label } from "@/components/ui/label";
import { apiRequest } from "@/lib/queryClient";
import { 
  ArrowLeft, 
  Building, 
  GraduationCap, 
  Shield,
  CheckCircle,
  Clock,
  FileText,
  Award,
  Users,
  Globe
} from "lucide-react";

export default function AccreditationApplication() {
  const [, setLocation] = useLocation();
  const [applicationSubmitted, setApplicationSubmitted] = useState(false);
  const [submissionData, setSubmissionData] = useState<any>(null);
  const { toast } = useToast();

  // Get application type from URL parameters
  const urlParams = new URLSearchParams(window.location.search);
  const applicationType = urlParams.get('type') || 'training_center';

  const submitApplicationMutation = useMutation({
    mutationFn: (data: any) => apiRequest("backend/api/accreditation/apply", "POST", data),
    onSuccess: (response) => {
      setSubmissionData(response);
      setApplicationSubmitted(true);
      toast({
        title: "Application Submitted Successfully",
        description: "Your accreditation application has been submitted for review.",
      });
    },
    onError: (error) => {
      toast({
        title: "Submission Failed",
        description: error.message || "Failed to submit application. Please try again.",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (data: any) => {
    submitApplicationMutation.mutate({
      ...data,
      status: "pending",
      submittedAt: new Date().toISOString(),
    });
  };

  const handleBackToHome = () => {
    setLocation("/");
  };

  if (applicationSubmitted) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 p-6">
        <div className="max-w-2xl mx-auto">
          <div className="text-center mb-8">
            <img 
              src="/logo.png" 
              alt="IBDL Logo" 
              className="h-16 mx-auto mb-4"
            />
            <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <CheckCircle className="w-8 h-8 text-green-600" />
            </div>
            <h1 className="text-3xl font-bold text-gray-900 mb-2">
              Application Submitted Successfully!
            </h1>
            <p className="text-gray-600">
              Your accreditation application has been received and is being reviewed by our team.
            </p>
          </div>

          <Card className="mb-6">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileText className="w-5 h-5" />
                Application Summary
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <Label className="text-sm font-medium text-gray-500">Application ID</Label>
                  <p className="font-mono text-sm">{submissionData?.id || "APP-" + Date.now()}</p>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Application Type</Label>
                  <p className="capitalize">{submissionData?.applicationType?.replace('_', ' ')}</p>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Organization</Label>
                  <p>{submissionData?.organizationName}</p>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Contact Person</Label>
                  <p>{submissionData?.contactPerson}</p>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Status</Label>
                  <Badge variant="secondary" className="bg-yellow-100 text-yellow-800">
                    <Clock className="w-3 h-3 mr-1" />
                    Under Review
                  </Badge>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-500">Submitted</Label>
                  <p>{new Date().toLocaleDateString()}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="mb-6">
            <CardHeader>
              <CardTitle>What Happens Next?</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-start gap-3">
                  <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span className="text-blue-600 font-semibold text-sm">1</span>
                  </div>
                  <div>
                    <h4 className="font-medium">Initial Review</h4>
                    <p className="text-sm text-gray-600">
                      Our team will review your application and documents within 5-7 business days.
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span className="text-blue-600 font-semibold text-sm">2</span>
                  </div>
                  <div>
                    <h4 className="font-medium">Document Verification</h4>
                    <p className="text-sm text-gray-600">
                      We'll verify all submitted documents and may request additional information.
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span className="text-blue-600 font-semibold text-sm">3</span>
                  </div>
                  <div>
                    <h4 className="font-medium">Site Visit (if applicable)</h4>
                    <p className="text-sm text-gray-600">
                      A site visit may be scheduled to verify your facilities and infrastructure.
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <div className="w-8 h-8 bg-green-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span className="text-green-600 font-semibold text-sm">4</span>
                  </div>
                  <div>
                    <h4 className="font-medium">Final Decision</h4>
                    <p className="text-sm text-gray-600">
                      You'll receive a final decision and, if approved, your accreditation certificate.
                    </p>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          <div className="text-center">
            <Button onClick={handleBackToHome} className="mr-4">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Home
            </Button>
            <Button variant="outline" onClick={() => window.print()}>
              <FileText className="w-4 h-4 mr-2" />
              Print Summary
            </Button>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100">
      <div className="container mx-auto px-4 py-8">
        <div className="mb-8">
          <div className="flex items-center gap-4 mb-6">
            <Button variant="ghost" onClick={handleBackToHome}>
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Home
            </Button>
            <img 
              src="/logo.png" 
              alt="IBDL Logo" 
              className="h-12"
            />
          </div>
          
          <div className="text-center mb-8">
            <h1 className="text-4xl font-bold text-gray-900 mb-2">
              IBDL Accreditation Application
            </h1>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Join our global network of accredited training and testing centers. 
              Enhance your credibility and expand your reach with IBDL certification.
            </p>
          </div>

          <div className="grid md:grid-cols-3 gap-6 mb-8">
            <Card>
              <CardHeader className="text-center">
                <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-2">
                  <GraduationCap className="w-6 h-6 text-blue-600" />
                </div>
                <CardTitle>Training Center</CardTitle>
                <CardDescription>
                  Deliver IBDL-certified training programs and workshops
                </CardDescription>
              </CardHeader>
            </Card>

            <Card>
              <CardHeader className="text-center">
                <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-2">
                  <Shield className="w-6 h-6 text-green-600" />
                </div>
                <CardTitle>Testing Center</CardTitle>
                <CardDescription>
                  Conduct secure IBDL certification examinations
                </CardDescription>
              </CardHeader>
            </Card>

            <Card>
              <CardHeader className="text-center">
                <div className="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-2">
                  <Building className="w-6 h-6 text-purple-600" />
                </div>
                <CardTitle>Both Services</CardTitle>
                <CardDescription>
                  Provide complete training and testing solutions
                </CardDescription>
              </CardHeader>
            </Card>
          </div>

          <Card className="mb-8">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Award className="w-5 h-5" />
                Accreditation Benefits
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <h4 className="font-semibold mb-2">Training Centers</h4>
                  <ul className="space-y-1 text-sm text-gray-600">
                    <li>• Access to IBDL curriculum and training materials</li>
                    <li>• Certified trainer qualification programs</li>
                    <li>• Marketing and promotional support</li>
                    <li>• Quality assurance and continuous improvement</li>
                  </ul>
                </div>
                <div>
                  <h4 className="font-semibold mb-2">Testing Centers</h4>
                  <ul className="space-y-1 text-sm text-gray-600">
                    <li>• Secure examination delivery platform</li>
                    <li>• Proctoring and monitoring tools</li>
                    <li>• Certificate issuance and verification</li>
                    <li>• Revenue sharing opportunities</li>
                  </ul>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="mb-8">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Users className="w-5 h-5" />
                Global Network
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid md:grid-cols-4 gap-4 text-center">
                <div>
                  <div className="text-2xl font-bold text-blue-600">735</div>
                  <div className="text-sm text-gray-600">Exam Providers</div>
                </div>
                <div>
                  <div className="text-2xl font-bold text-green-600">39</div>
                  <div className="text-sm text-gray-600">Countries</div>
                </div>
                <div>
                  <div className="text-2xl font-bold text-purple-600">222K</div>
                  <div className="text-sm text-gray-600">Exams Delivered</div>
                </div>
                <div>
                  <div className="text-2xl font-bold text-red-600">120</div>
                  <div className="text-sm text-gray-600">Strategic Partners</div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        <AccreditationApplicationForm
          onSubmit={handleSubmit}
          isLoading={submitApplicationMutation.isPending}
          onCancel={handleBackToHome}
          initialApplicationType={applicationType}
        />
      </div>
    </div>
  );
}