import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ProgressBar } from "@/components/ui/progress-bar";
import { BookOpen, Clock, Users, Star, Play } from "lucide-react";

interface CourseCardProps {
  id: string;
  title: string;
  description?: string;
  progress?: number;
  duration?: number;
  studentsEnrolled?: number;
  rating?: number;
  category?: string;
  difficulty?: 'beginner' | 'intermediate' | 'advanced';
  imageUrl?: string;
  status?: 'not_started' | 'in_progress' | 'completed';
  onEnroll?: () => void;
  onContinue?: () => void;
  onView?: () => void;
  className?: string;
}

export function CourseCard({
  id,
  title,
  description,
  progress = 0,
  duration,
  studentsEnrolled,
  rating,
  category,
  difficulty = 'beginner',
  imageUrl,
  status = 'not_started',
  onEnroll,
  onContinue,
  onView,
  className = ''
}: CourseCardProps) {
  const getDifficultyColor = () => {
    switch (difficulty) {
      case 'beginner': return 'bg-green-100 text-green-800';
      case 'intermediate': return 'bg-yellow-100 text-yellow-800';
      case 'advanced': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusAction = () => {
    switch (status) {
      case 'in_progress':
        return (
          <Button onClick={onContinue} className="w-full">
            <Play className="w-4 h-4 mr-2" />
            Continue Learning
          </Button>
        );
      case 'completed':
        return (
          <Button onClick={onView} variant="outline" className="w-full">
            View Certificate
          </Button>
        );
      default:
        return (
          <Button onClick={onEnroll} className="w-full">
            Enroll Now
          </Button>
        );
    }
  };

  return (
    <Card className={`course-card ${className}`}>
      {imageUrl ? (
        <img src={imageUrl} alt={title} className="course-image" />
      ) : (
        <div className="w-full h-32 bg-gradient-to-br from-primary to-purple-600 flex items-center justify-center">
          <BookOpen className="w-8 h-8 text-white" />
        </div>
      )}
      
      <CardContent className="course-content">
        <div className="flex items-start justify-between mb-2">
          <div className="flex-1">
            <h3 className="course-title">{title}</h3>
            {description && (
              <p className="text-xs text-muted-foreground mt-1 line-clamp-2">{description}</p>
            )}
          </div>
          {category && (
            <Badge variant="secondary" className="ml-2 text-xs">
              {category}
            </Badge>
          )}
        </div>

        <div className="flex items-center gap-4 mb-3 text-xs text-muted-foreground">
          {duration && (
            <div className="flex items-center">
              <Clock className="w-3 h-3 mr-1" />
              {duration}h
            </div>
          )}
          {studentsEnrolled && (
            <div className="flex items-center">
              <Users className="w-3 h-3 mr-1" />
              {studentsEnrolled}
            </div>
          )}
          {rating && (
            <div className="flex items-center">
              <Star className="w-3 h-3 mr-1 fill-current text-yellow-400" />
              {rating.toFixed(1)}
            </div>
          )}
        </div>

        <div className="flex items-center justify-between mb-4">
          <Badge variant="outline" className={`text-xs ${getDifficultyColor()}`}>
            {difficulty.charAt(0).toUpperCase() + difficulty.slice(1)}
          </Badge>
          {status === 'completed' && (
            <Badge className="bg-green-100 text-green-800 text-xs">
              Completed
            </Badge>
          )}
        </div>

        {status === 'in_progress' && (
          <div className="mb-4">
            <ProgressBar 
              value={progress} 
              label="Progress" 
              color="primary"
            />
          </div>
        )}

        {getStatusAction()}
      </CardContent>
    </Card>
  );
}
