import { useAuth } from "@/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Badge } from "@/components/ui/badge";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { 
  GraduationCap, 
  Home, 
  BookOpen, 
  FileCheck, 
  Award, 
  BarChart3,
  Users,
  Building,
  Settings,
  Bell,
  Search,
  LogOut,
  ChevronLeft,
  Shield,
  FileText
} from "lucide-react";
import { Link, useLocation } from "wouter";
import ibdlLogo from "@assets/logo_1751230646761.png";

interface DashboardLayoutProps {
  children: React.ReactNode;
  userType: 'learner' | 'corporate' | 'partner' | 'admin' | 'qa';
}

export default function DashboardLayout({ children, userType }: DashboardLayoutProps) {
  const { user } = useAuth() as { user: any };
  const [location] = useLocation();

  const getUserTypeInfo = () => {
    switch (userType) {
      case 'learner':
        return {
          title: 'Learning Dashboard',
          subtitle: 'Individual Learner Portal',
          bgColor: 'bg-primary',
          navigation: [
            { name: 'Dashboard', href: '/dashboard/learner', icon: Home },
            { name: 'Course Catalog', href: '/courses', icon: BookOpen },
            { name: 'Assessments', href: '/assessments', icon: FileCheck },
            { name: 'Examinations', href: '/examinations', icon: Shield },
            { name: 'Certifications', href: '/certificates', icon: Award },
            { name: 'Analytics', href: '/analytics', icon: BarChart3 },
          ]
        };
      case 'corporate':
        return {
          title: 'Enterprise Portal',
          subtitle: user?.organizationId || 'Corporate Dashboard',
          bgColor: 'bg-emerald-600',
          navigation: [
            { name: 'Overview', href: '/dashboard/corporate', icon: Home },
            { name: 'Account Management', href: '/corporate-accounts', icon: Users },
            { name: 'Learning Programs', href: '/programs', icon: BookOpen },
            { name: 'Analytics', href: '/analytics', icon: BarChart3 },
            { name: 'Compliance', href: '/compliance', icon: FileCheck },
          ]
        };
      case 'partner':
        return {
          title: 'Partner Portal',
          subtitle: user?.organizationId || 'Training Center',
          bgColor: 'bg-purple-600',
          navigation: [
            { name: 'Business Dashboard', href: '/dashboard/partner', icon: Home },
            { name: 'Question Bank', href: '/question-bank', icon: BookOpen },
            { name: 'Student Management', href: '/students', icon: Users },
            { name: 'Instructor Portal', href: '/instructors', icon: Users },
            { name: 'Revenue Analytics', href: '/revenue', icon: BarChart3 },
            { name: 'Quality Assurance', href: '/quality', icon: Award },
          ]
        };
      case 'admin':
        return {
          title: 'System Administration',
          subtitle: 'Platform Operations & Management',
          bgColor: 'bg-slate-800',
          navigation: [
            { name: 'Overview', href: '/dashboard/admin', icon: Home },
            { name: 'System Setup', href: '/system-setup', icon: Settings },
            { name: 'Examinations', href: '/examinations', icon: Shield },
            { name: 'Assessments', href: '/assessments', icon: FileCheck },
            { name: 'eLearning', href: '/elearning', icon: BookOpen },
            { name: 'eBooks', href: '/ebooks', icon: BookOpen },
            { name: 'Settings', href: '/admin/settings', icon: Settings },
          ]
        };
      case 'qa':
        return {
          title: 'Quality Assurance',
          subtitle: 'Accreditation & Compliance Management',
          bgColor: 'bg-blue-600',
          navigation: [
            { name: 'Accreditation Process', href: '/qa', icon: Award },
            { name: 'Quality Reports', href: '/qa/reports', icon: FileText },
            { name: 'Compliance Audit', href: '/qa/audit', icon: Shield },
            { name: 'QA Settings', href: '/qa/settings', icon: Settings },
          ]
        };
      default:
        return {
          title: 'Dashboard',
          subtitle: 'Platform',
          bgColor: 'bg-primary',
          navigation: []
        };
    }
  };

  const typeInfo = getUserTypeInfo();

  const handleLogout = async () => {
    try {
      await fetch('backend/api/logout', { method: 'POST' });
      window.location.href = "/";
    } catch (error) {
      console.error('Logout error:', error);
      window.location.href = "/";
    }
  };

  const getUserInitials = () => {
    if (user?.firstName && user?.lastName) {
      return `${user.firstName[0]}${user.lastName[0]}`;
    }
    if (user?.email) {
      return user.email.substring(0, 2).toUpperCase();
    }
    return 'U';
  };

  const getUserDisplayName = () => {
    if (user?.firstName && user?.lastName) {
      return `${user.firstName} ${user.lastName}`;
    }
    if (user?.email) {
      return user.email;
    }
    return 'User';
  };

  const getUserRole = () => {
    switch (userType) {
      case 'learner': return 'Learner';
      case 'corporate': return 'Corporate Admin';
      case 'partner': return 'Partner Admin';
      case 'admin': return 'System Admin';
      case 'qa': return 'QA Administrator';
      default: return 'User';
    }
  };

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Dashboard Header */}
      <header className="dashboard-header">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center space-x-4">
              <Link href="/">
                <Button variant="ghost" size="sm" className="text-slate-400 hover:text-slate-600">
                  <ChevronLeft className="w-5 h-5" />
                </Button>
              </Link>
              <div className="flex items-center space-x-3">
                <img src={ibdlLogo} alt="IBDL Logo" className="h-8 w-auto" />
                <div>
                  <h1 className="text-lg font-semibold text-slate-900">{typeInfo.title}</h1>
                  <p className="text-xs text-slate-500">{typeInfo.subtitle}</p>
                </div>
              </div>
            </div>
            <div className="flex items-center space-x-4">
              <Button variant="ghost" size="sm">
                <Search className="w-5 h-5" />
              </Button>
              <Button variant="ghost" size="sm" className="relative">
                <Bell className="w-5 h-5" />
                <span className="absolute -top-1 -right-1 w-3 h-3 bg-red-500 rounded-full"></span>
              </Button>
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="ghost" className="flex items-center space-x-3">
                    <div className="text-right">
                      <p className="text-sm font-medium text-slate-900">{getUserDisplayName()}</p>
                      <p className="text-xs text-slate-500">{getUserRole()}</p>
                    </div>
                    <Avatar className="w-8 h-8">
                      <AvatarImage src={user?.profileImageUrl || ''} />
                      <AvatarFallback className="text-xs">{getUserInitials()}</AvatarFallback>
                    </Avatar>
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end" className="w-56">
                  <DropdownMenuLabel>My Account</DropdownMenuLabel>
                  <DropdownMenuSeparator />
                  <DropdownMenuItem>
                    <Users className="w-4 h-4 mr-2" />
                    Profile
                  </DropdownMenuItem>
                  <DropdownMenuItem>
                    <Settings className="w-4 h-4 mr-2" />
                    Settings
                  </DropdownMenuItem>
                  <DropdownMenuSeparator />
                  <DropdownMenuItem onClick={handleLogout}>
                    <LogOut className="w-4 h-4 mr-2" />
                    Sign out
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            </div>
          </div>
        </div>
      </header>

      {/* Dashboard Navigation */}
      <nav className="bg-white border-b border-slate-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex space-x-8">
            {typeInfo.navigation.map((item) => {
              const isActive = location === item.href;
              return (
                <Link key={item.name} href={item.href}>
                  <button className={`dashboard-nav-item py-4 px-1 text-sm font-medium ${
                    isActive ? 'active' : ''
                  }`}>
                    <item.icon className="w-4 h-4 mr-2 inline" />
                    {item.name}
                  </button>
                </Link>
              );
            })}
          </div>
        </div>
      </nav>

      {/* Dashboard Content */}
      <main className="dashboard-content">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          {children}
        </div>
      </main>
    </div>
  );
}
